"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codeconnections = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codeconnections](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodeconnections.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codeconnections extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a Connection resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifProviderType()
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_CreateConnection.html
     */
    toCreateConnection() {
        return this.to('CreateConnection');
    }
    /**
     * Grants permission to create a host resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifProviderType()
     * - .ifVpcId()
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_CreateHost.html
     */
    toCreateHost() {
        return this.to('CreateHost');
    }
    /**
     * Grants permission to create a repository link
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - codeconnections:PassConnection
     * - codeconnections:UseConnection
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_CreateRepositoryLink.html
     */
    toCreateRepositoryLink() {
        return this.to('CreateRepositoryLink');
    }
    /**
     * Grants permission to create a template sync config
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBranch()
     *
     * Dependent actions:
     * - codeconnections:PassRepository
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_CreateSyncConfiguration.html
     */
    toCreateSyncConfiguration() {
        return this.to('CreateSyncConfiguration');
    }
    /**
     * Grants permission to delete a Connection resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection() {
        return this.to('DeleteConnection');
    }
    /**
     * Grants permission to delete a host resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_DeleteHost.html
     */
    toDeleteHost() {
        return this.to('DeleteHost');
    }
    /**
     * Grants permission to delete a repository link
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_DeleteRepositoryLink.html
     */
    toDeleteRepositoryLink() {
        return this.to('DeleteRepositoryLink');
    }
    /**
     * Grants permission to delete a sync configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_DeleteSyncConfiguration.html
     */
    toDeleteSyncConfiguration() {
        return this.to('DeleteSyncConfiguration');
    }
    /**
     * Grants permission to get details about a Connection resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_GetConnection.html
     */
    toGetConnection() {
        return this.to('GetConnection');
    }
    /**
     * Grants permission to get a Connection token to call provider actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-getconnectiontoken
     */
    toGetConnectionToken() {
        return this.to('GetConnectionToken');
    }
    /**
     * Grants permission to get details about a host resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_GetHost.html
     */
    toGetHost() {
        return this.to('GetHost');
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     *
     * Dependent actions:
     * - codeconnections:StartOAuthHandshake
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     */
    toGetIndividualAccessToken() {
        return this.to('GetIndividualAccessToken');
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     */
    toGetInstallationUrl() {
        return this.to('GetInstallationUrl');
    }
    /**
     * Grants permission to describe a repository link
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_GetRepositoryLink.html
     */
    toGetRepositoryLink() {
        return this.to('GetRepositoryLink');
    }
    /**
     * Grants permission to get the latest sync status for a repository
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifBranch()
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_GetRepositorySyncStatus.html
     */
    toGetRepositorySyncStatus() {
        return this.to('GetRepositorySyncStatus');
    }
    /**
     * Grants permission to get the latest sync status for a resource (cfn stack or other resources)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_GetResourceSyncStatus.html
     */
    toGetResourceSyncStatus() {
        return this.to('GetResourceSyncStatus');
    }
    /**
     * Grants permission to describe service sync blockers on a resource (cfn stack or other resources)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_GetSyncBlockerSummary.html
     */
    toGetSyncBlockerSummary() {
        return this.to('GetSyncBlockerSummary');
    }
    /**
     * Grants permission to describe a sync configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_GetSyncConfiguration.html
     */
    toGetSyncConfiguration() {
        return this.to('GetSyncConfiguration');
    }
    /**
     * Grants permission to list Connection resources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifProviderTypeFilter()
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_ListConnections.html
     */
    toListConnections() {
        return this.to('ListConnections');
    }
    /**
     * Grants permission to list host resources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifProviderTypeFilter()
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_ListHosts.html
     */
    toListHosts() {
        return this.to('ListHosts');
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: List
     *
     * Dependent actions:
     * - codeconnections:GetIndividualAccessToken
     * - codeconnections:StartOAuthHandshake
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     */
    toListInstallationTargets() {
        return this.to('ListInstallationTargets');
    }
    /**
     * Grants permission to list repository links
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_ListRepositoryLinks.html
     */
    toListRepositoryLinks() {
        return this.to('ListRepositoryLinks');
    }
    /**
     * Grants permission to list repository sync definitions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_ListRepositorySyncDefinitions.html
     */
    toListRepositorySyncDefinitions() {
        return this.to('ListRepositorySyncDefinitions');
    }
    /**
     * Grants permission to list sync configurations for a repository link
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_ListSyncConfigurations.html
     */
    toListSyncConfigurations() {
        return this.to('ListSyncConfigurations');
    }
    /**
     * Grants permission to the set of key-value pairs that are used to manage the resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to pass a Connection resource to an AWS service that accepts a Connection ARN as input, such as codepipeline:CreatePipeline
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPassedToService()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-passconnection
     */
    toPassConnection() {
        return this.to('PassConnection');
    }
    /**
     * Grants permission to pass a repository link resource to an AWS service that accepts a RepositoryLinkId as input, such as codeconnections:CreateSyncConfiguration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPassedToService()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-passrepository
     */
    toPassRepository() {
        return this.to('PassRepository');
    }
    /**
     * Grants permission to associate a third party server, such as a GitHub Enterprise Server instance, with a Host
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifHostArn()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#connections-permissions-actions-host-registration
     */
    toRegisterAppCode() {
        return this.to('RegisterAppCode');
    }
    /**
     * Grants permission to associate a third party server, such as a GitHub Enterprise Server instance, with a Host
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifHostArn()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#connections-permissions-actions-host-registration
     */
    toStartAppRegistrationHandshake() {
        return this.to('StartAppRegistrationHandshake');
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     */
    toStartOAuthHandshake() {
        return this.to('StartOAuthHandshake');
    }
    /**
     * Grants permission to add or modify the tags of the given resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from an AWS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a Connection resource with an installation of the CodeStar Connections App
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstallationId()
     *
     * Dependent actions:
     * - codeconnections:GetIndividualAccessToken
     * - codeconnections:GetInstallationUrl
     * - codeconnections:ListInstallationTargets
     * - codeconnections:StartOAuthHandshake
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     */
    toUpdateConnectionInstallation() {
        return this.to('UpdateConnectionInstallation');
    }
    /**
     * Grants permission to update a host resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVpcId()
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_UpdateHost.html
     */
    toUpdateHost() {
        return this.to('UpdateHost');
    }
    /**
     * Grants permission to update a repository link
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_UpdateRepositoryLink.html
     */
    toUpdateRepositoryLink() {
        return this.to('UpdateRepositoryLink');
    }
    /**
     * Grants permission to update a sync blocker for a resource (cfn stack or other resources)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_UpdateSyncBlocker.html
     */
    toUpdateSyncBlocker() {
        return this.to('UpdateSyncBlocker');
    }
    /**
     * Grants permission to update a sync configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBranch()
     *
     * https://docs.aws.amazon.com/codeconnections/latest/APIReference/API_UpdateSyncConfiguration.html
     */
    toUpdateSyncConfiguration() {
        return this.to('UpdateSyncConfiguration');
    }
    /**
     * Grants permission to use a Connection resource to call provider actions
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifBranchName()
     * - .ifFullRepositoryId()
     * - .ifOwnerId()
     * - .ifProviderAction()
     * - .ifProviderPermissionsRequired()
     * - .ifRepositoryName()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     */
    toUseConnection() {
        return this.to('UseConnection');
    }
    /**
     * Adds a resource of type Connection to the statement
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/connections.html
     *
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnection(connectionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codeconnections:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connection/${connectionId}`);
    }
    /**
     * Adds a resource of type Host to the statement
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/connections-hosts.html
     *
     * @param hostId - Identifier for the hostId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHost(hostId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codeconnections:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:host/${hostId}`);
    }
    /**
     * Adds a resource of type RepositoryLink to the statement
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/repositorylinks.html
     *
     * @param repositoryLinkId - Identifier for the repositoryLinkId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRepositoryLink(repositoryLinkId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codeconnections:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:repository-link/${repositoryLinkId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateConnection()
     * - .toCreateHost()
     * - .toCreateRepositoryLink()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - Connection
     * - Host
     * - RepositoryLink
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateConnection()
     * - .toCreateHost()
     * - .toCreateRepositoryLink()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the branch name that is passed in the request
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     *
     * Applies to actions:
     * - .toCreateSyncConfiguration()
     * - .toGetRepositorySyncStatus()
     * - .toUpdateSyncConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBranch(value, operator) {
        return this.if(`Branch`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the branch name that is passed in the request. Applies only to UseConnection requests for access to a specific repository branch
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBranchName(value, operator) {
        return this.if(`BranchName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the repository that is passed in the request. Applies only to UseConnection requests for access to a specific repository
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFullRepositoryId(value, operator) {
        return this.if(`FullRepositoryId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the host resource associated with the connection used in the request
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-hosts
     *
     * Applies to actions:
     * - .toRegisterAppCode()
     * - .toStartAppRegistrationHandshake()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifHostArn(value, operator) {
        return this.if(`HostArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the third-party ID (such as the Bitbucket App installation ID for CodeConnections) that is used to update a Connection. Allows you to restrict which third-party App installations can be used to make a Connection
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     *
     * Applies to actions:
     * - .toUpdateConnectionInstallation()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstallationId(value, operator) {
        return this.if(`InstallationId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the owner of the third-party repository. Applies only to UseConnection requests for access to repositories owned by a specific user
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOwnerId(value, operator) {
        return this.if(`OwnerId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the service to which the principal is allowed to pass a Connection or RepositoryLink
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-passconnection
     *
     * Applies to actions:
     * - .toPassConnection()
     * - .toPassRepository()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPassedToService(value, operator) {
        return this.if(`PassedToService`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the provider action in a UseConnection request such as ListRepositories. See documentation for all valid values
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-access
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderAction(value, operator) {
        return this.if(`ProviderAction`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the write permissions of a provider action in a UseConnection request. Valid types include read_only and read_write
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderPermissionsRequired(value, operator) {
        return this.if(`ProviderPermissionsRequired`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the type of third-party provider passed in the request
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-managing
     *
     * Applies to actions:
     * - .toCreateConnection()
     * - .toCreateHost()
     * - .toGetIndividualAccessToken()
     * - .toGetInstallationUrl()
     * - .toStartOAuthHandshake()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderType(value, operator) {
        return this.if(`ProviderType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the type of third-party provider used to filter results
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-managing
     *
     * Applies to actions:
     * - .toListConnections()
     * - .toListHosts()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderTypeFilter(value, operator) {
        return this.if(`ProviderTypeFilter`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the repository name that is passed in the request. Applies only to UseConnection requests for access to repositories owned by a specific user
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRepositoryName(value, operator) {
        return this.if(`RepositoryName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the VpcId passed in the request
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-hosts
     *
     * Applies to actions:
     * - .toCreateHost()
     * - .toUpdateHost()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcId(value, operator) {
        return this.if(`VpcId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [codeconnections](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodeconnections.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codeconnections';
        this.accessLevelList = {
            Write: [
                'CreateConnection',
                'CreateHost',
                'CreateRepositoryLink',
                'CreateSyncConfiguration',
                'DeleteConnection',
                'DeleteHost',
                'DeleteRepositoryLink',
                'DeleteSyncConfiguration',
                'UpdateConnectionInstallation',
                'UpdateHost',
                'UpdateRepositoryLink',
                'UpdateSyncBlocker',
                'UpdateSyncConfiguration'
            ],
            Read: [
                'GetConnection',
                'GetConnectionToken',
                'GetHost',
                'GetIndividualAccessToken',
                'GetInstallationUrl',
                'GetRepositoryLink',
                'GetRepositorySyncStatus',
                'GetResourceSyncStatus',
                'GetSyncBlockerSummary',
                'GetSyncConfiguration',
                'PassConnection',
                'PassRepository',
                'RegisterAppCode',
                'StartAppRegistrationHandshake',
                'StartOAuthHandshake',
                'UseConnection'
            ],
            List: [
                'ListConnections',
                'ListHosts',
                'ListInstallationTargets',
                'ListRepositoryLinks',
                'ListRepositorySyncDefinitions',
                'ListSyncConfigurations',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Codeconnections = Codeconnections;
//# sourceMappingURL=data:application/json;base64,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