"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codedeploy = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codedeploy](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodedeploy.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codedeploy extends shared_1.PolicyStatement {
    /**
     * Grants permission to add tags to one or more on-premises instances
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_AddTagsToOnPremisesInstances.html
     */
    toAddTagsToOnPremisesInstances() {
        return this.to('AddTagsToOnPremisesInstances');
    }
    /**
     * Grants permission to get information about one or more application revisions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetApplicationRevisions.html
     */
    toBatchGetApplicationRevisions() {
        return this.to('BatchGetApplicationRevisions');
    }
    /**
     * Grants permission to get information about multiple applications associated with the IAM user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetApplications.html
     */
    toBatchGetApplications() {
        return this.to('BatchGetApplications');
    }
    /**
     * Grants permission to get information about one or more deployment groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentGroups.html
     */
    toBatchGetDeploymentGroups() {
        return this.to('BatchGetDeploymentGroups');
    }
    /**
     * Grants permission to get information about one or more instance that are part of a deployment group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentInstances.html
     */
    toBatchGetDeploymentInstances() {
        return this.to('BatchGetDeploymentInstances');
    }
    /**
     * Grants permission to return an array of one or more targets associated with a deployment. This method works with all compute types and should be used instead of the deprecated BatchGetDeploymentInstances. The maximum number of targets that can be returned is 25
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentTargets.html
     */
    toBatchGetDeploymentTargets() {
        return this.to('BatchGetDeploymentTargets');
    }
    /**
     * Grants permission to get information about multiple deployments associated with the IAM user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeployments.html
     */
    toBatchGetDeployments() {
        return this.to('BatchGetDeployments');
    }
    /**
     * Grants permission to get information about one or more on-premises instances
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetOnPremisesInstances.html
     */
    toBatchGetOnPremisesInstances() {
        return this.to('BatchGetOnPremisesInstances');
    }
    /**
     * Grants permission to start the process of rerouting traffic from instances in the original environment to instances in thereplacement environment without waiting for a specified wait time to elapse
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ContinueDeployment.html
     */
    toContinueDeployment() {
        return this.to('ContinueDeployment');
    }
    /**
     * Grants permission to create an application associated with the IAM user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create CloudFormation deployment to cooperate ochestration for a CloudFormation stack update
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/codedeploy/latest/userguide/deployments-create-ecs-cfn.html
     */
    toCreateCloudFormationDeployment() {
        return this.to('CreateCloudFormationDeployment');
    }
    /**
     * Grants permission to create a deployment for an application associated with the IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeployment.html
     */
    toCreateDeployment() {
        return this.to('CreateDeployment');
    }
    /**
     * Grants permission to create a custom deployment configuration associated with the IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeploymentConfig.html
     */
    toCreateDeploymentConfig() {
        return this.to('CreateDeploymentConfig');
    }
    /**
     * Grants permission to create a deployment group for an application associated with the IAM user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeploymentGroup.html
     */
    toCreateDeploymentGroup() {
        return this.to('CreateDeploymentGroup');
    }
    /**
     * Grants permission to delete an application associated with the IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete a custom deployment configuration associated with the IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteDeploymentConfig.html
     */
    toDeleteDeploymentConfig() {
        return this.to('DeleteDeploymentConfig');
    }
    /**
     * Grants permission to delete a deployment group for an application associated with the IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteDeploymentGroup.html
     */
    toDeleteDeploymentGroup() {
        return this.to('DeleteDeploymentGroup');
    }
    /**
     * Grants permission to delete a GitHub account connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteGitHubAccountToken.html
     */
    toDeleteGitHubAccountToken() {
        return this.to('DeleteGitHubAccountToken');
    }
    /**
     * Grants permission to delete resources associated with the given external Id
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteResourcesByExternalId.html
     */
    toDeleteResourcesByExternalId() {
        return this.to('DeleteResourcesByExternalId');
    }
    /**
     * Grants permission to deregister an on-premises instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeregisterOnPremisesInstance.html
     */
    toDeregisterOnPremisesInstance() {
        return this.to('DeregisterOnPremisesInstance');
    }
    /**
     * Grants permission to get information about a single application associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetApplication.html
     */
    toGetApplication() {
        return this.to('GetApplication');
    }
    /**
     * Grants permission to get information about a single application revision for an application associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetApplicationRevision.html
     */
    toGetApplicationRevision() {
        return this.to('GetApplicationRevision');
    }
    /**
     * Grants permission to get information about a single deployment to a deployment group for an application associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeployment.html
     */
    toGetDeployment() {
        return this.to('GetDeployment');
    }
    /**
     * Grants permission to get information about a single deployment configuration associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentConfig.html
     */
    toGetDeploymentConfig() {
        return this.to('GetDeploymentConfig');
    }
    /**
     * Grants permission to get information about a single deployment group for an application associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentGroup.html
     */
    toGetDeploymentGroup() {
        return this.to('GetDeploymentGroup');
    }
    /**
     * Grants permission to get information about a single instance in a deployment associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentInstance.html
     */
    toGetDeploymentInstance() {
        return this.to('GetDeploymentInstance');
    }
    /**
     * Grants permission to return information about a deployment target
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentTarget.html
     */
    toGetDeploymentTarget() {
        return this.to('GetDeploymentTarget');
    }
    /**
     * Grants permission to get information about a single on-premises instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetOnPremisesInstance.html
     */
    toGetOnPremisesInstance() {
        return this.to('GetOnPremisesInstance');
    }
    /**
     * Grants permission to get information about all application revisions for an application associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListApplicationRevisions.html
     */
    toListApplicationRevisions() {
        return this.to('ListApplicationRevisions');
    }
    /**
     * Grants permission to get information about all applications associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to get information about all deployment configurations associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentConfigs.html
     */
    toListDeploymentConfigs() {
        return this.to('ListDeploymentConfigs');
    }
    /**
     * Grants permission to get information about all deployment groups for an application associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentGroups.html
     */
    toListDeploymentGroups() {
        return this.to('ListDeploymentGroups');
    }
    /**
     * Grants permission to get information about all instances in a deployment associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentInstances.html
     */
    toListDeploymentInstances() {
        return this.to('ListDeploymentInstances');
    }
    /**
     * Grants permission to return an array of target IDs that are associated a deployment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentTargets.html
     */
    toListDeploymentTargets() {
        return this.to('ListDeploymentTargets');
    }
    /**
     * Grants permission to get information about all deployments to a deployment group associated with the IAM user, or to get all deployments associated with the IAM user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeployments.html
     */
    toListDeployments() {
        return this.to('ListDeployments');
    }
    /**
     * Grants permission to list the names of stored connections to GitHub accounts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListGitHubAccountTokenNames.html
     */
    toListGitHubAccountTokenNames() {
        return this.to('ListGitHubAccountTokenNames');
    }
    /**
     * Grants permission to get a list of one or more on-premises instance names
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListOnPremisesInstances.html
     */
    toListOnPremisesInstances() {
        return this.to('ListOnPremisesInstances');
    }
    /**
     * Grants permission to return a list of tags for the resource identified by a specified ARN. Tags are used to organize and categorize your CodeDeploy resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to notify a lifecycle event hook execution status for associated deployment with the IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_PutLifecycleEventHookExecutionStatus.html
     */
    toPutLifecycleEventHookExecutionStatus() {
        return this.to('PutLifecycleEventHookExecutionStatus');
    }
    /**
     * Grants permission to register information about an application revision for an application associated with the IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RegisterApplicationRevision.html
     */
    toRegisterApplicationRevision() {
        return this.to('RegisterApplicationRevision');
    }
    /**
     * Grants permission to register an on-premises instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RegisterOnPremisesInstance.html
     */
    toRegisterOnPremisesInstance() {
        return this.to('RegisterOnPremisesInstance');
    }
    /**
     * Grants permission to remove tags from one or more on-premises instances
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RemoveTagsFromOnPremisesInstances.html
     */
    toRemoveTagsFromOnPremisesInstances() {
        return this.to('RemoveTagsFromOnPremisesInstances');
    }
    /**
     * Grants permission to override any specified wait time and starts terminating instances immediately after the traffic routing is complete. This action applies to blue-green deployments only
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_SkipWaitTimeForInstanceTermination.html
     */
    toSkipWaitTimeForInstanceTermination() {
        return this.to('SkipWaitTimeForInstanceTermination');
    }
    /**
     * Grants permission to stop a deployment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_StopDeployment.html
     */
    toStopDeployment() {
        return this.to('StopDeployment');
    }
    /**
     * Grants permission to associate the list of tags in the input Tags parameter with the resource identified by the ResourceArn input parameter
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to disassociate a resource from a list of tags. The resource is identified by the ResourceArn input parameter. The tags are identfied by the list of keys in the TagKeys input parameter
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Grants permission to change information about a single deployment group for an application associated with the IAM user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UpdateDeploymentGroup.html
     */
    toUpdateDeploymentGroup() {
        return this.to('UpdateDeploymentGroup');
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codedeploy:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application:${applicationName}`);
    }
    /**
     * Adds a resource of type deploymentconfig to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param deploymentConfigurationName - Identifier for the deploymentConfigurationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDeploymentconfig(deploymentConfigurationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codedeploy:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deploymentconfig:${deploymentConfigurationName}`);
    }
    /**
     * Adds a resource of type deploymentgroup to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param deploymentGroupName - Identifier for the deploymentGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeploymentgroup(applicationName, deploymentGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codedeploy:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:deploymentgroup:${applicationName}/${deploymentGroupName}`);
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param instanceName - Identifier for the instanceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onInstance(instanceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codedeploy:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:instance:${instanceName}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateDeploymentGroup()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - application
     * - deploymentgroup
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateDeploymentGroup()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [codedeploy](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodedeploy.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codedeploy';
        this.accessLevelList = {
            Tagging: [
                'AddTagsToOnPremisesInstances',
                'RemoveTagsFromOnPremisesInstances',
                'TagResource',
                'UntagResource'
            ],
            Read: [
                'BatchGetApplicationRevisions',
                'BatchGetApplications',
                'BatchGetDeploymentGroups',
                'BatchGetDeploymentInstances',
                'BatchGetDeploymentTargets',
                'BatchGetDeployments',
                'BatchGetOnPremisesInstances',
                'GetDeploymentTarget'
            ],
            Write: [
                'ContinueDeployment',
                'CreateApplication',
                'CreateCloudFormationDeployment',
                'CreateDeployment',
                'CreateDeploymentConfig',
                'CreateDeploymentGroup',
                'DeleteApplication',
                'DeleteDeploymentConfig',
                'DeleteDeploymentGroup',
                'DeleteGitHubAccountToken',
                'DeleteResourcesByExternalId',
                'DeregisterOnPremisesInstance',
                'PutLifecycleEventHookExecutionStatus',
                'RegisterApplicationRevision',
                'RegisterOnPremisesInstance',
                'SkipWaitTimeForInstanceTermination',
                'StopDeployment',
                'UpdateApplication',
                'UpdateDeploymentGroup'
            ],
            List: [
                'GetApplication',
                'GetApplicationRevision',
                'GetDeployment',
                'GetDeploymentConfig',
                'GetDeploymentGroup',
                'GetDeploymentInstance',
                'GetOnPremisesInstance',
                'ListApplicationRevisions',
                'ListApplications',
                'ListDeploymentConfigs',
                'ListDeploymentGroups',
                'ListDeploymentInstances',
                'ListDeploymentTargets',
                'ListDeployments',
                'ListGitHubAccountTokenNames',
                'ListOnPremisesInstances',
                'ListTagsForResource'
            ]
        };
    }
}
exports.Codedeploy = Codedeploy;
//# sourceMappingURL=data:application/json;base64,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