"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codepipeline = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codepipeline](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodepipeline.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codepipeline extends shared_1.PolicyStatement {
    /**
     * Grants permission to view information about a specified job and whether that job has been received by the job worker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_AcknowledgeJob.html
     */
    toAcknowledgeJob() {
        return this.to('AcknowledgeJob');
    }
    /**
     * Grants permission to confirm that a job worker has received the specified job (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_AcknowledgeThirdPartyJob.html
     */
    toAcknowledgeThirdPartyJob() {
        return this.to('AcknowledgeThirdPartyJob');
    }
    /**
     * Grants permission to create a custom action that you can use in the pipelines associated with your AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_CreateCustomActionType.html
     */
    toCreateCustomActionType() {
        return this.to('CreateCustomActionType');
    }
    /**
     * Grants permission to create a uniquely named pipeline
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_CreatePipeline.html
     */
    toCreatePipeline() {
        return this.to('CreatePipeline');
    }
    /**
     * Grants permission to delete a custom action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeleteCustomActionType.html
     */
    toDeleteCustomActionType() {
        return this.to('DeleteCustomActionType');
    }
    /**
     * Grants permission to delete a specified pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeletePipeline.html
     */
    toDeletePipeline() {
        return this.to('DeletePipeline');
    }
    /**
     * Grants permission to delete a specified webhook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeleteWebhook.html
     */
    toDeleteWebhook() {
        return this.to('DeleteWebhook');
    }
    /**
     * Grants permission to remove the registration of a webhook with the third party specified in its configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DeregisterWebhookWithThirdParty.html
     */
    toDeregisterWebhookWithThirdParty() {
        return this.to('DeregisterWebhookWithThirdParty');
    }
    /**
     * Grants permission to prevent revisions from transitioning to the next stage in a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_DisableStageTransition.html
     */
    toDisableStageTransition() {
        return this.to('DisableStageTransition');
    }
    /**
     * Grants permission to allow revisions to transition to the next stage in a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_EnableStageTransition.html
     */
    toEnableStageTransition() {
        return this.to('EnableStageTransition');
    }
    /**
     * Grants permission to view information about an action type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetActionType.html
     */
    toGetActionType() {
        return this.to('GetActionType');
    }
    /**
     * Grants permission to view information about a job (custom actions only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetJobDetails.html
     */
    toGetJobDetails() {
        return this.to('GetJobDetails');
    }
    /**
     * Grants permission to retrieve information about a pipeline structure
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipeline.html
     */
    toGetPipeline() {
        return this.to('GetPipeline');
    }
    /**
     * Grants permission to view information about an execution of a pipeline, including details about artifacts, the pipeline execution ID, and the name, version, and status of the pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipelineExecution.html
     */
    toGetPipelineExecution() {
        return this.to('GetPipelineExecution');
    }
    /**
     * Grants permission to view information about the current state of the stages and actions of a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetPipelineState.html
     */
    toGetPipelineState() {
        return this.to('GetPipelineState');
    }
    /**
     * Grants permission to view the details of a job for a third-party action (partner actions only)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_GetThirdPartyJobDetails.html
     */
    toGetThirdPartyJobDetails() {
        return this.to('GetThirdPartyJobDetails');
    }
    /**
     * Grants permission to list the action executions that have occurred in a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListActionExecutions.html
     */
    toListActionExecutions() {
        return this.to('ListActionExecutions');
    }
    /**
     * Grants permission to list a summary of all the action types available for pipelines in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListActionTypes.html
     */
    toListActionTypes() {
        return this.to('ListActionTypes');
    }
    /**
     * Grants permission to list the deployment details for deploy action executions that have occurred in a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListDeployActionExecutionTargets.html
     */
    toListDeployActionExecutionTargets() {
        return this.to('ListDeployActionExecutionTargets');
    }
    /**
     * Grants permission to list a summary of the most recent executions for a pipeline
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListPipelineExecutions.html
     */
    toListPipelineExecutions() {
        return this.to('ListPipelineExecutions');
    }
    /**
     * Grants permission to list a summary of all the pipelines associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListPipelines.html
     */
    toListPipelines() {
        return this.to('ListPipelines');
    }
    /**
     * Grants permission to list the rule executions that have occurred in a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListRuleExecutions.html
     */
    toListRuleExecutions() {
        return this.to('ListRuleExecutions');
    }
    /**
     * Grants permission to list a summary of all the rule types available for pipelines in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListRuleTypes.html
     */
    toListRuleTypes() {
        return this.to('ListRuleTypes');
    }
    /**
     * Grants permission to list tags for a CodePipeline resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all of the webhooks associated with your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_ListWebhooks.html
     */
    toListWebhooks() {
        return this.to('ListWebhooks');
    }
    /**
     * Grants permission to resume the pipeline execution by overriding a condition in a stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_OverrideStageCondition.html
     */
    toOverrideStageCondition() {
        return this.to('OverrideStageCondition');
    }
    /**
     * Grants permission to view information about any jobs for CodePipeline to act on
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PollForJobs.html
     */
    toPollForJobs() {
        return this.to('PollForJobs');
    }
    /**
     * Grants permission to determine whether there are any third-party jobs for a job worker to act on (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PollForThirdPartyJobs.html
     */
    toPollForThirdPartyJobs() {
        return this.to('PollForThirdPartyJobs');
    }
    /**
     * Grants permission to edit actions in a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutActionRevision.html
     */
    toPutActionRevision() {
        return this.to('PutActionRevision');
    }
    /**
     * Grants permission to provide a response (Approved or Rejected) to a manual approval request in CodePipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutApprovalResult.html
     */
    toPutApprovalResult() {
        return this.to('PutApprovalResult');
    }
    /**
     * Grants permission to represent the failure of a job as returned to the pipeline by a job worker (custom actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobFailureResult.html
     */
    toPutJobFailureResult() {
        return this.to('PutJobFailureResult');
    }
    /**
     * Grants permission to represent the success of a job as returned to the pipeline by a job worker (custom actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobSuccessResult.html
     */
    toPutJobSuccessResult() {
        return this.to('PutJobSuccessResult');
    }
    /**
     * Grants permission to represent the failure of a third-party job as returned to the pipeline by a job worker (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutThirdPartyJobFailureResult.html
     */
    toPutThirdPartyJobFailureResult() {
        return this.to('PutThirdPartyJobFailureResult');
    }
    /**
     * Grants permission to represent the success of a third-party job as returned to the pipeline by a job worker (partner actions only)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutThirdPartyJobSuccessResult.html
     */
    toPutThirdPartyJobSuccessResult() {
        return this.to('PutThirdPartyJobSuccessResult');
    }
    /**
     * Grants permission to create or update a webhook
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutWebhook.html
     */
    toPutWebhook() {
        return this.to('PutWebhook');
    }
    /**
     * Grants permission to register a webhook with the third party specified in its configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_RegisterWebhookWithThirdParty.html
     */
    toRegisterWebhookWithThirdParty() {
        return this.to('RegisterWebhookWithThirdParty');
    }
    /**
     * Grants permission to resume the pipeline execution by retrying the last failed actions in a stage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_RetryStageExecution.html
     */
    toRetryStageExecution() {
        return this.to('RetryStageExecution');
    }
    /**
     * Grants permission to rollback the stage to a previous successful execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_RollbackStage.html
     */
    toRollbackStage() {
        return this.to('RollbackStage');
    }
    /**
     * Grants permission to run the most recent revision through the pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_StartPipelineExecution.html
     */
    toStartPipelineExecution() {
        return this.to('StartPipelineExecution');
    }
    /**
     * Grants permission to stop an in-progress pipeline execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_StopPipelineExecution.html
     */
    toStopPipelineExecution() {
        return this.to('StopPipelineExecution');
    }
    /**
     * Grants permission to tag a CodePipeline resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a CodePipeline resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an action type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_UpdateActionType.html
     */
    toUpdateActionType() {
        return this.to('UpdateActionType');
    }
    /**
     * Grants permission to update a pipeline with changes to the structure of the pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_UpdatePipeline.html
     */
    toUpdatePipeline() {
        return this.to('UpdatePipeline');
    }
    /**
     * Adds a resource of type action to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param stageName - Identifier for the stageName.
     * @param actionName - Identifier for the actionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAction(pipelineName, stageName, actionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codepipeline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${pipelineName}/${stageName}/${actionName}`);
    }
    /**
     * Adds a resource of type actiontype to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param owner - Identifier for the owner.
     * @param category - Identifier for the category.
     * @param provider - Identifier for the provider.
     * @param version - Identifier for the version.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onActiontype(owner, category, provider, version, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codepipeline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:actiontype:${owner}/${category}/${provider}/${version}`);
    }
    /**
     * Adds a resource of type pipeline to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPipeline(pipelineName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codepipeline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${pipelineName}`);
    }
    /**
     * Adds a resource of type stage to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param pipelineName - Identifier for the pipelineName.
     * @param stageName - Identifier for the stageName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStage(pipelineName, stageName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codepipeline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${pipelineName}/${stageName}`);
    }
    /**
     * Adds a resource of type webhook to the statement
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/iam-access-control-identity-based.html#ACP_ARN_Format
     *
     * @param webhookName - Identifier for the webhookName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebhook(webhookName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codepipeline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:webhook:${webhookName}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCustomActionType()
     * - .toCreatePipeline()
     * - .toPutWebhook()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - action
     * - actiontype
     * - pipeline
     * - stage
     * - webhook
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCustomActionType()
     * - .toCreatePipeline()
     * - .toPutWebhook()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [codepipeline](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodepipeline.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codepipeline';
        this.accessLevelList = {
            Write: [
                'AcknowledgeJob',
                'AcknowledgeThirdPartyJob',
                'CreateCustomActionType',
                'CreatePipeline',
                'DeleteCustomActionType',
                'DeletePipeline',
                'DeleteWebhook',
                'DeregisterWebhookWithThirdParty',
                'DisableStageTransition',
                'EnableStageTransition',
                'OverrideStageCondition',
                'PollForJobs',
                'PollForThirdPartyJobs',
                'PutActionRevision',
                'PutApprovalResult',
                'PutJobFailureResult',
                'PutJobSuccessResult',
                'PutThirdPartyJobFailureResult',
                'PutThirdPartyJobSuccessResult',
                'PutWebhook',
                'RegisterWebhookWithThirdParty',
                'RetryStageExecution',
                'RollbackStage',
                'StartPipelineExecution',
                'StopPipelineExecution',
                'UpdateActionType',
                'UpdatePipeline'
            ],
            Read: [
                'GetActionType',
                'GetJobDetails',
                'GetPipeline',
                'GetPipelineExecution',
                'GetPipelineState',
                'GetThirdPartyJobDetails',
                'ListActionExecutions',
                'ListActionTypes',
                'ListDeployActionExecutionTargets',
                'ListRuleExecutions',
                'ListRuleTypes',
                'ListTagsForResource'
            ],
            List: [
                'ListPipelineExecutions',
                'ListPipelines',
                'ListWebhooks'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Codepipeline = Codepipeline;
//# sourceMappingURL=data:application/json;base64,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