"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodestarConnections = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [codestar-connections](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodestarconnections.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CodestarConnections extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a Connection resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifProviderType()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_CreateConnection.html
     */
    toCreateConnection() {
        return this.to('CreateConnection');
    }
    /**
     * Grants permission to create a host resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifProviderType()
     * - .ifVpcId()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_CreateHost.html
     */
    toCreateHost() {
        return this.to('CreateHost');
    }
    /**
     * Grants permission to create a repository link
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - codestar-connections:PassConnection
     * - codestar-connections:UseConnection
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_CreateRepositoryLink.html
     */
    toCreateRepositoryLink() {
        return this.to('CreateRepositoryLink');
    }
    /**
     * Grants permission to create a template sync config
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBranch()
     *
     * Dependent actions:
     * - codestar-connections:PassRepository
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_CreateSyncConfiguration.html
     */
    toCreateSyncConfiguration() {
        return this.to('CreateSyncConfiguration');
    }
    /**
     * Grants permission to delete a Connection resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection() {
        return this.to('DeleteConnection');
    }
    /**
     * Grants permission to delete a host resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_DeleteHost.html
     */
    toDeleteHost() {
        return this.to('DeleteHost');
    }
    /**
     * Grants permission to delete a repository link
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_DeleteRepositoryLink.html
     */
    toDeleteRepositoryLink() {
        return this.to('DeleteRepositoryLink');
    }
    /**
     * Grants permission to delete a sync configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_DeleteSyncConfiguration.html
     */
    toDeleteSyncConfiguration() {
        return this.to('DeleteSyncConfiguration');
    }
    /**
     * Grants permission to get details about a Connection resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetConnection.html
     */
    toGetConnection() {
        return this.to('GetConnection');
    }
    /**
     * Grants permission to get a Connection token to call provider actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-getconnectiontoken
     */
    toGetConnectionToken() {
        return this.to('GetConnectionToken');
    }
    /**
     * Grants permission to get details about a host resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetHost.html
     */
    toGetHost() {
        return this.to('GetHost');
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     *
     * Dependent actions:
     * - codestar-connections:StartOAuthHandshake
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     */
    toGetIndividualAccessToken() {
        return this.to('GetIndividualAccessToken');
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     */
    toGetInstallationUrl() {
        return this.to('GetInstallationUrl');
    }
    /**
     * Grants permission to describe a repository link
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetRepositoryLink.html
     */
    toGetRepositoryLink() {
        return this.to('GetRepositoryLink');
    }
    /**
     * Grants permission to get the latest sync status for a repository
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifBranch()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetRepositorySyncStatus.html
     */
    toGetRepositorySyncStatus() {
        return this.to('GetRepositorySyncStatus');
    }
    /**
     * Grants permission to get the latest sync status for a resource (cfn stack or other resources)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetResourceSyncStatus.html
     */
    toGetResourceSyncStatus() {
        return this.to('GetResourceSyncStatus');
    }
    /**
     * Grants permission to describe service sync blockers on a resource (cfn stack or other resources)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetSyncBlockerSummary.html
     */
    toGetSyncBlockerSummary() {
        return this.to('GetSyncBlockerSummary');
    }
    /**
     * Grants permission to describe a sync configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_GetSyncConfiguration.html
     */
    toGetSyncConfiguration() {
        return this.to('GetSyncConfiguration');
    }
    /**
     * Grants permission to list Connection resources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifProviderTypeFilter()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListConnections.html
     */
    toListConnections() {
        return this.to('ListConnections');
    }
    /**
     * Grants permission to list host resources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifProviderTypeFilter()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListHosts.html
     */
    toListHosts() {
        return this.to('ListHosts');
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: List
     *
     * Dependent actions:
     * - codestar-connections:GetIndividualAccessToken
     * - codestar-connections:StartOAuthHandshake
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     */
    toListInstallationTargets() {
        return this.to('ListInstallationTargets');
    }
    /**
     * Grants permission to list repository links
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListRepositoryLinks.html
     */
    toListRepositoryLinks() {
        return this.to('ListRepositoryLinks');
    }
    /**
     * Grants permission to list repository sync definitions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListRepositorySyncDefinitions.html
     */
    toListRepositorySyncDefinitions() {
        return this.to('ListRepositorySyncDefinitions');
    }
    /**
     * Grants permission to list sync configurations for a repository link
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListSyncConfigurations.html
     */
    toListSyncConfigurations() {
        return this.to('ListSyncConfigurations');
    }
    /**
     * Grants permission to the set of key-value pairs that are used to manage the resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to pass a Connection resource to an AWS service that accepts a Connection ARN as input, such as codepipeline:CreatePipeline
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPassedToService()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-passconnection
     */
    toPassConnection() {
        return this.to('PassConnection');
    }
    /**
     * Grants permission to pass a repository link resource to an AWS service that accepts a RepositoryLinkId as input, such as codestar-connections:CreateSyncConfiguration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifPassedToService()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-passrepository
     */
    toPassRepository() {
        return this.to('PassRepository');
    }
    /**
     * Grants permission to associate a third party server, such as a GitHub Enterprise Server instance, with a Host
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifHostArn()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#connections-permissions-actions-host-registration
     */
    toRegisterAppCode() {
        return this.to('RegisterAppCode');
    }
    /**
     * Grants permission to associate a third party server, such as a GitHub Enterprise Server instance, with a Host
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifHostArn()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#connections-permissions-actions-host-registration
     */
    toStartAppRegistrationHandshake() {
        return this.to('StartAppRegistrationHandshake');
    }
    /**
     * Grants permission to associate a third party, such as a Bitbucket App installation, with a Connection
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifProviderType()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     */
    toStartOAuthHandshake() {
        return this.to('StartOAuthHandshake');
    }
    /**
     * Grants permission to add or modify the tags of the given resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from an AWS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a Connection resource with an installation of the CodeStar Connections App
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstallationId()
     *
     * Dependent actions:
     * - codestar-connections:GetIndividualAccessToken
     * - codestar-connections:GetInstallationUrl
     * - codestar-connections:ListInstallationTargets
     * - codestar-connections:StartOAuthHandshake
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     */
    toUpdateConnectionInstallation() {
        return this.to('UpdateConnectionInstallation');
    }
    /**
     * Grants permission to update a host resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifVpcId()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_UpdateHost.html
     */
    toUpdateHost() {
        return this.to('UpdateHost');
    }
    /**
     * Grants permission to update a repository link
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_UpdateRepositoryLink.html
     */
    toUpdateRepositoryLink() {
        return this.to('UpdateRepositoryLink');
    }
    /**
     * Grants permission to update a sync blocker for a resource (cfn stack or other resources)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_UpdateSyncBlocker.html
     */
    toUpdateSyncBlocker() {
        return this.to('UpdateSyncBlocker');
    }
    /**
     * Grants permission to update a sync configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifBranch()
     *
     * https://docs.aws.amazon.com/codestar-connections/latest/APIReference/API_UpdateSyncConfiguration.html
     */
    toUpdateSyncConfiguration() {
        return this.to('UpdateSyncConfiguration');
    }
    /**
     * Grants permission to use a Connection resource to call provider actions
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifBranchName()
     * - .ifFullRepositoryId()
     * - .ifOwnerId()
     * - .ifProviderAction()
     * - .ifProviderPermissionsRequired()
     * - .ifRepositoryName()
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     */
    toUseConnection() {
        return this.to('UseConnection');
    }
    /**
     * Adds a resource of type Connection to the statement
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/connections.html
     *
     * @param connectionId - Identifier for the connectionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConnection(connectionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codestar-connections:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connection/${connectionId}`);
    }
    /**
     * Adds a resource of type Host to the statement
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/connections-hosts.html
     *
     * @param hostId - Identifier for the hostId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHost(hostId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codestar-connections:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:host/${hostId}`);
    }
    /**
     * Adds a resource of type RepositoryLink to the statement
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/repositorylinks.html
     *
     * @param repositoryLinkId - Identifier for the repositoryLinkId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRepositoryLink(repositoryLinkId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:codestar-connections:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:repository-link/${repositoryLinkId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateConnection()
     * - .toCreateHost()
     * - .toCreateRepositoryLink()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - Connection
     * - Host
     * - RepositoryLink
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateConnection()
     * - .toCreateHost()
     * - .toCreateRepositoryLink()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the branch name that is passed in the request
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     *
     * Applies to actions:
     * - .toCreateSyncConfiguration()
     * - .toGetRepositorySyncStatus()
     * - .toUpdateSyncConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBranch(value, operator) {
        return this.if(`Branch`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the branch name that is passed in the request. Applies only to UseConnection requests for access to a specific repository branch
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifBranchName(value, operator) {
        return this.if(`BranchName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the repository that is passed in the request. Applies only to UseConnection requests for access to a specific repository
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFullRepositoryId(value, operator) {
        return this.if(`FullRepositoryId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the host resource associated with the connection used in the request
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-hosts
     *
     * Applies to actions:
     * - .toRegisterAppCode()
     * - .toStartAppRegistrationHandshake()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifHostArn(value, operator) {
        return this.if(`HostArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the third-party ID (such as the Bitbucket App installation ID for CodeStar Connections) that is used to update a Connection. Allows you to restrict which third-party App installations can be used to make a Connection
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-handshake
     *
     * Applies to actions:
     * - .toUpdateConnectionInstallation()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstallationId(value, operator) {
        return this.if(`InstallationId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the owner of the third-party repository. Applies only to UseConnection requests for access to repositories owned by a specific user
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOwnerId(value, operator) {
        return this.if(`OwnerId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the service to which the principal is allowed to pass a Connection or RepositoryLink
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-passconnection
     *
     * Applies to actions:
     * - .toPassConnection()
     * - .toPassRepository()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPassedToService(value, operator) {
        return this.if(`PassedToService`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the provider action in a UseConnection request such as ListRepositories. See documentation for all valid values
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-access
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderAction(value, operator) {
        return this.if(`ProviderAction`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the write permissions of a provider action in a UseConnection request. Valid types include read_only and read_write
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderPermissionsRequired(value, operator) {
        return this.if(`ProviderPermissionsRequired`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the type of third-party provider passed in the request
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-managing
     *
     * Applies to actions:
     * - .toCreateConnection()
     * - .toCreateHost()
     * - .toGetIndividualAccessToken()
     * - .toGetInstallationUrl()
     * - .toStartOAuthHandshake()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderType(value, operator) {
        return this.if(`ProviderType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the type of third-party provider used to filter results
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-managing
     *
     * Applies to actions:
     * - .toListConnections()
     * - .toListHosts()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProviderTypeFilter(value, operator) {
        return this.if(`ProviderTypeFilter`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the repository name that is passed in the request. Applies only to UseConnection requests for access to repositories owned by a specific user
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-use
     *
     * Applies to actions:
     * - .toUseConnection()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRepositoryName(value, operator) {
        return this.if(`RepositoryName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the VpcId passed in the request
     *
     * https://docs.aws.amazon.com/dtconsole/latest/userguide/security-iam.html#permissions-reference-connections-hosts
     *
     * Applies to actions:
     * - .toCreateHost()
     * - .toUpdateHost()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVpcId(value, operator) {
        return this.if(`VpcId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [codestar-connections](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscodestarconnections.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'codestar-connections';
        this.accessLevelList = {
            Write: [
                'CreateConnection',
                'CreateHost',
                'CreateRepositoryLink',
                'CreateSyncConfiguration',
                'DeleteConnection',
                'DeleteHost',
                'DeleteRepositoryLink',
                'DeleteSyncConfiguration',
                'UpdateConnectionInstallation',
                'UpdateHost',
                'UpdateRepositoryLink',
                'UpdateSyncBlocker',
                'UpdateSyncConfiguration'
            ],
            Read: [
                'GetConnection',
                'GetConnectionToken',
                'GetHost',
                'GetIndividualAccessToken',
                'GetInstallationUrl',
                'GetRepositoryLink',
                'GetRepositorySyncStatus',
                'GetResourceSyncStatus',
                'GetSyncBlockerSummary',
                'GetSyncConfiguration',
                'PassConnection',
                'PassRepository',
                'RegisterAppCode',
                'StartAppRegistrationHandshake',
                'StartOAuthHandshake',
                'UseConnection'
            ],
            List: [
                'ListConnections',
                'ListHosts',
                'ListInstallationTargets',
                'ListRepositoryLinks',
                'ListRepositorySyncDefinitions',
                'ListSyncConfigurations',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.CodestarConnections = CodestarConnections;
//# sourceMappingURL=data:application/json;base64,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