"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoIdp = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cognito-idp](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncognitouserpools.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CognitoIdp extends shared_1.PolicyStatement {
    /**
     * Grants permission to add user attributes to the user pool schema
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AddCustomAttributes.html
     */
    toAddCustomAttributes() {
        return this.to('AddCustomAttributes');
    }
    /**
     * Grants permission to add any user to any group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminAddUserToGroup.html
     */
    toAdminAddUserToGroup() {
        return this.to('AdminAddUserToGroup');
    }
    /**
     * Grants permission to confirm any user's registration without a confirmation code
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminConfirmSignUp.html
     */
    toAdminConfirmSignUp() {
        return this.to('AdminConfirmSignUp');
    }
    /**
     * Grants permission to create new users and send welcome messages via email or SMS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminCreateUser.html
     */
    toAdminCreateUser() {
        return this.to('AdminCreateUser');
    }
    /**
     * Grants permission to delete any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDeleteUser.html
     */
    toAdminDeleteUser() {
        return this.to('AdminDeleteUser');
    }
    /**
     * Grants permission to delete attributes from any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDeleteUserAttributes.html
     */
    toAdminDeleteUserAttributes() {
        return this.to('AdminDeleteUserAttributes');
    }
    /**
     * Grants permission to unlink any user pool user from a third-party identity provider (IdP) user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDisableProviderForUser.html
     */
    toAdminDisableProviderForUser() {
        return this.to('AdminDisableProviderForUser');
    }
    /**
     * Grants permission to deactivate any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminDisableUser.html
     */
    toAdminDisableUser() {
        return this.to('AdminDisableUser');
    }
    /**
     * Grants permission to activate any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminEnableUser.html
     */
    toAdminEnableUser() {
        return this.to('AdminEnableUser');
    }
    /**
     * Grants permission to deregister any user's devices
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminForgetDevice.html
     */
    toAdminForgetDevice() {
        return this.to('AdminForgetDevice');
    }
    /**
     * Grants permission to get information about any user's devices
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminGetDevice.html
     */
    toAdminGetDevice() {
        return this.to('AdminGetDevice');
    }
    /**
     * Grants permission to look up any user by user name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminGetUser.html
     */
    toAdminGetUser() {
        return this.to('AdminGetUser');
    }
    /**
     * Grants permission to authenticate any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html
     */
    toAdminInitiateAuth() {
        return this.to('AdminInitiateAuth');
    }
    /**
     * Grants permission to link any user pool user to a third-party IdP user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminLinkProviderForUser.html
     */
    toAdminLinkProviderForUser() {
        return this.to('AdminLinkProviderForUser');
    }
    /**
     * Grants permission to list any user's remembered devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminListDevices.html
     */
    toAdminListDevices() {
        return this.to('AdminListDevices');
    }
    /**
     * Grants permission to list the groups that any user belongs to
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminListGroupsForUser.html
     */
    toAdminListGroupsForUser() {
        return this.to('AdminListGroupsForUser');
    }
    /**
     * Grants permission to lists sign-in events for any user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminListUserAuthEvents.html
     */
    toAdminListUserAuthEvents() {
        return this.to('AdminListUserAuthEvents');
    }
    /**
     * Grants permission to remove any user from any group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminRemoveUserFromGroup.html
     */
    toAdminRemoveUserFromGroup() {
        return this.to('AdminRemoveUserFromGroup');
    }
    /**
     * Grants permission to reset any user's password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminResetUserPassword.html
     */
    toAdminResetUserPassword() {
        return this.to('AdminResetUserPassword');
    }
    /**
     * Grants permission to respond to an authentication challenge during the authentication of any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminRespondToAuthChallenge.html
     */
    toAdminRespondToAuthChallenge() {
        return this.to('AdminRespondToAuthChallenge');
    }
    /**
     * Grants permission to set any user's preferred MFA method
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminSetUserMFAPreference.html
     */
    toAdminSetUserMFAPreference() {
        return this.to('AdminSetUserMFAPreference');
    }
    /**
     * Grants permission to set any user's password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminSetUserPassword.html
     */
    toAdminSetUserPassword() {
        return this.to('AdminSetUserPassword');
    }
    /**
     * Grants permission to set user settings for any user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminSetUserSettings.html
     */
    toAdminSetUserSettings() {
        return this.to('AdminSetUserSettings');
    }
    /**
     * Grants permission to update advanced security feedback for any user's authentication event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateAuthEventFeedback.html
     */
    toAdminUpdateAuthEventFeedback() {
        return this.to('AdminUpdateAuthEventFeedback');
    }
    /**
     * Grants permission to update the status of any user's remembered devices
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateDeviceStatus.html
     */
    toAdminUpdateDeviceStatus() {
        return this.to('AdminUpdateDeviceStatus');
    }
    /**
     * Grants permission to updates any user's standard or custom attributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html
     */
    toAdminUpdateUserAttributes() {
        return this.to('AdminUpdateUserAttributes');
    }
    /**
     * Grants permission to sign out any user from all sessions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUserGlobalSignOut.html
     */
    toAdminUserGlobalSignOut() {
        return this.to('AdminUserGlobalSignOut');
    }
    /**
     * Grants permission to return a unique generated shared secret key code for the user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AssociateSoftwareToken.html
     */
    toAssociateSoftwareToken() {
        return this.to('AssociateSoftwareToken');
    }
    /**
     * Grants permission to associate the user pool with an AWS WAF web ACL
     *
     * Access Level: Write
     */
    toAssociateWebACL() {
        return this.to('AssociateWebACL');
    }
    /**
     * Grants permission to change the password for a specified user in a user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ChangePassword.html
     */
    toChangePassword() {
        return this.to('ChangePassword');
    }
    /**
     * Grants permission to confirm tracking of the device. This API call is the call that begins device tracking
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ConfirmDevice.html
     */
    toConfirmDevice() {
        return this.to('ConfirmDevice');
    }
    /**
     * Grants permission to allow a user to enter a confirmation code to reset a forgotten password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ConfirmForgotPassword.html
     */
    toConfirmForgotPassword() {
        return this.to('ConfirmForgotPassword');
    }
    /**
     * Grants permission to confirm registration of a user and handles the existing alias from a previous user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ConfirmSignUp.html
     */
    toConfirmSignUp() {
        return this.to('ConfirmSignUp');
    }
    /**
     * Grants permission to create new user pool groups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        return this.to('CreateGroup');
    }
    /**
     * Grants permission to add identity providers to user pools
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateIdentityProvider.html
     */
    toCreateIdentityProvider() {
        return this.to('CreateIdentityProvider');
    }
    /**
     * Grants permission to create a branding settings for managed login and associate it with an app client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateManagedLoginBranding.html
     */
    toCreateManagedLoginBranding() {
        return this.to('CreateManagedLoginBranding');
    }
    /**
     * Grants permission to create and configure scopes for OAuth 2.0 resource servers
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateResourceServer.html
     */
    toCreateResourceServer() {
        return this.to('CreateResourceServer');
    }
    /**
     * Grants permission to create user CSV import jobs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserImportJob.html
     */
    toCreateUserImportJob() {
        return this.to('CreateUserImportJob');
    }
    /**
     * Grants permission to create and set password policy for user pools
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html
     */
    toCreateUserPool() {
        return this.to('CreateUserPool');
    }
    /**
     * Grants permission to create user pool app clients
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPoolClient.html
     */
    toCreateUserPoolClient() {
        return this.to('CreateUserPoolClient');
    }
    /**
     * Grants permission to add user pool domains
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPoolDomain.html
     */
    toCreateUserPoolDomain() {
        return this.to('CreateUserPoolDomain');
    }
    /**
     * Grants permission to delete any empty user pool group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to delete any identity provider from user pools
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteIdentityProvider.html
     */
    toDeleteIdentityProvider() {
        return this.to('DeleteIdentityProvider');
    }
    /**
     * Grants permission to delete the managed login branding style for any app client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteManagedLoginBranding.html
     */
    toDeleteManagedLoginBranding() {
        return this.to('DeleteManagedLoginBranding');
    }
    /**
     * Grants permission to delete any OAuth 2.0 resource server from user pools
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteResourceServer.html
     */
    toDeleteResourceServer() {
        return this.to('DeleteResourceServer');
    }
    /**
     * Grants permission to allow a user to delete one's self
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to delete the attributes for a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserAttributes.html
     */
    toDeleteUserAttributes() {
        return this.to('DeleteUserAttributes');
    }
    /**
     * Grants permission to delete user pools
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserPool.html
     */
    toDeleteUserPool() {
        return this.to('DeleteUserPool');
    }
    /**
     * Grants permission to delete any user pool app client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserPoolClient.html
     */
    toDeleteUserPoolClient() {
        return this.to('DeleteUserPoolClient');
    }
    /**
     * Grants permission to delete any user pool domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DeleteUserPoolDomain.html
     */
    toDeleteUserPoolDomain() {
        return this.to('DeleteUserPoolDomain');
    }
    /**
     * Grants permission to describe any user pool identity provider
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeIdentityProvider.html
     */
    toDescribeIdentityProvider() {
        return this.to('DescribeIdentityProvider');
    }
    /**
     * Grants permission to get the detailed information about the branding style of managed login
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeManagedLoginBranding.html
     */
    toDescribeManagedLoginBranding() {
        return this.to('DescribeManagedLoginBranding');
    }
    /**
     * Grants permission to get the detailed information about the branding style of managed login associated with an appclient
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeManagedLoginBrandingByClient.html
     */
    toDescribeManagedLoginBrandingByClient() {
        return this.to('DescribeManagedLoginBrandingByClient');
    }
    /**
     * Grants permission to describe any OAuth 2.0 resource server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeResourceServer.html
     */
    toDescribeResourceServer() {
        return this.to('DescribeResourceServer');
    }
    /**
     * Grants permission to describe the risk configuration settings of user pools and app clients
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeRiskConfiguration.html
     */
    toDescribeRiskConfiguration() {
        return this.to('DescribeRiskConfiguration');
    }
    /**
     * Grants permission to describe any user import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserImportJob.html
     */
    toDescribeUserImportJob() {
        return this.to('DescribeUserImportJob');
    }
    /**
     * Grants permission to describe user pools
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPool.html
     */
    toDescribeUserPool() {
        return this.to('DescribeUserPool');
    }
    /**
     * Grants permission to describe any user pool app client
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPoolClient.html
     */
    toDescribeUserPoolClient() {
        return this.to('DescribeUserPoolClient');
    }
    /**
     * Grants permission to describe any user pool domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPoolDomain.html
     */
    toDescribeUserPoolDomain() {
        return this.to('DescribeUserPoolDomain');
    }
    /**
     * Grants permission to disassociate the user pool with an AWS WAF web ACL
     *
     * Access Level: Write
     */
    toDisassociateWebACL() {
        return this.to('DisassociateWebACL');
    }
    /**
     * Grants permission to forget the specified device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ForgetDevice.html
     */
    toForgetDevice() {
        return this.to('ForgetDevice');
    }
    /**
     * Grants permission to send a message to the end user with a confirmation code that is required to change the user's password
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ForgotPassword.html
     */
    toForgotPassword() {
        return this.to('ForgotPassword');
    }
    /**
     * Grants permission to generate headers for a user import .csv file
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetCSVHeader.html
     */
    toGetCSVHeader() {
        return this.to('GetCSVHeader');
    }
    /**
     * Grants permission to get the device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetDevice.html
     */
    toGetDevice() {
        return this.to('GetDevice');
    }
    /**
     * Grants permission to describe a user pool group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetGroup.html
     */
    toGetGroup() {
        return this.to('GetGroup');
    }
    /**
     * Grants permission to correlate a user pool IdP identifier to the IdP Name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetIdentityProviderByIdentifier.html
     */
    toGetIdentityProviderByIdentifier() {
        return this.to('GetIdentityProviderByIdentifier');
    }
    /**
     * Grants permission to get the detailed activity logging configuration for a user pool
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetLogDeliveryConfiguration.html
     */
    toGetLogDeliveryConfiguration() {
        return this.to('GetLogDeliveryConfiguration');
    }
    /**
     * Grants permission to look up signing certificates for user pools
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetSigningCertificate.html
     */
    toGetSigningCertificate() {
        return this.to('GetSigningCertificate');
    }
    /**
     * Grants permission to update user tokens with refresh tokens
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetTokensFromRefreshToken.html
     */
    toGetTokensFromRefreshToken() {
        return this.to('GetTokensFromRefreshToken');
    }
    /**
     * Grants permission to get UI customization information for the hosted UI of any app client
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUICustomization.html
     */
    toGetUICustomization() {
        return this.to('GetUICustomization');
    }
    /**
     * Grants permission to get the user attributes and metadata for a user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUser.html
     */
    toGetUser() {
        return this.to('GetUser');
    }
    /**
     * Grants permission to get the user attribute verification code for the specified attribute name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUserAttributeVerificationCode.html
     */
    toGetUserAttributeVerificationCode() {
        return this.to('GetUserAttributeVerificationCode');
    }
    /**
     * Grants permission to look up the MFA configuration of user pools
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUserPoolMfaConfig.html
     */
    toGetUserPoolMfaConfig() {
        return this.to('GetUserPoolMfaConfig');
    }
    /**
     * Grants permission to get the AWS WAF web ACL that is associated with an Amazon Cognito user pool
     *
     * Access Level: Read
     */
    toGetWebACLForResource() {
        return this.to('GetWebACLForResource');
    }
    /**
     * Grants permission to sign out users from all devices
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GlobalSignOut.html
     */
    toGlobalSignOut() {
        return this.to('GlobalSignOut');
    }
    /**
     * Grants permission to initiate the authentication flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_InitiateAuth.html
     */
    toInitiateAuth() {
        return this.to('InitiateAuth');
    }
    /**
     * Grants permission to list the devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListDevices.html
     */
    toListDevices() {
        return this.to('ListDevices');
    }
    /**
     * Grants permission to list all groups in user pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListGroups.html
     */
    toListGroups() {
        return this.to('ListGroups');
    }
    /**
     * Grants permission to list all identity providers in user pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListIdentityProviders.html
     */
    toListIdentityProviders() {
        return this.to('ListIdentityProviders');
    }
    /**
     * Grants permission to list all resource servers in user pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListResourceServers.html
     */
    toListResourceServers() {
        return this.to('ListResourceServers');
    }
    /**
     * Grants permission to list the user pools that are associated with an AWS WAF web ACL
     *
     * Access Level: List
     */
    toListResourcesForWebACL() {
        return this.to('ListResourcesForWebACL');
    }
    /**
     * Grants permission to list the tags that are assigned to an Amazon Cognito user pool
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all user import jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUserImportJobs.html
     */
    toListUserImportJobs() {
        return this.to('ListUserImportJobs');
    }
    /**
     * Grants permission to list all app clients in user pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUserPoolClients.html
     */
    toListUserPoolClients() {
        return this.to('ListUserPoolClients');
    }
    /**
     * Grants permission to list all user pools
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUserPools.html
     */
    toListUserPools() {
        return this.to('ListUserPools');
    }
    /**
     * Grants permission to list all user pool users
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUsers.html
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to list the users in any group
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUsersInGroup.html
     */
    toListUsersInGroup() {
        return this.to('ListUsersInGroup');
    }
    /**
     * Grants permission to resend the confirmation (for confirmation of registration) to a specific user in the user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ResendConfirmationCode.html
     */
    toResendConfirmationCode() {
        return this.to('ResendConfirmationCode');
    }
    /**
     * Grants permission to respond to the authentication challenge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_RespondToAuthChallenge.html
     */
    toRespondToAuthChallenge() {
        return this.to('RespondToAuthChallenge');
    }
    /**
     * Grants permission to revoke all of the access tokens generated by the specified refresh token
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_RevokeToken.html
     */
    toRevokeToken() {
        return this.to('RevokeToken');
    }
    /**
     * Grants permission to set up or modify the detailed activity logging configuration of a user pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetLogDeliveryConfiguration.html
     */
    toSetLogDeliveryConfiguration() {
        return this.to('SetLogDeliveryConfiguration');
    }
    /**
     * Grants permission to set risk configuration for user pools and app clients
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetRiskConfiguration.html
     */
    toSetRiskConfiguration() {
        return this.to('SetRiskConfiguration');
    }
    /**
     * Grants permission to customize the hosted UI for any app client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUICustomization.html
     */
    toSetUICustomization() {
        return this.to('SetUICustomization');
    }
    /**
     * Grants permission to set MFA preference for the user in the userpool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUserMFAPreference.html
     */
    toSetUserMFAPreference() {
        return this.to('SetUserMFAPreference');
    }
    /**
     * Grants permission to set user pool MFA configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUserPoolMfaConfig.html
     */
    toSetUserPoolMfaConfig() {
        return this.to('SetUserPoolMfaConfig');
    }
    /**
     * Grants permission to set the user settings like multi-factor authentication (MFA)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUserSettings.html
     */
    toSetUserSettings() {
        return this.to('SetUserSettings');
    }
    /**
     * Grants permission to register the user in the specified user pool and creates a user name, password, and user attributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SignUp.html
     */
    toSignUp() {
        return this.to('SignUp');
    }
    /**
     * Grants permission to start any user import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_StartUserImportJob.html
     */
    toStartUserImportJob() {
        return this.to('StartUserImportJob');
    }
    /**
     * Grants permission to stop any user import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_StopUserImportJob.html
     */
    toStopUserImportJob() {
        return this.to('StopUserImportJob');
    }
    /**
     * Grants permission to tag a user pool
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a user pool
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the feedback for the user authentication event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateAuthEventFeedback.html
     */
    toUpdateAuthEventFeedback() {
        return this.to('UpdateAuthEventFeedback');
    }
    /**
     * Grants permission to update the device status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateDeviceStatus.html
     */
    toUpdateDeviceStatus() {
        return this.to('UpdateDeviceStatus');
    }
    /**
     * Grants permission to update the configuration of any group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        return this.to('UpdateGroup');
    }
    /**
     * Grants permission to update the configuration of any user pool IdP
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateIdentityProvider.html
     */
    toUpdateIdentityProvider() {
        return this.to('UpdateIdentityProvider');
    }
    /**
     * Grants permission to update the branding settings of a managed login
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateManagedLoginBranding.html
     */
    toUpdateManagedLoginBranding() {
        return this.to('UpdateManagedLoginBranding');
    }
    /**
     * Grants permission to update the configuration of any OAuth 2.0 resource server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateResourceServer.html
     */
    toUpdateResourceServer() {
        return this.to('UpdateResourceServer');
    }
    /**
     * Grants permission to allow a user to update a specific attribute (one at a time)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserAttributes.html
     */
    toUpdateUserAttributes() {
        return this.to('UpdateUserAttributes');
    }
    /**
     * Grants permission to updates the configuration of user pools
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPool.html
     */
    toUpdateUserPool() {
        return this.to('UpdateUserPool');
    }
    /**
     * Grants permission to update any user pool client
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPoolClient.html
     */
    toUpdateUserPoolClient() {
        return this.to('UpdateUserPoolClient');
    }
    /**
     * Grants permission to replace the certificate for any custom domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPoolDomain.html
     */
    toUpdateUserPoolDomain() {
        return this.to('UpdateUserPoolDomain');
    }
    /**
     * Grants permission to register a user's entered TOTP code and mark the user's software token MFA status as verified if successful
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_VerifySoftwareToken.html
     */
    toVerifySoftwareToken() {
        return this.to('VerifySoftwareToken');
    }
    /**
     * Grants permission to verify a user attribute using a one time verification code
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_VerifyUserAttribute.html
     */
    toVerifyUserAttribute() {
        return this.to('VerifyUserAttribute');
    }
    /**
     * Adds a resource of type userpool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html
     *
     * @param userPoolId - Identifier for the userPoolId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUserpool(userPoolId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cognito-idp:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:userpool/${userPoolId}`);
    }
    /**
     * Adds a resource of type webacl to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-waf.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onWebacl(scope, name, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wafv2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${scope}/webacl/${name}/${id}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateUserPool()
     * - .toTagResource()
     * - .toUpdateUserPool()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateUserPool()
     *
     * Applies to resource types:
     * - userpool
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a key that is present in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateUserPool()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateUserPool()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cognito-idp](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoncognitouserpools.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cognito-idp';
        this.accessLevelList = {
            Write: [
                'AddCustomAttributes',
                'AdminAddUserToGroup',
                'AdminConfirmSignUp',
                'AdminCreateUser',
                'AdminDeleteUser',
                'AdminDeleteUserAttributes',
                'AdminDisableProviderForUser',
                'AdminDisableUser',
                'AdminEnableUser',
                'AdminForgetDevice',
                'AdminInitiateAuth',
                'AdminLinkProviderForUser',
                'AdminRemoveUserFromGroup',
                'AdminResetUserPassword',
                'AdminRespondToAuthChallenge',
                'AdminSetUserMFAPreference',
                'AdminSetUserPassword',
                'AdminSetUserSettings',
                'AdminUpdateAuthEventFeedback',
                'AdminUpdateDeviceStatus',
                'AdminUpdateUserAttributes',
                'AdminUserGlobalSignOut',
                'AssociateSoftwareToken',
                'AssociateWebACL',
                'ChangePassword',
                'ConfirmDevice',
                'ConfirmForgotPassword',
                'ConfirmSignUp',
                'CreateGroup',
                'CreateIdentityProvider',
                'CreateManagedLoginBranding',
                'CreateResourceServer',
                'CreateUserImportJob',
                'CreateUserPool',
                'CreateUserPoolClient',
                'CreateUserPoolDomain',
                'DeleteGroup',
                'DeleteIdentityProvider',
                'DeleteManagedLoginBranding',
                'DeleteResourceServer',
                'DeleteUser',
                'DeleteUserAttributes',
                'DeleteUserPool',
                'DeleteUserPoolClient',
                'DeleteUserPoolDomain',
                'DisassociateWebACL',
                'ForgetDevice',
                'ForgotPassword',
                'GetTokensFromRefreshToken',
                'GlobalSignOut',
                'InitiateAuth',
                'ResendConfirmationCode',
                'RespondToAuthChallenge',
                'RevokeToken',
                'SetLogDeliveryConfiguration',
                'SetRiskConfiguration',
                'SetUICustomization',
                'SetUserMFAPreference',
                'SetUserPoolMfaConfig',
                'SetUserSettings',
                'SignUp',
                'StartUserImportJob',
                'StopUserImportJob',
                'UpdateAuthEventFeedback',
                'UpdateDeviceStatus',
                'UpdateGroup',
                'UpdateIdentityProvider',
                'UpdateManagedLoginBranding',
                'UpdateResourceServer',
                'UpdateUserAttributes',
                'UpdateUserPool',
                'UpdateUserPoolClient',
                'UpdateUserPoolDomain',
                'VerifySoftwareToken',
                'VerifyUserAttribute'
            ],
            Read: [
                'AdminGetDevice',
                'AdminGetUser',
                'AdminListUserAuthEvents',
                'DescribeIdentityProvider',
                'DescribeManagedLoginBranding',
                'DescribeManagedLoginBrandingByClient',
                'DescribeResourceServer',
                'DescribeRiskConfiguration',
                'DescribeUserImportJob',
                'DescribeUserPool',
                'DescribeUserPoolClient',
                'DescribeUserPoolDomain',
                'GetCSVHeader',
                'GetDevice',
                'GetGroup',
                'GetIdentityProviderByIdentifier',
                'GetLogDeliveryConfiguration',
                'GetSigningCertificate',
                'GetUICustomization',
                'GetUser',
                'GetUserAttributeVerificationCode',
                'GetUserPoolMfaConfig',
                'GetWebACLForResource'
            ],
            List: [
                'AdminListDevices',
                'AdminListGroupsForUser',
                'ListDevices',
                'ListGroups',
                'ListIdentityProviders',
                'ListResourceServers',
                'ListResourcesForWebACL',
                'ListTagsForResource',
                'ListUserImportJobs',
                'ListUserPoolClients',
                'ListUserPools',
                'ListUsers',
                'ListUsersInGroup'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.CognitoIdp = CognitoIdp;
//# sourceMappingURL=data:application/json;base64,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