"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cases = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cases](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnectcases.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cases extends shared_1.PolicyStatement {
    /**
     * Grants permission to retrieve information about the case rules in the case domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_BatchGetCaseRule.html
     */
    toBatchGetCaseRule() {
        return this.to('BatchGetCaseRule');
    }
    /**
     * Grants permission to retrieve information about the fields in the case domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_BatchGetField.html
     */
    toBatchGetField() {
        return this.to('BatchGetField');
    }
    /**
     * Grants permission to update the field options in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_BatchPutFieldOptions.html
     */
    toBatchPutFieldOptions() {
        return this.to('BatchPutFieldOptions');
    }
    /**
     * Grants permission to create a case in the case domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifConnectUserArn()
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_CreateCase.html
     */
    toCreateCase() {
        return this.to('CreateCase');
    }
    /**
     * Grants permission to create a case rule in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_CreateCaseRule.html
     */
    toCreateCaseRule() {
        return this.to('CreateCaseRule');
    }
    /**
     * Grants permission to create a new case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_CreateDomain.html
     */
    toCreateDomain() {
        return this.to('CreateDomain');
    }
    /**
     * Grants permission to create a field in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_CreateField.html
     */
    toCreateField() {
        return this.to('CreateField');
    }
    /**
     * Grants permission to create a layout in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_CreateLayout.html
     */
    toCreateLayout() {
        return this.to('CreateLayout');
    }
    /**
     * Grants permission to create a related item associated to a case in the case domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifConnectUserArn()
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_CreateRelatedItem.html
     */
    toCreateRelatedItem() {
        return this.to('CreateRelatedItem');
    }
    /**
     * Grants permission to create a template in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_CreateTemplate.html
     */
    toCreateTemplate() {
        return this.to('CreateTemplate');
    }
    /**
     * Grants permission to delete the case in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_DeleteCase.html
     */
    toDeleteCase() {
        return this.to('DeleteCase');
    }
    /**
     * Grants permission to delete the case rule in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_DeleteCaseRule.html
     */
    toDeleteCaseRule() {
        return this.to('DeleteCaseRule');
    }
    /**
     * Grants permission to delete the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        return this.to('DeleteDomain');
    }
    /**
     * Grants permission to delete the field in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_DeleteField.html
     */
    toDeleteField() {
        return this.to('DeleteField');
    }
    /**
     * Grants permission to delete the layout in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_DeleteLayout.html
     */
    toDeleteLayout() {
        return this.to('DeleteLayout');
    }
    /**
     * Grants permission to delete the related item associated to the case in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_DeleteRelatedItem.html
     */
    toDeleteRelatedItem() {
        return this.to('DeleteRelatedItem');
    }
    /**
     * Grants permission to delete the template in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_DeleteTemplate.html
     */
    toDeleteTemplate() {
        return this.to('DeleteTemplate');
    }
    /**
     * Grants permission to retrieve information about a case in the case domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_GetCase.html
     */
    toGetCase() {
        return this.to('GetCase');
    }
    /**
     * Grants permission to view audit history of a case
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_GetCaseAuditEvents.html
     */
    toGetCaseAuditEvents() {
        return this.to('GetCaseAuditEvents');
    }
    /**
     * Grants permission to retrieve information about the case event configuraton in the case domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_GetCaseEventConfiguration.html
     */
    toGetCaseEventConfiguration() {
        return this.to('GetCaseEventConfiguration');
    }
    /**
     * Grants permission to retrieve information about the case domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_GetDomain.html
     */
    toGetDomain() {
        return this.to('GetDomain');
    }
    /**
     * Grants permission to retrieve information about the layout in the case domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_GetLayout.html
     */
    toGetLayout() {
        return this.to('GetLayout');
    }
    /**
     * Grants permission to retrieve information about the template in the case domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_GetTemplate.html
     */
    toGetTemplate() {
        return this.to('GetTemplate');
    }
    /**
     * Grants permission to list case rules in the case domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_ListCaseRules.html
     */
    toListCaseRules() {
        return this.to('ListCaseRules');
    }
    /**
     * Grants permission to list cases for a specific contact in the case domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_ListCasesForContact.html
     */
    toListCasesForContact() {
        return this.to('ListCasesForContact');
    }
    /**
     * Grants permission to list all domains in the aws account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_ListDomains.html
     */
    toListDomains() {
        return this.to('ListDomains');
    }
    /**
     * Grants permission to list field options for a single select field in the case domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_ListFieldOptions.html
     */
    toListFieldOptions() {
        return this.to('ListFieldOptions');
    }
    /**
     * Grants permission to list fields in the case domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_ListFields.html
     */
    toListFields() {
        return this.to('ListFields');
    }
    /**
     * Grants permission to list layouts in the case domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_ListLayouts.html
     */
    toListLayouts() {
        return this.to('ListLayouts');
    }
    /**
     * Grants permission to list the tags for the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list templates in the case domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_ListTemplates.html
     */
    toListTemplates() {
        return this.to('ListTemplates');
    }
    /**
     * Grants permission to insert or update the case event configuration in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_PutCaseEventConfiguration.html
     */
    toPutCaseEventConfiguration() {
        return this.to('PutCaseEventConfiguration');
    }
    /**
     * Grants permission to search for cases in the case domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_SearchCases.html
     */
    toSearchCases() {
        return this.to('SearchCases');
    }
    /**
     * Grants permission to search for related items associated to the case in the case domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_SearchRelatedItems.html
     */
    toSearchRelatedItems() {
        return this.to('SearchRelatedItems');
    }
    /**
     * Grants permission to add the specified tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the specified tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the field values on the case in the case domain
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifConnectUserArn()
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_UpdateCase.html
     */
    toUpdateCase() {
        return this.to('UpdateCase');
    }
    /**
     * Grants permission to update the case rule in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_UpdateCaseRule.html
     */
    toUpdateCaseRule() {
        return this.to('UpdateCaseRule');
    }
    /**
     * Grants permission to update the field in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_UpdateField.html
     */
    toUpdateField() {
        return this.to('UpdateField');
    }
    /**
     * Grants permission to update the layout in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_UpdateLayout.html
     */
    toUpdateLayout() {
        return this.to('UpdateLayout');
    }
    /**
     * Grants permission to update the template in the case domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cases/latest/APIReference/API_UpdateTemplate.html
     */
    toUpdateTemplate() {
        return this.to('UpdateTemplate');
    }
    /**
     * Adds a resource of type Case to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/cases.html
     *
     * @param domainId - Identifier for the domainId.
     * @param caseId - Identifier for the caseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCase(domainId, caseId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cases:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domain/${domainId}/case/${caseId}`);
    }
    /**
     * Adds a resource of type Domain to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/cases.html
     *
     * @param domainId - Identifier for the domainId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomain(domainId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cases:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domain/${domainId}`);
    }
    /**
     * Adds a resource of type Field to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/case-fields.html
     *
     * @param domainId - Identifier for the domainId.
     * @param fieldId - Identifier for the fieldId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onField(domainId, fieldId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cases:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domain/${domainId}/field/${fieldId}`);
    }
    /**
     * Adds a resource of type Layout to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/case-layouts.html
     *
     * @param domainId - Identifier for the domainId.
     * @param layoutId - Identifier for the layoutId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLayout(domainId, layoutId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cases:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domain/${domainId}/layout/${layoutId}`);
    }
    /**
     * Adds a resource of type RelatedItem to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/associatecontactandcase.html
     *
     * @param domainId - Identifier for the domainId.
     * @param caseId - Identifier for the caseId.
     * @param relatedItemId - Identifier for the relatedItemId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRelatedItem(domainId, caseId, relatedItemId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cases:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domain/${domainId}/case/${caseId}/related-item/${relatedItemId}`);
    }
    /**
     * Adds a resource of type Template to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/case-templates.html
     *
     * @param domainId - Identifier for the domainId.
     * @param templateId - Identifier for the templateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTemplate(domainId, templateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cases:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domain/${domainId}/template/${templateId}`);
    }
    /**
     * Adds a resource of type CaseRule to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/case-rules.html
     *
     * @param domainId - Identifier for the domainId.
     * @param caseRuleId - Identifier for the caseRuleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCaseRule(domainId, caseRuleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cases:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domain/${domainId}/case-rule/${caseRuleId}`);
    }
    /**
     * Filters access by tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - Case
     * - Domain
     * - Field
     * - Layout
     * - RelatedItem
     * - Template
     * - CaseRule
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by connect's UserArn
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_User.html
     *
     * Applies to actions:
     * - .toCreateCase()
     * - .toCreateRelatedItem()
     * - .toUpdateCase()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifConnectUserArn(value, operator) {
        return this.if(`connect:UserArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [cases](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnectcases.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cases';
        this.accessLevelList = {
            Read: [
                'BatchGetCaseRule',
                'BatchGetField',
                'GetCase',
                'GetCaseAuditEvents',
                'GetCaseEventConfiguration',
                'GetDomain',
                'GetLayout',
                'GetTemplate',
                'ListTagsForResource',
                'SearchCases',
                'SearchRelatedItems'
            ],
            Write: [
                'BatchPutFieldOptions',
                'CreateCase',
                'CreateCaseRule',
                'CreateDomain',
                'CreateField',
                'CreateLayout',
                'CreateRelatedItem',
                'CreateTemplate',
                'DeleteCase',
                'DeleteCaseRule',
                'DeleteDomain',
                'DeleteField',
                'DeleteLayout',
                'DeleteRelatedItem',
                'DeleteTemplate',
                'PutCaseEventConfiguration',
                'UpdateCase',
                'UpdateCaseRule',
                'UpdateField',
                'UpdateLayout',
                'UpdateTemplate'
            ],
            List: [
                'ListCaseRules',
                'ListCasesForContact',
                'ListDomains',
                'ListFieldOptions',
                'ListFields',
                'ListLayouts',
                'ListTemplates'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Cases = Cases;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29ubmVjdGNhc2VzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY29ubmVjdGNhc2VzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxLQUFNLFNBQVEsd0JBQWU7SUFHeEM7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFNBQVM7UUFDZCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDNUIsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQXNERDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksTUFBTSxDQUFDLFFBQWdCLEVBQUUsTUFBYyxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ25HLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFVBQVcsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFdBQVksUUFBUyxTQUFVLE1BQU8sRUFBRSxDQUFDLENBQUM7SUFDbEwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFFBQVEsQ0FBQyxRQUFnQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3JGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFVBQVcsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFdBQVksUUFBUyxFQUFFLENBQUMsQ0FBQztJQUNqSyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLE9BQU8sQ0FBQyxRQUFnQixFQUFFLE9BQWUsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNyRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixVQUFXLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxXQUFZLFFBQVMsVUFBVyxPQUFRLEVBQUUsQ0FBQyxDQUFDO0lBQ3BMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksUUFBUSxDQUFDLFFBQWdCLEVBQUUsUUFBZ0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN2RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixVQUFXLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxXQUFZLFFBQVMsV0FBWSxRQUFTLEVBQUUsQ0FBQyxDQUFDO0lBQ3RMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLGFBQWEsQ0FBQyxRQUFnQixFQUFFLE1BQWMsRUFBRSxhQUFxQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2pJLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFVBQVcsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFdBQVksUUFBUyxTQUFVLE1BQU8saUJBQWtCLGFBQWMsRUFBRSxDQUFDLENBQUM7SUFDbE4sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxVQUFVLENBQUMsUUFBZ0IsRUFBRSxVQUFrQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFVBQVcsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFdBQVksUUFBUyxhQUFjLFVBQVcsRUFBRSxDQUFDLENBQUM7SUFDMUwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxVQUFVLENBQUMsUUFBZ0IsRUFBRSxVQUFrQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLFVBQVcsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFdBQVksUUFBUyxjQUFlLFVBQVcsRUFBRSxDQUFDLENBQUM7SUFDM0wsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksZUFBZSxDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBbUIsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNoRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksZ0JBQWdCLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDNUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFvQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksZ0JBQWdCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxTQUFTLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFudEJSLGtCQUFhLEdBQUcsT0FBTyxDQUFDO1FBcWRyQixvQkFBZSxHQUFvQjtZQUMzQyxJQUFJLEVBQUU7Z0JBQ0osa0JBQWtCO2dCQUNsQixlQUFlO2dCQUNmLFNBQVM7Z0JBQ1Qsb0JBQW9CO2dCQUNwQiwyQkFBMkI7Z0JBQzNCLFdBQVc7Z0JBQ1gsV0FBVztnQkFDWCxhQUFhO2dCQUNiLHFCQUFxQjtnQkFDckIsYUFBYTtnQkFDYixvQkFBb0I7YUFDckI7WUFDRCxLQUFLLEVBQUU7Z0JBQ0wsc0JBQXNCO2dCQUN0QixZQUFZO2dCQUNaLGdCQUFnQjtnQkFDaEIsY0FBYztnQkFDZCxhQUFhO2dCQUNiLGNBQWM7Z0JBQ2QsbUJBQW1CO2dCQUNuQixnQkFBZ0I7Z0JBQ2hCLFlBQVk7Z0JBQ1osZ0JBQWdCO2dCQUNoQixjQUFjO2dCQUNkLGFBQWE7Z0JBQ2IsY0FBYztnQkFDZCxtQkFBbUI7Z0JBQ25CLGdCQUFnQjtnQkFDaEIsMkJBQTJCO2dCQUMzQixZQUFZO2dCQUNaLGdCQUFnQjtnQkFDaEIsYUFBYTtnQkFDYixjQUFjO2dCQUNkLGdCQUFnQjthQUNqQjtZQUNELElBQUksRUFBRTtnQkFDSixlQUFlO2dCQUNmLHFCQUFxQjtnQkFDckIsYUFBYTtnQkFDYixrQkFBa0I7Z0JBQ2xCLFlBQVk7Z0JBQ1osYUFBYTtnQkFDYixlQUFlO2FBQ2hCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLGFBQWE7Z0JBQ2IsZUFBZTthQUNoQjtTQUNGLENBQUM7SUE2TUYsQ0FBQztDQUNGO0FBdHRCRCxzQkFzdEJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtjYXNlc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uY29ubmVjdGNhc2VzLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgQ2FzZXMgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdjYXNlcyc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IHRoZSBjYXNlIHJ1bGVzIGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0JhdGNoR2V0Q2FzZVJ1bGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQmF0Y2hHZXRDYXNlUnVsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hHZXRDYXNlUnVsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IHRoZSBmaWVsZHMgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQmF0Y2hHZXRGaWVsZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaEdldEZpZWxkKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdCYXRjaEdldEZpZWxkJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBmaWVsZCBvcHRpb25zIGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9CYXRjaFB1dEZpZWxkT3B0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaFB1dEZpZWxkT3B0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hQdXRGaWVsZE9wdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBjYXNlIGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQ29ubmVjdFVzZXJBcm4oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQ2FzZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDYXNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVDYXNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgY2FzZSBydWxlIGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVDYXNlUnVsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDYXNlUnVsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQ2FzZVJ1bGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXcgY2FzZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlRG9tYWluLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZURvbWFpbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlRG9tYWluJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgZmllbGQgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUZpZWxkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUZpZWxkKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVGaWVsZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGxheW91dCBpbiB0aGUgY2FzZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlTGF5b3V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUxheW91dCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlTGF5b3V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgcmVsYXRlZCBpdGVtIGFzc29jaWF0ZWQgdG8gYSBjYXNlIGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQ29ubmVjdFVzZXJBcm4oKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlUmVsYXRlZEl0ZW0uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUmVsYXRlZEl0ZW0oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVJlbGF0ZWRJdGVtJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgdGVtcGxhdGUgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVRlbXBsYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVUZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgY2FzZSBpbiB0aGUgY2FzZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQ2FzZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVDYXNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVDYXNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBjYXNlIHJ1bGUgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUNhc2VSdWxlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUNhc2VSdWxlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVDYXNlUnVsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZURvbWFpbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEb21haW4oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZURvbWFpbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgZmllbGQgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUZpZWxkLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUZpZWxkKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVGaWVsZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgbGF5b3V0IGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVMYXlvdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlTGF5b3V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVMYXlvdXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIHJlbGF0ZWQgaXRlbSBhc3NvY2lhdGVkIHRvIHRoZSBjYXNlIGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVSZWxhdGVkSXRlbS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVSZWxhdGVkSXRlbSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlUmVsYXRlZEl0ZW0nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgdGhlIHRlbXBsYXRlIGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVUZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVUZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlVGVtcGxhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCBhIGNhc2UgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0Q2FzZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRDYXNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRDYXNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyBhdWRpdCBoaXN0b3J5IG9mIGEgY2FzZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldENhc2VBdWRpdEV2ZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRDYXNlQXVkaXRFdmVudHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldENhc2VBdWRpdEV2ZW50cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IHRoZSBjYXNlIGV2ZW50IGNvbmZpZ3VyYXRvbiBpbiB0aGUgY2FzZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRDYXNlRXZlbnRDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldENhc2VFdmVudENvbmZpZ3VyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldENhc2VFdmVudENvbmZpZ3VyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgY2FzZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXREb21haW4uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RG9tYWluKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXREb21haW4nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgbGF5b3V0IGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldExheW91dC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRMYXlvdXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldExheW91dCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGFib3V0IHRoZSB0ZW1wbGF0ZSBpbiB0aGUgY2FzZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRUZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRUZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0VGVtcGxhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGNhc2UgcnVsZXMgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdENhc2VSdWxlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0Q2FzZVJ1bGVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0Q2FzZVJ1bGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBjYXNlcyBmb3IgYSBzcGVjaWZpYyBjb250YWN0IGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RDYXNlc0ZvckNvbnRhY3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdENhc2VzRm9yQ29udGFjdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdENhc2VzRm9yQ29udGFjdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgYWxsIGRvbWFpbnMgaW4gdGhlIGF3cyBhY2NvdW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdERvbWFpbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdERvbWFpbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3REb21haW5zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBmaWVsZCBvcHRpb25zIGZvciBhIHNpbmdsZSBzZWxlY3QgZmllbGQgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEZpZWxkT3B0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RmllbGRPcHRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RmllbGRPcHRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBmaWVsZHMgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEZpZWxkcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RmllbGRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0RmllbGRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBsYXlvdXRzIGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RMYXlvdXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RMYXlvdXRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0TGF5b3V0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgdGhlIHRhZ3MgZm9yIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRlbXBsYXRlcyBpbiB0aGUgY2FzZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGVtcGxhdGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUZW1wbGF0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUZW1wbGF0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBpbnNlcnQgb3IgdXBkYXRlIHRoZSBjYXNlIGV2ZW50IGNvbmZpZ3VyYXRpb24gaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dENhc2VFdmVudENvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0Q2FzZUV2ZW50Q29uZmlndXJhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0Q2FzZUV2ZW50Q29uZmlndXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHNlYXJjaCBmb3IgY2FzZXMgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2VhcmNoQ2FzZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU2VhcmNoQ2FzZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1NlYXJjaENhc2VzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc2VhcmNoIGZvciByZWxhdGVkIGl0ZW1zIGFzc29jaWF0ZWQgdG8gdGhlIGNhc2UgaW4gdGhlIGNhc2UgZG9tYWluXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU2VhcmNoUmVsYXRlZEl0ZW1zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1NlYXJjaFJlbGF0ZWRJdGVtcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU2VhcmNoUmVsYXRlZEl0ZW1zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYWRkIHRoZSBzcGVjaWZpZWQgdGFncyB0byB0aGUgc3BlY2lmaWVkIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1RhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVtb3ZlIHRoZSBzcGVjaWZpZWQgdGFncyBmcm9tIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VudGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGZpZWxkIHZhbHVlcyBvbiB0aGUgY2FzZSBpbiB0aGUgY2FzZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkNvbm5lY3RVc2VyQXJuKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nhc2VzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUNhc2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQ2FzZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlQ2FzZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgY2FzZSBydWxlIGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVDYXNlUnVsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDYXNlUnVsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlQ2FzZVJ1bGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGZpZWxkIGluIHRoZSBjYXNlIGRvbWFpblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jYXNlcy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVGaWVsZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVGaWVsZCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlRmllbGQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIGxheW91dCBpbiB0aGUgY2FzZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlTGF5b3V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUxheW91dCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlTGF5b3V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSB0ZW1wbGF0ZSBpbiB0aGUgY2FzZSBkb21haW5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2FzZXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVRlbXBsYXRlJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgUmVhZDogW1xuICAgICAgJ0JhdGNoR2V0Q2FzZVJ1bGUnLFxuICAgICAgJ0JhdGNoR2V0RmllbGQnLFxuICAgICAgJ0dldENhc2UnLFxuICAgICAgJ0dldENhc2VBdWRpdEV2ZW50cycsXG4gICAgICAnR2V0Q2FzZUV2ZW50Q29uZmlndXJhdGlvbicsXG4gICAgICAnR2V0RG9tYWluJyxcbiAgICAgICdHZXRMYXlvdXQnLFxuICAgICAgJ0dldFRlbXBsYXRlJyxcbiAgICAgICdMaXN0VGFnc0ZvclJlc291cmNlJyxcbiAgICAgICdTZWFyY2hDYXNlcycsXG4gICAgICAnU2VhcmNoUmVsYXRlZEl0ZW1zJ1xuICAgIF0sXG4gICAgV3JpdGU6IFtcbiAgICAgICdCYXRjaFB1dEZpZWxkT3B0aW9ucycsXG4gICAgICAnQ3JlYXRlQ2FzZScsXG4gICAgICAnQ3JlYXRlQ2FzZVJ1bGUnLFxuICAgICAgJ0NyZWF0ZURvbWFpbicsXG4gICAgICAnQ3JlYXRlRmllbGQnLFxuICAgICAgJ0NyZWF0ZUxheW91dCcsXG4gICAgICAnQ3JlYXRlUmVsYXRlZEl0ZW0nLFxuICAgICAgJ0NyZWF0ZVRlbXBsYXRlJyxcbiAgICAgICdEZWxldGVDYXNlJyxcbiAgICAgICdEZWxldGVDYXNlUnVsZScsXG4gICAgICAnRGVsZXRlRG9tYWluJyxcbiAgICAgICdEZWxldGVGaWVsZCcsXG4gICAgICAnRGVsZXRlTGF5b3V0JyxcbiAgICAgICdEZWxldGVSZWxhdGVkSXRlbScsXG4gICAgICAnRGVsZXRlVGVtcGxhdGUnLFxuICAgICAgJ1B1dENhc2VFdmVudENvbmZpZ3VyYXRpb24nLFxuICAgICAgJ1VwZGF0ZUNhc2UnLFxuICAgICAgJ1VwZGF0ZUNhc2VSdWxlJyxcbiAgICAgICdVcGRhdGVGaWVsZCcsXG4gICAgICAnVXBkYXRlTGF5b3V0JyxcbiAgICAgICdVcGRhdGVUZW1wbGF0ZSdcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0Q2FzZVJ1bGVzJyxcbiAgICAgICdMaXN0Q2FzZXNGb3JDb250YWN0JyxcbiAgICAgICdMaXN0RG9tYWlucycsXG4gICAgICAnTGlzdEZpZWxkT3B0aW9ucycsXG4gICAgICAnTGlzdEZpZWxkcycsXG4gICAgICAnTGlzdExheW91dHMnLFxuICAgICAgJ0xpc3RUZW1wbGF0ZXMnXG4gICAgXSxcbiAgICBUYWdnaW5nOiBbXG4gICAgICAnVGFnUmVzb3VyY2UnLFxuICAgICAgJ1VudGFnUmVzb3VyY2UnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBDYXNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nvbm5lY3QvbGF0ZXN0L2FkbWluZ3VpZGUvY2FzZXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluSWQuXG4gICAqIEBwYXJhbSBjYXNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgY2FzZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25DYXNlKGRvbWFpbklkOiBzdHJpbmcsIGNhc2VJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNhc2VzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpkb21haW4vJHsgZG9tYWluSWQgfS9jYXNlLyR7IGNhc2VJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgRG9tYWluIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nvbm5lY3QvbGF0ZXN0L2FkbWluZ3VpZGUvY2FzZXMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkRvbWFpbihkb21haW5JZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNhc2VzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpkb21haW4vJHsgZG9tYWluSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIEZpZWxkIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nvbm5lY3QvbGF0ZXN0L2FkbWluZ3VpZGUvY2FzZS1maWVsZHMuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluSWQuXG4gICAqIEBwYXJhbSBmaWVsZElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGZpZWxkSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkZpZWxkKGRvbWFpbklkOiBzdHJpbmcsIGZpZWxkSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpjYXNlczokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZG9tYWluLyR7IGRvbWFpbklkIH0vZmllbGQvJHsgZmllbGRJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgTGF5b3V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nvbm5lY3QvbGF0ZXN0L2FkbWluZ3VpZGUvY2FzZS1sYXlvdXRzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRvbWFpbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbklkLlxuICAgKiBAcGFyYW0gbGF5b3V0SWQgLSBJZGVudGlmaWVyIGZvciB0aGUgbGF5b3V0SWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkxheW91dChkb21haW5JZDogc3RyaW5nLCBsYXlvdXRJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNhc2VzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpkb21haW4vJHsgZG9tYWluSWQgfS9sYXlvdXQvJHsgbGF5b3V0SWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFJlbGF0ZWRJdGVtIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nvbm5lY3QvbGF0ZXN0L2FkbWluZ3VpZGUvYXNzb2NpYXRlY29udGFjdGFuZGNhc2UuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZG9tYWluSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZG9tYWluSWQuXG4gICAqIEBwYXJhbSBjYXNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgY2FzZUlkLlxuICAgKiBAcGFyYW0gcmVsYXRlZEl0ZW1JZCAtIElkZW50aWZpZXIgZm9yIHRoZSByZWxhdGVkSXRlbUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SZWxhdGVkSXRlbShkb21haW5JZDogc3RyaW5nLCBjYXNlSWQ6IHN0cmluZywgcmVsYXRlZEl0ZW1JZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNhc2VzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpkb21haW4vJHsgZG9tYWluSWQgfS9jYXNlLyR7IGNhc2VJZCB9L3JlbGF0ZWQtaXRlbS8keyByZWxhdGVkSXRlbUlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBUZW1wbGF0ZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb25uZWN0L2xhdGVzdC9hZG1pbmd1aWRlL2Nhc2UtdGVtcGxhdGVzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRvbWFpbklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRvbWFpbklkLlxuICAgKiBAcGFyYW0gdGVtcGxhdGVJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB0ZW1wbGF0ZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25UZW1wbGF0ZShkb21haW5JZDogc3RyaW5nLCB0ZW1wbGF0ZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06Y2FzZXM6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmRvbWFpbi8keyBkb21haW5JZCB9L3RlbXBsYXRlLyR7IHRlbXBsYXRlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIENhc2VSdWxlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nvbm5lY3QvbGF0ZXN0L2FkbWluZ3VpZGUvY2FzZS1ydWxlcy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBkb21haW5JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBkb21haW5JZC5cbiAgICogQHBhcmFtIGNhc2VSdWxlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgY2FzZVJ1bGVJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ2FzZVJ1bGUoZG9tYWluSWQ6IHN0cmluZywgY2FzZVJ1bGVJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmNhc2VzOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpkb21haW4vJHsgZG9tYWluSWQgfS9jYXNlLXJ1bGUvJHsgY2FzZVJ1bGVJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGFncyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRhZ3MgYXNzb2NpYXRlZCB3aXRoIHRoZSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVzb3VyY2V0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBDYXNlXG4gICAqIC0gRG9tYWluXG4gICAqIC0gRmllbGRcbiAgICogLSBMYXlvdXRcbiAgICogLSBSZWxhdGVkSXRlbVxuICAgKiAtIFRlbXBsYXRlXG4gICAqIC0gQ2FzZVJ1bGVcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGFnIGtleXMgdGhhdCBhcmUgcGFzc2VkIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy10YWdrZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqIC0gLnRvVW50YWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQXdzVGFnS2V5cyh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlRhZ0tleXNgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBjb25uZWN0J3MgVXNlckFyblxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29ubmVjdC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9Vc2VyLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUNhc2UoKVxuICAgKiAtIC50b0NyZWF0ZVJlbGF0ZWRJdGVtKClcbiAgICogLSAudG9VcGRhdGVDYXNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkxpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZDb25uZWN0VXNlckFybih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgY29ubmVjdDpVc2VyQXJuYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdBcm5MaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtjYXNlc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9uY29ubmVjdGNhc2VzLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==