"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Profile = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [profile](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnectcustomerprofiles.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Profile extends shared_1.PolicyStatement {
    /**
     * Grants permission to add a profile key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_AddProfileKey.html
     */
    toAddProfileKey() {
        return this.to('AddProfileKey');
    }
    /**
     * Grants permission to retrieve a calculated attribute for the specific profiles in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_BatchGetCalculatedAttributeForProfile.html
     */
    toBatchGetCalculatedAttributeForProfile() {
        return this.to('BatchGetCalculatedAttributeForProfile');
    }
    /**
     * Grants permission to get profiles in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_BatchGetProfile.html
     */
    toBatchGetProfile() {
        return this.to('BatchGetProfile');
    }
    /**
     * Grants permission to create a calculated attribute definition in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateCalculatedAttributeDefinition.html
     */
    toCreateCalculatedAttributeDefinition() {
        return this.to('CreateCalculatedAttributeDefinition');
    }
    /**
     * Grants permission to create a Domain
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateDomain.html
     */
    toCreateDomain() {
        return this.to('CreateDomain');
    }
    /**
     * Grants permission to create a layout in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateDomainLayout.html
     */
    toCreateDomainLayout() {
        return this.to('CreateDomainLayout');
    }
    /**
     * Grants permission to put an event stream in a domain
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PutRolePolicy
     * - kinesis:DescribeStreamSummary
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateEventStream.html
     */
    toCreateEventStream() {
        return this.to('CreateEventStream');
    }
    /**
     * Grants permission to create an event trigger in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateEventTrigger.html
     */
    toCreateEventTrigger() {
        return this.to('CreateEventTrigger');
    }
    /**
     * Grants permission to create an integration workflow in a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateIntegrationWorkflow.html
     */
    toCreateIntegrationWorkflow() {
        return this.to('CreateIntegrationWorkflow');
    }
    /**
     * Grants permission to create a profile in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateProfile.html
     */
    toCreateProfile() {
        return this.to('CreateProfile');
    }
    /**
     * Grants permission to create a segment definition in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateSegmentDefinition.html
     */
    toCreateSegmentDefinition() {
        return this.to('CreateSegmentDefinition');
    }
    /**
     * Grants permission to create a segment estimate in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateSegmentEstimate.html
     */
    toCreateSegmentEstimate() {
        return this.to('CreateSegmentEstimate');
    }
    /**
     * Grants permission to create a segment snapshot in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateSegmentSnapshot.html
     */
    toCreateSegmentSnapshot() {
        return this.to('CreateSegmentSnapshot');
    }
    /**
     * Grants permission to create a snapshot in the domain
     *
     * Access Level: Write
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permission to create an upload job in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_CreateUploadJob.html
     */
    toCreateUploadJob() {
        return this.to('CreateUploadJob');
    }
    /**
     * Grants permission to delete a calculated attribute definition in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteCalculatedAttributeDefinition.html
     */
    toDeleteCalculatedAttributeDefinition() {
        return this.to('DeleteCalculatedAttributeDefinition');
    }
    /**
     * Grants permission to delete a Domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteDomain.html
     */
    toDeleteDomain() {
        return this.to('DeleteDomain');
    }
    /**
     * Grants permission to delete a layout in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteDomainLayout.html
     */
    toDeleteDomainLayout() {
        return this.to('DeleteDomainLayout');
    }
    /**
     * Grants permission to delete an event stream in a domain
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:DeleteRolePolicy
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteEventStream.html
     */
    toDeleteEventStream() {
        return this.to('DeleteEventStream');
    }
    /**
     * Grants permission to delete an event trigger in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteEventTrigger.html
     */
    toDeleteEventTrigger() {
        return this.to('DeleteEventTrigger');
    }
    /**
     * Grants permission to delete a integration in a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteIntegration.html
     */
    toDeleteIntegration() {
        return this.to('DeleteIntegration');
    }
    /**
     * Grants permission to delete a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteProfile.html
     */
    toDeleteProfile() {
        return this.to('DeleteProfile');
    }
    /**
     * Grants permission to delete a profile key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteProfileKey.html
     */
    toDeleteProfileKey() {
        return this.to('DeleteProfileKey');
    }
    /**
     * Grants permission to delete a profile object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteProfileObject.html
     */
    toDeleteProfileObject() {
        return this.to('DeleteProfileObject');
    }
    /**
     * Grants permission to delete a specific profile object type in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteProfileObjectType.html
     */
    toDeleteProfileObjectType() {
        return this.to('DeleteProfileObjectType');
    }
    /**
     * Grants permission to delete a segment definition in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteSegmentDefinition.html
     */
    toDeleteSegmentDefinition() {
        return this.to('DeleteSegmentDefinition');
    }
    /**
     * Grants permission to delete a workflow in a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DeleteWorkflow.html
     */
    toDeleteWorkflow() {
        return this.to('DeleteWorkflow');
    }
    /**
     * Grants permission to auto detect object type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_DetectProfileObjectType.html
     */
    toDetectProfileObjectType() {
        return this.to('DetectProfileObjectType');
    }
    /**
     * Grants permission to get a preview of auto merging in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetAutoMergingPreview.html
     */
    toGetAutoMergingPreview() {
        return this.to('GetAutoMergingPreview');
    }
    /**
     * Grants permission to get a calculated attribute definition in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetCalculatedAttributeDefinition.html
     */
    toGetCalculatedAttributeDefinition() {
        return this.to('GetCalculatedAttributeDefinition');
    }
    /**
     * Grants permission to retrieve a calculated attribute for a specific profile in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetCalculatedAttributeForProfile.html
     */
    toGetCalculatedAttributeForProfile() {
        return this.to('GetCalculatedAttributeForProfile');
    }
    /**
     * Grants permission to get a specific domain in an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetDomain.html
     */
    toGetDomain() {
        return this.to('GetDomain');
    }
    /**
     * Grants permission to get a layout in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetDomainLayout.html
     */
    toGetDomainLayout() {
        return this.to('GetDomainLayout');
    }
    /**
     * Grants permission to get a specific event stream in a domain
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kinesis:DescribeStreamSummary
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetEventStream.html
     */
    toGetEventStream() {
        return this.to('GetEventStream');
    }
    /**
     * Grants permission to get an event trigger in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetEventTrigger.html
     */
    toGetEventTrigger() {
        return this.to('GetEventTrigger');
    }
    /**
     * Grants permission to get an identity resolution job in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetIdentityResolutionJob.html
     */
    toGetIdentityResolutionJob() {
        return this.to('GetIdentityResolutionJob');
    }
    /**
     * Grants permission to get a specific integrations in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetIntegration.html
     */
    toGetIntegration() {
        return this.to('GetIntegration');
    }
    /**
     * Grants permission to get profile matches in a domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetMatches.html
     */
    toGetMatches() {
        return this.to('GetMatches');
    }
    /**
     * Grants permission to get a specific profile object type in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetProfileObjectType.html
     */
    toGetProfileObjectType() {
        return this.to('GetProfileObjectType');
    }
    /**
     * Grants permission to get a specific object type template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetProfileObjectTypeTemplate.html
     */
    toGetProfileObjectTypeTemplate() {
        return this.to('GetProfileObjectTypeTemplate');
    }
    /**
     * Grants permission to get a segment definition in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetSegmentDefinition.html
     */
    toGetSegmentDefinition() {
        return this.to('GetSegmentDefinition');
    }
    /**
     * Grants permission to get a segment estimate in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetSegmentEstimate.html
     */
    toGetSegmentEstimate() {
        return this.to('GetSegmentEstimate');
    }
    /**
     * Grants permission to determine if the given profiles are part of a segment in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetSegmentMembership.html
     */
    toGetSegmentMembership() {
        return this.to('GetSegmentMembership');
    }
    /**
     * Grants permission to get a segment snapshot in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetSegmentSnapshot.html
     */
    toGetSegmentSnapshot() {
        return this.to('GetSegmentSnapshot');
    }
    /**
     * Grants permission to get all the similar profiles in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetSimilarProfiles.html
     */
    toGetSimilarProfiles() {
        return this.to('GetSimilarProfiles');
    }
    /**
     * Grants permission to get a snapshot in the domain
     *
     * Access Level: Read
     */
    toGetSnapshot() {
        return this.to('GetSnapshot');
    }
    /**
     * Grants permission to get details of an upload job in the domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetUploadJob.html
     */
    toGetUploadJob() {
        return this.to('GetUploadJob');
    }
    /**
     * Grants permission to get a pre-signed URL to upload file for an upload job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetUploadJobPath.html
     */
    toGetUploadJobPath() {
        return this.to('GetUploadJobPath');
    }
    /**
     * Grants permission to get workflow details in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetWorkflow.html
     */
    toGetWorkflow() {
        return this.to('GetWorkflow');
    }
    /**
     * Grants permission to get workflow step details in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_GetWorkflowSteps.html
     */
    toGetWorkflowSteps() {
        return this.to('GetWorkflowSteps');
    }
    /**
     * Grants permission to list all the integrations in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListAccountIntegrations.html
     */
    toListAccountIntegrations() {
        return this.to('ListAccountIntegrations');
    }
    /**
     * Grants permission to list all the calculated attribute definitions in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListCalculatedAttributeDefinitions.html
     */
    toListCalculatedAttributeDefinitions() {
        return this.to('ListCalculatedAttributeDefinitions');
    }
    /**
     * Grants permission to list all calculated attributes for a specific profile in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListCalculatedAttributesForProfile.html
     */
    toListCalculatedAttributesForProfile() {
        return this.to('ListCalculatedAttributesForProfile');
    }
    /**
     * Grants permission to list all the layouts in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListDomainLayouts.html
     */
    toListDomainLayouts() {
        return this.to('ListDomainLayouts');
    }
    /**
     * Grants permission to list all the domains in an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListDomains.html
     */
    toListDomains() {
        return this.to('ListDomains');
    }
    /**
     * Grants permission to list all the event streams in a specific domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListEventStreams.html
     */
    toListEventStreams() {
        return this.to('ListEventStreams');
    }
    /**
     * Grants permission to list all the event triggers in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListEventTriggers.html
     */
    toListEventTriggers() {
        return this.to('ListEventTriggers');
    }
    /**
     * Grants permission to list identity resolution jobs in a domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListIdentityResolutionJobs.html
     */
    toListIdentityResolutionJobs() {
        return this.to('ListIdentityResolutionJobs');
    }
    /**
     * Grants permission to list all the integrations in a specific domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListIntegrations.html
     */
    toListIntegrations() {
        return this.to('ListIntegrations');
    }
    /**
     * Grants permission to list all the attributes of a specific object type in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListObjectTypeAttributes.html
     */
    toListObjectTypeAttributes() {
        return this.to('ListObjectTypeAttributes');
    }
    /**
     * Grants permission to list all the values of a profile attribute in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListProfileAttributeValues.html
     */
    toListProfileAttributeValues() {
        return this.to('ListProfileAttributeValues');
    }
    /**
     * Grants permission to list all the profile object type templates in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListProfileObjectTypeTemplates.html
     */
    toListProfileObjectTypeTemplates() {
        return this.to('ListProfileObjectTypeTemplates');
    }
    /**
     * Grants permission to list all the profile object types in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListProfileObjectTypes.html
     */
    toListProfileObjectTypes() {
        return this.to('ListProfileObjectTypes');
    }
    /**
     * Grants permission to list all the profile objects for a profile
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListProfileObjects.html
     */
    toListProfileObjects() {
        return this.to('ListProfileObjects');
    }
    /**
     * Grants permission to list all the rule-based matching result in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListRuleBasedMatches.html
     */
    toListRuleBasedMatches() {
        return this.to('ListRuleBasedMatches');
    }
    /**
     * Grants permission to list all the segment definitions in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListSegmentDefinitions.html
     */
    toListSegmentDefinitions() {
        return this.to('ListSegmentDefinitions');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all upload jobs in the domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListUploadJobs.html
     */
    toListUploadJobs() {
        return this.to('ListUploadJobs');
    }
    /**
     * Grants permission to list all the workflows in a specific domain
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_ListWorkflows.html
     */
    toListWorkflows() {
        return this.to('ListWorkflows');
    }
    /**
     * Grants permission to merge profiles in a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_MergeProfiles.html
     */
    toMergeProfiles() {
        return this.to('MergeProfiles');
    }
    /**
     * Grants permission to put a integration in a domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_PutIntegration.html
     */
    toPutIntegration() {
        return this.to('PutIntegration');
    }
    /**
     * Grants permission to put an object for a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_PutProfileObject.html
     */
    toPutProfileObject() {
        return this.to('PutProfileObject');
    }
    /**
     * Grants permission to put a specific profile object type in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_PutProfileObjectType.html
     */
    toPutProfileObjectType() {
        return this.to('PutProfileObjectType');
    }
    /**
     * Grants permission to search for profiles in a domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_SearchProfiles.html
     */
    toSearchProfiles() {
        return this.to('SearchProfiles');
    }
    /**
     * Grants permission to start an upload job in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_StartUploadJob.html
     */
    toStartUploadJob() {
        return this.to('StartUploadJob');
    }
    /**
     * Grants permission to stop an upload job in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_StopUploadJob.html
     */
    toStopUploadJob() {
        return this.to('StopUploadJob');
    }
    /**
     * Grants permission to adds tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a calculated attribute definition in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UpdateCalculatedAttributeDefinition.html
     */
    toUpdateCalculatedAttributeDefinition() {
        return this.to('UpdateCalculatedAttributeDefinition');
    }
    /**
     * Grants permission to update a Domain
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UpdateDomain.html
     */
    toUpdateDomain() {
        return this.to('UpdateDomain');
    }
    /**
     * Grants permission to update a layout in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UpdateDomainLayout.html
     */
    toUpdateDomainLayout() {
        return this.to('UpdateDomainLayout');
    }
    /**
     * Grants permission to update an event trigger in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UpdateEventTrigger.html
     */
    toUpdateEventTrigger() {
        return this.to('UpdateEventTrigger');
    }
    /**
     * Grants permission to update a profile in the domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/API_UpdateProfile.html
     */
    toUpdateProfile() {
        return this.to('UpdateProfile');
    }
    /**
     * Adds a resource of type domains to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDomains(domainName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}`);
    }
    /**
     * Adds a resource of type object-types to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param objectTypeName - Identifier for the objectTypeName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onObjectTypes(domainName, objectTypeName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/object-types/${objectTypeName}`);
    }
    /**
     * Adds a resource of type integrations to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param uri - Identifier for the uri.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegrations(domainName, uri, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/integrations/${uri}`);
    }
    /**
     * Adds a resource of type event-streams to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param eventStreamName - Identifier for the eventStreamName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventStreams(domainName, eventStreamName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/event-streams/${eventStreamName}`);
    }
    /**
     * Adds a resource of type calculated-attributes to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param calculatedAttributeName - Identifier for the calculatedAttributeName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCalculatedAttributes(domainName, calculatedAttributeName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/calculated-attributes/${calculatedAttributeName}`);
    }
    /**
     * Adds a resource of type segment-definitions to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param segmentDefinitionName - Identifier for the segmentDefinitionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSegmentDefinitions(domainName, segmentDefinitionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/segment-definitions/${segmentDefinitionName}`);
    }
    /**
     * Adds a resource of type event-triggers to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param eventTriggerName - Identifier for the eventTriggerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventTriggers(domainName, eventTriggerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/event-triggers/${eventTriggerName}`);
    }
    /**
     * Adds a resource of type layouts to the statement
     *
     * https://docs.aws.amazon.com/customerprofiles/latest/APIReference/
     *
     * @param domainName - Identifier for the domainName.
     * @param layoutDefinitionName - Identifier for the layoutDefinitionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLayouts(domainName, layoutDefinitionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:profile:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:domains/${domainName}/layouts/${layoutDefinitionName}`);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the customer profile service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateCalculatedAttributeDefinition()
     * - .toCreateDomain()
     * - .toCreateDomainLayout()
     * - .toCreateEventStream()
     * - .toCreateEventTrigger()
     * - .toCreateIntegrationWorkflow()
     * - .toCreateSegmentDefinition()
     * - .toPutIntegration()
     * - .toPutProfileObjectType()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to resource types:
     * - domains
     * - object-types
     * - integrations
     * - event-streams
     * - calculated-attributes
     * - segment-definitions
     * - event-triggers
     * - layouts
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names present in the request the user makes to the customer profile service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateCalculatedAttributeDefinition()
     * - .toCreateDomain()
     * - .toCreateDomainLayout()
     * - .toCreateEventStream()
     * - .toCreateEventTrigger()
     * - .toCreateIntegrationWorkflow()
     * - .toCreateSegmentDefinition()
     * - .toPutIntegration()
     * - .toPutProfileObjectType()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [profile](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnectcustomerprofiles.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'profile';
        this.accessLevelList = {
            Write: [
                'AddProfileKey',
                'CreateCalculatedAttributeDefinition',
                'CreateDomain',
                'CreateDomainLayout',
                'CreateEventStream',
                'CreateEventTrigger',
                'CreateIntegrationWorkflow',
                'CreateProfile',
                'CreateSegmentDefinition',
                'CreateSegmentEstimate',
                'CreateSegmentSnapshot',
                'CreateSnapshot',
                'CreateUploadJob',
                'DeleteCalculatedAttributeDefinition',
                'DeleteDomain',
                'DeleteDomainLayout',
                'DeleteEventStream',
                'DeleteEventTrigger',
                'DeleteIntegration',
                'DeleteProfile',
                'DeleteProfileKey',
                'DeleteProfileObject',
                'DeleteProfileObjectType',
                'DeleteSegmentDefinition',
                'DeleteWorkflow',
                'MergeProfiles',
                'PutIntegration',
                'PutProfileObject',
                'PutProfileObjectType',
                'StartUploadJob',
                'StopUploadJob',
                'UpdateCalculatedAttributeDefinition',
                'UpdateDomain',
                'UpdateDomainLayout',
                'UpdateEventTrigger',
                'UpdateProfile'
            ],
            Read: [
                'BatchGetCalculatedAttributeForProfile',
                'BatchGetProfile',
                'DetectProfileObjectType',
                'GetAutoMergingPreview',
                'GetCalculatedAttributeDefinition',
                'GetCalculatedAttributeForProfile',
                'GetDomain',
                'GetDomainLayout',
                'GetEventStream',
                'GetEventTrigger',
                'GetIdentityResolutionJob',
                'GetIntegration',
                'GetProfileObjectType',
                'GetProfileObjectTypeTemplate',
                'GetSegmentDefinition',
                'GetSegmentEstimate',
                'GetSegmentMembership',
                'GetSegmentSnapshot',
                'GetSnapshot',
                'GetUploadJob',
                'GetUploadJobPath',
                'GetWorkflow',
                'GetWorkflowSteps',
                'ListTagsForResource',
                'SearchProfiles'
            ],
            List: [
                'GetMatches',
                'GetSimilarProfiles',
                'ListAccountIntegrations',
                'ListCalculatedAttributeDefinitions',
                'ListCalculatedAttributesForProfile',
                'ListDomainLayouts',
                'ListDomains',
                'ListEventStreams',
                'ListEventTriggers',
                'ListIdentityResolutionJobs',
                'ListIntegrations',
                'ListObjectTypeAttributes',
                'ListProfileAttributeValues',
                'ListProfileObjectTypeTemplates',
                'ListProfileObjectTypes',
                'ListProfileObjects',
                'ListRuleBasedMatches',
                'ListSegmentDefinitions',
                'ListUploadJobs',
                'ListWorkflows'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Profile = Profile;
//# sourceMappingURL=data:application/json;base64,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