"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectCampaigns = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [connect-campaigns](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnectoutboundcampaigns.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class ConnectCampaigns extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a campaign
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toCreateCampaign() {
        return this.to('CreateCampaign');
    }
    /**
     * Grants permission to delete a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteCampaign() {
        return this.to('DeleteCampaign');
    }
    /**
     * Grants permission to delete the channel subtype configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteCampaignChannelSubtypeConfig() {
        return this.to('DeleteCampaignChannelSubtypeConfig');
    }
    /**
     * Grants permission to delete the communication limits configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteCampaignCommunicationLimits() {
        return this.to('DeleteCampaignCommunicationLimits');
    }
    /**
     * Grants permission to delete the communication time configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteCampaignCommunicationTime() {
        return this.to('DeleteCampaignCommunicationTime');
    }
    /**
     * Grants permission to remove configuration information for an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteConnectInstanceConfig() {
        return this.to('DeleteConnectInstanceConfig');
    }
    /**
     * Grants permission to remove integration information for an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteConnectInstanceIntegration() {
        return this.to('DeleteConnectInstanceIntegration');
    }
    /**
     * Grants permission to remove onboarding job for an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDeleteInstanceOnboardingJob() {
        return this.to('DeleteInstanceOnboardingJob');
    }
    /**
     * Grants permission to describe a specific campaign
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toDescribeCampaign() {
        return this.to('DescribeCampaign');
    }
    /**
     * Grants permission to get state of a campaign
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toGetCampaignState() {
        return this.to('GetCampaignState');
    }
    /**
     * Grants permission to get state of campaigns
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toGetCampaignStateBatch() {
        return this.to('GetCampaignStateBatch');
    }
    /**
     * Grants permission to get configuration information for an Amazon Connect instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toGetConnectInstanceConfig() {
        return this.to('GetConnectInstanceConfig');
    }
    /**
     * Grants permission to get the communication limits configuration of an Amazon Connect instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toGetInstanceCommunicationLimits() {
        return this.to('GetInstanceCommunicationLimits');
    }
    /**
     * Grants permission to get onboarding job status for an Amazon Connect instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toGetInstanceOnboardingJobStatus() {
        return this.to('GetInstanceOnboardingJobStatus');
    }
    /**
     * Grants permission to provide summary of all campaigns
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toListCampaigns() {
        return this.to('ListCampaigns');
    }
    /**
     * Grants permission to provide summary of all integrations with an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toListConnectInstanceIntegrations() {
        return this.to('ListConnectInstanceIntegrations');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to pause a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPauseCampaign() {
        return this.to('PauseCampaign');
    }
    /**
     * Grants permission to put an integration configuration with an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPutConnectInstanceIntegration() {
        return this.to('PutConnectInstanceIntegration');
    }
    /**
     * Grants permission to create dial requests for the specified campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPutDialRequestBatch() {
        return this.to('PutDialRequestBatch');
    }
    /**
     * Grants permission to put the communication limits configuration of an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPutInstanceCommunicationLimits() {
        return this.to('PutInstanceCommunicationLimits');
    }
    /**
     * Grants permission to create dial requests for the specified campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPutOutboundRequestBatch() {
        return this.to('PutOutboundRequestBatch');
    }
    /**
     * Grants permission to create profile outbound requests for the specified campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toPutProfileOutboundRequestBatch() {
        return this.to('PutProfileOutboundRequestBatch');
    }
    /**
     * Grants permission to resume a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toResumeCampaign() {
        return this.to('ResumeCampaign');
    }
    /**
     * Grants permission to start a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toStartCampaign() {
        return this.to('StartCampaign');
    }
    /**
     * Grants permission to start onboarding job for an Amazon Connect instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toStartInstanceOnboardingJob() {
        return this.to('StartInstanceOnboardingJob');
    }
    /**
     * Grants permission to stop a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toStopCampaign() {
        return this.to('StopCampaign');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the channel subtype configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignChannelSubtypeConfig() {
        return this.to('UpdateCampaignChannelSubtypeConfig');
    }
    /**
     * Grants permission to update the communication limits configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignCommunicationLimits() {
        return this.to('UpdateCampaignCommunicationLimits');
    }
    /**
     * Grants permission to update the communication time configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignCommunicationTime() {
        return this.to('UpdateCampaignCommunicationTime');
    }
    /**
     * Grants permission to update the dialer configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignDialerConfig() {
        return this.to('UpdateCampaignDialerConfig');
    }
    /**
     * Grants permission to update the flow association of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignFlowAssociation() {
        return this.to('UpdateCampaignFlowAssociation');
    }
    /**
     * Grants permission to update the name of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignName() {
        return this.to('UpdateCampaignName');
    }
    /**
     * Grants permission to update the outbound call configuration of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignOutboundCallConfig() {
        return this.to('UpdateCampaignOutboundCallConfig');
    }
    /**
     * Grants permission to update the schedule of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignSchedule() {
        return this.to('UpdateCampaignSchedule');
    }
    /**
     * Grants permission to update the source of a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     */
    toUpdateCampaignSource() {
        return this.to('UpdateCampaignSource');
    }
    /**
     * Adds a resource of type campaign to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/enable-outbound-campaigns.html
     *
     * @param campaignId - Identifier for the campaignId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCampaign(campaignId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:connect-campaigns:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:campaign/${campaignId}`);
    }
    /**
     * Filters access by actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCampaign()
     * - .toDescribeCampaign()
     * - .toGetCampaignState()
     * - .toGetCampaignStateBatch()
     * - .toListCampaigns()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     *
     * Applies to resource types:
     * - campaign
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCampaign()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [connect-campaigns](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnectoutboundcampaigns.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'connect-campaigns';
        this.accessLevelList = {
            Write: [
                'CreateCampaign',
                'DeleteCampaign',
                'DeleteCampaignChannelSubtypeConfig',
                'DeleteCampaignCommunicationLimits',
                'DeleteCampaignCommunicationTime',
                'DeleteConnectInstanceConfig',
                'DeleteConnectInstanceIntegration',
                'DeleteInstanceOnboardingJob',
                'PauseCampaign',
                'PutConnectInstanceIntegration',
                'PutDialRequestBatch',
                'PutInstanceCommunicationLimits',
                'PutOutboundRequestBatch',
                'PutProfileOutboundRequestBatch',
                'ResumeCampaign',
                'StartCampaign',
                'StartInstanceOnboardingJob',
                'StopCampaign',
                'UpdateCampaignChannelSubtypeConfig',
                'UpdateCampaignCommunicationLimits',
                'UpdateCampaignCommunicationTime',
                'UpdateCampaignDialerConfig',
                'UpdateCampaignFlowAssociation',
                'UpdateCampaignName',
                'UpdateCampaignOutboundCallConfig',
                'UpdateCampaignSchedule',
                'UpdateCampaignSource'
            ],
            Read: [
                'DescribeCampaign',
                'GetCampaignState',
                'GetCampaignStateBatch',
                'GetConnectInstanceConfig',
                'GetInstanceCommunicationLimits',
                'GetInstanceOnboardingJobStatus',
                'ListTagsForResource'
            ],
            List: [
                'ListCampaigns',
                'ListConnectInstanceIntegrations'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.ConnectCampaigns = ConnectCampaigns;
//# sourceMappingURL=data:application/json;base64,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