"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ce = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ce](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscostexplorerservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ce extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a new Anomaly Monitor
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_CreateAnomalyMonitor.html
     */
    toCreateAnomalyMonitor() {
        return this.to('CreateAnomalyMonitor');
    }
    /**
     * Grants permission to create a new Anomaly Subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_CreateAnomalySubscription.html
     */
    toCreateAnomalySubscription() {
        return this.to('CreateAnomalySubscription');
    }
    /**
     * Grants permission to create a new Cost Category with the requested name and rules
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_CreateCostCategoryDefinition.html
     */
    toCreateCostCategoryDefinition() {
        return this.to('CreateCostCategoryDefinition');
    }
    /**
     * Grants permission to create Reservation expiration alerts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toCreateNotificationSubscription() {
        return this.to('CreateNotificationSubscription');
    }
    /**
     * Grants permission to create Cost Explorer Reports
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toCreateReport() {
        return this.to('CreateReport');
    }
    /**
     * Grants permission to delete an Anomaly Monitor
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DeleteAnomalyMonitor.html
     */
    toDeleteAnomalyMonitor() {
        return this.to('DeleteAnomalyMonitor');
    }
    /**
     * Grants permission to delete an Anomaly Subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DeleteAnomalySubscription.html
     */
    toDeleteAnomalySubscription() {
        return this.to('DeleteAnomalySubscription');
    }
    /**
     * Grants permission to delete a Cost Category
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DeleteCostCategoryDefinition.html
     */
    toDeleteCostCategoryDefinition() {
        return this.to('DeleteCostCategoryDefinition');
    }
    /**
     * Grants permission to delete Reservation expiration alerts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toDeleteNotificationSubscription() {
        return this.to('DeleteNotificationSubscription');
    }
    /**
     * Grants permission to delete Cost Explorer Reports
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toDeleteReport() {
        return this.to('DeleteReport');
    }
    /**
     * Grants permission to retrieve descriptions such as the name, ARN, rules, definition, and effective dates of a Cost Category
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DescribeCostCategoryDefinition.html
     */
    toDescribeCostCategoryDefinition() {
        return this.to('DescribeCostCategoryDefinition');
    }
    /**
     * Grants permission to view Reservation expiration alerts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toDescribeNotificationSubscription() {
        return this.to('DescribeNotificationSubscription');
    }
    /**
     * Grants permission to view Cost Explorer Reports page
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toDescribeReport() {
        return this.to('DescribeReport');
    }
    /**
     * Grants permission to retrieve anomalies
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetAnomalies.html
     */
    toGetAnomalies() {
        return this.to('GetAnomalies');
    }
    /**
     * Grants permission to query Anomaly Monitors
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetAnomalyMonitors.html
     */
    toGetAnomalyMonitors() {
        return this.to('GetAnomalyMonitors');
    }
    /**
     * Grants permission to query Anomaly Subscriptions
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetAnomalySubscriptions.html
     */
    toGetAnomalySubscriptions() {
        return this.to('GetAnomalySubscriptions');
    }
    /**
     * Grants permission to retrieve approximate usage record count for the chosen resource, level, and hourly granularity preferences, derived from the past month's usage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toGetApproximateUsageRecords() {
        return this.to('GetApproximateUsageRecords');
    }
    /**
     * Grants permission to retrieve the commitment purchase analysis for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCommitmentPurchaseAnalysis.html
     */
    toGetCommitmentPurchaseAnalysis() {
        return this.to('GetCommitmentPurchaseAnalysis');
    }
    /**
     * Grants permission to view whether existing or fine-grained IAM actions are being used to control authorization to Billing, Cost Management, and Account consoles
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toGetConsoleActionSetEnforced() {
        return this.to('GetConsoleActionSetEnforced');
    }
    /**
     * Grants permission to retrieve the cost and usage metrics for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostAndUsage.html
     */
    toGetCostAndUsage() {
        return this.to('GetCostAndUsage');
    }
    /**
     * Grants permission to retrieve the cost and usage comparisons for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostAndUsageComparisons.html
     */
    toGetCostAndUsageComparisons() {
        return this.to('GetCostAndUsageComparisons');
    }
    /**
     * Grants permission to retrieve the cost and usage metrics with resources for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostAndUsageWithResources.html
     */
    toGetCostAndUsageWithResources() {
        return this.to('GetCostAndUsageWithResources');
    }
    /**
     * Grants permission to query Cost Catagory names and values for a specified time period
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostCategories.html
     */
    toGetCostCategories() {
        return this.to('GetCostCategories');
    }
    /**
     * Grants permission to retrieve the cost drivers for your account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostComparisonDrivers.html
     */
    toGetCostComparisonDrivers() {
        return this.to('GetCostComparisonDrivers');
    }
    /**
     * Grants permission to retrieve a cost forecast for a forecast time period
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostForecast.html
     */
    toGetCostForecast() {
        return this.to('GetCostForecast');
    }
    /**
     * Grants permission to retrieve all available filter values for a filter for a period of time
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetDimensionValues.html
     */
    toGetDimensionValues() {
        return this.to('GetDimensionValues');
    }
    /**
     * Grants permission to view Cost Explorer Preferences page
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toGetPreferences() {
        return this.to('GetPreferences');
    }
    /**
     * Grants permission to retrieve the reservation coverage for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetReservationCoverage.html
     */
    toGetReservationCoverage() {
        return this.to('GetReservationCoverage');
    }
    /**
     * Grants permission to retrieve the reservation recommendations for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetReservationPurchaseRecommendation.html
     */
    toGetReservationPurchaseRecommendation() {
        return this.to('GetReservationPurchaseRecommendation');
    }
    /**
     * Grants permission to retrieve the reservation utilization for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetReservationUtilization.html
     */
    toGetReservationUtilization() {
        return this.to('GetReservationUtilization');
    }
    /**
     * Grants permission to retrieve the rightsizing recommendations for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetRightsizingRecommendation.html
     */
    toGetRightsizingRecommendation() {
        return this.to('GetRightsizingRecommendation');
    }
    /**
     * Grants permission to retrieve the Savings Plan recommendation details for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlanPurchaseRecommendationDetails.html
     */
    toGetSavingsPlanPurchaseRecommendationDetails() {
        return this.to('GetSavingsPlanPurchaseRecommendationDetails');
    }
    /**
     * Grants permission to retrieve the Savings Plans coverage for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansCoverage.html
     */
    toGetSavingsPlansCoverage() {
        return this.to('GetSavingsPlansCoverage');
    }
    /**
     * Grants permission to retrieve the Savings Plans recommendations for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansPurchaseRecommendation.html
     */
    toGetSavingsPlansPurchaseRecommendation() {
        return this.to('GetSavingsPlansPurchaseRecommendation');
    }
    /**
     * Grants permission to retrieve the Savings Plans utilization for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansUtilization.html
     */
    toGetSavingsPlansUtilization() {
        return this.to('GetSavingsPlansUtilization');
    }
    /**
     * Grants permission to retrieve the Savings Plans utilization details for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansUtilizationDetails.html
     */
    toGetSavingsPlansUtilizationDetails() {
        return this.to('GetSavingsPlansUtilizationDetails');
    }
    /**
     * Grants permission to query tags for a specified time period
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetTags.html
     */
    toGetTags() {
        return this.to('GetTags');
    }
    /**
     * Grants permission to retrieve a usage forecast for a forecast time period
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetUsageForecast.html
     */
    toGetUsageForecast() {
        return this.to('GetUsageForecast');
    }
    /**
     * Grants permission to retrieve a list of your historical commitment purchase analyses
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ListCommitmentPurchaseAnalyses.html
     */
    toListCommitmentPurchaseAnalyses() {
        return this.to('ListCommitmentPurchaseAnalyses');
    }
    /**
     * Grants permission to list Cost Allocation Tag backfill history
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ListCostAllocationTagBackfillHistory.html
     */
    toListCostAllocationTagBackfillHistory() {
        return this.to('ListCostAllocationTagBackfillHistory');
    }
    /**
     * Grants permission to list Cost Allocation Tags
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ListCostAllocationTags.html
     */
    toListCostAllocationTags() {
        return this.to('ListCostAllocationTags');
    }
    /**
     * Grants permission to retrieve names, ARN, and effective dates for all Cost Categories
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ListCostCategoryDefinitions.html
     */
    toListCostCategoryDefinitions() {
        return this.to('ListCostCategoryDefinitions');
    }
    /**
     * Grants permission to retrieve a list of your historical recommendation generations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ListSavingsPlansPurchaseRecommendationGeneration.html
     */
    toListSavingsPlansPurchaseRecommendationGeneration() {
        return this.to('ListSavingsPlansPurchaseRecommendationGeneration');
    }
    /**
     * Grants permission to list tags for a Cost Explorer resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to provide feedback on detected anomalies
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ProvideAnomalyFeedback.html
     */
    toProvideAnomalyFeedback() {
        return this.to('ProvideAnomalyFeedback');
    }
    /**
     * Grants permission to request a commitment purchase analysis
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_StartCommitmentPurchaseAnalysis.html
     */
    toStartCommitmentPurchaseAnalysis() {
        return this.to('StartCommitmentPurchaseAnalysis');
    }
    /**
     * Grants permission to request a Cost Allocation Tag backfill
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_StartCostAllocationTagBackfill.html
     */
    toStartCostAllocationTagBackfill() {
        return this.to('StartCostAllocationTagBackfill');
    }
    /**
     * Grants permission to request a Savings Plans recommendation generation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_StartSavingsPlansPurchaseRecommendationGeneration.html
     */
    toStartSavingsPlansPurchaseRecommendationGeneration() {
        return this.to('StartSavingsPlansPurchaseRecommendationGeneration');
    }
    /**
     * Grants permission to tag a Cost Explorer resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a Cost Explorer resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an existing Anomaly Monitor
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UpdateAnomalyMonitor.html
     */
    toUpdateAnomalyMonitor() {
        return this.to('UpdateAnomalyMonitor');
    }
    /**
     * Grants permission to update an existing Anomaly Subscription
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UpdateAnomalySubscription.html
     */
    toUpdateAnomalySubscription() {
        return this.to('UpdateAnomalySubscription');
    }
    /**
     * Grants permission to change whether existing or fine-grained IAM actions will be used to control authorization to Billing, Cost Management, and Account consoles
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toUpdateConsoleActionSetEnforced() {
        return this.to('UpdateConsoleActionSetEnforced');
    }
    /**
     * Grants permission to update existing Cost Allocation Tags status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UpdateCostAllocationTagsStatus.html
     */
    toUpdateCostAllocationTagsStatus() {
        return this.to('UpdateCostAllocationTagsStatus');
    }
    /**
     * Grants permission to update an existing Cost Category
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UpdateCostCategoryDefinition.html
     */
    toUpdateCostCategoryDefinition() {
        return this.to('UpdateCostCategoryDefinition');
    }
    /**
     * Grants permission to update Reservation expiration alerts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toUpdateNotificationSubscription() {
        return this.to('UpdateNotificationSubscription');
    }
    /**
     * Grants permission to edit Cost Explorer Preferences page
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toUpdatePreferences() {
        return this.to('UpdatePreferences');
    }
    /**
     * Grants permission to update Cost Explorer Reports
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/billing-permissions-ref.html
     */
    toUpdateReport() {
        return this.to('UpdateReport');
    }
    /**
     * Adds a resource of type anomalysubscription to the statement
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AnomalySubscription.html
     *
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnomalysubscription(identifier, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ce::${account ?? this.defaultAccount}:anomalysubscription/${identifier}`);
    }
    /**
     * Adds a resource of type anomalymonitor to the statement
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_AnomalyMonitor.html
     *
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnomalymonitor(identifier, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ce::${account ?? this.defaultAccount}:anomalymonitor/${identifier}`);
    }
    /**
     * Adds a resource of type costcategory to the statement
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_CostCategory.html
     *
     * @param identifier - Identifier for the identifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCostcategory(identifier, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ce::${account ?? this.defaultAccount}:costcategory/${identifier}`);
    }
    /**
     * Adds a resource of type billingview to the statement
     *
     * https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBillingview(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:billing::${account ?? this.defaultAccount}:billingview/${resourceId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAnomalyMonitor()
     * - .toCreateAnomalySubscription()
     * - .toCreateCostCategoryDefinition()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteAnomalyMonitor()
     * - .toDeleteAnomalySubscription()
     * - .toDeleteCostCategoryDefinition()
     * - .toDescribeCostCategoryDefinition()
     * - .toGetAnomalies()
     * - .toGetAnomalyMonitors()
     * - .toGetAnomalySubscriptions()
     * - .toGetCostAndUsage()
     * - .toGetCostAndUsageComparisons()
     * - .toGetCostAndUsageWithResources()
     * - .toGetCostCategories()
     * - .toGetCostComparisonDrivers()
     * - .toGetCostForecast()
     * - .toGetDimensionValues()
     * - .toGetTags()
     * - .toGetUsageForecast()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAnomalyMonitor()
     * - .toUpdateAnomalySubscription()
     * - .toUpdateCostCategoryDefinition()
     *
     * Applies to resource types:
     * - anomalysubscription
     * - anomalymonitor
     * - costcategory
     * - billingview
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAnomalyMonitor()
     * - .toCreateAnomalySubscription()
     * - .toCreateCostCategoryDefinition()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ce](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awscostexplorerservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ce';
        this.accessLevelList = {
            Write: [
                'CreateAnomalyMonitor',
                'CreateAnomalySubscription',
                'CreateCostCategoryDefinition',
                'CreateNotificationSubscription',
                'CreateReport',
                'DeleteAnomalyMonitor',
                'DeleteAnomalySubscription',
                'DeleteCostCategoryDefinition',
                'DeleteNotificationSubscription',
                'DeleteReport',
                'ProvideAnomalyFeedback',
                'StartCommitmentPurchaseAnalysis',
                'StartCostAllocationTagBackfill',
                'StartSavingsPlansPurchaseRecommendationGeneration',
                'UpdateAnomalyMonitor',
                'UpdateAnomalySubscription',
                'UpdateConsoleActionSetEnforced',
                'UpdateCostAllocationTagsStatus',
                'UpdateCostCategoryDefinition',
                'UpdateNotificationSubscription',
                'UpdatePreferences',
                'UpdateReport'
            ],
            Read: [
                'DescribeCostCategoryDefinition',
                'DescribeNotificationSubscription',
                'DescribeReport',
                'GetAnomalies',
                'GetAnomalyMonitors',
                'GetAnomalySubscriptions',
                'GetApproximateUsageRecords',
                'GetCommitmentPurchaseAnalysis',
                'GetConsoleActionSetEnforced',
                'GetCostAndUsage',
                'GetCostAndUsageComparisons',
                'GetCostAndUsageWithResources',
                'GetCostCategories',
                'GetCostComparisonDrivers',
                'GetCostForecast',
                'GetDimensionValues',
                'GetPreferences',
                'GetReservationCoverage',
                'GetReservationPurchaseRecommendation',
                'GetReservationUtilization',
                'GetRightsizingRecommendation',
                'GetSavingsPlanPurchaseRecommendationDetails',
                'GetSavingsPlansCoverage',
                'GetSavingsPlansPurchaseRecommendation',
                'GetSavingsPlansUtilization',
                'GetSavingsPlansUtilizationDetails',
                'GetTags',
                'GetUsageForecast',
                'ListTagsForResource'
            ],
            List: [
                'ListCommitmentPurchaseAnalyses',
                'ListCostAllocationTagBackfillHistory',
                'ListCostAllocationTags',
                'ListCostCategoryDefinitions',
                'ListSavingsPlansPurchaseRecommendationGeneration'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Ce = Ce;
//# sourceMappingURL=data:application/json;base64,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