"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dataexchange = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [dataexchange](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdataexchange.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dataexchange extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept a data grant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_AcceptDataGrant.html
     */
    toAcceptDataGrant() {
        return this.to('AcceptDataGrant');
    }
    /**
     * Grants permission to cancel a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_CancelJob.html
     */
    toCancelJob() {
        return this.to('CancelJob');
    }
    /**
     * Grants permission to create an asset (for example, in a Job)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/api-permissions-ref.html
     */
    toCreateAsset() {
        return this.to('CreateAsset');
    }
    /**
     * Grants permission to create a data grant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - dataexchange:PublishToDataGrant
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_CreateDataGrant.html
     */
    toCreateDataGrant() {
        return this.to('CreateDataGrant');
    }
    /**
     * Grants permission to create a data set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_CreateDataSet.html
     */
    toCreateDataSet() {
        return this.to('CreateDataSet');
    }
    /**
     * Grants permission to create an event action
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_CreateEventAction.html
     */
    toCreateEventAction() {
        return this.to('CreateEventAction');
    }
    /**
     * Grants permission to create a job to import or export assets
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifJobType()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_CreateJob.html
     */
    toCreateJob() {
        return this.to('CreateJob');
    }
    /**
     * Grants permission to create a revision
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_CreateRevision.html
     */
    toCreateRevision() {
        return this.to('CreateRevision');
    }
    /**
     * Grants permission to delete an asset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_DeleteAsset.html
     */
    toDeleteAsset() {
        return this.to('DeleteAsset');
    }
    /**
     * Grants permission to delete a data grant
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_DeleteDataGrant.html
     */
    toDeleteDataGrant() {
        return this.to('DeleteDataGrant');
    }
    /**
     * Grants permission to delete a data set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_DeleteDataSet.html
     */
    toDeleteDataSet() {
        return this.to('DeleteDataSet');
    }
    /**
     * Grants permission to delete an event action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_DeleteEventAction.html
     */
    toDeleteEventAction() {
        return this.to('DeleteEventAction');
    }
    /**
     * Grants permission to delete a revision
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_DeleteRevision.html
     */
    toDeleteRevision() {
        return this.to('DeleteRevision');
    }
    /**
     * Grants permission to get information about an asset and to export it (for example, in a Job)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_GetAsset.html
     */
    toGetAsset() {
        return this.to('GetAsset');
    }
    /**
     * Grants permission to get a data grant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_GetDataGrant.html
     */
    toGetDataGrant() {
        return this.to('GetDataGrant');
    }
    /**
     * Grants permission to get information about a data set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_GetDataSet.html
     */
    toGetDataSet() {
        return this.to('GetDataSet');
    }
    /**
     * Grants permission to get an event action
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_GetEventAction.html
     */
    toGetEventAction() {
        return this.to('GetEventAction');
    }
    /**
     * Grants permission to get information about a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_GetJob.html
     */
    toGetJob() {
        return this.to('GetJob');
    }
    /**
     * Grants permission to get a received data grant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_GetReceivedDataGrant.html
     */
    toGetReceivedDataGrant() {
        return this.to('GetReceivedDataGrant');
    }
    /**
     * Grants permission to get information about a revision
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_GetRevision.html
     */
    toGetRevision() {
        return this.to('GetRevision');
    }
    /**
     * Grants permission to list data grants for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_ListDataGrants.html
     */
    toListDataGrants() {
        return this.to('ListDataGrants');
    }
    /**
     * Grants permission to list the revisions of a data set
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_ListDataSetRevisions.html
     */
    toListDataSetRevisions() {
        return this.to('ListDataSetRevisions');
    }
    /**
     * Grants permission to list data sets for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_ListDataSets.html
     */
    toListDataSets() {
        return this.to('ListDataSets');
    }
    /**
     * Grants permission to list event actions for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_ListEventActions.html
     */
    toListEventActions() {
        return this.to('ListEventActions');
    }
    /**
     * Grants permission to list jobs for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_ListJobs.html
     */
    toListJobs() {
        return this.to('ListJobs');
    }
    /**
     * Grants permission to list received data grants for the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_ListReceivedDataGrants.html
     */
    toListReceivedDataGrants() {
        return this.to('ListReceivedDataGrants');
    }
    /**
     * Grants permission to get list the assets of a revision
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_ListRevisionAssets.html
     */
    toListRevisionAssets() {
        return this.to('ListRevisionAssets');
    }
    /**
     * Grants permission to list the tags that you associated with the specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to publish a data set to a product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/api-permissions-ref.html
     */
    toPublishDataSet() {
        return this.to('PublishDataSet');
    }
    /**
     * Grants permission to publish a data set to a data grant
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/api-permissions-ref.html
     */
    toPublishToDataGrant() {
        return this.to('PublishToDataGrant');
    }
    /**
     * Grants permission to revoke subscriber access to a revision
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_RevokeRevision.html
     */
    toRevokeRevision() {
        return this.to('RevokeRevision');
    }
    /**
     * Grants permission to send a request to an API asset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_SendApiAsset.html
     */
    toSendApiAsset() {
        return this.to('SendApiAsset');
    }
    /**
     * Grants permission to send a notification to subscribers of a data set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_SendDataSetNotification.html
     */
    toSendDataSetNotification() {
        return this.to('SendDataSetNotification');
    }
    /**
     * Grants permission to start a job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dataexchange:CreateAsset
     * - dataexchange:DeleteDataSet
     * - dataexchange:GetAsset
     * - dataexchange:GetDataSet
     * - dataexchange:GetRevision
     * - dataexchange:PublishDataSet
     * - redshift:AuthorizeDataShare
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_StartJob.html
     */
    toStartJob() {
        return this.to('StartJob');
    }
    /**
     * Grants permission to add one or more tags to a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags from a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to get update information about an asset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_UpdateAsset.html
     */
    toUpdateAsset() {
        return this.to('UpdateAsset');
    }
    /**
     * Grants permission to update information about a data set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_UpdateDataSet.html
     */
    toUpdateDataSet() {
        return this.to('UpdateDataSet');
    }
    /**
     * Grants permission to update information for an event action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_UpdateEventAction.html
     */
    toUpdateEventAction() {
        return this.to('UpdateEventAction');
    }
    /**
     * Grants permission to update information about a revision
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dataexchange:PublishDataSet
     * - dataexchange:PublishToDataGrant
     *
     * https://docs.aws.amazon.com/data-exchange/latest/apireference/API_UpdateRevision.html
     */
    toUpdateRevision() {
        return this.to('UpdateRevision');
    }
    /**
     * Adds a resource of type jobs to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifJobType()
     */
    onJobs(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dataexchange:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:jobs/${jobId}`);
    }
    /**
     * Adds a resource of type data-sets to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html
     *
     * @param dataSetId - Identifier for the dataSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataSets(dataSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dataexchange:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:data-sets/${dataSetId}`);
    }
    /**
     * Adds a resource of type entitled-data-sets to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html
     *
     * @param dataSetId - Identifier for the dataSetId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEntitledDataSets(dataSetId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dataexchange:${region ?? this.defaultRegion}::data-sets/${dataSetId}`);
    }
    /**
     * Adds a resource of type revisions to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html#revisions
     *
     * @param dataSetId - Identifier for the dataSetId.
     * @param revisionId - Identifier for the revisionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRevisions(dataSetId, revisionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dataexchange:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:data-sets/${dataSetId}/revisions/${revisionId}`);
    }
    /**
     * Adds a resource of type entitled-revisions to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html#revisions
     *
     * @param dataSetId - Identifier for the dataSetId.
     * @param revisionId - Identifier for the revisionId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEntitledRevisions(dataSetId, revisionId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dataexchange:${region ?? this.defaultRegion}::data-sets/${dataSetId}/revisions/${revisionId}`);
    }
    /**
     * Adds a resource of type assets to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html#assets
     *
     * @param dataSetId - Identifier for the dataSetId.
     * @param revisionId - Identifier for the revisionId.
     * @param assetId - Identifier for the assetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAssets(dataSetId, revisionId, assetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dataexchange:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:data-sets/${dataSetId}/revisions/${revisionId}/assets/${assetId}`);
    }
    /**
     * Adds a resource of type entitled-assets to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html#assets
     *
     * @param dataSetId - Identifier for the dataSetId.
     * @param revisionId - Identifier for the revisionId.
     * @param assetId - Identifier for the assetId.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEntitledAssets(dataSetId, revisionId, assetId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dataexchange:${region ?? this.defaultRegion}::data-sets/${dataSetId}/revisions/${revisionId}/assets/${assetId}`);
    }
    /**
     * Adds a resource of type event-actions to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html
     *
     * @param eventActionId - Identifier for the eventActionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventActions(eventActionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dataexchange:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:event-actions/${eventActionId}`);
    }
    /**
     * Adds a resource of type data-grants to the statement
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/data-sets.html
     *
     * @param dataGrantId - Identifier for the dataGrantId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataGrants(dataGrantId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dataexchange:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:data-grants/${dataGrantId}`);
    }
    /**
     * Filters access by the allowed set of values for each of the mandatory tags in the create request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateDataGrant()
     * - .toCreateDataSet()
     * - .toCreateEventAction()
     * - .toCreateRevision()
     * - .toPublishToDataGrant()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to resource types:
     * - data-sets
     * - revisions
     * - event-actions
     * - data-grants
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of mandatory tags in the create request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateDataGrant()
     * - .toCreateDataSet()
     * - .toCreateEventAction()
     * - .toCreateRevision()
     * - .toPublishToDataGrant()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the specified job type
     *
     * https://docs.aws.amazon.com/data-exchange/latest/userguide/access-control.html
     *
     * Applies to actions:
     * - .toCreateJob()
     *
     * Applies to resource types:
     * - jobs
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobType(value, operator) {
        return this.if(`JobType`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [dataexchange](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdataexchange.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'dataexchange';
        this.accessLevelList = {
            Write: [
                'AcceptDataGrant',
                'CancelJob',
                'CreateAsset',
                'CreateDataGrant',
                'CreateDataSet',
                'CreateEventAction',
                'CreateJob',
                'CreateRevision',
                'DeleteAsset',
                'DeleteDataGrant',
                'DeleteDataSet',
                'DeleteEventAction',
                'DeleteRevision',
                'PublishDataSet',
                'PublishToDataGrant',
                'RevokeRevision',
                'SendApiAsset',
                'SendDataSetNotification',
                'StartJob',
                'UpdateAsset',
                'UpdateDataSet',
                'UpdateEventAction',
                'UpdateRevision'
            ],
            Read: [
                'GetAsset',
                'GetDataGrant',
                'GetDataSet',
                'GetEventAction',
                'GetJob',
                'GetReceivedDataGrant',
                'GetRevision'
            ],
            List: [
                'ListDataGrants',
                'ListDataSetRevisions',
                'ListDataSets',
                'ListEventActions',
                'ListJobs',
                'ListReceivedDataGrants',
                'ListRevisionAssets',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Dataexchange = Dataexchange;
//# sourceMappingURL=data:application/json;base64,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