"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Datasync = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [datasync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatasync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Datasync extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a storage system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_AddStorageSystem.html
     */
    toAddStorageSystem() {
        return this.to('AddStorageSystem');
    }
    /**
     * Grants permission to cancel execution of a sync task
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CancelTaskExecution.html
     */
    toCancelTaskExecution() {
        return this.to('CancelTaskExecution');
    }
    /**
     * Grants permission to activate an agent that you have deployed on your host
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateAgent.html
     */
    toCreateAgent() {
        return this.to('CreateAgent');
    }
    /**
     * Grants permission to create an endpoint for a Microsoft Azure Blob Storage container
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationAzureBlob.html
     */
    toCreateLocationAzureBlob() {
        return this.to('CreateLocationAzureBlob');
    }
    /**
     * Grants permission to create an endpoint for an Amazon EFS file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationEfs.html
     */
    toCreateLocationEfs() {
        return this.to('CreateLocationEfs');
    }
    /**
     * Grants permission to create an endpoint for an Amazon Fsx Lustre
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxLustre.html
     */
    toCreateLocationFsxLustre() {
        return this.to('CreateLocationFsxLustre');
    }
    /**
     * Grants permission to create an endpoint for Amazon FSx for ONTAP
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxOntap.html
     */
    toCreateLocationFsxOntap() {
        return this.to('CreateLocationFsxOntap');
    }
    /**
     * Grants permission to create an endpoint for Amazon FSx for OpenZFS
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxOpenZfs.html
     */
    toCreateLocationFsxOpenZfs() {
        return this.to('CreateLocationFsxOpenZfs');
    }
    /**
     * Grants permission to create an endpoint for an Amazon FSx Windows File Server file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationFsxWindows.html
     */
    toCreateLocationFsxWindows() {
        return this.to('CreateLocationFsxWindows');
    }
    /**
     * Grants permission to create an endpoint for an Amazon Hdfs
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationHdfs.html
     */
    toCreateLocationHdfs() {
        return this.to('CreateLocationHdfs');
    }
    /**
     * Grants permission to create an endpoint for a NFS file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationNfs.html
     */
    toCreateLocationNfs() {
        return this.to('CreateLocationNfs');
    }
    /**
     * Grants permission to create an endpoint for a self-managed object storage bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationObjectStorage.html
     */
    toCreateLocationObjectStorage() {
        return this.to('CreateLocationObjectStorage');
    }
    /**
     * Grants permission to create an endpoint for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationS3.html
     */
    toCreateLocationS3() {
        return this.to('CreateLocationS3');
    }
    /**
     * Grants permission to create an endpoint for an SMB file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateLocationSmb.html
     */
    toCreateLocationSmb() {
        return this.to('CreateLocationSmb');
    }
    /**
     * Grants permission to create a sync task
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_CreateTask.html
     */
    toCreateTask() {
        return this.to('CreateTask');
    }
    /**
     * Grants permission to delete an agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DeleteAgent.html
     */
    toDeleteAgent() {
        return this.to('DeleteAgent');
    }
    /**
     * Grants permission to delete a location used by AWS DataSync
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DeleteLocation.html
     */
    toDeleteLocation() {
        return this.to('DeleteLocation');
    }
    /**
     * Grants permission to delete a sync task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DeleteTask.html
     */
    toDeleteTask() {
        return this.to('DeleteTask');
    }
    /**
     * Grants permission to view metadata such as name, network interfaces, and the status (that is, whether the agent is running or not) about a sync agent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeAgent.html
     */
    toDescribeAgent() {
        return this.to('DescribeAgent');
    }
    /**
     * Grants permission to describe metadata about a discovery job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeDiscoveryJob.html
     */
    toDescribeDiscoveryJob() {
        return this.to('DescribeDiscoveryJob');
    }
    /**
     * Grants permission to view metadata, such as the path information about an Azure Blob Storage sync location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationAzureBlob.html
     */
    toDescribeLocationAzureBlob() {
        return this.to('DescribeLocationAzureBlob');
    }
    /**
     * Grants permission to view metadata, such as the path information about an Amazon EFS sync location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationEfs.html
     */
    toDescribeLocationEfs() {
        return this.to('DescribeLocationEfs');
    }
    /**
     * Grants permission to view metadata, such as the path information about an Amazon FSx Lustre sync location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationFsxLustre.html
     */
    toDescribeLocationFsxLustre() {
        return this.to('DescribeLocationFsxLustre');
    }
    /**
     * Grants permission to view metadata, such as the path information about an Amazon FSx for ONTAP sync location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationFsxOntap.html
     */
    toDescribeLocationFsxOntap() {
        return this.to('DescribeLocationFsxOntap');
    }
    /**
     * Grants permission to view metadata, such as the path information about an Amazon FSx OpenZFS sync location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationFsxOpenZfs.html
     */
    toDescribeLocationFsxOpenZfs() {
        return this.to('DescribeLocationFsxOpenZfs');
    }
    /**
     * Grants permission to view metadata, such as the path information about an Amazon FSx Windows sync location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationFsxWindows.html
     */
    toDescribeLocationFsxWindows() {
        return this.to('DescribeLocationFsxWindows');
    }
    /**
     * Grants permission to view metadata, such as the path information about an Amazon HDFS sync location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationHdfs.html
     */
    toDescribeLocationHdfs() {
        return this.to('DescribeLocationHdfs');
    }
    /**
     * Grants permission to view metadata, such as the path information, about a NFS sync location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationNfs.html
     */
    toDescribeLocationNfs() {
        return this.to('DescribeLocationNfs');
    }
    /**
     * Grants permission to view metadata about a self-managed object storage server location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationObjectStorage.html
     */
    toDescribeLocationObjectStorage() {
        return this.to('DescribeLocationObjectStorage');
    }
    /**
     * Grants permission to view metadata, such as bucket name, about an Amazon S3 bucket sync location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationS3.html
     */
    toDescribeLocationS3() {
        return this.to('DescribeLocationS3');
    }
    /**
     * Grants permission to view metadata, such as the path information, about an SMB sync location
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeLocationSmb.html
     */
    toDescribeLocationSmb() {
        return this.to('DescribeLocationSmb');
    }
    /**
     * Grants permission to view metadata about a storage system
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeStorageSystem.html
     */
    toDescribeStorageSystem() {
        return this.to('DescribeStorageSystem');
    }
    /**
     * Grants permission to describe resource metrics collected by a discovery job
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeStorageSystemResourceMetrics.html
     */
    toDescribeStorageSystemResourceMetrics() {
        return this.to('DescribeStorageSystemResourceMetrics');
    }
    /**
     * Grants permission to describe resources identified by a discovery job
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeStorageSystemResources.html
     */
    toDescribeStorageSystemResources() {
        return this.to('DescribeStorageSystemResources');
    }
    /**
     * Grants permission to view metadata about a sync task
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeTask.html
     */
    toDescribeTask() {
        return this.to('DescribeTask');
    }
    /**
     * Grants permission to view metadata about a sync task that is being executed
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_DescribeTaskExecution.html
     */
    toDescribeTaskExecution() {
        return this.to('DescribeTaskExecution');
    }
    /**
     * Grants permission to generate recommendations for a resource identified by a discovery job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_GenerateRecommendations.html
     */
    toGenerateRecommendations() {
        return this.to('GenerateRecommendations');
    }
    /**
     * Grants permission to list agents owned by an AWS account in a region specified in the request
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListAgents.html
     */
    toListAgents() {
        return this.to('ListAgents');
    }
    /**
     * Grants permission to list discovery jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListDiscoveryJobs.html
     */
    toListDiscoveryJobs() {
        return this.to('ListDiscoveryJobs');
    }
    /**
     * Grants permission to list source and destination sync locations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListLocations.html
     */
    toListLocations() {
        return this.to('ListLocations');
    }
    /**
     * Grants permission to list storage systems
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListStorageSystems.html
     */
    toListStorageSystems() {
        return this.to('ListStorageSystems');
    }
    /**
     * Grants permission to list tags that have been added to the specified resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list executed sync tasks
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListTaskExecutions.html
     */
    toListTaskExecutions() {
        return this.to('ListTaskExecutions');
    }
    /**
     * Grants permission to list of all the sync tasks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_ListTasks.html
     */
    toListTasks() {
        return this.to('ListTasks');
    }
    /**
     * Grants permission to delete a storage system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_RemoveStorageSystem.html
     */
    toRemoveStorageSystem() {
        return this.to('RemoveStorageSystem');
    }
    /**
     * Grants permission to start a discovery job for a storage system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_StartDiscoveryJob.html
     */
    toStartDiscoveryJob() {
        return this.to('StartDiscoveryJob');
    }
    /**
     * Grants permission to start a specific invocation of a sync task
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_StartTaskExecution.html
     */
    toStartTaskExecution() {
        return this.to('StartTaskExecution');
    }
    /**
     * Grants permission to stop a discovery job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_StopDiscoveryJob.html
     */
    toStopDiscoveryJob() {
        return this.to('StopDiscoveryJob');
    }
    /**
     * Grants permission to apply a key-value pair to an AWS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove one or more tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the name of an agent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateAgent.html
     */
    toUpdateAgent() {
        return this.to('UpdateAgent');
    }
    /**
     * Grants permission to update a discovery job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateDiscoveryJob.html
     */
    toUpdateDiscoveryJob() {
        return this.to('UpdateDiscoveryJob');
    }
    /**
     * Grants permission to update an Azure Blob Storage sync location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationAzureBlob.html
     */
    toUpdateLocationAzureBlob() {
        return this.to('UpdateLocationAzureBlob');
    }
    /**
     * Grants permission to update an EFS sync Location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationEfs.html
     */
    toUpdateLocationEfs() {
        return this.to('UpdateLocationEfs');
    }
    /**
     * Grants permission to update an FSx Lustre sync Location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationFsxLustre.html
     */
    toUpdateLocationFsxLustre() {
        return this.to('UpdateLocationFsxLustre');
    }
    /**
     * Grants permission to update an FSx ONTAP sync Location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationFsxOntap.html
     */
    toUpdateLocationFsxOntap() {
        return this.to('UpdateLocationFsxOntap');
    }
    /**
     * Grants permission to update an FSx OpenZFS sync Location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationFsxOpenZfs.html
     */
    toUpdateLocationFsxOpenZfs() {
        return this.to('UpdateLocationFsxOpenZfs');
    }
    /**
     * Grants permission to update an FSx Windows sync Location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationFsxWindows.html
     */
    toUpdateLocationFsxWindows() {
        return this.to('UpdateLocationFsxWindows');
    }
    /**
     * Grants permission to update an HDFS sync Location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationHdfs.html
     */
    toUpdateLocationHdfs() {
        return this.to('UpdateLocationHdfs');
    }
    /**
     * Grants permission to update an NFS sync Location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationNfs.html
     */
    toUpdateLocationNfs() {
        return this.to('UpdateLocationNfs');
    }
    /**
     * Grants permission to update a self-managed object storage server location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationObjectStorage.html
     */
    toUpdateLocationObjectStorage() {
        return this.to('UpdateLocationObjectStorage');
    }
    /**
     * Grants permission to update an S3 sync Location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationS3.html
     */
    toUpdateLocationS3() {
        return this.to('UpdateLocationS3');
    }
    /**
     * Grants permission to update a SMB sync location
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateLocationSmb.html
     */
    toUpdateLocationSmb() {
        return this.to('UpdateLocationSmb');
    }
    /**
     * Grants permission to update a storage system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateStorageSystem.html
     */
    toUpdateStorageSystem() {
        return this.to('UpdateStorageSystem');
    }
    /**
     * Grants permission to update metadata associated with a sync task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateTask.html
     */
    toUpdateTask() {
        return this.to('UpdateTask');
    }
    /**
     * Grants permission to update execution of a sync task
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/API_UpdateTaskExecution.html
     */
    toUpdateTaskExecution() {
        return this.to('UpdateTaskExecution');
    }
    /**
     * Adds a resource of type agent to the statement
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/working-with-agents.html
     *
     * @param agentId - Identifier for the agentId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAgent(agentId, accountId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:datasync:${region ?? this.defaultRegion}:${accountId ?? this.defaultAccount}:agent/${agentId}`);
    }
    /**
     * Adds a resource of type location to the statement
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/working-with-locations.html
     *
     * @param locationId - Identifier for the locationId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLocation(locationId, accountId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:datasync:${region ?? this.defaultRegion}:${accountId ?? this.defaultAccount}:location/${locationId}`);
    }
    /**
     * Adds a resource of type task to the statement
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/working-with-tasks.html
     *
     * @param taskId - Identifier for the taskId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTask(taskId, accountId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:datasync:${region ?? this.defaultRegion}:${accountId ?? this.defaultAccount}:task/${taskId}`);
    }
    /**
     * Adds a resource of type taskexecution to the statement
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/working-with-task-executions.html
     *
     * @param taskId - Identifier for the taskId.
     * @param executionId - Identifier for the executionId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTaskexecution(taskId, executionId, accountId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:datasync:${region ?? this.defaultRegion}:${accountId ?? this.defaultAccount}:task/${taskId}/execution/${executionId}`);
    }
    /**
     * Adds a resource of type storagesystem to the statement
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/discovery-configure-storage.html
     *
     * @param storageSystemId - Identifier for the storageSystemId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStoragesystem(storageSystemId, accountId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:datasync:${region ?? this.defaultRegion}:${accountId ?? this.defaultAccount}:system/${storageSystemId}`);
    }
    /**
     * Adds a resource of type discoveryjob to the statement
     *
     * https://docs.aws.amazon.com/datasync/latest/userguide/discovery-job-create.html
     *
     * @param storageSystemId - Identifier for the storageSystemId.
     * @param discoveryJobId - Identifier for the discoveryJobId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDiscoveryjob(storageSystemId, discoveryJobId, accountId, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:datasync:${region ?? this.defaultRegion}:${accountId ?? this.defaultAccount}:system/${storageSystemId}/job/${discoveryJobId}`);
    }
    /**
     * Filters access by the tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAddStorageSystem()
     * - .toCreateAgent()
     * - .toCreateLocationAzureBlob()
     * - .toCreateLocationEfs()
     * - .toCreateLocationFsxLustre()
     * - .toCreateLocationFsxOntap()
     * - .toCreateLocationFsxOpenZfs()
     * - .toCreateLocationFsxWindows()
     * - .toCreateLocationHdfs()
     * - .toCreateLocationNfs()
     * - .toCreateLocationObjectStorage()
     * - .toCreateLocationS3()
     * - .toCreateLocationSmb()
     * - .toCreateTask()
     * - .toStartTaskExecution()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag key-value pairs associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAddStorageSystem()
     * - .toCancelTaskExecution()
     * - .toDescribeTaskExecution()
     * - .toListTaskExecutions()
     * - .toStartTaskExecution()
     * - .toUpdateTaskExecution()
     *
     * Applies to resource types:
     * - agent
     * - location
     * - task
     * - taskexecution
     * - storagesystem
     * - discoveryjob
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAddStorageSystem()
     * - .toCreateAgent()
     * - .toCreateLocationAzureBlob()
     * - .toCreateLocationEfs()
     * - .toCreateLocationFsxLustre()
     * - .toCreateLocationFsxOntap()
     * - .toCreateLocationFsxOpenZfs()
     * - .toCreateLocationFsxWindows()
     * - .toCreateLocationHdfs()
     * - .toCreateLocationNfs()
     * - .toCreateLocationObjectStorage()
     * - .toCreateLocationS3()
     * - .toCreateLocationSmb()
     * - .toCreateTask()
     * - .toStartTaskExecution()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [datasync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdatasync.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'datasync';
        this.accessLevelList = {
            Write: [
                'AddStorageSystem',
                'CancelTaskExecution',
                'CreateAgent',
                'CreateLocationAzureBlob',
                'CreateLocationEfs',
                'CreateLocationFsxLustre',
                'CreateLocationFsxOntap',
                'CreateLocationFsxOpenZfs',
                'CreateLocationFsxWindows',
                'CreateLocationHdfs',
                'CreateLocationNfs',
                'CreateLocationObjectStorage',
                'CreateLocationS3',
                'CreateLocationSmb',
                'CreateTask',
                'DeleteAgent',
                'DeleteLocation',
                'DeleteTask',
                'GenerateRecommendations',
                'RemoveStorageSystem',
                'StartDiscoveryJob',
                'StartTaskExecution',
                'StopDiscoveryJob',
                'UpdateAgent',
                'UpdateDiscoveryJob',
                'UpdateLocationAzureBlob',
                'UpdateLocationEfs',
                'UpdateLocationFsxLustre',
                'UpdateLocationFsxOntap',
                'UpdateLocationFsxOpenZfs',
                'UpdateLocationFsxWindows',
                'UpdateLocationHdfs',
                'UpdateLocationNfs',
                'UpdateLocationObjectStorage',
                'UpdateLocationS3',
                'UpdateLocationSmb',
                'UpdateStorageSystem',
                'UpdateTask',
                'UpdateTaskExecution'
            ],
            Read: [
                'DescribeAgent',
                'DescribeDiscoveryJob',
                'DescribeLocationAzureBlob',
                'DescribeLocationEfs',
                'DescribeLocationFsxLustre',
                'DescribeLocationFsxOntap',
                'DescribeLocationFsxOpenZfs',
                'DescribeLocationFsxWindows',
                'DescribeLocationHdfs',
                'DescribeLocationNfs',
                'DescribeLocationObjectStorage',
                'DescribeLocationS3',
                'DescribeLocationSmb',
                'DescribeStorageSystem',
                'DescribeTask',
                'DescribeTaskExecution',
                'ListTagsForResource'
            ],
            List: [
                'DescribeStorageSystemResourceMetrics',
                'DescribeStorageSystemResources',
                'ListAgents',
                'ListDiscoveryJobs',
                'ListLocations',
                'ListStorageSystems',
                'ListTaskExecutions',
                'ListTasks'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Datasync = Datasync;
//# sourceMappingURL=data:application/json;base64,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