"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Deadline = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [deadline](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdeadlinecloud.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Deadline extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a member to a farm
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     * - .ifMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssociateMemberToFarm.html
     */
    toAssociateMemberToFarm() {
        return this.to('AssociateMemberToFarm');
    }
    /**
     * Grants permission to associate a member to a fleet
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     * - .ifMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssociateMemberToFleet.html
     */
    toAssociateMemberToFleet() {
        return this.to('AssociateMemberToFleet');
    }
    /**
     * Grants permission to associate a member to a job
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     * - .ifMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssociateMemberToJob.html
     */
    toAssociateMemberToJob() {
        return this.to('AssociateMemberToJob');
    }
    /**
     * Grants permission to associate a member to a queue
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     * - .ifMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssociateMemberToQueue.html
     */
    toAssociateMemberToQueue() {
        return this.to('AssociateMemberToQueue');
    }
    /**
     * Grants permission to assume a fleet role for read-only access
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssumeFleetRoleForRead.html
     */
    toAssumeFleetRoleForRead() {
        return this.to('AssumeFleetRoleForRead');
    }
    /**
     * Grants permission to assume a fleet role for a worker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssumeFleetRoleForWorker.html
     */
    toAssumeFleetRoleForWorker() {
        return this.to('AssumeFleetRoleForWorker');
    }
    /**
     * Grants permission to assume a queue role for read-only access
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssumeQueueRoleForRead.html
     */
    toAssumeQueueRoleForRead() {
        return this.to('AssumeQueueRoleForRead');
    }
    /**
     * Grants permission to assume a queue role for a user
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssumeQueueRoleForUser.html
     */
    toAssumeQueueRoleForUser() {
        return this.to('AssumeQueueRoleForUser');
    }
    /**
     * Grants permission to assume a queue role for a worker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_AssumeQueueRoleForWorker.html
     */
    toAssumeQueueRoleForWorker() {
        return this.to('AssumeQueueRoleForWorker');
    }
    /**
     * Grants permission to get a job entity for a worker
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_BatchGetJobEntity.html
     */
    toBatchGetJobEntity() {
        return this.to('BatchGetJobEntity');
    }
    /**
     * Grants permission to copy a job template to an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CopyJobTemplate.html
     */
    toCopyJobTemplate() {
        return this.to('CopyJobTemplate');
    }
    /**
     * Grants permission to create a budget
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateBudget.html
     */
    toCreateBudget() {
        return this.to('CreateBudget');
    }
    /**
     * Grants permission to create a farm
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:TagResource
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateFarm.html
     */
    toCreateFarm() {
        return this.to('CreateFarm');
    }
    /**
     * Grants permission to create a fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:TagResource
     * - ec2:CreateVpcEndpoint
     * - iam:PassRole
     * - identitystore:ListGroupMembershipsForMember
     * - logs:CreateLogGroup
     * - vpc-lattice:GetResourceConfiguration
     * - vpc-lattice:GetResourceGateway
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet() {
        return this.to('CreateFleet');
    }
    /**
     * Grants permission to create a job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - deadline:GetJobTemplate
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateJob.html
     */
    toCreateJob() {
        return this.to('CreateJob');
    }
    /**
     * Grants permission to create a license endpoint for licensed software or products
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:TagResource
     * - ec2:CreateTags
     * - ec2:CreateVpcEndpoint
     * - ec2:DescribeVpcEndpoints
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateLicenseEndpoint.html
     */
    toCreateLicenseEndpoint() {
        return this.to('CreateLicenseEndpoint');
    }
    /**
     * Grants permission to create a limit for a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateLimit.html
     */
    toCreateLimit() {
        return this.to('CreateLimit');
    }
    /**
     * Grants permission to create a monitor
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:TagResource
     * - iam:PassRole
     * - sso:CreateApplication
     * - sso:DeleteApplication
     * - sso:PutApplicationAssignmentConfiguration
     * - sso:PutApplicationAuthenticationMethod
     * - sso:PutApplicationGrant
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateMonitor.html
     */
    toCreateMonitor() {
        return this.to('CreateMonitor');
    }
    /**
     * Grants permission to create a queue
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:TagResource
     * - iam:PassRole
     * - identitystore:ListGroupMembershipsForMember
     * - logs:CreateLogGroup
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateQueue.html
     */
    toCreateQueue() {
        return this.to('CreateQueue');
    }
    /**
     * Grants permission to create a queue environment
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateQueueEnvironment.html
     */
    toCreateQueueEnvironment() {
        return this.to('CreateQueueEnvironment');
    }
    /**
     * Grants permission to create a queue-fleet association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateQueueFleetAssociation.html
     */
    toCreateQueueFleetAssociation() {
        return this.to('CreateQueueFleetAssociation');
    }
    /**
     * Grants permission to create a queue-limit association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateQueueLimitAssociation.html
     */
    toCreateQueueLimitAssociation() {
        return this.to('CreateQueueLimitAssociation');
    }
    /**
     * Grants permission to create a storage profile for a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateStorageProfile.html
     */
    toCreateStorageProfile() {
        return this.to('CreateStorageProfile');
    }
    /**
     * Grants permission to create a worker
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - deadline:ListTagsForResource
     * - deadline:TagResource
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_CreateWorker.html
     */
    toCreateWorker() {
        return this.to('CreateWorker');
    }
    /**
     * Grants permission to delete a budget
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteBudget.html
     */
    toDeleteBudget() {
        return this.to('DeleteBudget');
    }
    /**
     * Grants permission to delete a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteFarm.html
     */
    toDeleteFarm() {
        return this.to('DeleteFarm');
    }
    /**
     * Grants permission to delete a fleet
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteFleet.html
     */
    toDeleteFleet() {
        return this.to('DeleteFleet');
    }
    /**
     * Grants permission to delete a license endpoint
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteVpcEndpoints
     * - ec2:DescribeVpcEndpoints
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteLicenseEndpoint.html
     */
    toDeleteLicenseEndpoint() {
        return this.to('DeleteLicenseEndpoint');
    }
    /**
     * Grants permission to delete a limit
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteLimit.html
     */
    toDeleteLimit() {
        return this.to('DeleteLimit');
    }
    /**
     * Grants permission to delete a metered product
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteMeteredProduct.html
     */
    toDeleteMeteredProduct() {
        return this.to('DeleteMeteredProduct');
    }
    /**
     * Grants permission to delete a monitor
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sso:DeleteApplication
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteMonitor.html
     */
    toDeleteMonitor() {
        return this.to('DeleteMonitor');
    }
    /**
     * Grants permission to delete a queue
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteQueue.html
     */
    toDeleteQueue() {
        return this.to('DeleteQueue');
    }
    /**
     * Grants permission to delete a queue environment
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteQueueEnvironment.html
     */
    toDeleteQueueEnvironment() {
        return this.to('DeleteQueueEnvironment');
    }
    /**
     * Grants permission to delete a queue-fleet association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteQueueFleetAssociation.html
     */
    toDeleteQueueFleetAssociation() {
        return this.to('DeleteQueueFleetAssociation');
    }
    /**
     * Grants permission to delete a queue-limit association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteQueueLimitAssociation.html
     */
    toDeleteQueueLimitAssociation() {
        return this.to('DeleteQueueLimitAssociation');
    }
    /**
     * Grants permission to delete a storage profile
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteStorageProfile.html
     */
    toDeleteStorageProfile() {
        return this.to('DeleteStorageProfile');
    }
    /**
     * Grants permission to delete a worker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DeleteWorker.html
     */
    toDeleteWorker() {
        return this.to('DeleteWorker');
    }
    /**
     * Grants permission to disassociate a member from a farm
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DisassociateMemberFromFarm.html
     */
    toDisassociateMemberFromFarm() {
        return this.to('DisassociateMemberFromFarm');
    }
    /**
     * Grants permission to disassociate a member from a fleet
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DisassociateMemberFromFleet.html
     */
    toDisassociateMemberFromFleet() {
        return this.to('DisassociateMemberFromFleet');
    }
    /**
     * Grants permission to disassociate a member from a job
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DisassociateMemberFromJob.html
     */
    toDisassociateMemberFromJob() {
        return this.to('DisassociateMemberFromJob');
    }
    /**
     * Grants permission to disassociate a member from a queue
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAssociatedMembershipLevel()
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_DisassociateMemberFromQueue.html
     */
    toDisassociateMemberFromQueue() {
        return this.to('DisassociateMemberFromQueue');
    }
    /**
     * Grants permission to get the latest version of an application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetApplicationVersion.html
     */
    toGetApplicationVersion() {
        return this.to('GetApplicationVersion');
    }
    /**
     * Grants permission to get a budget
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetBudget.html
     */
    toGetBudget() {
        return this.to('GetBudget');
    }
    /**
     * Grants permission to get a farm
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetFarm.html
     */
    toGetFarm() {
        return this.to('GetFarm');
    }
    /**
     * Grants permission to get a fleet
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetFleet.html
     */
    toGetFleet() {
        return this.to('GetFleet');
    }
    /**
     * Grants permission to get a job
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetJob.html
     */
    toGetJob() {
        return this.to('GetJob');
    }
    /**
     * Grants permission to read job template
     *
     * Access Level: Read
     */
    toGetJobTemplate() {
        return this.to('GetJobTemplate');
    }
    /**
     * Grants permission to get a license endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetLicenseEndpoint.html
     */
    toGetLicenseEndpoint() {
        return this.to('GetLicenseEndpoint');
    }
    /**
     * Grants permission to get a limit
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetLimit.html
     */
    toGetLimit() {
        return this.to('GetLimit');
    }
    /**
     * Grants permission to get a monitor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetMonitor.html
     */
    toGetMonitor() {
        return this.to('GetMonitor');
    }
    /**
     * Grants permission to get a queue
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetQueue.html
     */
    toGetQueue() {
        return this.to('GetQueue');
    }
    /**
     * Grants permission to get a queue environment
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetQueueEnvironment.html
     */
    toGetQueueEnvironment() {
        return this.to('GetQueueEnvironment');
    }
    /**
     * Grants permission to get a queue-fleet association
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetQueueFleetAssociation.html
     */
    toGetQueueFleetAssociation() {
        return this.to('GetQueueFleetAssociation');
    }
    /**
     * Grants permission to get a queue-limit association
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetQueueLimitAssociation.html
     */
    toGetQueueLimitAssociation() {
        return this.to('GetQueueLimitAssociation');
    }
    /**
     * Grants permission to get a session for a job
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetSession.html
     */
    toGetSession() {
        return this.to('GetSession');
    }
    /**
     * Grants permission to get a session action for a job
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetSessionAction.html
     */
    toGetSessionAction() {
        return this.to('GetSessionAction');
    }
    /**
     * Grants permission to get all collected statistics for sessions
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetSessionsStatisticsAggregation.html
     */
    toGetSessionsStatisticsAggregation() {
        return this.to('GetSessionsStatisticsAggregation');
    }
    /**
     * Grants permission to get a step in a job
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetStep.html
     */
    toGetStep() {
        return this.to('GetStep');
    }
    /**
     * Grants permission to get a storage profile
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetStorageProfile.html
     */
    toGetStorageProfile() {
        return this.to('GetStorageProfile');
    }
    /**
     * Grants permission to get a storage profile for a queue
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetStorageProfileForQueue.html
     */
    toGetStorageProfileForQueue() {
        return this.to('GetStorageProfileForQueue');
    }
    /**
     * Grants permission to get a job task
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetTask.html
     */
    toGetTask() {
        return this.to('GetTask');
    }
    /**
     * Grants permission to get a worker
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_GetWorker.html
     */
    toGetWorker() {
        return this.to('GetWorker');
    }
    /**
     * Grants permission to list all available metered products within a license endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListAvailableMeteredProducts.html
     */
    toListAvailableMeteredProducts() {
        return this.to('ListAvailableMeteredProducts');
    }
    /**
     * Grants permission to list all budgets for a farm
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListBudgets.html
     */
    toListBudgets() {
        return this.to('ListBudgets');
    }
    /**
     * Grants permission to list all members of a farm
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListFarmMembers.html
     */
    toListFarmMembers() {
        return this.to('ListFarmMembers');
    }
    /**
     * Grants permission to list all farms
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPrincipalId()
     * - .ifRequesterPrincipalId()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListFarms.html
     */
    toListFarms() {
        return this.to('ListFarms');
    }
    /**
     * Grants permission to list all members of a fleet
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListFleetMembers.html
     */
    toListFleetMembers() {
        return this.to('ListFleetMembers');
    }
    /**
     * Grants permission to list all fleets
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPrincipalId()
     * - .ifRequesterPrincipalId()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListFleets.html
     */
    toListFleets() {
        return this.to('ListFleets');
    }
    /**
     * Grants permission to list all members of a job
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListJobMembers.html
     */
    toListJobMembers() {
        return this.to('ListJobMembers');
    }
    /**
     * Grants permission to get a job's parameter definitions in the job template
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListJobParameterDefinitions.html
     */
    toListJobParameterDefinitions() {
        return this.to('ListJobParameterDefinitions');
    }
    /**
     * Grants permission to list all jobs in a queue
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPrincipalId()
     * - .ifRequesterPrincipalId()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListJobs.html
     */
    toListJobs() {
        return this.to('ListJobs');
    }
    /**
     * Grants permission to list all license endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListLicenseEndpoints.html
     */
    toListLicenseEndpoints() {
        return this.to('ListLicenseEndpoints');
    }
    /**
     * Grants permission to list all limits in a farm
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListLimits.html
     */
    toListLimits() {
        return this.to('ListLimits');
    }
    /**
     * Grants permission to list all metered products in a license endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListMeteredProducts.html
     */
    toListMeteredProducts() {
        return this.to('ListMeteredProducts');
    }
    /**
     * Grants permission to list all monitors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListMonitors.html
     */
    toListMonitors() {
        return this.to('ListMonitors');
    }
    /**
     * Grants permission to list all queue environments to which a queue is associated
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListQueueEnvironments.html
     */
    toListQueueEnvironments() {
        return this.to('ListQueueEnvironments');
    }
    /**
     * Grants permission to list all queue-fleet associations
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListQueueFleetAssociations.html
     */
    toListQueueFleetAssociations() {
        return this.to('ListQueueFleetAssociations');
    }
    /**
     * Grants permission to list all queue-limit associations
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListQueueLimitAssociations.html
     */
    toListQueueLimitAssociations() {
        return this.to('ListQueueLimitAssociations');
    }
    /**
     * Grants permission to list all members in a queue
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListQueueMembers.html
     */
    toListQueueMembers() {
        return this.to('ListQueueMembers');
    }
    /**
     * Grants permission to list all queues on a farm
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifPrincipalId()
     * - .ifRequesterPrincipalId()
     *
     * Dependent actions:
     * - identitystore:DescribeGroup
     * - identitystore:DescribeUser
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListQueues.html
     */
    toListQueues() {
        return this.to('ListQueues');
    }
    /**
     * Grants permission to list all session actions for a job
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListSessionActions.html
     */
    toListSessionActions() {
        return this.to('ListSessionActions');
    }
    /**
     * Grants permission to list all sessions for a job
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListSessions.html
     */
    toListSessions() {
        return this.to('ListSessions');
    }
    /**
     * Grants permission to list all sessions for a worker
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListSessionsForWorker.html
     */
    toListSessionsForWorker() {
        return this.to('ListSessionsForWorker');
    }
    /**
     * Grants permission to list the step consumers for a job step
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListStepConsumers.html
     */
    toListStepConsumers() {
        return this.to('ListStepConsumers');
    }
    /**
     * Grants permission to list dependencies for a job step
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListStepDependencies.html
     */
    toListStepDependencies() {
        return this.to('ListStepDependencies');
    }
    /**
     * Grants permission to list all steps for a job
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListSteps.html
     */
    toListSteps() {
        return this.to('ListSteps');
    }
    /**
     * Grants permission to list all storage profiles in a farm
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListStorageProfiles.html
     */
    toListStorageProfiles() {
        return this.to('ListStorageProfiles');
    }
    /**
     * Grants permission to list all storage profiles in a queue
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListStorageProfilesForQueue.html
     */
    toListStorageProfilesForQueue() {
        return this.to('ListStorageProfilesForQueue');
    }
    /**
     * Grants permission to list all tags on specified Deadline Cloud resources
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCalledAction()
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all tasks for a job
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListTasks.html
     */
    toListTasks() {
        return this.to('ListTasks');
    }
    /**
     * Grants permission to list all workers in a fleet
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_ListWorkers.html
     */
    toListWorkers() {
        return this.to('ListWorkers');
    }
    /**
     * Grants permission to add a metered product to a license endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_PutMeteredProduct.html
     */
    toPutMeteredProduct() {
        return this.to('PutMeteredProduct');
    }
    /**
     * Grants permission to search for jobs in multiple queues
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_SearchJobs.html
     */
    toSearchJobs() {
        return this.to('SearchJobs');
    }
    /**
     * Grants permission to search the steps within a single job or to search the steps for multiple queues
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_SearchSteps.html
     */
    toSearchSteps() {
        return this.to('SearchSteps');
    }
    /**
     * Grants permission to search the tasks within a single job or to search the tasks for multiple queues
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_SearchTasks.html
     */
    toSearchTasks() {
        return this.to('SearchTasks');
    }
    /**
     * Grants permission to search for workers in multiple fleets
     *
     * Access Level: List
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_SearchWorkers.html
     */
    toSearchWorkers() {
        return this.to('SearchWorkers');
    }
    /**
     * Grants permission to get all collected statistics for sessions
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_StartSessionsStatisticsAggregation.html
     */
    toStartSessionsStatisticsAggregation() {
        return this.to('StartSessionsStatisticsAggregation');
    }
    /**
     * Grants permission to add or overwrite one or more tags for the specified Deadline Cloud resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCalledAction()
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to disassociate one or more tags from the specified Deadline Cloud resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a budget
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateBudget.html
     */
    toUpdateBudget() {
        return this.to('UpdateBudget');
    }
    /**
     * Grants permission to update a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateFarm.html
     */
    toUpdateFarm() {
        return this.to('UpdateFarm');
    }
    /**
     * Grants permission to update a fleet
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateVpcEndpoint
     * - iam:PassRole
     * - identitystore:ListGroupMembershipsForMember
     * - vpc-lattice:GetResourceConfiguration
     * - vpc-lattice:GetResourceGateway
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateFleet.html
     */
    toUpdateFleet() {
        return this.to('UpdateFleet');
    }
    /**
     * Grants permission to update a job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateJob.html
     */
    toUpdateJob() {
        return this.to('UpdateJob');
    }
    /**
     * Grants permission to update a limit for a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateLimit.html
     */
    toUpdateLimit() {
        return this.to('UpdateLimit');
    }
    /**
     * Grants permission to update a monitor
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - sso:PutApplicationGrant
     * - sso:UpdateApplication
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateMonitor.html
     */
    toUpdateMonitor() {
        return this.to('UpdateMonitor');
    }
    /**
     * Grants permission to update a queue
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateQueue.html
     */
    toUpdateQueue() {
        return this.to('UpdateQueue');
    }
    /**
     * Grants permission to update a queue environment
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateQueueEnvironment.html
     */
    toUpdateQueueEnvironment() {
        return this.to('UpdateQueueEnvironment');
    }
    /**
     * Grants permission to update a queue-fleet association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateQueueFleetAssociation.html
     */
    toUpdateQueueFleetAssociation() {
        return this.to('UpdateQueueFleetAssociation');
    }
    /**
     * Grants permission to update a queue-limit association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateQueueLimitAssociation.html
     */
    toUpdateQueueLimitAssociation() {
        return this.to('UpdateQueueLimitAssociation');
    }
    /**
     * Grants permission to update a session for a job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateSession.html
     */
    toUpdateSession() {
        return this.to('UpdateSession');
    }
    /**
     * Grants permission to update a step for a job
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateStep.html
     */
    toUpdateStep() {
        return this.to('UpdateStep');
    }
    /**
     * Grants permission to update a storage profile for a farm
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateStorageProfile.html
     */
    toUpdateStorageProfile() {
        return this.to('UpdateStorageProfile');
    }
    /**
     * Grants permission to update a task
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:ListGroupMembershipsForMember
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateTask.html
     */
    toUpdateTask() {
        return this.to('UpdateTask');
    }
    /**
     * Grants permission to update a worker
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - logs:CreateLogStream
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateWorker.html
     */
    toUpdateWorker() {
        return this.to('UpdateWorker');
    }
    /**
     * Grants permission to update the schedule for a worker
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - logs:CreateLogStream
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/APIReference/API_UpdateWorkerSchedule.html
     */
    toUpdateWorkerSchedule() {
        return this.to('UpdateWorkerSchedule');
    }
    /**
     * Adds a resource of type budget to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/manage-costs.html
     *
     * @param farmId - Identifier for the farmId.
     * @param budgetId - Identifier for the budgetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifFarmMembershipLevels()
     */
    onBudget(farmId, budgetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deadline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:farm/${farmId}/budget/${budgetId}`);
    }
    /**
     * Adds a resource of type farm to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/farms.html
     *
     * @param farmId - Identifier for the farmId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifFarmMembershipLevels()
     */
    onFarm(farmId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deadline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:farm/${farmId}`);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/manage-fleets.html
     *
     * @param farmId - Identifier for the farmId.
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifFarmMembershipLevels()
     * - .ifFleetMembershipLevels()
     */
    onFleet(farmId, fleetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deadline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:farm/${farmId}/fleet/${fleetId}`);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/deadline-cloud-jobs.html
     *
     * @param farmId - Identifier for the farmId.
     * @param queueId - Identifier for the queueId.
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifFarmMembershipLevels()
     * - .ifJobMembershipLevels()
     * - .ifQueueMembershipLevels()
     */
    onJob(farmId, queueId, jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deadline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:farm/${farmId}/queue/${queueId}/job/${jobId}`);
    }
    /**
     * Adds a resource of type license-endpoint to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/cmf-ubl.html
     *
     * @param licenseEndpointId - Identifier for the licenseEndpointId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLicenseEndpoint(licenseEndpointId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deadline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:license-endpoint/${licenseEndpointId}`);
    }
    /**
     * Adds a resource of type metered-product to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/cmf-ubl.html
     *
     * @param licenseEndpointId - Identifier for the licenseEndpointId.
     * @param productId - Identifier for the productId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMeteredProduct(licenseEndpointId, productId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deadline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:license-endpoint/${licenseEndpointId}/metered-product/${productId}`);
    }
    /**
     * Adds a resource of type monitor to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/working-with-deadline-monitor.html
     *
     * @param monitorId - Identifier for the monitorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMonitor(monitorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deadline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:monitor/${monitorId}`);
    }
    /**
     * Adds a resource of type queue to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/queues.html
     *
     * @param farmId - Identifier for the farmId.
     * @param queueId - Identifier for the queueId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifFarmMembershipLevels()
     * - .ifQueueMembershipLevels()
     */
    onQueue(farmId, queueId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deadline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:farm/${farmId}/queue/${queueId}`);
    }
    /**
     * Adds a resource of type worker to the statement
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam.html
     *
     * @param farmId - Identifier for the farmId.
     * @param fleetId - Identifier for the fleetId.
     * @param workerId - Identifier for the workerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifFarmMembershipLevels()
     * - .ifFleetMembershipLevels()
     */
    onWorker(farmId, fleetId, workerId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:deadline:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:farm/${farmId}/fleet/${fleetId}/worker/${workerId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateFarm()
     * - .toCreateFleet()
     * - .toCreateLicenseEndpoint()
     * - .toCreateMonitor()
     * - .toCreateQueue()
     * - .toCreateWorker()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - farm
     * - fleet
     * - license-endpoint
     * - monitor
     * - queue
     * - worker
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateFarm()
     * - .toCreateFleet()
     * - .toCreateLicenseEndpoint()
     * - .toCreateMonitor()
     * - .toCreateQueue()
     * - .toCreateWorker()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the associated membership level of the principal provided in the request
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toAssociateMemberToFarm()
     * - .toAssociateMemberToFleet()
     * - .toAssociateMemberToJob()
     * - .toAssociateMemberToQueue()
     * - .toDisassociateMemberFromFarm()
     * - .toDisassociateMemberFromFleet()
     * - .toDisassociateMemberFromJob()
     * - .toDisassociateMemberFromQueue()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssociatedMembershipLevel(value, operator) {
        return this.if(`AssociatedMembershipLevel`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the allowed action in the request
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCalledAction(value, operator) {
        return this.if(`CalledAction`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by membership levels on the farm
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to resource types:
     * - budget
     * - farm
     * - fleet
     * - job
     * - queue
     * - worker
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFarmMembershipLevels(value, operator) {
        return this.if(`FarmMembershipLevels`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by membership levels on the fleet
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to resource types:
     * - fleet
     * - worker
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifFleetMembershipLevels(value, operator) {
        return this.if(`FleetMembershipLevels`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by membership levels on the job
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to resource types:
     * - job
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobMembershipLevels(value, operator) {
        return this.if(`JobMembershipLevels`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the membership level passed in the request
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toAssociateMemberToFarm()
     * - .toAssociateMemberToFleet()
     * - .toAssociateMemberToJob()
     * - .toAssociateMemberToQueue()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMembershipLevel(value, operator) {
        return this.if(`MembershipLevel`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the principle ID provided in the request
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toListFarms()
     * - .toListFleets()
     * - .toListJobs()
     * - .toListQueues()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrincipalId(value, operator) {
        return this.if(`PrincipalId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by membership levels on the queue
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to resource types:
     * - job
     * - queue
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifQueueMembershipLevels(value, operator) {
        return this.if(`QueueMembershipLevels`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the user calling the Deadline Cloud API
     *
     * https://docs.aws.amazon.com/deadline-cloud/latest/userguide/security-iam-service-with-iam.html
     *
     * Applies to actions:
     * - .toListFarms()
     * - .toListFleets()
     * - .toListJobs()
     * - .toListQueues()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequesterPrincipalId(value, operator) {
        return this.if(`RequesterPrincipalId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [deadline](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdeadlinecloud.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'deadline';
        this.accessLevelList = {
            'Permissions management': [
                'AssociateMemberToFarm',
                'AssociateMemberToFleet',
                'AssociateMemberToJob',
                'AssociateMemberToQueue',
                'DisassociateMemberFromFarm',
                'DisassociateMemberFromFleet',
                'DisassociateMemberFromJob',
                'DisassociateMemberFromQueue'
            ],
            Write: [
                'AssumeFleetRoleForRead',
                'AssumeFleetRoleForWorker',
                'AssumeQueueRoleForRead',
                'AssumeQueueRoleForUser',
                'AssumeQueueRoleForWorker',
                'CopyJobTemplate',
                'CreateBudget',
                'CreateFarm',
                'CreateFleet',
                'CreateJob',
                'CreateLicenseEndpoint',
                'CreateLimit',
                'CreateMonitor',
                'CreateQueue',
                'CreateQueueEnvironment',
                'CreateQueueFleetAssociation',
                'CreateQueueLimitAssociation',
                'CreateStorageProfile',
                'CreateWorker',
                'DeleteBudget',
                'DeleteFarm',
                'DeleteFleet',
                'DeleteLicenseEndpoint',
                'DeleteLimit',
                'DeleteMeteredProduct',
                'DeleteMonitor',
                'DeleteQueue',
                'DeleteQueueEnvironment',
                'DeleteQueueFleetAssociation',
                'DeleteQueueLimitAssociation',
                'DeleteStorageProfile',
                'DeleteWorker',
                'PutMeteredProduct',
                'UpdateBudget',
                'UpdateFarm',
                'UpdateFleet',
                'UpdateJob',
                'UpdateLimit',
                'UpdateMonitor',
                'UpdateQueue',
                'UpdateQueueEnvironment',
                'UpdateQueueFleetAssociation',
                'UpdateQueueLimitAssociation',
                'UpdateSession',
                'UpdateStep',
                'UpdateStorageProfile',
                'UpdateTask',
                'UpdateWorker',
                'UpdateWorkerSchedule'
            ],
            Read: [
                'BatchGetJobEntity',
                'GetApplicationVersion',
                'GetBudget',
                'GetFarm',
                'GetFleet',
                'GetJob',
                'GetJobTemplate',
                'GetLicenseEndpoint',
                'GetLimit',
                'GetMonitor',
                'GetQueue',
                'GetQueueEnvironment',
                'GetQueueFleetAssociation',
                'GetQueueLimitAssociation',
                'GetSession',
                'GetSessionAction',
                'GetSessionsStatisticsAggregation',
                'GetStep',
                'GetStorageProfile',
                'GetStorageProfileForQueue',
                'GetTask',
                'GetWorker',
                'StartSessionsStatisticsAggregation'
            ],
            List: [
                'ListAvailableMeteredProducts',
                'ListBudgets',
                'ListFarmMembers',
                'ListFarms',
                'ListFleetMembers',
                'ListFleets',
                'ListJobMembers',
                'ListJobParameterDefinitions',
                'ListJobs',
                'ListLicenseEndpoints',
                'ListLimits',
                'ListMeteredProducts',
                'ListMonitors',
                'ListQueueEnvironments',
                'ListQueueFleetAssociations',
                'ListQueueLimitAssociations',
                'ListQueueMembers',
                'ListQueues',
                'ListSessionActions',
                'ListSessions',
                'ListSessionsForWorker',
                'ListStepConsumers',
                'ListStepDependencies',
                'ListSteps',
                'ListStorageProfiles',
                'ListStorageProfilesForQueue',
                'ListTagsForResource',
                'ListTasks',
                'ListWorkers',
                'SearchJobs',
                'SearchSteps',
                'SearchTasks',
                'SearchWorkers'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Deadline = Deadline;
//# sourceMappingURL=data:application/json;base64,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