"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DevopsGuru = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [devops-guru](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondevopsguru.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class DevopsGuru extends shared_1.PolicyStatement {
    /**
     * Grants permission to add a notification channel to DevOps Guru
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sns:GetTopicAttributes
     * - sns:SetTopicAttributes
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_AddNotificationChannel.html
     */
    toAddNotificationChannel() {
        return this.to('AddNotificationChannel');
    }
    /**
     * Grants permission to delete specified insight in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DeleteInsight.html
     */
    toDeleteInsight() {
        return this.to('DeleteInsight');
    }
    /**
     * Grants permission to view the health of operations in your AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeAccountHealth.html
     */
    toDescribeAccountHealth() {
        return this.to('DescribeAccountHealth');
    }
    /**
     * Grants permission to view the health of operations within a time range in your AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeAccountOverview.html
     */
    toDescribeAccountOverview() {
        return this.to('DescribeAccountOverview');
    }
    /**
     * Grants permission to list the details of a specified anomaly
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeAnomaly.html
     */
    toDescribeAnomaly() {
        return this.to('DescribeAnomaly');
    }
    /**
     * Grants permission to retrieve details about event sources for DevOps Guru
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeEventSourcesConfig.html
     */
    toDescribeEventSourcesConfig() {
        return this.to('DescribeEventSourcesConfig');
    }
    /**
     * Grants permission to view the feedback details of a specified insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeFeedback.html
     */
    toDescribeFeedback() {
        return this.to('DescribeFeedback');
    }
    /**
     * Grants permission to list the details of a specified insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeInsight.html
     */
    toDescribeInsight() {
        return this.to('DescribeInsight');
    }
    /**
     * Grants permission to view the health of operations in your organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeOrganizationHealth.html
     */
    toDescribeOrganizationHealth() {
        return this.to('DescribeOrganizationHealth');
    }
    /**
     * Grants permission to view the health of operations within a time range in your organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeOrganizationOverview.html
     */
    toDescribeOrganizationOverview() {
        return this.to('DescribeOrganizationOverview');
    }
    /**
     * Grants permission to view the health of operations for each AWS CloudFormation stack or AWS Services or accounts specified in DevOps Guru in your organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeOrganizationResourceCollectionHealth.html
     */
    toDescribeOrganizationResourceCollectionHealth() {
        return this.to('DescribeOrganizationResourceCollectionHealth');
    }
    /**
     * Grants permission to view the health of operations for each AWS CloudFormation stack specified in DevOps Guru
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeResourceCollectionHealth.html
     */
    toDescribeResourceCollectionHealth() {
        return this.to('DescribeResourceCollectionHealth');
    }
    /**
     * Grants permission to view the integration status of services that can be integrated with DevOps Guru
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_DescribeServiceIntegration.html
     */
    toDescribeServiceIntegration() {
        return this.to('DescribeServiceIntegration');
    }
    /**
     * Grants permission to list service resource cost estimates
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_GetCostEstimation.html
     */
    toGetCostEstimation() {
        return this.to('GetCostEstimation');
    }
    /**
     * Grants permission to list AWS CloudFormation stacks that DevOps Guru is configured to use
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_GetResourceCollection.html
     */
    toGetResourceCollection() {
        return this.to('GetResourceCollection');
    }
    /**
     * Grants permission to list anomalies of a given insight in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServiceNames()
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_ListAnomaliesForInsight.html
     */
    toListAnomaliesForInsight() {
        return this.to('ListAnomaliesForInsight');
    }
    /**
     * Grants permission to list log anomalies of a given insight in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_ListAnomalousLogGroups.html
     */
    toListAnomalousLogGroups() {
        return this.to('ListAnomalousLogGroups');
    }
    /**
     * Grants permission to list resource events that are evaluated by DevOps Guru
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_ListEvents.html
     */
    toListEvents() {
        return this.to('ListEvents');
    }
    /**
     * Grants permission to list insights in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_ListInsights.html
     */
    toListInsights() {
        return this.to('ListInsights');
    }
    /**
     * Grants permission to list resource monitored by DevOps Guru in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_ListMonitoredResources.html
     */
    toListMonitoredResources() {
        return this.to('ListMonitoredResources');
    }
    /**
     * Grants permission to list notification channels configured for DevOps Guru in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_ListNotificationChannels.html
     */
    toListNotificationChannels() {
        return this.to('ListNotificationChannels');
    }
    /**
     * Grants permission to list insights in your organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_ListOrganizationInsights.html
     */
    toListOrganizationInsights() {
        return this.to('ListOrganizationInsights');
    }
    /**
     * Grants permission to list a specified insight's recommendations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_ListRecommendations.html
     */
    toListRecommendations() {
        return this.to('ListRecommendations');
    }
    /**
     * Grants permission to submit a feedback to DevOps Guru
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PutFeedback.html
     */
    toPutFeedback() {
        return this.to('PutFeedback');
    }
    /**
     * Grants permission to remove a notification channel from DevOps Guru
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sns:GetTopicAttributes
     * - sns:SetTopicAttributes
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_RemoveNotificationChannel.html
     */
    toRemoveNotificationChannel() {
        return this.to('RemoveNotificationChannel');
    }
    /**
     * Grants permission to search insights in your account
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifServiceNames()
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_SearchInsights.html
     */
    toSearchInsights() {
        return this.to('SearchInsights');
    }
    /**
     * Grants permission to search insights in your organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_SearchOrganizationInsights.html
     */
    toSearchOrganizationInsights() {
        return this.to('SearchOrganizationInsights');
    }
    /**
     * Grants permission to start the creation of an estimate of the monthly cost
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_StartCostEstimation.html
     */
    toStartCostEstimation() {
        return this.to('StartCostEstimation');
    }
    /**
     * Grants permission to update an event source for DevOps Guru
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_UpdateEventSourcesConfig.html
     */
    toUpdateEventSourcesConfig() {
        return this.to('UpdateEventSourcesConfig');
    }
    /**
     * Grants permission to update the list of AWS CloudFormation stacks that are used to specify which AWS resources in your account are analyzed by DevOps Guru
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_UpdateResourceCollection.html
     */
    toUpdateResourceCollection() {
        return this.to('UpdateResourceCollection');
    }
    /**
     * Grants permission to enable or disable a service that integrates with DevOps Guru
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_UpdateServiceIntegration.html
     */
    toUpdateServiceIntegration() {
        return this.to('UpdateServiceIntegration');
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/devops-guru/latest/userguide/setting-up.html#setting-up-notifications
     *
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTopic(topicName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sns:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${topicName}`);
    }
    /**
     * Filters access by API to restrict access to given AWS service names
     *
     * https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_ServiceCollection.html
     *
     * Applies to actions:
     * - .toListAnomaliesForInsight()
     * - .toSearchInsights()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifServiceNames(value, operator) {
        return this.if(`ServiceNames`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [devops-guru](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondevopsguru.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'devops-guru';
        this.accessLevelList = {
            Write: [
                'AddNotificationChannel',
                'DeleteInsight',
                'PutFeedback',
                'RemoveNotificationChannel',
                'UpdateEventSourcesConfig',
                'UpdateResourceCollection',
                'UpdateServiceIntegration'
            ],
            Read: [
                'DescribeAccountHealth',
                'DescribeAccountOverview',
                'DescribeAnomaly',
                'DescribeEventSourcesConfig',
                'DescribeFeedback',
                'DescribeInsight',
                'DescribeOrganizationHealth',
                'DescribeOrganizationOverview',
                'DescribeOrganizationResourceCollectionHealth',
                'DescribeResourceCollectionHealth',
                'DescribeServiceIntegration',
                'GetCostEstimation',
                'GetResourceCollection',
                'StartCostEstimation'
            ],
            List: [
                'ListAnomaliesForInsight',
                'ListAnomalousLogGroups',
                'ListEvents',
                'ListInsights',
                'ListMonitoredResources',
                'ListNotificationChannels',
                'ListOrganizationInsights',
                'ListRecommendations',
                'SearchInsights',
                'SearchOrganizationInsights'
            ]
        };
    }
}
exports.DevopsGuru = DevopsGuru;
//# sourceMappingURL=data:application/json;base64,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