"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ds = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ds](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdirectoryservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ds extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept a directory sharing request that was sent from the directory owner account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AcceptSharedDirectory.html
     */
    toAcceptSharedDirectory() {
        return this.to('AcceptSharedDirectory');
    }
    /**
     * Grants permission to access directory data using the Directory Service Data API
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toAccessDSData() {
        return this.to('AccessDSData');
    }
    /**
     * Grants permission to add a CIDR address block to correctly route traffic to and from your Microsoft AD on Amazon Web Services
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:DescribeSecurityGroups
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddIpRoutes.html
     */
    toAddIpRoutes() {
        return this.to('AddIpRoutes');
    }
    /**
     * Grants permission to add two domain controllers in the specified Region for the specified directory
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddRegion.html
     */
    toAddRegion() {
        return this.to('AddRegion');
    }
    /**
     * Grants permission to add or overwrite one or more tags for the specified Amazon Directory Services directory
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateTags
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        return this.to('AddTagsToResource');
    }
    /**
     * Grants permission to authorize an application for your AWS Directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toAuthorizeApplication() {
        return this.to('AuthorizeApplication');
    }
    /**
     * Grants permission to cancel an in-progress schema extension to a Microsoft AD directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CancelSchemaExtension.html
     */
    toCancelSchemaExtension() {
        return this.to('CancelSchemaExtension');
    }
    /**
     * Grants permission to verify that the alias is available for use
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toCheckAlias() {
        return this.to('CheckAlias');
    }
    /**
     * Grants permission to create an AD Connector to connect to an on-premises directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ConnectDirectory.html
     */
    toConnectDirectory() {
        return this.to('ConnectDirectory');
    }
    /**
     * Grants permission to create an alias for a directory and assigns the alias to the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateAlias.html
     */
    toCreateAlias() {
        return this.to('CreateAlias');
    }
    /**
     * Grants permission to create a computer account in the specified directory, and joins the computer to the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateComputer.html
     */
    toCreateComputer() {
        return this.to('CreateComputer');
    }
    /**
     * Grants permission to create a conditional forwarder associated with your AWS directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateConditionalForwarder.html
     */
    toCreateConditionalForwarder() {
        return this.to('CreateConditionalForwarder');
    }
    /**
     * Grants permission to create a Simple AD directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateDirectory.html
     */
    toCreateDirectory() {
        return this.to('CreateDirectory');
    }
    /**
     * Grants permission to create a Hybrid Managed AD directory
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateNetworkInterfacePermission
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - iam:CreateServiceLinkedRole
     * - iam:GetRole
     * - secretsmanager:DescribeSecret
     * - secretsmanager:GetSecretValue
     * - ssm:GetCommandInvocation
     * - ssm:GetConnectionStatus
     * - ssm:ListCommands
     * - ssm:SendCommand
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateHybridAD.html
     */
    toCreateHybridAD() {
        return this.to('CreateHybridAD');
    }
    /**
     * Grants permission to create an IdentityPool Directory in the AWS cloud
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toCreateIdentityPoolDirectory() {
        return this.to('CreateIdentityPoolDirectory');
    }
    /**
     * Grants permission to create a subscription to forward real time Directory Service domain controller security logs to the specified CloudWatch log group in your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateLogSubscription.html
     */
    toCreateLogSubscription() {
        return this.to('CreateLogSubscription');
    }
    /**
     * Grants permission to create a Microsoft AD in the AWS cloud
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateMicrosoftAD.html
     */
    toCreateMicrosoftAD() {
        return this.to('CreateMicrosoftAD');
    }
    /**
     * Grants permission to create a snapshot of a Simple AD or Microsoft AD directory in the AWS cloud
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permission to initiate the creation of the AWS side of a trust relationship between a Microsoft AD in the AWS cloud and an external domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_CreateTrust.html
     */
    toCreateTrust() {
        return this.to('CreateTrust');
    }
    /**
     * Grants permission to delete a directory assessment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteADAssessment.html
     */
    toDeleteADAssessment() {
        return this.to('DeleteADAssessment');
    }
    /**
     * Grants permission to delete a conditional forwarder that has been set up for your AWS directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteConditionalForwarder.html
     */
    toDeleteConditionalForwarder() {
        return this.to('DeleteConditionalForwarder');
    }
    /**
     * Grants permission to delete an AWS Directory Service directory
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteNetworkInterface
     * - ec2:DeleteSecurityGroup
     * - ec2:DescribeNetworkInterfaces
     * - ec2:RevokeSecurityGroupEgress
     * - ec2:RevokeSecurityGroupIngress
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteDirectory.html
     */
    toDeleteDirectory() {
        return this.to('DeleteDirectory');
    }
    /**
     * Grants permission to delete the specified log subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteLogSubscription.html
     */
    toDeleteLogSubscription() {
        return this.to('DeleteLogSubscription');
    }
    /**
     * Grants permission to delete a directory snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        return this.to('DeleteSnapshot');
    }
    /**
     * Grants permission to delete an existing trust relationship between your Microsoft AD in the AWS cloud and an external domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeleteTrust.html
     */
    toDeleteTrust() {
        return this.to('DeleteTrust');
    }
    /**
     * Grants permission to delete from the system the certificate that was registered for a secured LDAP connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterCertificate.html
     */
    toDeregisterCertificate() {
        return this.to('DeregisterCertificate');
    }
    /**
     * Grants permission to remove the specified directory as a publisher to the specified SNS topic
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DeregisterEventTopic.html
     */
    toDeregisterEventTopic() {
        return this.to('DeregisterEventTopic');
    }
    /**
     * Grants permission to describe a directory assessment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeADAssessment.html
     */
    toDescribeADAssessment() {
        return this.to('DescribeADAssessment');
    }
    /**
     * Grants permission to describe the ca enrollment status of a specified directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeCAEnrollmentPolicy.html
     */
    toDescribeCAEnrollmentPolicy() {
        return this.to('DescribeCAEnrollmentPolicy');
    }
    /**
     * Grants permission to display information about the certificate registered for a secured LDAP connection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeCertificate.html
     */
    toDescribeCertificate() {
        return this.to('DescribeCertificate');
    }
    /**
     * Grants permission to retrieve information about the type of client authentication for the specified directory, if the type is specified. If no type is specified, information about all client authentication types that are supported for the specified directory is retrieved. Currently, only SmartCard is supported
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeClientAuthenticationSettings.html
     */
    toDescribeClientAuthenticationSettings() {
        return this.to('DescribeClientAuthenticationSettings');
    }
    /**
     * Grants permission to obtain information about the conditional forwarders for this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeConditionalForwarders.html
     */
    toDescribeConditionalForwarders() {
        return this.to('DescribeConditionalForwarders');
    }
    /**
     * Grants permission to obtain information about the directories that belong to this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectories.html
     */
    toDescribeDirectories() {
        return this.to('DescribeDirectories');
    }
    /**
     * Grants permission to describe the Directory Service Data API status for the specified directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDirectoryDataAccess.html
     */
    toDescribeDirectoryDataAccess() {
        return this.to('DescribeDirectoryDataAccess');
    }
    /**
     * Grants permission to provide information about any domain controllers in your directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeDomainControllers.html
     */
    toDescribeDomainControllers() {
        return this.to('DescribeDomainControllers');
    }
    /**
     * Grants permission to obtain information about which SNS topics receive status messages from the specified directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeEventTopics.html
     */
    toDescribeEventTopics() {
        return this.to('DescribeEventTopics');
    }
    /**
     * Grants permission to describe the updates of a specified hybrid directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeHybridADUpdate.html
     */
    toDescribeHybridADUpdate() {
        return this.to('DescribeHybridADUpdate');
    }
    /**
     * Grants permission to describe the status of LDAP security for the specified directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeLDAPSSettings.html
     */
    toDescribeLDAPSSettings() {
        return this.to('DescribeLDAPSSettings');
    }
    /**
     * Grants permission to provide information about the Regions that are configured for multi-Region replication
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeRegions.html
     */
    toDescribeRegions() {
        return this.to('DescribeRegions');
    }
    /**
     * Grants permission to retrieve information about the configurable settings for the specified directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSettings.html
     */
    toDescribeSettings() {
        return this.to('DescribeSettings');
    }
    /**
     * Grants permission to return the shared directories in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSharedDirectories.html
     */
    toDescribeSharedDirectories() {
        return this.to('DescribeSharedDirectories');
    }
    /**
     * Grants permission to obtain information about the directory snapshots that belong to this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        return this.to('DescribeSnapshots');
    }
    /**
     * Grants permission to obtain information about the trust relationships for this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeTrusts.html
     */
    toDescribeTrusts() {
        return this.to('DescribeTrusts');
    }
    /**
     * Grants permission to describe the updates of a directory for a particular update type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DescribeUpdateDirectory.html
     */
    toDescribeUpdateDirectory() {
        return this.to('DescribeUpdateDirectory');
    }
    /**
     * Grants permission to disable the ca enrollment of a specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableCAEnrollmentPolicy.html
     */
    toDisableCAEnrollmentPolicy() {
        return this.to('DisableCAEnrollmentPolicy');
    }
    /**
     * Grants permission to disable alternative client authentication methods for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableClientAuthentication.html
     */
    toDisableClientAuthentication() {
        return this.to('DisableClientAuthentication');
    }
    /**
     * Grants permission to disable the Directory Service Data API for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableDirectoryDataAccess.html
     */
    toDisableDirectoryDataAccess() {
        return this.to('DisableDirectoryDataAccess');
    }
    /**
     * Grants permission to deactivate LDAP secure calls for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableLDAPS.html
     */
    toDisableLDAPS() {
        return this.to('DisableLDAPS');
    }
    /**
     * Grants permission to disable multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableRadius.html
     */
    toDisableRadius() {
        return this.to('DisableRadius');
    }
    /**
     * Grants permission to disable AWS Management Console access for identity in your AWS Directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toDisableRoleAccess() {
        return this.to('DisableRoleAccess');
    }
    /**
     * Grants permission to disable single-sign on for a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_DisableSso.html
     */
    toDisableSso() {
        return this.to('DisableSso');
    }
    /**
     * Grants permission to enable the ca enrollment of a specified directory
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - acm-pca:DescribeCertificateAuthority
     * - pca-connector-ad:GetConnector
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableCAEnrollmentPolicy.html
     */
    toEnableCAEnrollmentPolicy() {
        return this.to('EnableCAEnrollmentPolicy');
    }
    /**
     * Grants permission to enable alternative client authentication methods for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableClientAuthentication.html
     */
    toEnableClientAuthentication() {
        return this.to('EnableClientAuthentication');
    }
    /**
     * Grants permission to enable the Directory Service Data API for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableDirectoryDataAccess.html
     */
    toEnableDirectoryDataAccess() {
        return this.to('EnableDirectoryDataAccess');
    }
    /**
     * Grants permission to activate the switch for the specific directory to always use LDAP secure calls
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableLDAPS.html
     */
    toEnableLDAPS() {
        return this.to('EnableLDAPS');
    }
    /**
     * Grants permission to enable multi-factor authentication (MFA) with the Remote Authentication Dial In User Service (RADIUS) server for an AD Connector directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableRadius.html
     */
    toEnableRadius() {
        return this.to('EnableRadius');
    }
    /**
     * Grants permission to enable AWS Management Console access for identity in your AWS Directory
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toEnableRoleAccess() {
        return this.to('EnableRoleAccess');
    }
    /**
     * Grants permission to enable single-sign on for a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_EnableSso.html
     */
    toEnableSso() {
        return this.to('EnableSso');
    }
    /**
     * Grants permission to retrieve the details of the authorized applications on a directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toGetAuthorizedApplicationDetails() {
        return this.to('GetAuthorizedApplicationDetails');
    }
    /**
     * Grants permission to obtain directory limit information for the current region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetDirectoryLimits.html
     */
    toGetDirectoryLimits() {
        return this.to('GetDirectoryLimits');
    }
    /**
     * Grants permission to obtain the manual snapshot limits for a directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_GetSnapshotLimits.html
     */
    toGetSnapshotLimits() {
        return this.to('GetSnapshotLimits');
    }
    /**
     * Grants permission to list directory assessments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListADAssessments.html
     */
    toListADAssessments() {
        return this.to('ListADAssessments');
    }
    /**
     * Grants permission to obtain the AWS applications authorized for a directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toListAuthorizedApplications() {
        return this.to('ListAuthorizedApplications');
    }
    /**
     * Grants permission to list all the certificates registered for a secured LDAP connection, for the specified directory
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListCertificates.html
     */
    toListCertificates() {
        return this.to('ListCertificates');
    }
    /**
     * Grants permission to list the address blocks that you have added to a directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListIpRoutes.html
     */
    toListIpRoutes() {
        return this.to('ListIpRoutes');
    }
    /**
     * Grants permission to list the active log subscriptions for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListLogSubscriptions.html
     */
    toListLogSubscriptions() {
        return this.to('ListLogSubscriptions');
    }
    /**
     * Grants permission to list all schema extensions applied to a Microsoft AD Directory
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListSchemaExtensions.html
     */
    toListSchemaExtensions() {
        return this.to('ListSchemaExtensions');
    }
    /**
     * Grants permission to list all tags on an Amazon Directory Services directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to register a certificate for secured LDAP connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterCertificate.html
     */
    toRegisterCertificate() {
        return this.to('RegisterCertificate');
    }
    /**
     * Grants permission to associate a directory with an SNS topic
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sns:GetTopicAttributes
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RegisterEventTopic.html
     */
    toRegisterEventTopic() {
        return this.to('RegisterEventTopic');
    }
    /**
     * Grants permission to reject a directory sharing request that was sent from the directory owner account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RejectSharedDirectory.html
     */
    toRejectSharedDirectory() {
        return this.to('RejectSharedDirectory');
    }
    /**
     * Grants permission to remove IP address blocks from a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveIpRoutes.html
     */
    toRemoveIpRoutes() {
        return this.to('RemoveIpRoutes');
    }
    /**
     * Grants permission to stop all replication and removes the domain controllers from the specified Region. You cannot remove the primary Region with this operation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveRegion.html
     */
    toRemoveRegion() {
        return this.to('RemoveRegion');
    }
    /**
     * Grants permission to remove tags from an Amazon Directory Services directory
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DeleteTags
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        return this.to('RemoveTagsFromResource');
    }
    /**
     * Grants permission to reset the password for any user in your AWS Managed Microsoft AD or Simple AD directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ResetUserPassword.html
     */
    toResetUserPassword() {
        return this.to('ResetUserPassword');
    }
    /**
     * Grants permission to restore a directory using an existing directory snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_RestoreFromSnapshot.html
     */
    toRestoreFromSnapshot() {
        return this.to('RestoreFromSnapshot');
    }
    /**
     * Grants permission to share a specified directory in your AWS account (directory owner) with another AWS account (directory consumer). With this operation you can use your directory from any AWS account and from any Amazon VPC within an AWS Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_ShareDirectory.html
     */
    toShareDirectory() {
        return this.to('ShareDirectory');
    }
    /**
     * Grants permission to start a directory assessment
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateNetworkInterfacePermission
     * - ec2:CreateSecurityGroup
     * - ec2:DeleteNetworkInterface
     * - ec2:DeleteSecurityGroup
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - ssm:GetCommandInvocation
     * - ssm:GetConnectionStatus
     * - ssm:ListCommands
     * - ssm:SendCommand
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_StartADAssessment.html
     */
    toStartADAssessment() {
        return this.to('StartADAssessment');
    }
    /**
     * Grants permission to apply a schema extension to a Microsoft AD directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_StartSchemaExtension.html
     */
    toStartSchemaExtension() {
        return this.to('StartSchemaExtension');
    }
    /**
     * Grants permission to unauthorize an application from your AWS Directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toUnauthorizeApplication() {
        return this.to('UnauthorizeApplication');
    }
    /**
     * Grants permission to stop the directory sharing between the directory owner and consumer accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UnshareDirectory.html
     */
    toUnshareDirectory() {
        return this.to('UnshareDirectory');
    }
    /**
     * Grants permission to update an authorized application for your AWS Directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toUpdateAuthorizedApplication() {
        return this.to('UpdateAuthorizedApplication');
    }
    /**
     * Grants permission to update a conditional forwarder that has been set up for your AWS directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateConditionalForwarder.html
     */
    toUpdateConditionalForwarder() {
        return this.to('UpdateConditionalForwarder');
    }
    /**
     * Grants permission to update the configurations like service account credentials or DNS server IP addresses for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/admin-guide/UsingWithDS_IAM_ResourcePermissions.html
     */
    toUpdateDirectory() {
        return this.to('UpdateDirectory');
    }
    /**
     * Grants permission to update the directory for a particular update type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateDirectorySetup.html
     */
    toUpdateDirectorySetup() {
        return this.to('UpdateDirectorySetup');
    }
    /**
     * Grants permission to update configurations for a specified hybrid directory
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupEgress
     * - ec2:AuthorizeSecurityGroupIngress
     * - ec2:CreateNetworkInterface
     * - ec2:CreateNetworkInterfacePermission
     * - ec2:CreateSecurityGroup
     * - ec2:CreateTags
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - secretsmanager:DescribeSecret
     * - secretsmanager:GetSecretValue
     * - ssm:GetCommandInvocation
     * - ssm:GetConnectionStatus
     * - ssm:ListCommands
     * - ssm:SendCommand
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateHybridAD.html
     */
    toUpdateHybridAD() {
        return this.to('UpdateHybridAD');
    }
    /**
     * Grants permission to add or remove domain controllers to or from the directory. Based on the difference between current value and new value (provided through this API call), domain controllers will be added or removed. It may take up to 45 minutes for any new domain controllers to become fully active once the requested number of domain controllers is updated. During this time, you cannot make another update request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateNumberOfDomainControllers.html
     */
    toUpdateNumberOfDomainControllers() {
        return this.to('UpdateNumberOfDomainControllers');
    }
    /**
     * Grants permission to update the Remote Authentication Dial In User Service (RADIUS) server information for an AD Connector directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateRadius.html
     */
    toUpdateRadius() {
        return this.to('UpdateRadius');
    }
    /**
     * Grants permission to update the configurable settings for the specified directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateSettings.html
     */
    toUpdateSettings() {
        return this.to('UpdateSettings');
    }
    /**
     * Grants permission to update the trust that has been set up between your AWS Managed Microsoft AD directory and an on-premises Active Directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_UpdateTrust.html
     */
    toUpdateTrust() {
        return this.to('UpdateTrust');
    }
    /**
     * Grants permission to verify a trust relationship between your Microsoft AD in the AWS cloud and an external domain
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_VerifyTrust.html
     */
    toVerifyTrust() {
        return this.to('VerifyTrust');
    }
    /**
     * Adds a resource of type directory to the statement
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/welcome.html
     *
     * @param directoryId - Identifier for the directoryId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDirectory(directoryId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ds:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:directory/${directoryId}`);
    }
    /**
     * Filters access by the value of the request to AWS DS
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_Tag.html
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toConnectDirectory()
     * - .toCreateDirectory()
     * - .toCreateHybridAD()
     * - .toCreateIdentityPoolDirectory()
     * - .toCreateMicrosoftAD()
     * - .toRemoveTagsFromResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AWS DS Resource being acted upon
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_Tag.html
     *
     * Applies to resource types:
     * - directory
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/directoryservice/latest/devguide/API_Tag.html
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toConnectDirectory()
     * - .toCreateDirectory()
     * - .toCreateHybridAD()
     * - .toCreateIdentityPoolDirectory()
     * - .toCreateMicrosoftAD()
     * - .toRemoveTagsFromResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ds](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsdirectoryservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ds';
        this.accessLevelList = {
            Write: [
                'AcceptSharedDirectory',
                'AddIpRoutes',
                'AddRegion',
                'AuthorizeApplication',
                'CancelSchemaExtension',
                'ConnectDirectory',
                'CreateAlias',
                'CreateComputer',
                'CreateConditionalForwarder',
                'CreateDirectory',
                'CreateHybridAD',
                'CreateIdentityPoolDirectory',
                'CreateLogSubscription',
                'CreateMicrosoftAD',
                'CreateSnapshot',
                'CreateTrust',
                'DeleteADAssessment',
                'DeleteConditionalForwarder',
                'DeleteDirectory',
                'DeleteLogSubscription',
                'DeleteSnapshot',
                'DeleteTrust',
                'DeregisterCertificate',
                'DeregisterEventTopic',
                'DisableCAEnrollmentPolicy',
                'DisableClientAuthentication',
                'DisableDirectoryDataAccess',
                'DisableLDAPS',
                'DisableRadius',
                'DisableRoleAccess',
                'DisableSso',
                'EnableCAEnrollmentPolicy',
                'EnableClientAuthentication',
                'EnableDirectoryDataAccess',
                'EnableLDAPS',
                'EnableRadius',
                'EnableRoleAccess',
                'EnableSso',
                'RegisterCertificate',
                'RegisterEventTopic',
                'RejectSharedDirectory',
                'RemoveIpRoutes',
                'RemoveRegion',
                'ResetUserPassword',
                'RestoreFromSnapshot',
                'ShareDirectory',
                'StartADAssessment',
                'StartSchemaExtension',
                'UnauthorizeApplication',
                'UnshareDirectory',
                'UpdateAuthorizedApplication',
                'UpdateConditionalForwarder',
                'UpdateDirectory',
                'UpdateDirectorySetup',
                'UpdateHybridAD',
                'UpdateNumberOfDomainControllers',
                'UpdateRadius',
                'UpdateSettings',
                'UpdateTrust'
            ],
            'Permissions management': [
                'AccessDSData'
            ],
            Tagging: [
                'AddTagsToResource',
                'RemoveTagsFromResource'
            ],
            Read: [
                'CheckAlias',
                'DescribeADAssessment',
                'DescribeCAEnrollmentPolicy',
                'DescribeCertificate',
                'DescribeClientAuthenticationSettings',
                'DescribeConditionalForwarders',
                'DescribeDirectoryDataAccess',
                'DescribeDomainControllers',
                'DescribeEventTopics',
                'DescribeHybridADUpdate',
                'DescribeLDAPSSettings',
                'DescribeRegions',
                'DescribeSettings',
                'DescribeSharedDirectories',
                'DescribeSnapshots',
                'DescribeTrusts',
                'DescribeUpdateDirectory',
                'GetAuthorizedApplicationDetails',
                'GetDirectoryLimits',
                'GetSnapshotLimits',
                'ListAuthorizedApplications',
                'ListIpRoutes',
                'ListLogSubscriptions',
                'ListTagsForResource',
                'VerifyTrust'
            ],
            List: [
                'DescribeDirectories',
                'ListADAssessments',
                'ListCertificates',
                'ListSchemaExtensions'
            ]
        };
    }
}
exports.Ds = Ds;
//# sourceMappingURL=data:application/json;base64,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