"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dynamodb = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [dynamodb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondynamodb.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Dynamodb extends shared_1.PolicyStatement {
    /**
     * Grants permission to return the attributes of one or more items from one or more tables
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchGetItem.html
     */
    toBatchGetItem() {
        return this.to('BatchGetItem');
    }
    /**
     * Grants permission to put or delete multiple items in one or more tables
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html
     */
    toBatchWriteItem() {
        return this.to('BatchWriteItem');
    }
    /**
     * Grants permission to the ConditionCheckItem operation checks the existence of a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ConditionCheck.html
     */
    toConditionCheckItem() {
        return this.to('ConditionCheckItem');
    }
    /**
     * Grants permission to create a backup for an existing table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateBackup.html
     */
    toCreateBackup() {
        return this.to('CreateBackup');
    }
    /**
     * Grants permission to create a global table from an existing table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateGlobalTable.html
     */
    toCreateGlobalTable() {
        return this.to('CreateGlobalTable');
    }
    /**
     * Grants permission to add a Witness to a Global Table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
     */
    toCreateGlobalTableWitness() {
        return this.to('CreateGlobalTableWitness');
    }
    /**
     * Grants permission to the CreateTable operation adds a new table to your account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_CreateTable.html
     */
    toCreateTable() {
        return this.to('CreateTable');
    }
    /**
     * Grants permission to add a new replica table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
     */
    toCreateTableReplica() {
        return this.to('CreateTableReplica');
    }
    /**
     * Grants permission to delete an existing backup of a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteBackup.html
     */
    toDeleteBackup() {
        return this.to('DeleteBackup');
    }
    /**
     * Grants permission to remove a Witness from a Global Table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
     */
    toDeleteGlobalTableWitness() {
        return this.to('DeleteGlobalTableWitness');
    }
    /**
     * Grants permission to deletes a single item in a table by primary key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteItem.html
     */
    toDeleteItem() {
        return this.to('DeleteItem');
    }
    /**
     * Grants permission to delete the resource-based policy attached to the resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to the DeleteTable operation which deletes a table and all of its items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DeleteTable.html
     */
    toDeleteTable() {
        return this.to('DeleteTable');
    }
    /**
     * Grants permission to delete a replica table and all of its items
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
     */
    toDeleteTableReplica() {
        return this.to('DeleteTableReplica');
    }
    /**
     * Grants permission to describe an existing backup of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeBackup.html
     */
    toDescribeBackup() {
        return this.to('DescribeBackup');
    }
    /**
     * Grants permission to check the status of the backup restore settings on the specified table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContinuousBackups.html
     */
    toDescribeContinuousBackups() {
        return this.to('DescribeContinuousBackups');
    }
    /**
     * Grants permission to describe the contributor insights status and related details for a given table or global secondary index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeContributorInsights.html
     */
    toDescribeContributorInsights() {
        return this.to('DescribeContributorInsights');
    }
    /**
     * Grants permission to return the regional endpoint information
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeEndpoints.html
     */
    toDescribeEndpoints() {
        return this.to('DescribeEndpoints');
    }
    /**
     * Grants permission to describe an existing Export of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeExport.html
     */
    toDescribeExport() {
        return this.to('DescribeExport');
    }
    /**
     * Grants permission to return information about the specified global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTable.html
     */
    toDescribeGlobalTable() {
        return this.to('DescribeGlobalTable');
    }
    /**
     * Grants permission to return settings information about the specified global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeGlobalTableSettings.html
     */
    toDescribeGlobalTableSettings() {
        return this.to('DescribeGlobalTableSettings');
    }
    /**
     * Grants permission to describe an existing import
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeImport.html
     */
    toDescribeImport() {
        return this.to('DescribeImport');
    }
    /**
     * Grants permission to grant permission to describe the status of Kinesis streaming and related details for a given table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeKinesisStreamingDestination.html
     */
    toDescribeKinesisStreamingDestination() {
        return this.to('DescribeKinesisStreamingDestination');
    }
    /**
     * Grants permission to return the current provisioned-capacity limits for your AWS account in a region, both for the region as a whole and for any one DynamoDB table that you create there
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeLimits.html
     */
    toDescribeLimits() {
        return this.to('DescribeLimits');
    }
    /**
     * Grants permission to describe one or more of the Reserved Capacity purchased
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/iam-policy-prevent-purchase-reserved-capacity.html
     */
    toDescribeReservedCapacity() {
        return this.to('DescribeReservedCapacity');
    }
    /**
     * Grants permission to describe Reserved Capacity offerings that are available for purchase
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/iam-policy-prevent-purchase-reserved-capacity.html
     */
    toDescribeReservedCapacityOfferings() {
        return this.to('DescribeReservedCapacityOfferings');
    }
    /**
     * Grants permission to return information about a stream, including the current status of the stream, its Amazon Resource Name (ARN), the composition of its shards, and its corresponding DynamoDB table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_streams_DescribeStream.html
     */
    toDescribeStream() {
        return this.to('DescribeStream');
    }
    /**
     * Grants permission to return information about the table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTable.html
     */
    toDescribeTable() {
        return this.to('DescribeTable');
    }
    /**
     * Grants permission to describe the auto scaling settings across all replicas of the global table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTableReplicaAutoScaling.html
     */
    toDescribeTableReplicaAutoScaling() {
        return this.to('DescribeTableReplicaAutoScaling');
    }
    /**
     * Grants permission to give a description of the Time to Live (TTL) status on the specified table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTimeToLive.html
     */
    toDescribeTimeToLive() {
        return this.to('DescribeTimeToLive');
    }
    /**
     * Grants permission to grant permission to stop replication from the DynamoDB table to the Kinesis data stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DisableKinesisStreamingDestination.html
     */
    toDisableKinesisStreamingDestination() {
        return this.to('DisableKinesisStreamingDestination');
    }
    /**
     * Grants permission to grant permission to start table data replication to the specified Kinesis data stream at a timestamp chosen during the enable workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_EnableKinesisStreamingDestination.html
     */
    toEnableKinesisStreamingDestination() {
        return this.to('EnableKinesisStreamingDestination');
    }
    /**
     * Grants permission to initiate an Export of a DynamoDB table to S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExportTableToPointInTime.html
     */
    toExportTableToPointInTime() {
        return this.to('ExportTableToPointInTime');
    }
    /**
     * Grants permission to view the status of Attribute Based Access Control for the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/abac-enable-ddb.html
     */
    toGetAbacStatus() {
        return this.to('GetAbacStatus');
    }
    /**
     * Grants permission to the GetItem operation that returns a set of attributes for the item with the given primary key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetItem.html
     */
    toGetItem() {
        return this.to('GetItem');
    }
    /**
     * Grants permission to retrieve the stream records from a given shard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_streams_GetRecords.html
     */
    toGetRecords() {
        return this.to('GetRecords');
    }
    /**
     * Grants permission to view a resource-based policy for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_GetResourcePolicy.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to return a shard iterator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_streams_GetShardIterator.html
     */
    toGetShardIterator() {
        return this.to('GetShardIterator');
    }
    /**
     * Grants permission to initiate an import from S3 to a DynamoDB table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ImportTable.html
     */
    toImportTable() {
        return this.to('ImportTable');
    }
    /**
     * Grants permission to list backups associated with the account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListBackups.html
     */
    toListBackups() {
        return this.to('ListBackups');
    }
    /**
     * Grants permission to list the ContributorInsightsSummary for all tables and global secondary indexes associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListContributorInsights.html
     */
    toListContributorInsights() {
        return this.to('ListContributorInsights');
    }
    /**
     * Grants permission to list exports associated with the account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListExports.html
     */
    toListExports() {
        return this.to('ListExports');
    }
    /**
     * Grants permission to list all global tables that have a replica in the specified region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListGlobalTables.html
     */
    toListGlobalTables() {
        return this.to('ListGlobalTables');
    }
    /**
     * Grants permission to list imports associated with the account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListImports.html
     */
    toListImports() {
        return this.to('ListImports');
    }
    /**
     * Grants permission to return an array of stream ARNs associated with the current account and endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_streams_ListStreams.html
     */
    toListStreams() {
        return this.to('ListStreams');
    }
    /**
     * Grants permission to return an array of table names associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTables.html
     */
    toListTables() {
        return this.to('ListTables');
    }
    /**
     * Grants permission to list all tags on an Amazon DynamoDB resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ListTagsOfResource.html
     */
    toListTagsOfResource() {
        return this.to('ListTagsOfResource');
    }
    /**
     * Grants permission to delete a single item in a table by primary key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExecuteStatement.html
     */
    toPartiQLDelete() {
        return this.to('PartiQLDelete');
    }
    /**
     * Grants permission to create a new item, if an item with same primary key does not exist in the table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExecuteStatement.html
     */
    toPartiQLInsert() {
        return this.to('PartiQLInsert');
    }
    /**
     * Grants permission to read a set of attributes for items from a table or index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifFullTableScan()
     * - .ifLeadingKeys()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExecuteStatement.html
     */
    toPartiQLSelect() {
        return this.to('PartiQLSelect');
    }
    /**
     * Grants permission to edit an existing item's attributes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_ExecuteStatement.html
     */
    toPartiQLUpdate() {
        return this.to('PartiQLUpdate');
    }
    /**
     * Grants permission to purchases reserved capacity for use with your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/iam-policy-prevent-purchase-reserved-capacity.html
     */
    toPurchaseReservedCapacityOfferings() {
        return this.to('PurchaseReservedCapacityOfferings');
    }
    /**
     * Grants permission to create a new item, or replace an old item with a new item
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutItem.html
     */
    toPutItem() {
        return this.to('PutItem');
    }
    /**
     * Grants permission to attach a resource-based policy to the resource
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to use the primary key of a table or a secondary index to directly access items from that table or index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Query.html
     */
    toQuery() {
        return this.to('Query');
    }
    /**
     * Grants permission to create a new table from recovery point on AWS Backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/UsageNotesAWS.html
     */
    toRestoreTableFromAwsBackup() {
        return this.to('RestoreTableFromAwsBackup');
    }
    /**
     * Grants permission to create a new table from an existing backup
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dynamodb:BatchWriteItem
     * - dynamodb:DeleteItem
     * - dynamodb:GetItem
     * - dynamodb:PutItem
     * - dynamodb:Query
     * - dynamodb:Scan
     * - dynamodb:UpdateItem
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableFromBackup.html
     */
    toRestoreTableFromBackup() {
        return this.to('RestoreTableFromBackup');
    }
    /**
     * Grants permission to restore a table to a point in time
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - dynamodb:BatchWriteItem
     * - dynamodb:DeleteItem
     * - dynamodb:GetItem
     * - dynamodb:PutItem
     * - dynamodb:Query
     * - dynamodb:Scan
     * - dynamodb:UpdateItem
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_RestoreTableToPointInTime.html
     */
    toRestoreTableToPointInTime() {
        return this.to('RestoreTableToPointInTime');
    }
    /**
     * Grants permission to return one or more items and item attributes by accessing every item in a table or a secondary index
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     * - .ifSelect()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Scan.html
     */
    toScan() {
        return this.to('Scan');
    }
    /**
     * Grants permission to create a backup on AWS Backup with advanced features enabled
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/UsageNotesAWS.html
     */
    toStartAwsBackupJob() {
        return this.to('StartAwsBackupJob');
    }
    /**
     * Grants permission to associate a set of tags with an Amazon DynamoDB resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the association of tags from an Amazon DynamoDB resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the status of Attribute Based Access Control for the account
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/abac-enable-ddb.html
     */
    toUpdateAbacStatus() {
        return this.to('UpdateAbacStatus');
    }
    /**
     * Grants permission to enable or disable continuous backups
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContinuousBackups.html
     */
    toUpdateContinuousBackups() {
        return this.to('UpdateContinuousBackups');
    }
    /**
     * Grants permission to update the status for contributor insights for a specific table or global secondary index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateContributorInsights.html
     */
    toUpdateContributorInsights() {
        return this.to('UpdateContributorInsights');
    }
    /**
     * Grants permission to add or remove replicas in the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTable.html
     */
    toUpdateGlobalTable() {
        return this.to('UpdateGlobalTable');
    }
    /**
     * Grants permission to update settings of the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateGlobalTableSettings.html
     */
    toUpdateGlobalTableSettings() {
        return this.to('UpdateGlobalTableSettings');
    }
    /**
     * Grants permission to update version of the specified global table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2globaltables_upgrade.html
     */
    toUpdateGlobalTableVersion() {
        return this.to('UpdateGlobalTableVersion');
    }
    /**
     * Grants permission to edit an existing item's attributes, or adds a new item to the table if it does not already exist
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributes()
     * - .ifEnclosingOperation()
     * - .ifLeadingKeys()
     * - .ifReturnConsumedCapacity()
     * - .ifReturnValues()
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateItem.html
     */
    toUpdateItem() {
        return this.to('UpdateItem');
    }
    /**
     * Grants permission to update data replication configurations for the specified Kinesis data stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateKinesisStreamingDestination.html
     */
    toUpdateKinesisStreamingDestination() {
        return this.to('UpdateKinesisStreamingDestination');
    }
    /**
     * Grants permission to modify the provisioned throughput settings, global secondary indexes, or DynamoDB Streams settings for a given table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTable.html
     */
    toUpdateTable() {
        return this.to('UpdateTable');
    }
    /**
     * Grants permission to update auto scaling settings on your replica table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTableReplicaAutoScaling.html
     */
    toUpdateTableReplicaAutoScaling() {
        return this.to('UpdateTableReplicaAutoScaling');
    }
    /**
     * Grants permission to enable or disable TTL for the specified table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTimeToLive.html
     */
    toUpdateTimeToLive() {
        return this.to('UpdateTimeToLive');
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.html#HowItWorks.CoreComponents.PrimaryKey
     *
     * @param tableName - Identifier for the tableName.
     * @param indexName - Identifier for the indexName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIndex(tableName, indexName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}/index/${indexName}`);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.html#HowItWorks.CoreComponents.Streams
     *
     * @param tableName - Identifier for the tableName.
     * @param streamLabel - Identifier for the streamLabel.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onStream(tableName, streamLabel, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}/stream/${streamLabel}`);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.CoreComponents.html#HowItWorks.CoreComponents.TablesItemsAttributes
     *
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTable(tableName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}`);
    }
    /**
     * Adds a resource of type backup to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/backuprestore_HowItWorks.html
     *
     * @param tableName - Identifier for the tableName.
     * @param backupName - Identifier for the backupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBackup(tableName, backupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}/backup/${backupName}`);
    }
    /**
     * Adds a resource of type export to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/S3DataExport.HowItWorks.html
     *
     * @param tableName - Identifier for the tableName.
     * @param exportName - Identifier for the exportName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExport(tableName, exportName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}/export/${exportName}`);
    }
    /**
     * Adds a resource of type global-table to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables_HowItWorks.html
     *
     * @param globalTableName - Identifier for the globalTableName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGlobalTable(globalTableName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb::${account ?? this.defaultAccount}:global-table/${globalTableName}`);
    }
    /**
     * Adds a resource of type import to the statement
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/S3DataImport.HowItWorks.html
     *
     * @param tableName - Identifier for the tableName.
     * @param importName - Identifier for the importName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onImport(tableName, importName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:dynamodb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:table/${tableName}/import/${importName}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateTable()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - index
     * - table
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateTable()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by attribute (field or column) names of the table
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPartiQLDelete()
     * - .toPartiQLInsert()
     * - .toPartiQLSelect()
     * - .toPartiQLUpdate()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAttributes(value, operator) {
        return this.if(`Attributes`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by blocking Transactions APIs calls and allow the non-Transaction APIs calls and vice-versa
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPartiQLDelete()
     * - .toPartiQLInsert()
     * - .toPartiQLSelect()
     * - .toPartiQLUpdate()
     * - .toPutItem()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEnclosingOperation(value, operator) {
        return this.if(`EnclosingOperation`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by blocking full table scan
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ql-iam.html
     *
     * Applies to actions:
     * - .toPartiQLSelect()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifFullTableScan(value) {
        return this.if(`FullTableScan`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the partition key of the table
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPartiQLDelete()
     * - .toPartiQLInsert()
     * - .toPartiQLSelect()
     * - .toPartiQLUpdate()
     * - .toPutItem()
     * - .toQuery()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLeadingKeys(value, operator) {
        return this.if(`LeadingKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ReturnConsumedCapacity parameter of a request. Contains either "TOTAL" or "NONE"
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toBatchWriteItem()
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toGetItem()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReturnConsumedCapacity(value, operator) {
        return this.if(`ReturnConsumedCapacity`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ReturnValues parameter of request. Contains one of the following: "ALL_OLD", "UPDATED_OLD","ALL_NEW","UPDATED_NEW", or "NONE"
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toConditionCheckItem()
     * - .toDeleteItem()
     * - .toPartiQLDelete()
     * - .toPartiQLUpdate()
     * - .toPutItem()
     * - .toQuery()
     * - .toScan()
     * - .toUpdateItem()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReturnValues(value, operator) {
        return this.if(`ReturnValues`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the Select parameter of a Query or Scan request
     *
     * https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/specifying-conditions.html#FGAC_DDB.ConditionKeys
     *
     * Applies to actions:
     * - .toBatchGetItem()
     * - .toGetItem()
     * - .toPartiQLSelect()
     * - .toQuery()
     * - .toScan()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSelect(value, operator) {
        return this.if(`Select`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [dynamodb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazondynamodb.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'dynamodb';
        this.accessLevelList = {
            Read: [
                'BatchGetItem',
                'ConditionCheckItem',
                'DescribeBackup',
                'DescribeContinuousBackups',
                'DescribeContributorInsights',
                'DescribeEndpoints',
                'DescribeExport',
                'DescribeGlobalTable',
                'DescribeGlobalTableSettings',
                'DescribeImport',
                'DescribeKinesisStreamingDestination',
                'DescribeLimits',
                'DescribeReservedCapacity',
                'DescribeReservedCapacityOfferings',
                'DescribeStream',
                'DescribeTable',
                'DescribeTableReplicaAutoScaling',
                'DescribeTimeToLive',
                'GetAbacStatus',
                'GetItem',
                'GetRecords',
                'GetResourcePolicy',
                'GetShardIterator',
                'ListStreams',
                'ListTagsOfResource',
                'PartiQLSelect',
                'Query',
                'Scan'
            ],
            Write: [
                'BatchWriteItem',
                'CreateBackup',
                'CreateGlobalTable',
                'CreateGlobalTableWitness',
                'CreateTable',
                'CreateTableReplica',
                'DeleteBackup',
                'DeleteGlobalTableWitness',
                'DeleteItem',
                'DeleteTable',
                'DeleteTableReplica',
                'DisableKinesisStreamingDestination',
                'EnableKinesisStreamingDestination',
                'ExportTableToPointInTime',
                'ImportTable',
                'PartiQLDelete',
                'PartiQLInsert',
                'PartiQLUpdate',
                'PurchaseReservedCapacityOfferings',
                'PutItem',
                'RestoreTableFromAwsBackup',
                'RestoreTableFromBackup',
                'RestoreTableToPointInTime',
                'StartAwsBackupJob',
                'UpdateContinuousBackups',
                'UpdateContributorInsights',
                'UpdateGlobalTable',
                'UpdateGlobalTableSettings',
                'UpdateGlobalTableVersion',
                'UpdateItem',
                'UpdateKinesisStreamingDestination',
                'UpdateTable',
                'UpdateTableReplicaAutoScaling',
                'UpdateTimeToLive'
            ],
            'Permissions management': [
                'DeleteResourcePolicy',
                'PutResourcePolicy',
                'UpdateAbacStatus'
            ],
            List: [
                'ListBackups',
                'ListContributorInsights',
                'ListExports',
                'ListGlobalTables',
                'ListImports',
                'ListTables'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Dynamodb = Dynamodb;
//# sourceMappingURL=data:application/json;base64,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