"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticache = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [elasticache](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticache extends shared_1.PolicyStatement {
    /**
     * Grants permission to add tags to an ElastiCache resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        return this.to('AddTagsToResource');
    }
    /**
     * Grants permission to authorize an EC2 security group on a ElastiCache security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupIngress
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_AuthorizeCacheSecurityGroupIngress.html
     */
    toAuthorizeCacheSecurityGroupIngress() {
        return this.to('AuthorizeCacheSecurityGroupIngress');
    }
    /**
     * Grants permission to apply ElastiCache service updates to sets of clusters and replication groups
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_BatchApplyUpdateAction.html
     */
    toBatchApplyUpdateAction() {
        return this.to('BatchApplyUpdateAction');
    }
    /**
     * Grants permission to stop ElastiCache service updates from being executed on a set of clusters
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_BatchStopUpdateAction.html
     */
    toBatchStopUpdateAction() {
        return this.to('BatchStopUpdateAction');
    }
    /**
     * Grants permission to complete an online migration of data from hosted Redis on Amazon EC2 to ElastiCache
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CompleteMigration.html
     */
    toCompleteMigration() {
        return this.to('CompleteMigration');
    }
    /**
     * Grants permission to connect as a specified ElastiCache user to an ElastiCache Replication Group or ElastiCache serverless cache
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/auth-iam.html
     */
    toConnect() {
        return this.to('Connect');
    }
    /**
     * Grants permission to make a copy of an existing serverless cache snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CopyServerlessCacheSnapshot.html
     */
    toCopyServerlessCacheSnapshot() {
        return this.to('CopyServerlessCacheSnapshot');
    }
    /**
     * Grants permission to make a copy of an existing snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifKmsKeyId()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        return this.to('CopySnapshot');
    }
    /**
     * Grants permission to create a cache cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheCluster.html
     */
    toCreateCacheCluster() {
        return this.to('CreateCacheCluster');
    }
    /**
     * Grants permission to create a parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCacheParameterGroupName()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheParameterGroup.html
     */
    toCreateCacheParameterGroup() {
        return this.to('CreateCacheParameterGroup');
    }
    /**
     * Grants permission to create a cache security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheSecurityGroup.html
     */
    toCreateCacheSecurityGroup() {
        return this.to('CreateCacheSecurityGroup');
    }
    /**
     * Grants permission to create a cache subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheSubnetGroup.html
     */
    toCreateCacheSubnetGroup() {
        return this.to('CreateCacheSubnetGroup');
    }
    /**
     * Grants permission to create a global replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateGlobalReplicationGroup.html
     */
    toCreateGlobalReplicationGroup() {
        return this.to('CreateGlobalReplicationGroup');
    }
    /**
     * Grants permission to create a replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateReplicationGroup.html
     */
    toCreateReplicationGroup() {
        return this.to('CreateReplicationGroup');
    }
    /**
     * Grants permission to create a serverless cache
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateTags
     * - ec2:CreateVpcEndpoint
     * - ec2:DeleteVpcEndpoints
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeTags
     * - ec2:DescribeVpcEndpoints
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateServerlessCache.html
     */
    toCreateServerlessCache() {
        return this.to('CreateServerlessCache');
    }
    /**
     * Grants permission to create a copy of a serverless cache at a specific moment in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateServerlessCacheSnapshot.html
     */
    toCreateServerlessCacheSnapshot() {
        return this.to('CreateServerlessCacheSnapshot');
    }
    /**
     * Grants permission to create a copy of an entire Redis cluster at a specific moment in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permission to create a user for Redis. Users are supported from Redis 6.0 onwards
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifUserAuthenticationMode()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to create a user group for Redis. Groups are supported from Redis 6.0 onwards
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateUserGroup.html
     */
    toCreateUserGroup() {
        return this.to('CreateUserGroup');
    }
    /**
     * Grants permission to decrease the number of node groups in global replication groups
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNumNodeGroups()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DecreaseNodeGroupsInGlobalReplicationGroup.html
     */
    toDecreaseNodeGroupsInGlobalReplicationGroup() {
        return this.to('DecreaseNodeGroupsInGlobalReplicationGroup');
    }
    /**
     * Grants permission to decrease the number of replicas in a Redis (cluster mode disabled) replication group or the number of replica nodes in one or more node groups (shards) of a Redis (cluster mode enabled) replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifReplicasPerNodeGroup()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DecreaseReplicaCount.html
     */
    toDecreaseReplicaCount() {
        return this.to('DecreaseReplicaCount');
    }
    /**
     * Grants permission to delete a previously provisioned cluster
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheCluster.html
     */
    toDeleteCacheCluster() {
        return this.to('DeleteCacheCluster');
    }
    /**
     * Grants permission to delete the specified cache parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifCacheParameterGroupName()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheParameterGroup.html
     */
    toDeleteCacheParameterGroup() {
        return this.to('DeleteCacheParameterGroup');
    }
    /**
     * Grants permission to delete a cache security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheSecurityGroup.html
     */
    toDeleteCacheSecurityGroup() {
        return this.to('DeleteCacheSecurityGroup');
    }
    /**
     * Grants permission to delete a cache subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheSubnetGroup.html
     */
    toDeleteCacheSubnetGroup() {
        return this.to('DeleteCacheSubnetGroup');
    }
    /**
     * Grants permission to delete an existing global replication group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteGlobalReplicationGroup.html
     */
    toDeleteGlobalReplicationGroup() {
        return this.to('DeleteGlobalReplicationGroup');
    }
    /**
     * Grants permission to delete an existing replication group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteReplicationGroup.html
     */
    toDeleteReplicationGroup() {
        return this.to('DeleteReplicationGroup');
    }
    /**
     * Grants permission to delete a serverless cache
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeTags
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteServerlessCache.html
     */
    toDeleteServerlessCache() {
        return this.to('DeleteServerlessCache');
    }
    /**
     * Grants permission to delete a serverless cache snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteServerlessCacheSnapshot.html
     */
    toDeleteServerlessCacheSnapshot() {
        return this.to('DeleteServerlessCacheSnapshot');
    }
    /**
     * Grants permission to delete an existing snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        return this.to('DeleteSnapshot');
    }
    /**
     * Grants permission to delete an existing user and thus remove it from all user groups and replication groups where it was assigned
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to delete an existing user group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteUserGroup.html
     */
    toDeleteUserGroup() {
        return this.to('DeleteUserGroup');
    }
    /**
     * Grants permission to list information about provisioned cache clusters
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheClusters.html
     */
    toDescribeCacheClusters() {
        return this.to('DescribeCacheClusters');
    }
    /**
     * Grants permission to list available cache engines and their versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheEngineVersions.html
     */
    toDescribeCacheEngineVersions() {
        return this.to('DescribeCacheEngineVersions');
    }
    /**
     * Grants permission to list cache parameter group descriptions
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheParameterGroups.html
     */
    toDescribeCacheParameterGroups() {
        return this.to('DescribeCacheParameterGroups');
    }
    /**
     * Grants permission to retrieve the detailed parameter list for a particular cache parameter group
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheParameters.html
     */
    toDescribeCacheParameters() {
        return this.to('DescribeCacheParameters');
    }
    /**
     * Grants permission to list cache security group descriptions
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheSecurityGroups.html
     */
    toDescribeCacheSecurityGroups() {
        return this.to('DescribeCacheSecurityGroups');
    }
    /**
     * Grants permission to list cache subnet group descriptions
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheSubnetGroups.html
     */
    toDescribeCacheSubnetGroups() {
        return this.to('DescribeCacheSubnetGroups');
    }
    /**
     * Grants permission to retrieve the default engine and system parameter information for the specified cache engine
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeEngineDefaultParameters.html
     */
    toDescribeEngineDefaultParameters() {
        return this.to('DescribeEngineDefaultParameters');
    }
    /**
     * Grants permission to list events related to clusters, cache security groups, and cache parameter groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        return this.to('DescribeEvents');
    }
    /**
     * Grants permission to list information about global replication groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeGlobalReplicationGroups.html
     */
    toDescribeGlobalReplicationGroups() {
        return this.to('DescribeGlobalReplicationGroups');
    }
    /**
     * Grants permission to list information about provisioned replication groups
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReplicationGroups.html
     */
    toDescribeReplicationGroups() {
        return this.to('DescribeReplicationGroups');
    }
    /**
     * Grants permission to list information about purchased reserved cache nodes
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReservedCacheNodes.html
     */
    toDescribeReservedCacheNodes() {
        return this.to('DescribeReservedCacheNodes');
    }
    /**
     * Grants permission to list available reserved cache node offerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReservedCacheNodesOfferings.html
     */
    toDescribeReservedCacheNodesOfferings() {
        return this.to('DescribeReservedCacheNodesOfferings');
    }
    /**
     * Grants permission to list information about serverless cache snapshots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeServerlessCacheSnapshots.html
     */
    toDescribeServerlessCacheSnapshots() {
        return this.to('DescribeServerlessCacheSnapshots');
    }
    /**
     * Grants permission to list serverless caches
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeServerlessCaches.html
     */
    toDescribeServerlessCaches() {
        return this.to('DescribeServerlessCaches');
    }
    /**
     * Grants permission to list details of the service updates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeServiceUpdates.html
     */
    toDescribeServiceUpdates() {
        return this.to('DescribeServiceUpdates');
    }
    /**
     * Grants permission to list information about cluster or replication group snapshots
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        return this.to('DescribeSnapshots');
    }
    /**
     * Grants permission to list details of the update actions for a set of clusters or replication groups
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUpdateActions.html
     */
    toDescribeUpdateActions() {
        return this.to('DescribeUpdateActions');
    }
    /**
     * Grants permission to list information about Redis user groups
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUserGroups.html
     */
    toDescribeUserGroups() {
        return this.to('DescribeUserGroups');
    }
    /**
     * Grants permission to list information about Redis users
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUsers.html
     */
    toDescribeUsers() {
        return this.to('DescribeUsers');
    }
    /**
     * Grants permission to remove a secondary replication group from the global replication group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DisassociateGlobalReplicationGroup.html
     */
    toDisassociateGlobalReplicationGroup() {
        return this.to('DisassociateGlobalReplicationGroup');
    }
    /**
     * Grants permission to export a copy of a serverless cache at a specific moment in time to s3 bucket
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:DeleteObject
     * - s3:ListAllMyBuckets
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ExportServerlessCacheSnapshot.html
     */
    toExportServerlessCacheSnapshot() {
        return this.to('ExportServerlessCacheSnapshot');
    }
    /**
     * Grants permission to failover the primary region to a selected secondary region of a global replication group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_FailoverGlobalReplicationGroup.html
     */
    toFailoverGlobalReplicationGroup() {
        return this.to('FailoverGlobalReplicationGroup');
    }
    /**
     * Grants permission to increase the number of node groups in a global replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifNumNodeGroups()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_IncreaseNodeGroupsInGlobalReplicationGroup.html
     */
    toIncreaseNodeGroupsInGlobalReplicationGroup() {
        return this.to('IncreaseNodeGroupsInGlobalReplicationGroup');
    }
    /**
     * Grants permission to increase the number of replicas in a Redis (cluster mode disabled) replication group or the number of replica nodes in one or more node groups (shards) of a Redis (cluster mode enabled) replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifReplicasPerNodeGroup()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_IncreaseReplicaCount.html
     */
    toIncreaseReplicaCount() {
        return this.to('IncreaseReplicaCount');
    }
    /**
     * Grants permission to test an AZ power interruption for an ElastiCache resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/fis-actions-reference.html#elasticache-actions-reference
     */
    toInterruptClusterAzPower() {
        return this.to('InterruptClusterAzPower');
    }
    /**
     * Grants permission to list available node type that can be used to scale a particular Redis cluster or replication group
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ListAllowedNodeTypeModifications.html
     */
    toListAllowedNodeTypeModifications() {
        return this.to('ListAllowedNodeTypeModifications');
    }
    /**
     * Grants permission to list tags for an ElastiCache resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to modify settings for a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheCluster.html
     */
    toModifyCacheCluster() {
        return this.to('ModifyCacheCluster');
    }
    /**
     * Grants permission to modify parameters of a cache parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifCacheParameterGroupName()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheParameterGroup.html
     */
    toModifyCacheParameterGroup() {
        return this.to('ModifyCacheParameterGroup');
    }
    /**
     * Grants permission to modify an existing cache subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheSubnetGroup.html
     */
    toModifyCacheSubnetGroup() {
        return this.to('ModifyCacheSubnetGroup');
    }
    /**
     * Grants permission to modify settings for a global replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCacheNodeType()
     * - .ifEngineVersion()
     * - .ifAutomaticFailoverEnabled()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyGlobalReplicationGroup.html
     */
    toModifyGlobalReplicationGroup() {
        return this.to('ModifyGlobalReplicationGroup');
    }
    /**
     * Grants permission to modify the settings for a replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyReplicationGroup.html
     */
    toModifyReplicationGroup() {
        return this.to('ModifyReplicationGroup');
    }
    /**
     * Grants permission to add shards, remove shards, or rebalance the keyspaces among existing shards of a replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifNumNodeGroups()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyReplicationGroupShardConfiguration.html
     */
    toModifyReplicationGroupShardConfiguration() {
        return this.to('ModifyReplicationGroupShardConfiguration');
    }
    /**
     * Grants permission to modify parameters for a serverless cache
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeTags
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyServerlessCache.html
     */
    toModifyServerlessCache() {
        return this.to('ModifyServerlessCache');
    }
    /**
     * Grants permission to change Redis user password(s) and/or access string
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifUserAuthenticationMode()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyUser.html
     */
    toModifyUser() {
        return this.to('ModifyUser');
    }
    /**
     * Grants permission to change list of users that belong to the user group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyUserGroup.html
     */
    toModifyUserGroup() {
        return this.to('ModifyUserGroup');
    }
    /**
     * Grants permission to purchase a reserved cache node offering
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_PurchaseReservedCacheNodesOffering.html
     */
    toPurchaseReservedCacheNodesOffering() {
        return this.to('PurchaseReservedCacheNodesOffering');
    }
    /**
     * Grants permission to perform a key space rebalance operation to redistribute slots and ensure uniform key distribution across existing shards in a global replication group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RebalanceSlotsInGlobalReplicationGroup.html
     */
    toRebalanceSlotsInGlobalReplicationGroup() {
        return this.to('RebalanceSlotsInGlobalReplicationGroup');
    }
    /**
     * Grants permission to reboot some, or all, of the cache nodes within a provisioned cache cluster or replication group (cluster mode disabled)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RebootCacheCluster.html
     */
    toRebootCacheCluster() {
        return this.to('RebootCacheCluster');
    }
    /**
     * Grants permission to remove tags from a ElastiCache resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        return this.to('RemoveTagsFromResource');
    }
    /**
     * Grants permission to modify parameters of a cache parameter group back to their default values
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifCacheParameterGroupName()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ResetCacheParameterGroup.html
     */
    toResetCacheParameterGroup() {
        return this.to('ResetCacheParameterGroup');
    }
    /**
     * Grants permission to remove an EC2 security group ingress from a ElastiCache security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RevokeCacheSecurityGroupIngress.html
     */
    toRevokeCacheSecurityGroupIngress() {
        return this.to('RevokeCacheSecurityGroupIngress');
    }
    /**
     * Grants permission to start a migration of data from hosted Redis on Amazon EC2 to ElastiCache for Redis
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_StartMigration.html
     */
    toStartMigration() {
        return this.to('StartMigration');
    }
    /**
     * Grants permission to test automatic failover on a specified node group in a replication group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_TestFailover.html
     */
    toTestFailover() {
        return this.to('TestFailover');
    }
    /**
     * Grants permission to test a migration of data from hosted Redis on Amazon EC2 to ElastiCache for Redis
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_TestMigration.html
     */
    toTestMigration() {
        return this.to('TestMigration');
    }
    /**
     * Adds a resource of type parametergroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.ParameterGroups
     *
     * @param cacheParameterGroupName - Identifier for the cacheParameterGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifCacheParameterGroupName()
     */
    onParametergroup(cacheParameterGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:parametergroup:${cacheParameterGroupName}`);
    }
    /**
     * Adds a resource of type securitygroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.SecurityGroups
     *
     * @param cacheSecurityGroupName - Identifier for the cacheSecurityGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onSecuritygroup(cacheSecurityGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:securitygroup:${cacheSecurityGroupName}`);
    }
    /**
     * Adds a resource of type subnetgroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.SubnetGroups
     *
     * @param cacheSubnetGroupName - Identifier for the cacheSubnetGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onSubnetgroup(cacheSubnetGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:subnetgroup:${cacheSubnetGroupName}`);
    }
    /**
     * Adds a resource of type replicationgroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.ReplicationGroups
     *
     * @param replicationGroupId - Identifier for the replicationGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAtRestEncryptionEnabled()
     * - .ifAuthTokenEnabled()
     * - .ifAutomaticFailoverEnabled()
     * - .ifCacheNodeType()
     * - .ifCacheParameterGroupName()
     * - .ifClusterModeEnabled()
     * - .ifEngineType()
     * - .ifEngineVersion()
     * - .ifKmsKeyId()
     * - .ifMultiAZEnabled()
     * - .ifNumNodeGroups()
     * - .ifReplicasPerNodeGroup()
     * - .ifSnapshotRetentionLimit()
     * - .ifTransitEncryptionEnabled()
     */
    onReplicationgroup(replicationGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:replicationgroup:${replicationGroupId}`);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.Clusters
     *
     * @param cacheClusterId - Identifier for the cacheClusterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifAuthTokenEnabled()
     * - .ifCacheNodeType()
     * - .ifCacheParameterGroupName()
     * - .ifEngineType()
     * - .ifEngineVersion()
     * - .ifMultiAZEnabled()
     * - .ifSnapshotRetentionLimit()
     */
    onCluster(cacheClusterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster:${cacheClusterId}`);
    }
    /**
     * Adds a resource of type reserved-instance to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/reserved-nodes.html
     *
     * @param reservedCacheNodeId - Identifier for the reservedCacheNodeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onReservedInstance(reservedCacheNodeId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:reserved-instance:${reservedCacheNodeId}`);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.Components.html#WhatIs.Components.Snapshots
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifKmsKeyId()
     */
    onSnapshot(snapshotName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:snapshot:${snapshotName}`);
    }
    /**
     * Adds a resource of type globalreplicationgroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Redis-Global-Datastore.html
     *
     * @param globalReplicationGroupId - Identifier for the globalReplicationGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAtRestEncryptionEnabled()
     * - .ifAuthTokenEnabled()
     * - .ifAutomaticFailoverEnabled()
     * - .ifCacheNodeType()
     * - .ifCacheParameterGroupName()
     * - .ifClusterModeEnabled()
     * - .ifEngineType()
     * - .ifEngineVersion()
     * - .ifKmsKeyId()
     * - .ifMultiAZEnabled()
     * - .ifNumNodeGroups()
     * - .ifReplicasPerNodeGroup()
     * - .ifSnapshotRetentionLimit()
     * - .ifTransitEncryptionEnabled()
     */
    onGlobalreplicationgroup(globalReplicationGroupId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache::${account ?? this.defaultAccount}:globalreplicationgroup:${globalReplicationGroupId}`);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Clusters.RBAC.html
     *
     * @param userId - Identifier for the userId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifUserAuthenticationMode()
     */
    onUser(userId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:user:${userId}`);
    }
    /**
     * Adds a resource of type usergroup to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/Clusters.RBAC.html
     *
     * @param userGroupId - Identifier for the userGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onUsergroup(userGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:usergroup:${userGroupId}`);
    }
    /**
     * Adds a resource of type serverlesscache to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.html
     *
     * @param serverlessCacheName - Identifier for the serverlessCacheName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifDataStorageUnit()
     * - .ifEngineType()
     * - .ifEngineVersion()
     * - .ifKmsKeyId()
     * - .ifMaximumDataStorage()
     * - .ifMaximumECPUPerSecond()
     * - .ifMinimumDataStorage()
     * - .ifMinimumECPUPerSecond()
     * - .ifSnapshotRetentionLimit()
     */
    onServerlesscache(serverlessCacheName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:serverlesscache:${serverlessCacheName}`);
    }
    /**
     * Adds a resource of type serverlesscachesnapshot to the statement
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/WhatIs.html
     *
     * @param serverlessCacheSnapshotName - Identifier for the serverlessCacheSnapshotName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifKmsKeyId()
     */
    onServerlesscachesnapshot(serverlessCacheSnapshotName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticache:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:serverlesscachesnapshot:${serverlessCacheSnapshotName}`);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCopyServerlessCacheSnapshot()
     * - .toCopySnapshot()
     * - .toCreateCacheCluster()
     * - .toCreateCacheParameterGroup()
     * - .toCreateCacheSecurityGroup()
     * - .toCreateCacheSubnetGroup()
     * - .toCreateReplicationGroup()
     * - .toCreateServerlessCache()
     * - .toCreateServerlessCacheSnapshot()
     * - .toCreateSnapshot()
     * - .toCreateUser()
     * - .toCreateUserGroup()
     * - .toPurchaseReservedCacheNodesOffering()
     *
     * Applies to resource types:
     * - parametergroup
     * - securitygroup
     * - subnetgroup
     * - replicationgroup
     * - cluster
     * - reserved-instance
     * - snapshot
     * - user
     * - usergroup
     * - serverlesscache
     * - serverlesscachesnapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toAuthorizeCacheSecurityGroupIngress()
     * - .toBatchApplyUpdateAction()
     * - .toBatchStopUpdateAction()
     * - .toCompleteMigration()
     * - .toConnect()
     * - .toCopyServerlessCacheSnapshot()
     * - .toCopySnapshot()
     * - .toCreateCacheCluster()
     * - .toCreateCacheParameterGroup()
     * - .toCreateCacheSecurityGroup()
     * - .toCreateCacheSubnetGroup()
     * - .toCreateGlobalReplicationGroup()
     * - .toCreateReplicationGroup()
     * - .toCreateServerlessCache()
     * - .toCreateServerlessCacheSnapshot()
     * - .toCreateSnapshot()
     * - .toCreateUser()
     * - .toCreateUserGroup()
     * - .toDecreaseReplicaCount()
     * - .toDeleteCacheCluster()
     * - .toDeleteCacheParameterGroup()
     * - .toDeleteCacheSecurityGroup()
     * - .toDeleteCacheSubnetGroup()
     * - .toDeleteReplicationGroup()
     * - .toDeleteServerlessCache()
     * - .toDeleteServerlessCacheSnapshot()
     * - .toDeleteSnapshot()
     * - .toDeleteUser()
     * - .toDeleteUserGroup()
     * - .toDescribeCacheClusters()
     * - .toDescribeCacheParameterGroups()
     * - .toDescribeCacheParameters()
     * - .toDescribeCacheSecurityGroups()
     * - .toDescribeCacheSubnetGroups()
     * - .toDescribeReplicationGroups()
     * - .toDescribeReservedCacheNodes()
     * - .toDescribeServerlessCacheSnapshots()
     * - .toDescribeServerlessCaches()
     * - .toDescribeSnapshots()
     * - .toDescribeUpdateActions()
     * - .toDescribeUserGroups()
     * - .toDescribeUsers()
     * - .toExportServerlessCacheSnapshot()
     * - .toIncreaseReplicaCount()
     * - .toInterruptClusterAzPower()
     * - .toListAllowedNodeTypeModifications()
     * - .toListTagsForResource()
     * - .toModifyCacheCluster()
     * - .toModifyCacheParameterGroup()
     * - .toModifyCacheSubnetGroup()
     * - .toModifyReplicationGroup()
     * - .toModifyReplicationGroupShardConfiguration()
     * - .toModifyServerlessCache()
     * - .toModifyUser()
     * - .toModifyUserGroup()
     * - .toPurchaseReservedCacheNodesOffering()
     * - .toRebootCacheCluster()
     * - .toRemoveTagsFromResource()
     * - .toResetCacheParameterGroup()
     * - .toRevokeCacheSecurityGroupIngress()
     * - .toStartMigration()
     * - .toTestFailover()
     * - .toTestMigration()
     *
     * Applies to resource types:
     * - parametergroup
     * - securitygroup
     * - subnetgroup
     * - replicationgroup
     * - cluster
     * - reserved-instance
     * - snapshot
     * - user
     * - usergroup
     * - serverlesscache
     * - serverlesscachesnapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAddTagsToResource()
     * - .toCopyServerlessCacheSnapshot()
     * - .toCopySnapshot()
     * - .toCreateCacheCluster()
     * - .toCreateCacheParameterGroup()
     * - .toCreateCacheSecurityGroup()
     * - .toCreateCacheSubnetGroup()
     * - .toCreateReplicationGroup()
     * - .toCreateServerlessCache()
     * - .toCreateServerlessCacheSnapshot()
     * - .toCreateSnapshot()
     * - .toCreateUser()
     * - .toCreateUserGroup()
     * - .toPurchaseReservedCacheNodesOffering()
     * - .toRemoveTagsFromResource()
     *
     * Applies to resource types:
     * - parametergroup
     * - securitygroup
     * - subnetgroup
     * - replicationgroup
     * - cluster
     * - reserved-instance
     * - snapshot
     * - user
     * - usergroup
     * - serverlesscache
     * - serverlesscachesnapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AtRestEncryptionEnabled parameter present in the request or default false value if parameter is not present
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAtRestEncryptionEnabled(value) {
        return this.if(`AtRestEncryptionEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the presence of non empty AuthToken parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toModifyCacheCluster()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAuthTokenEnabled(value) {
        return this.if(`AuthTokenEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the AutomaticFailoverEnabled parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     * - .toModifyGlobalReplicationGroup()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAutomaticFailoverEnabled(value) {
        return this.if(`AutomaticFailoverEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the cacheNodeType parameter present in the request. This key can be used to restrict which cache node types can be used on cluster creation or scaling operations
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toModifyCacheCluster()
     * - .toModifyGlobalReplicationGroup()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCacheNodeType(value, operator) {
        return this.if(`CacheNodeType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the CacheParameterGroupName parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateCacheParameterGroup()
     * - .toCreateReplicationGroup()
     * - .toDeleteCacheParameterGroup()
     * - .toModifyCacheCluster()
     * - .toModifyCacheParameterGroup()
     * - .toModifyReplicationGroup()
     * - .toResetCacheParameterGroup()
     *
     * Applies to resource types:
     * - parametergroup
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCacheParameterGroupName(value, operator) {
        return this.if(`CacheParameterGroupName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the cluster mode parameter present in the request. Default value for single node group (shard) creations is false
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifClusterModeEnabled(value) {
        return this.if(`ClusterModeEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the CacheUsageLimits.DataStorage.Unit parameter in the CreateServerlessCache and ModifyServerlessCache request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateServerlessCache()
     * - .toModifyServerlessCache()
     *
     * Applies to resource types:
     * - serverlesscache
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataStorageUnit(value, operator) {
        return this.if(`DataStorageUnit`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the engine type present in creation requests. For replication group creations, default engine 'redis' is used as key if parameter is not present
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toCreateServerlessCache()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     * - serverlesscache
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEngineType(value, operator) {
        return this.if(`EngineType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the engineVersion parameter present in creation or cluster modification requests
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toCreateServerlessCache()
     * - .toModifyCacheCluster()
     * - .toModifyGlobalReplicationGroup()
     * - .toModifyReplicationGroup()
     * - .toModifyServerlessCache()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     * - serverlesscache
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEngineVersion(value, operator) {
        return this.if(`EngineVersion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the Key ID of the KMS key
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCopyServerlessCacheSnapshot()
     * - .toCopySnapshot()
     * - .toCreateReplicationGroup()
     * - .toCreateServerlessCache()
     * - .toCreateServerlessCacheSnapshot()
     * - .toCreateSnapshot()
     *
     * Applies to resource types:
     * - replicationgroup
     * - snapshot
     * - globalreplicationgroup
     * - serverlesscache
     * - serverlesscachesnapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKmsKeyId(value, operator) {
        return this.if(`KmsKeyId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the CacheUsageLimits.DataStorage.Maximum parameter in the CreateServerlessCache and ModifyServerlessCache request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateServerlessCache()
     * - .toModifyServerlessCache()
     *
     * Applies to resource types:
     * - serverlesscache
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaximumDataStorage(value, operator) {
        return this.if(`MaximumDataStorage`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the CacheUsageLimits.ECPUPerSecond.Maximum parameter in the CreateServerlessCache and ModifyServerlessCache request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateServerlessCache()
     * - .toModifyServerlessCache()
     *
     * Applies to resource types:
     * - serverlesscache
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaximumECPUPerSecond(value, operator) {
        return this.if(`MaximumECPUPerSecond`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the CacheUsageLimits.DataStorage.Minimum parameter in the CreateServerlessCache and ModifyServerlessCache request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateServerlessCache()
     * - .toModifyServerlessCache()
     *
     * Applies to resource types:
     * - serverlesscache
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMinimumDataStorage(value, operator) {
        return this.if(`MinimumDataStorage`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the CacheUsageLimits.ECPUPerSecond.Minimum parameter in the CreateServerlessCache and ModifyServerlessCache request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateServerlessCache()
     * - .toModifyServerlessCache()
     *
     * Applies to resource types:
     * - serverlesscache
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMinimumECPUPerSecond(value, operator) {
        return this.if(`MinimumECPUPerSecond`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the AZMode parameter, MultiAZEnabled parameter or the number of availability zones that the cluster or replication group can be placed in
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toModifyCacheCluster()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifMultiAZEnabled(value) {
        return this.if(`MultiAZEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the NumNodeGroups or NodeGroupCount parameter specified in the request. This key can be used to restrict the number of node groups (shards) clusters can have after creation or scaling operations
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     * - .toDecreaseNodeGroupsInGlobalReplicationGroup()
     * - .toIncreaseNodeGroupsInGlobalReplicationGroup()
     * - .toModifyReplicationGroupShardConfiguration()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifNumNodeGroups(value, operator) {
        return this.if(`NumNodeGroups`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the number of replicas per node group (shards) specified in creations or scaling requests
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     * - .toDecreaseReplicaCount()
     * - .toIncreaseReplicaCount()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifReplicasPerNodeGroup(value, operator) {
        return this.if(`ReplicasPerNodeGroup`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the SnapshotRetentionLimit parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCacheCluster()
     * - .toCreateReplicationGroup()
     * - .toCreateServerlessCache()
     * - .toModifyCacheCluster()
     * - .toModifyReplicationGroup()
     * - .toModifyServerlessCache()
     *
     * Applies to resource types:
     * - replicationgroup
     * - cluster
     * - globalreplicationgroup
     * - serverlesscache
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSnapshotRetentionLimit(value, operator) {
        return this.if(`SnapshotRetentionLimit`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access by the TransitEncryptionEnabled parameter present in the request. For replication group creations, default value 'false' is used as key if parameter is not present
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateReplicationGroup()
     * - .toModifyReplicationGroup()
     *
     * Applies to resource types:
     * - replicationgroup
     * - globalreplicationgroup
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifTransitEncryptionEnabled(value) {
        return this.if(`TransitEncryptionEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the UserAuthenticationMode parameter in the request
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateUser()
     * - .toModifyUser()
     *
     * Applies to resource types:
     * - user
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserAuthenticationMode(value, operator) {
        return this.if(`UserAuthenticationMode`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [elasticache](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'elasticache';
        this.accessLevelList = {
            Tagging: [
                'AddTagsToResource',
                'RemoveTagsFromResource'
            ],
            Write: [
                'AuthorizeCacheSecurityGroupIngress',
                'BatchApplyUpdateAction',
                'BatchStopUpdateAction',
                'CompleteMigration',
                'Connect',
                'CopyServerlessCacheSnapshot',
                'CopySnapshot',
                'CreateCacheCluster',
                'CreateCacheParameterGroup',
                'CreateCacheSecurityGroup',
                'CreateCacheSubnetGroup',
                'CreateGlobalReplicationGroup',
                'CreateReplicationGroup',
                'CreateServerlessCache',
                'CreateServerlessCacheSnapshot',
                'CreateSnapshot',
                'CreateUser',
                'CreateUserGroup',
                'DecreaseNodeGroupsInGlobalReplicationGroup',
                'DecreaseReplicaCount',
                'DeleteCacheCluster',
                'DeleteCacheParameterGroup',
                'DeleteCacheSecurityGroup',
                'DeleteCacheSubnetGroup',
                'DeleteGlobalReplicationGroup',
                'DeleteReplicationGroup',
                'DeleteServerlessCache',
                'DeleteServerlessCacheSnapshot',
                'DeleteSnapshot',
                'DeleteUser',
                'DeleteUserGroup',
                'DisassociateGlobalReplicationGroup',
                'ExportServerlessCacheSnapshot',
                'FailoverGlobalReplicationGroup',
                'IncreaseNodeGroupsInGlobalReplicationGroup',
                'IncreaseReplicaCount',
                'InterruptClusterAzPower',
                'ModifyCacheCluster',
                'ModifyCacheParameterGroup',
                'ModifyCacheSubnetGroup',
                'ModifyGlobalReplicationGroup',
                'ModifyReplicationGroup',
                'ModifyReplicationGroupShardConfiguration',
                'ModifyServerlessCache',
                'ModifyUser',
                'ModifyUserGroup',
                'PurchaseReservedCacheNodesOffering',
                'RebalanceSlotsInGlobalReplicationGroup',
                'RebootCacheCluster',
                'ResetCacheParameterGroup',
                'RevokeCacheSecurityGroupIngress',
                'StartMigration',
                'TestFailover',
                'TestMigration'
            ],
            List: [
                'DescribeCacheClusters',
                'DescribeCacheEngineVersions',
                'DescribeCacheParameterGroups',
                'DescribeCacheParameters',
                'DescribeCacheSecurityGroups',
                'DescribeCacheSubnetGroups',
                'DescribeEngineDefaultParameters',
                'DescribeEvents',
                'DescribeGlobalReplicationGroups',
                'DescribeReplicationGroups',
                'DescribeReservedCacheNodes',
                'DescribeReservedCacheNodesOfferings',
                'DescribeServerlessCacheSnapshots',
                'DescribeServerlessCaches',
                'DescribeServiceUpdates',
                'DescribeSnapshots',
                'DescribeUpdateActions',
                'DescribeUserGroups',
                'DescribeUsers',
                'ListAllowedNodeTypeModifications'
            ],
            Read: [
                'ListTagsForResource'
            ]
        };
    }
}
exports.Elasticache = Elasticache;
//# sourceMappingURL=data:application/json;base64,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