"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticbeanstalk = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [elasticbeanstalk](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselasticbeanstalk.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticbeanstalk extends shared_1.PolicyStatement {
    /**
     * Grants permission to cancel in-progress environment configuration update or application version deployment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_AbortEnvironmentUpdate.html
     */
    toAbortEnvironmentUpdate() {
        return this.to('AbortEnvironmentUpdate');
    }
    /**
     * Grants permission to add tags to an Elastic Beanstalk resource and to update tag values
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateTagsForResource.html
     */
    toAddTags() {
        return this.to('AddTags');
    }
    /**
     * Grants permission to apply a scheduled managed action immediately
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ApplyEnvironmentManagedAction.html
     */
    toApplyEnvironmentManagedAction() {
        return this.to('ApplyEnvironmentManagedAction');
    }
    /**
     * Grants permission to associate an operations role with an environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_AssociateEnvironmentOperationsRole.html
     */
    toAssociateEnvironmentOperationsRole() {
        return this.to('AssociateEnvironmentOperationsRole');
    }
    /**
     * Grants permission to check CNAME availability
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CheckDNSAvailability.html
     */
    toCheckDNSAvailability() {
        return this.to('CheckDNSAvailability');
    }
    /**
     * Grants permission to create or update a group of environments, each running a separate component of a single application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ComposeEnvironments.html
     */
    toComposeEnvironments() {
        return this.to('ComposeEnvironments');
    }
    /**
     * Grants permission to create a new application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create an application version for an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateApplicationVersion.html
     */
    toCreateApplicationVersion() {
        return this.to('CreateApplicationVersion');
    }
    /**
     * Grants permission to create a configuration template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromApplication()
     * - .ifFromApplicationVersion()
     * - .ifFromConfigurationTemplate()
     * - .ifFromEnvironment()
     * - .ifFromSolutionStack()
     * - .ifFromPlatform()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateConfigurationTemplate.html
     */
    toCreateConfigurationTemplate() {
        return this.to('CreateConfigurationTemplate');
    }
    /**
     * Grants permission to launch an environment for an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromApplicationVersion()
     * - .ifFromConfigurationTemplate()
     * - .ifFromSolutionStack()
     * - .ifFromPlatform()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateEnvironment.html
     */
    toCreateEnvironment() {
        return this.to('CreateEnvironment');
    }
    /**
     * Grants permission to create a new version of a custom platform
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreatePlatformVersion.html
     */
    toCreatePlatformVersion() {
        return this.to('CreatePlatformVersion');
    }
    /**
     * Grants permission to create the Amazon S3 storage location for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_CreateStorageLocation.html
     */
    toCreateStorageLocation() {
        return this.to('CreateStorageLocation');
    }
    /**
     * Grants permission to delete an application along with all associated versions and configurations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete an application version from an application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteApplicationVersion.html
     */
    toDeleteApplicationVersion() {
        return this.to('DeleteApplicationVersion');
    }
    /**
     * Grants permission to delete a configuration template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteConfigurationTemplate.html
     */
    toDeleteConfigurationTemplate() {
        return this.to('DeleteConfigurationTemplate');
    }
    /**
     * Grants permission to delete the draft configuration associated with the running environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeleteEnvironmentConfiguration.html
     */
    toDeleteEnvironmentConfiguration() {
        return this.to('DeleteEnvironmentConfiguration');
    }
    /**
     * Grants permission to delete a version of a custom platform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DeletePlatformVersion.html
     */
    toDeletePlatformVersion() {
        return this.to('DeletePlatformVersion');
    }
    /**
     * Grants permission to retrieve a list of account attributes, including resource quotas
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeAccountAttributes.html
     */
    toDescribeAccountAttributes() {
        return this.to('DescribeAccountAttributes');
    }
    /**
     * Grants permission to retrieve a list of application versions stored in an AWS Elastic Beanstalk storage bucket
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeApplicationVersions.html
     */
    toDescribeApplicationVersions() {
        return this.to('DescribeApplicationVersions');
    }
    /**
     * Grants permission to retrieve the descriptions of existing applications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeApplications.html
     */
    toDescribeApplications() {
        return this.to('DescribeApplications');
    }
    /**
     * Grants permission to retrieve descriptions of environment configuration options
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeConfigurationOptions.html
     */
    toDescribeConfigurationOptions() {
        return this.to('DescribeConfigurationOptions');
    }
    /**
     * Grants permission to retrieve a description of the settings for a configuration set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeConfigurationSettings.html
     */
    toDescribeConfigurationSettings() {
        return this.to('DescribeConfigurationSettings');
    }
    /**
     * Grants permission to retrieve information about the overall health of an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentHealth.html
     */
    toDescribeEnvironmentHealth() {
        return this.to('DescribeEnvironmentHealth');
    }
    /**
     * Grants permission to retrieve a list of an environment's completed and failed managed actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentManagedActionHistory.html
     */
    toDescribeEnvironmentManagedActionHistory() {
        return this.to('DescribeEnvironmentManagedActionHistory');
    }
    /**
     * Grants permission to retrieve a list of an environment's upcoming and in-progress managed actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentManagedActions.html
     */
    toDescribeEnvironmentManagedActions() {
        return this.to('DescribeEnvironmentManagedActions');
    }
    /**
     * Grants permission to retrieve a list of AWS resources for an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironmentResources.html
     */
    toDescribeEnvironmentResources() {
        return this.to('DescribeEnvironmentResources');
    }
    /**
     * Grants permission to retrieve descriptions for existing environments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEnvironments.html
     */
    toDescribeEnvironments() {
        return this.to('DescribeEnvironments');
    }
    /**
     * Grants permission to retrieve a list of event descriptions matching a set of criteria
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeEvents.html
     */
    toDescribeEvents() {
        return this.to('DescribeEvents');
    }
    /**
     * Grants permission to retrieve more detailed information about the health of environment instances
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribeInstancesHealth.html
     */
    toDescribeInstancesHealth() {
        return this.to('DescribeInstancesHealth');
    }
    /**
     * Grants permission to retrieve a description of a managed platform version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DescribePlatformVersion.html
     */
    toDescribePlatformVersion() {
        return this.to('DescribePlatformVersion');
    }
    /**
     * Grants permission to disassociate an operations role with an environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_DisassociateEnvironmentOperationsRole.html
     */
    toDisassociateEnvironmentOperationsRole() {
        return this.to('DisassociateEnvironmentOperationsRole');
    }
    /**
     * Grants permission to retrieve a list of the available solution stack names
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListAvailableSolutionStacks.html
     */
    toListAvailableSolutionStacks() {
        return this.to('ListAvailableSolutionStacks');
    }
    /**
     * Grants permission to retrieve a list of the available platform branches
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListPlatformBranches.html
     */
    toListPlatformBranches() {
        return this.to('ListPlatformBranches');
    }
    /**
     * Grants permission to retrieve a list of the available platforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListPlatformVersions.html
     */
    toListPlatformVersions() {
        return this.to('ListPlatformVersions');
    }
    /**
     * Grants permission to retrieve a list of tags of an Elastic Beanstalk resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to submit instance statistics for enhanced health
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/health-enhanced.html#health-enhanced-authz
     */
    toPutInstanceStatistics() {
        return this.to('PutInstanceStatistics');
    }
    /**
     * Grants permission to delete and recreate all of the AWS resources for an environment and to force a restart
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RebuildEnvironment.html
     */
    toRebuildEnvironment() {
        return this.to('RebuildEnvironment');
    }
    /**
     * Grants permission to remove tags from an Elastic Beanstalk resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateTagsForResource.html
     */
    toRemoveTags() {
        return this.to('RemoveTags');
    }
    /**
     * Grants permission to initiate a request to compile information of the deployed environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RequestEnvironmentInfo.html
     */
    toRequestEnvironmentInfo() {
        return this.to('RequestEnvironmentInfo');
    }
    /**
     * Grants permission to request an environment to restart the application container server running on each Amazon EC2 instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RestartAppServer.html
     */
    toRestartAppServer() {
        return this.to('RestartAppServer');
    }
    /**
     * Grants permission to retrieve the compiled information from a RequestEnvironmentInfo request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_RetrieveEnvironmentInfo.html
     */
    toRetrieveEnvironmentInfo() {
        return this.to('RetrieveEnvironmentInfo');
    }
    /**
     * Grants permission to swap the CNAMEs of two environments
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromEnvironment()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_SwapEnvironmentCNAMEs.html
     */
    toSwapEnvironmentCNAMEs() {
        return this.to('SwapEnvironmentCNAMEs');
    }
    /**
     * Grants permission to terminate an environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_TerminateEnvironment.html
     */
    toTerminateEnvironment() {
        return this.to('TerminateEnvironment');
    }
    /**
     * Grants permission to update an application with specified properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Grants permission to update the application version lifecycle policy associated with the application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplicationResourceLifecycle.html
     */
    toUpdateApplicationResourceLifecycle() {
        return this.to('UpdateApplicationResourceLifecycle');
    }
    /**
     * Grants permission to update an application version with specified properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateApplicationVersion.html
     */
    toUpdateApplicationVersion() {
        return this.to('UpdateApplicationVersion');
    }
    /**
     * Grants permission to update a configuration template with specified properties or configuration option values
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromApplication()
     * - .ifFromApplicationVersion()
     * - .ifFromConfigurationTemplate()
     * - .ifFromEnvironment()
     * - .ifFromSolutionStack()
     * - .ifFromPlatform()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateConfigurationTemplate.html
     */
    toUpdateConfigurationTemplate() {
        return this.to('UpdateConfigurationTemplate');
    }
    /**
     * Grants permission to update an environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifFromApplicationVersion()
     * - .ifFromConfigurationTemplate()
     * - .ifFromSolutionStack()
     * - .ifFromPlatform()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateEnvironment.html
     */
    toUpdateEnvironment() {
        return this.to('UpdateEnvironment');
    }
    /**
     * Doesn't grant permission to update tags. To grant permission to add tags to an Elastic Beanstalk resource, remove tags, and to update tag values, specify elasticbeanstalk:AddTags and elasticbeanstalk:RemoveTags
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_UpdateTagsForResource.html
     */
    toUpdateTagsForResource() {
        return this.to('UpdateTagsForResource');
    }
    /**
     * Grants permission to check the validity of a set of configuration settings for a configuration template or an environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/api/API_ValidateConfigurationSettings.html
     */
    toValidateConfigurationSettings() {
        return this.to('ValidateConfigurationSettings');
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onApplication(applicationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticbeanstalk:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:application/${applicationName}`);
    }
    /**
     * Adds a resource of type applicationversion to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param versionLabel - Identifier for the versionLabel.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInApplication()
     */
    onApplicationversion(applicationName, versionLabel, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticbeanstalk:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:applicationversion/${applicationName}/${versionLabel}`);
    }
    /**
     * Adds a resource of type configurationtemplate to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param templateName - Identifier for the templateName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInApplication()
     */
    onConfigurationtemplate(applicationName, templateName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticbeanstalk:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configurationtemplate/${applicationName}/${templateName}`);
    }
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param environmentName - Identifier for the environmentName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInApplication()
     */
    onEnvironment(applicationName, environmentName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticbeanstalk:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:environment/${applicationName}/${environmentName}`);
    }
    /**
     * Adds a resource of type solutionstack to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param solutionStackName - Identifier for the solutionStackName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSolutionstack(solutionStackName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticbeanstalk:${region ?? this.defaultRegion}::solutionstack/${solutionStackName}`);
    }
    /**
     * Adds a resource of type platform to the statement
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.arn.html
     *
     * @param platformNameWithVersion - Identifier for the platformNameWithVersion.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onPlatform(platformNameWithVersion, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticbeanstalk:${region ?? this.defaultRegion}::platform/${platformNameWithVersion}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateApplication()
     * - .toCreateApplicationVersion()
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toCreatePlatformVersion()
     * - .toUpdateTagsForResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to resource types:
     * - application
     * - applicationversion
     * - configurationtemplate
     * - environment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateApplication()
     * - .toCreateApplicationVersion()
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toCreatePlatformVersion()
     * - .toRemoveTags()
     * - .toUpdateTagsForResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by an application as a dependency or a constraint on an input parameter
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toUpdateConfigurationTemplate()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromApplication(value, operator) {
        return this.if(`FromApplication`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by an application version as a dependency or a constraint on an input parameter
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toUpdateConfigurationTemplate()
     * - .toUpdateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromApplicationVersion(value, operator) {
        return this.if(`FromApplicationVersion`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by a configuration template as a dependency or a constraint on an input parameter
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toUpdateConfigurationTemplate()
     * - .toUpdateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromConfigurationTemplate(value, operator) {
        return this.if(`FromConfigurationTemplate`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by an environment as a dependency or a constraint on an input parameter
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toSwapEnvironmentCNAMEs()
     * - .toUpdateConfigurationTemplate()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromEnvironment(value, operator) {
        return this.if(`FromEnvironment`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by a platform as a dependency or a constraint on an input parameter
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toUpdateConfigurationTemplate()
     * - .toUpdateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromPlatform(value, operator) {
        return this.if(`FromPlatform`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by a solution stack as a dependency or a constraint on an input parameter
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toUpdateConfigurationTemplate()
     * - .toUpdateEnvironment()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifFromSolutionStack(value, operator) {
        return this.if(`FromSolutionStack`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the application that contains the resource that the action operates on
     *
     * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/AWSHowTo.iam.policies.actions.html#AWSHowTo.iam.policies.conditions
     *
     * Applies to actions:
     * - .toAbortEnvironmentUpdate()
     * - .toApplyEnvironmentManagedAction()
     * - .toComposeEnvironments()
     * - .toCreateApplicationVersion()
     * - .toCreateConfigurationTemplate()
     * - .toCreateEnvironment()
     * - .toDeleteApplicationVersion()
     * - .toDeleteConfigurationTemplate()
     * - .toDeleteEnvironmentConfiguration()
     * - .toDescribeApplicationVersions()
     * - .toDescribeConfigurationOptions()
     * - .toDescribeConfigurationSettings()
     * - .toDescribeEnvironmentManagedActionHistory()
     * - .toDescribeEnvironmentManagedActions()
     * - .toDescribeEnvironmentResources()
     * - .toDescribeEnvironments()
     * - .toDescribeEvents()
     * - .toRebuildEnvironment()
     * - .toRequestEnvironmentInfo()
     * - .toRestartAppServer()
     * - .toRetrieveEnvironmentInfo()
     * - .toSwapEnvironmentCNAMEs()
     * - .toTerminateEnvironment()
     * - .toUpdateApplicationVersion()
     * - .toUpdateConfigurationTemplate()
     * - .toUpdateEnvironment()
     * - .toValidateConfigurationSettings()
     *
     * Applies to resource types:
     * - applicationversion
     * - configurationtemplate
     * - environment
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifInApplication(value, operator) {
        return this.if(`InApplication`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [elasticbeanstalk](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselasticbeanstalk.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'elasticbeanstalk';
        this.accessLevelList = {
            Write: [
                'AbortEnvironmentUpdate',
                'ApplyEnvironmentManagedAction',
                'AssociateEnvironmentOperationsRole',
                'ComposeEnvironments',
                'CreateApplication',
                'CreateApplicationVersion',
                'CreateConfigurationTemplate',
                'CreateEnvironment',
                'CreatePlatformVersion',
                'CreateStorageLocation',
                'DeleteApplication',
                'DeleteApplicationVersion',
                'DeleteConfigurationTemplate',
                'DeleteEnvironmentConfiguration',
                'DeletePlatformVersion',
                'DisassociateEnvironmentOperationsRole',
                'PutInstanceStatistics',
                'RebuildEnvironment',
                'RestartAppServer',
                'SwapEnvironmentCNAMEs',
                'TerminateEnvironment',
                'UpdateApplication',
                'UpdateApplicationResourceLifecycle',
                'UpdateApplicationVersion',
                'UpdateConfigurationTemplate',
                'UpdateEnvironment'
            ],
            Tagging: [
                'AddTags',
                'RemoveTags',
                'UpdateTagsForResource'
            ],
            Read: [
                'CheckDNSAvailability',
                'DescribeAccountAttributes',
                'DescribeConfigurationOptions',
                'DescribeConfigurationSettings',
                'DescribeEnvironmentHealth',
                'DescribeEnvironmentManagedActionHistory',
                'DescribeEnvironmentManagedActions',
                'DescribeEnvironmentResources',
                'DescribeEvents',
                'DescribeInstancesHealth',
                'DescribePlatformVersion',
                'ListTagsForResource',
                'RequestEnvironmentInfo',
                'RetrieveEnvironmentInfo',
                'ValidateConfigurationSettings'
            ],
            List: [
                'DescribeApplicationVersions',
                'DescribeApplications',
                'DescribeEnvironments',
                'ListAvailableSolutionStacks',
                'ListPlatformBranches',
                'ListPlatformVersions'
            ]
        };
    }
}
exports.Elasticbeanstalk = Elasticbeanstalk;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWxhc3RpY2JlYW5zdGFsay5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImVsYXN0aWNiZWFuc3RhbGsudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EseUNBQXlEO0FBR3pEOzs7O0dBSUc7QUFDSCxNQUFhLGdCQUFpQixTQUFRLHdCQUFlO0lBR25EOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFNBQVM7UUFDZCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDNUIsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7OztPQWdCRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0NBQWdDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlDQUF5QztRQUM5QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUNBQXlDLENBQUMsQ0FBQztJQUM1RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUNBQXVDO1FBQzVDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztJQUMxQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBK0REOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGFBQWEsQ0FBQyxlQUF1QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2pHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLHFCQUFzQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZ0JBQWlCLGVBQWdCLEVBQUUsQ0FBQyxDQUFDO0lBQ3hMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLG9CQUFvQixDQUFDLGVBQXVCLEVBQUUsWUFBb0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM5SCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixxQkFBc0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLHVCQUF3QixlQUFnQixJQUFLLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDak4sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksdUJBQXVCLENBQUMsZUFBdUIsRUFBRSxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2pJLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLHFCQUFzQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsMEJBQTJCLGVBQWdCLElBQUssWUFBYSxFQUFFLENBQUMsQ0FBQztJQUNwTixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxhQUFhLENBQUMsZUFBdUIsRUFBRSxlQUF1QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzFILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLHFCQUFzQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZ0JBQWlCLGVBQWdCLElBQUssZUFBZ0IsRUFBRSxDQUFDLENBQUM7SUFDN00sQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksZUFBZSxDQUFDLGlCQUF5QixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixxQkFBc0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLG1CQUFvQixpQkFBa0IsRUFBRSxDQUFDLENBQUM7SUFDekosQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksVUFBVSxDQUFDLHVCQUErQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNwRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixxQkFBc0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLGNBQWUsdUJBQXdCLEVBQUUsQ0FBQyxDQUFDO0lBQzFKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGlCQUFpQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDN0UsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksU0FBUyxDQUFDLENBQUM7SUFDbEUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSx3QkFBd0IsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3BGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFNBQVMsQ0FBQyxDQUFDO0lBQ3pFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksMkJBQTJCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUN2RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxTQUFTLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksaUJBQWlCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUM3RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxTQUFTLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGNBQWMsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzFFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxTQUFTLENBQUMsQ0FBQztJQUMvRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLG1CQUFtQixDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDL0UsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksU0FBUyxDQUFDLENBQUM7SUFDcEUsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXlDRztJQUNJLGVBQWUsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzNFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxTQUFTLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUExOUJSLGtCQUFhLEdBQUcsa0JBQWtCLENBQUM7UUE4bEJoQyxvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsd0JBQXdCO2dCQUN4QiwrQkFBK0I7Z0JBQy9CLG9DQUFvQztnQkFDcEMscUJBQXFCO2dCQUNyQixtQkFBbUI7Z0JBQ25CLDBCQUEwQjtnQkFDMUIsNkJBQTZCO2dCQUM3QixtQkFBbUI7Z0JBQ25CLHVCQUF1QjtnQkFDdkIsdUJBQXVCO2dCQUN2QixtQkFBbUI7Z0JBQ25CLDBCQUEwQjtnQkFDMUIsNkJBQTZCO2dCQUM3QixnQ0FBZ0M7Z0JBQ2hDLHVCQUF1QjtnQkFDdkIsdUNBQXVDO2dCQUN2Qyx1QkFBdUI7Z0JBQ3ZCLG9CQUFvQjtnQkFDcEIsa0JBQWtCO2dCQUNsQix1QkFBdUI7Z0JBQ3ZCLHNCQUFzQjtnQkFDdEIsbUJBQW1CO2dCQUNuQixvQ0FBb0M7Z0JBQ3BDLDBCQUEwQjtnQkFDMUIsNkJBQTZCO2dCQUM3QixtQkFBbUI7YUFDcEI7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsU0FBUztnQkFDVCxZQUFZO2dCQUNaLHVCQUF1QjthQUN4QjtZQUNELElBQUksRUFBRTtnQkFDSixzQkFBc0I7Z0JBQ3RCLDJCQUEyQjtnQkFDM0IsOEJBQThCO2dCQUM5QiwrQkFBK0I7Z0JBQy9CLDJCQUEyQjtnQkFDM0IseUNBQXlDO2dCQUN6QyxtQ0FBbUM7Z0JBQ25DLDhCQUE4QjtnQkFDOUIsZ0JBQWdCO2dCQUNoQix5QkFBeUI7Z0JBQ3pCLHlCQUF5QjtnQkFDekIscUJBQXFCO2dCQUNyQix3QkFBd0I7Z0JBQ3hCLHlCQUF5QjtnQkFDekIsK0JBQStCO2FBQ2hDO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLDZCQUE2QjtnQkFDN0Isc0JBQXNCO2dCQUN0QixzQkFBc0I7Z0JBQ3RCLDZCQUE2QjtnQkFDN0Isc0JBQXNCO2dCQUN0QixzQkFBc0I7YUFDdkI7U0FDRixDQUFDO0lBa1VGLENBQUM7Q0FDRjtBQTc5QkQsNENBNjlCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbZWxhc3RpY2JlYW5zdGFsa10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzZWxhc3RpY2JlYW5zdGFsay5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIEVsYXN0aWNiZWFuc3RhbGsgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdlbGFzdGljYmVhbnN0YWxrJztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY2FuY2VsIGluLXByb2dyZXNzIGVudmlyb25tZW50IGNvbmZpZ3VyYXRpb24gdXBkYXRlIG9yIGFwcGxpY2F0aW9uIHZlcnNpb24gZGVwbG95bWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0Fib3J0RW52aXJvbm1lbnRVcGRhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWJvcnRFbnZpcm9ubWVudFVwZGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQWJvcnRFbnZpcm9ubWVudFVwZGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZCB0YWdzIHRvIGFuIEVsYXN0aWMgQmVhbnN0YWxrIHJlc291cmNlIGFuZCB0byB1cGRhdGUgdGFnIHZhbHVlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfVXBkYXRlVGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FkZFRhZ3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0FkZFRhZ3MnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhcHBseSBhIHNjaGVkdWxlZCBtYW5hZ2VkIGFjdGlvbiBpbW1lZGlhdGVseVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0FwcGx5RW52aXJvbm1lbnRNYW5hZ2VkQWN0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FwcGx5RW52aXJvbm1lbnRNYW5hZ2VkQWN0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBcHBseUVudmlyb25tZW50TWFuYWdlZEFjdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSBhbiBvcGVyYXRpb25zIHJvbGUgd2l0aCBhbiBlbnZpcm9ubWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0Fzc29jaWF0ZUVudmlyb25tZW50T3BlcmF0aW9uc1JvbGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQXNzb2NpYXRlRW52aXJvbm1lbnRPcGVyYXRpb25zUm9sZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQXNzb2NpYXRlRW52aXJvbm1lbnRPcGVyYXRpb25zUm9sZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNoZWNrIENOQU1FIGF2YWlsYWJpbGl0eVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfQ2hlY2tETlNBdmFpbGFiaWxpdHkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ2hlY2tETlNBdmFpbGFiaWxpdHkoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NoZWNrRE5TQXZhaWxhYmlsaXR5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIG9yIHVwZGF0ZSBhIGdyb3VwIG9mIGVudmlyb25tZW50cywgZWFjaCBydW5uaW5nIGEgc2VwYXJhdGUgY29tcG9uZW50IG9mIGEgc2luZ2xlIGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfQ29tcG9zZUVudmlyb25tZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Db21wb3NlRW52aXJvbm1lbnRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDb21wb3NlRW52aXJvbm1lbnRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfQ3JlYXRlQXBwbGljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUFwcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIGFwcGxpY2F0aW9uIHZlcnNpb24gZm9yIGFuIGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfQ3JlYXRlQXBwbGljYXRpb25WZXJzaW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUFwcGxpY2F0aW9uVmVyc2lvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQXBwbGljYXRpb25WZXJzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgY29uZmlndXJhdGlvbiB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRnJvbUFwcGxpY2F0aW9uKClcbiAgICogLSAuaWZGcm9tQXBwbGljYXRpb25WZXJzaW9uKClcbiAgICogLSAuaWZGcm9tQ29uZmlndXJhdGlvblRlbXBsYXRlKClcbiAgICogLSAuaWZGcm9tRW52aXJvbm1lbnQoKVxuICAgKiAtIC5pZkZyb21Tb2x1dGlvblN0YWNrKClcbiAgICogLSAuaWZGcm9tUGxhdGZvcm0oKVxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9DcmVhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ29uZmlndXJhdGlvblRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsYXVuY2ggYW4gZW52aXJvbm1lbnQgZm9yIGFuIGFwcGxpY2F0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZGcm9tQXBwbGljYXRpb25WZXJzaW9uKClcbiAgICogLSAuaWZGcm9tQ29uZmlndXJhdGlvblRlbXBsYXRlKClcbiAgICogLSAuaWZGcm9tU29sdXRpb25TdGFjaygpXG4gICAqIC0gLmlmRnJvbVBsYXRmb3JtKClcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfQ3JlYXRlRW52aXJvbm1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRW52aXJvbm1lbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUVudmlyb25tZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbmV3IHZlcnNpb24gb2YgYSBjdXN0b20gcGxhdGZvcm1cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9DcmVhdGVQbGF0Zm9ybVZlcnNpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUGxhdGZvcm1WZXJzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVQbGF0Zm9ybVZlcnNpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgdGhlIEFtYXpvbiBTMyBzdG9yYWdlIGxvY2F0aW9uIGZvciB0aGUgYWNjb3VudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0NyZWF0ZVN0b3JhZ2VMb2NhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTdG9yYWdlTG9jYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVN0b3JhZ2VMb2NhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBhcHBsaWNhdGlvbiBhbG9uZyB3aXRoIGFsbCBhc3NvY2lhdGVkIHZlcnNpb25zIGFuZCBjb25maWd1cmF0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0RlbGV0ZUFwcGxpY2F0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUFwcGxpY2F0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVBcHBsaWNhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhbiBhcHBsaWNhdGlvbiB2ZXJzaW9uIGZyb20gYW4gYXBwbGljYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9EZWxldGVBcHBsaWNhdGlvblZlcnNpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQXBwbGljYXRpb25WZXJzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVBcHBsaWNhdGlvblZlcnNpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBjb25maWd1cmF0aW9uIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfRGVsZXRlQ29uZmlndXJhdGlvblRlbXBsYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUNvbmZpZ3VyYXRpb25UZW1wbGF0ZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQ29uZmlndXJhdGlvblRlbXBsYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRoZSBkcmFmdCBjb25maWd1cmF0aW9uIGFzc29jaWF0ZWQgd2l0aCB0aGUgcnVubmluZyBlbnZpcm9ubWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0RlbGV0ZUVudmlyb25tZW50Q29uZmlndXJhdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVFbnZpcm9ubWVudENvbmZpZ3VyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUVudmlyb25tZW50Q29uZmlndXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHZlcnNpb24gb2YgYSBjdXN0b20gcGxhdGZvcm1cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9EZWxldGVQbGF0Zm9ybVZlcnNpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUGxhdGZvcm1WZXJzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVQbGF0Zm9ybVZlcnNpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3Qgb2YgYWNjb3VudCBhdHRyaWJ1dGVzLCBpbmNsdWRpbmcgcmVzb3VyY2UgcXVvdGFzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9EZXNjcmliZUFjY291bnRBdHRyaWJ1dGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQWNjb3VudEF0dHJpYnV0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlQWNjb3VudEF0dHJpYnV0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3Qgb2YgYXBwbGljYXRpb24gdmVyc2lvbnMgc3RvcmVkIGluIGFuIEFXUyBFbGFzdGljIEJlYW5zdGFsayBzdG9yYWdlIGJ1Y2tldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVBcHBsaWNhdGlvblZlcnNpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQXBwbGljYXRpb25WZXJzaW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVBcHBsaWNhdGlvblZlcnNpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIGRlc2NyaXB0aW9ucyBvZiBleGlzdGluZyBhcHBsaWNhdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlQXBwbGljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQXBwbGljYXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUFwcGxpY2F0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGRlc2NyaXB0aW9ucyBvZiBlbnZpcm9ubWVudCBjb25maWd1cmF0aW9uIG9wdGlvbnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlQ29uZmlndXJhdGlvbk9wdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVDb25maWd1cmF0aW9uT3B0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVDb25maWd1cmF0aW9uT3B0aW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgZGVzY3JpcHRpb24gb2YgdGhlIHNldHRpbmdzIGZvciBhIGNvbmZpZ3VyYXRpb24gc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9EZXNjcmliZUNvbmZpZ3VyYXRpb25TZXR0aW5ncy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUNvbmZpZ3VyYXRpb25TZXR0aW5ncygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVDb25maWd1cmF0aW9uU2V0dGluZ3MnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBpbmZvcm1hdGlvbiBhYm91dCB0aGUgb3ZlcmFsbCBoZWFsdGggb2YgYW4gZW52aXJvbm1lbnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0Rlc2NyaWJlRW52aXJvbm1lbnRIZWFsdGguaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVFbnZpcm9ubWVudEhlYWx0aCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVFbnZpcm9ubWVudEhlYWx0aCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiBhbiBlbnZpcm9ubWVudCdzIGNvbXBsZXRlZCBhbmQgZmFpbGVkIG1hbmFnZWQgYWN0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVFbnZpcm9ubWVudE1hbmFnZWRBY3Rpb25IaXN0b3J5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRW52aXJvbm1lbnRNYW5hZ2VkQWN0aW9uSGlzdG9yeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVFbnZpcm9ubWVudE1hbmFnZWRBY3Rpb25IaXN0b3J5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGFuIGVudmlyb25tZW50J3MgdXBjb21pbmcgYW5kIGluLXByb2dyZXNzIG1hbmFnZWQgYWN0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVFbnZpcm9ubWVudE1hbmFnZWRBY3Rpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRW52aXJvbm1lbnRNYW5hZ2VkQWN0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVFbnZpcm9ubWVudE1hbmFnZWRBY3Rpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIEFXUyByZXNvdXJjZXMgZm9yIGFuIGVudmlyb25tZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9EZXNjcmliZUVudmlyb25tZW50UmVzb3VyY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlRW52aXJvbm1lbnRSZXNvdXJjZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRW52aXJvbm1lbnRSZXNvdXJjZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBkZXNjcmlwdGlvbnMgZm9yIGV4aXN0aW5nIGVudmlyb25tZW50c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVFbnZpcm9ubWVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVFbnZpcm9ubWVudHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRW52aXJvbm1lbnRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIGV2ZW50IGRlc2NyaXB0aW9ucyBtYXRjaGluZyBhIHNldCBvZiBjcml0ZXJpYVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVFdmVudHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVFdmVudHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlRXZlbnRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgbW9yZSBkZXRhaWxlZCBpbmZvcm1hdGlvbiBhYm91dCB0aGUgaGVhbHRoIG9mIGVudmlyb25tZW50IGluc3RhbmNlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVJbnN0YW5jZXNIZWFsdGguaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVJbnN0YW5jZXNIZWFsdGgoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlSW5zdGFuY2VzSGVhbHRoJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBkZXNjcmlwdGlvbiBvZiBhIG1hbmFnZWQgcGxhdGZvcm0gdmVyc2lvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfRGVzY3JpYmVQbGF0Zm9ybVZlcnNpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVQbGF0Zm9ybVZlcnNpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlUGxhdGZvcm1WZXJzaW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIGFuIG9wZXJhdGlvbnMgcm9sZSB3aXRoIGFuIGVudmlyb25tZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfRGlzYXNzb2NpYXRlRW52aXJvbm1lbnRPcGVyYXRpb25zUm9sZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EaXNhc3NvY2lhdGVFbnZpcm9ubWVudE9wZXJhdGlvbnNSb2xlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhc3NvY2lhdGVFbnZpcm9ubWVudE9wZXJhdGlvbnNSb2xlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIHRoZSBhdmFpbGFibGUgc29sdXRpb24gc3RhY2sgbmFtZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0xpc3RBdmFpbGFibGVTb2x1dGlvblN0YWNrcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QXZhaWxhYmxlU29sdXRpb25TdGFja3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RBdmFpbGFibGVTb2x1dGlvblN0YWNrcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiB0aGUgYXZhaWxhYmxlIHBsYXRmb3JtIGJyYW5jaGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9MaXN0UGxhdGZvcm1CcmFuY2hlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UGxhdGZvcm1CcmFuY2hlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFBsYXRmb3JtQnJhbmNoZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhIGxpc3Qgb2YgdGhlIGF2YWlsYWJsZSBwbGF0Zm9ybXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX0xpc3RQbGF0Zm9ybVZlcnNpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RQbGF0Zm9ybVZlcnNpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0UGxhdGZvcm1WZXJzaW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiB0YWdzIG9mIGFuIEVsYXN0aWMgQmVhbnN0YWxrIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdWJtaXQgaW5zdGFuY2Ugc3RhdGlzdGljcyBmb3IgZW5oYW5jZWQgaGVhbHRoXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2RnL2hlYWx0aC1lbmhhbmNlZC5odG1sI2hlYWx0aC1lbmhhbmNlZC1hdXRoelxuICAgKi9cbiAgcHVibGljIHRvUHV0SW5zdGFuY2VTdGF0aXN0aWNzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQdXRJbnN0YW5jZVN0YXRpc3RpY3MnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW5kIHJlY3JlYXRlIGFsbCBvZiB0aGUgQVdTIHJlc291cmNlcyBmb3IgYW4gZW52aXJvbm1lbnQgYW5kIHRvIGZvcmNlIGEgcmVzdGFydFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX1JlYnVpbGRFbnZpcm9ubWVudC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZWJ1aWxkRW52aXJvbm1lbnQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1JlYnVpbGRFbnZpcm9ubWVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlbW92ZSB0YWdzIGZyb20gYW4gRWxhc3RpYyBCZWFuc3RhbGsgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZVRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZW1vdmVUYWdzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZW1vdmVUYWdzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gaW5pdGlhdGUgYSByZXF1ZXN0IHRvIGNvbXBpbGUgaW5mb3JtYXRpb24gb2YgdGhlIGRlcGxveWVkIGVudmlyb25tZW50XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9SZXF1ZXN0RW52aXJvbm1lbnRJbmZvLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlcXVlc3RFbnZpcm9ubWVudEluZm8oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1JlcXVlc3RFbnZpcm9ubWVudEluZm8nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXF1ZXN0IGFuIGVudmlyb25tZW50IHRvIHJlc3RhcnQgdGhlIGFwcGxpY2F0aW9uIGNvbnRhaW5lciBzZXJ2ZXIgcnVubmluZyBvbiBlYWNoIEFtYXpvbiBFQzIgaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9SZXN0YXJ0QXBwU2VydmVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Jlc3RhcnRBcHBTZXJ2ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1Jlc3RhcnRBcHBTZXJ2ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUgY29tcGlsZWQgaW5mb3JtYXRpb24gZnJvbSBhIFJlcXVlc3RFbnZpcm9ubWVudEluZm8gcmVxdWVzdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfUmV0cmlldmVFbnZpcm9ubWVudEluZm8uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUmV0cmlldmVFbnZpcm9ubWVudEluZm8oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1JldHJpZXZlRW52aXJvbm1lbnRJbmZvJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3dhcCB0aGUgQ05BTUVzIG9mIHR3byBlbnZpcm9ubWVudHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkZyb21FbnZpcm9ubWVudCgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX1N3YXBFbnZpcm9ubWVudENOQU1Fcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Td2FwRW52aXJvbm1lbnRDTkFNRXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N3YXBFbnZpcm9ubWVudENOQU1FcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHRlcm1pbmF0ZSBhbiBlbnZpcm9ubWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX1Rlcm1pbmF0ZUVudmlyb25tZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1Rlcm1pbmF0ZUVudmlyb25tZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdUZXJtaW5hdGVFbnZpcm9ubWVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBhcHBsaWNhdGlvbiB3aXRoIHNwZWNpZmllZCBwcm9wZXJ0aWVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfVXBkYXRlQXBwbGljYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQXBwbGljYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUFwcGxpY2F0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBhcHBsaWNhdGlvbiB2ZXJzaW9uIGxpZmVjeWNsZSBwb2xpY3kgYXNzb2NpYXRlZCB3aXRoIHRoZSBhcHBsaWNhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZUFwcGxpY2F0aW9uUmVzb3VyY2VMaWZlY3ljbGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQXBwbGljYXRpb25SZXNvdXJjZUxpZmVjeWNsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlQXBwbGljYXRpb25SZXNvdXJjZUxpZmVjeWNsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBhcHBsaWNhdGlvbiB2ZXJzaW9uIHdpdGggc3BlY2lmaWVkIHByb3BlcnRpZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvYXBpL0FQSV9VcGRhdGVBcHBsaWNhdGlvblZlcnNpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQXBwbGljYXRpb25WZXJzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVBcHBsaWNhdGlvblZlcnNpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBjb25maWd1cmF0aW9uIHRlbXBsYXRlIHdpdGggc3BlY2lmaWVkIHByb3BlcnRpZXMgb3IgY29uZmlndXJhdGlvbiBvcHRpb24gdmFsdWVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZGcm9tQXBwbGljYXRpb24oKVxuICAgKiAtIC5pZkZyb21BcHBsaWNhdGlvblZlcnNpb24oKVxuICAgKiAtIC5pZkZyb21Db25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKiAtIC5pZkZyb21FbnZpcm9ubWVudCgpXG4gICAqIC0gLmlmRnJvbVNvbHV0aW9uU3RhY2soKVxuICAgKiAtIC5pZkZyb21QbGF0Zm9ybSgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZUNvbmZpZ3VyYXRpb25UZW1wbGF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUNvbmZpZ3VyYXRpb25UZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBlbnZpcm9ubWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmRnJvbUFwcGxpY2F0aW9uVmVyc2lvbigpXG4gICAqIC0gLmlmRnJvbUNvbmZpZ3VyYXRpb25UZW1wbGF0ZSgpXG4gICAqIC0gLmlmRnJvbVNvbHV0aW9uU3RhY2soKVxuICAgKiAtIC5pZkZyb21QbGF0Zm9ybSgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9hcGkvQVBJX1VwZGF0ZUVudmlyb25tZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUVudmlyb25tZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVFbnZpcm9ubWVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIERvZXNuJ3QgZ3JhbnQgcGVybWlzc2lvbiB0byB1cGRhdGUgdGFncy4gVG8gZ3JhbnQgcGVybWlzc2lvbiB0byBhZGQgdGFncyB0byBhbiBFbGFzdGljIEJlYW5zdGFsayByZXNvdXJjZSwgcmVtb3ZlIHRhZ3MsIGFuZCB0byB1cGRhdGUgdGFnIHZhbHVlcywgc3BlY2lmeSBlbGFzdGljYmVhbnN0YWxrOkFkZFRhZ3MgYW5kIGVsYXN0aWNiZWFuc3RhbGs6UmVtb3ZlVGFnc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfVXBkYXRlVGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlVGFnc0ZvclJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY2hlY2sgdGhlIHZhbGlkaXR5IG9mIGEgc2V0IG9mIGNvbmZpZ3VyYXRpb24gc2V0dGluZ3MgZm9yIGEgY29uZmlndXJhdGlvbiB0ZW1wbGF0ZSBvciBhbiBlbnZpcm9ubWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2FwaS9BUElfVmFsaWRhdGVDb25maWd1cmF0aW9uU2V0dGluZ3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVmFsaWRhdGVDb25maWd1cmF0aW9uU2V0dGluZ3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1ZhbGlkYXRlQ29uZmlndXJhdGlvblNldHRpbmdzJyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgV3JpdGU6IFtcbiAgICAgICdBYm9ydEVudmlyb25tZW50VXBkYXRlJyxcbiAgICAgICdBcHBseUVudmlyb25tZW50TWFuYWdlZEFjdGlvbicsXG4gICAgICAnQXNzb2NpYXRlRW52aXJvbm1lbnRPcGVyYXRpb25zUm9sZScsXG4gICAgICAnQ29tcG9zZUVudmlyb25tZW50cycsXG4gICAgICAnQ3JlYXRlQXBwbGljYXRpb24nLFxuICAgICAgJ0NyZWF0ZUFwcGxpY2F0aW9uVmVyc2lvbicsXG4gICAgICAnQ3JlYXRlQ29uZmlndXJhdGlvblRlbXBsYXRlJyxcbiAgICAgICdDcmVhdGVFbnZpcm9ubWVudCcsXG4gICAgICAnQ3JlYXRlUGxhdGZvcm1WZXJzaW9uJyxcbiAgICAgICdDcmVhdGVTdG9yYWdlTG9jYXRpb24nLFxuICAgICAgJ0RlbGV0ZUFwcGxpY2F0aW9uJyxcbiAgICAgICdEZWxldGVBcHBsaWNhdGlvblZlcnNpb24nLFxuICAgICAgJ0RlbGV0ZUNvbmZpZ3VyYXRpb25UZW1wbGF0ZScsXG4gICAgICAnRGVsZXRlRW52aXJvbm1lbnRDb25maWd1cmF0aW9uJyxcbiAgICAgICdEZWxldGVQbGF0Zm9ybVZlcnNpb24nLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZUVudmlyb25tZW50T3BlcmF0aW9uc1JvbGUnLFxuICAgICAgJ1B1dEluc3RhbmNlU3RhdGlzdGljcycsXG4gICAgICAnUmVidWlsZEVudmlyb25tZW50JyxcbiAgICAgICdSZXN0YXJ0QXBwU2VydmVyJyxcbiAgICAgICdTd2FwRW52aXJvbm1lbnRDTkFNRXMnLFxuICAgICAgJ1Rlcm1pbmF0ZUVudmlyb25tZW50JyxcbiAgICAgICdVcGRhdGVBcHBsaWNhdGlvbicsXG4gICAgICAnVXBkYXRlQXBwbGljYXRpb25SZXNvdXJjZUxpZmVjeWNsZScsXG4gICAgICAnVXBkYXRlQXBwbGljYXRpb25WZXJzaW9uJyxcbiAgICAgICdVcGRhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUnLFxuICAgICAgJ1VwZGF0ZUVudmlyb25tZW50J1xuICAgIF0sXG4gICAgVGFnZ2luZzogW1xuICAgICAgJ0FkZFRhZ3MnLFxuICAgICAgJ1JlbW92ZVRhZ3MnLFxuICAgICAgJ1VwZGF0ZVRhZ3NGb3JSZXNvdXJjZSdcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdDaGVja0ROU0F2YWlsYWJpbGl0eScsXG4gICAgICAnRGVzY3JpYmVBY2NvdW50QXR0cmlidXRlcycsXG4gICAgICAnRGVzY3JpYmVDb25maWd1cmF0aW9uT3B0aW9ucycsXG4gICAgICAnRGVzY3JpYmVDb25maWd1cmF0aW9uU2V0dGluZ3MnLFxuICAgICAgJ0Rlc2NyaWJlRW52aXJvbm1lbnRIZWFsdGgnLFxuICAgICAgJ0Rlc2NyaWJlRW52aXJvbm1lbnRNYW5hZ2VkQWN0aW9uSGlzdG9yeScsXG4gICAgICAnRGVzY3JpYmVFbnZpcm9ubWVudE1hbmFnZWRBY3Rpb25zJyxcbiAgICAgICdEZXNjcmliZUVudmlyb25tZW50UmVzb3VyY2VzJyxcbiAgICAgICdEZXNjcmliZUV2ZW50cycsXG4gICAgICAnRGVzY3JpYmVJbnN0YW5jZXNIZWFsdGgnLFxuICAgICAgJ0Rlc2NyaWJlUGxhdGZvcm1WZXJzaW9uJyxcbiAgICAgICdMaXN0VGFnc0ZvclJlc291cmNlJyxcbiAgICAgICdSZXF1ZXN0RW52aXJvbm1lbnRJbmZvJyxcbiAgICAgICdSZXRyaWV2ZUVudmlyb25tZW50SW5mbycsXG4gICAgICAnVmFsaWRhdGVDb25maWd1cmF0aW9uU2V0dGluZ3MnXG4gICAgXSxcbiAgICBMaXN0OiBbXG4gICAgICAnRGVzY3JpYmVBcHBsaWNhdGlvblZlcnNpb25zJyxcbiAgICAgICdEZXNjcmliZUFwcGxpY2F0aW9ucycsXG4gICAgICAnRGVzY3JpYmVFbnZpcm9ubWVudHMnLFxuICAgICAgJ0xpc3RBdmFpbGFibGVTb2x1dGlvblN0YWNrcycsXG4gICAgICAnTGlzdFBsYXRmb3JtQnJhbmNoZXMnLFxuICAgICAgJ0xpc3RQbGF0Zm9ybVZlcnNpb25zJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYXBwbGljYXRpb24gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvZGcvQVdTSG93VG8uaWFtLnBvbGljaWVzLmFybi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBhcHBsaWNhdGlvbk5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgYXBwbGljYXRpb25OYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25BcHBsaWNhdGlvbihhcHBsaWNhdGlvbk5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTplbGFzdGljYmVhbnN0YWxrOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTphcHBsaWNhdGlvbi8keyBhcHBsaWNhdGlvbk5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGFwcGxpY2F0aW9udmVyc2lvbiB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9kZy9BV1NIb3dUby5pYW0ucG9saWNpZXMuYXJuLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwcGxpY2F0aW9uTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBhcHBsaWNhdGlvbk5hbWUuXG4gICAqIEBwYXJhbSB2ZXJzaW9uTGFiZWwgLSBJZGVudGlmaWVyIGZvciB0aGUgdmVyc2lvbkxhYmVsLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmSW5BcHBsaWNhdGlvbigpXG4gICAqL1xuICBwdWJsaWMgb25BcHBsaWNhdGlvbnZlcnNpb24oYXBwbGljYXRpb25OYW1lOiBzdHJpbmcsIHZlcnNpb25MYWJlbDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmVsYXN0aWNiZWFuc3RhbGs6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmFwcGxpY2F0aW9udmVyc2lvbi8keyBhcHBsaWNhdGlvbk5hbWUgfS8keyB2ZXJzaW9uTGFiZWwgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNvbmZpZ3VyYXRpb250ZW1wbGF0ZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9kZy9BV1NIb3dUby5pYW0ucG9saWNpZXMuYXJuLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwcGxpY2F0aW9uTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBhcHBsaWNhdGlvbk5hbWUuXG4gICAqIEBwYXJhbSB0ZW1wbGF0ZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgdGVtcGxhdGVOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqIC0gLmlmSW5BcHBsaWNhdGlvbigpXG4gICAqL1xuICBwdWJsaWMgb25Db25maWd1cmF0aW9udGVtcGxhdGUoYXBwbGljYXRpb25OYW1lOiBzdHJpbmcsIHRlbXBsYXRlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmVsYXN0aWNiZWFuc3RhbGs6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmNvbmZpZ3VyYXRpb250ZW1wbGF0ZS8keyBhcHBsaWNhdGlvbk5hbWUgfS8keyB0ZW1wbGF0ZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGVudmlyb25tZW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2RnL0FXU0hvd1RvLmlhbS5wb2xpY2llcy5hcm4uaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYXBwbGljYXRpb25OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwcGxpY2F0aW9uTmFtZS5cbiAgICogQHBhcmFtIGVudmlyb25tZW50TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBlbnZpcm9ubWVudE5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZJbkFwcGxpY2F0aW9uKClcbiAgICovXG4gIHB1YmxpYyBvbkVudmlyb25tZW50KGFwcGxpY2F0aW9uTmFtZTogc3RyaW5nLCBlbnZpcm9ubWVudE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTplbGFzdGljYmVhbnN0YWxrOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTplbnZpcm9ubWVudC8keyBhcHBsaWNhdGlvbk5hbWUgfS8keyBlbnZpcm9ubWVudE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHNvbHV0aW9uc3RhY2sgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvZGcvQVdTSG93VG8uaWFtLnBvbGljaWVzLmFybi5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzb2x1dGlvblN0YWNrTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzb2x1dGlvblN0YWNrTmFtZS5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uU29sdXRpb25zdGFjayhzb2x1dGlvblN0YWNrTmFtZTogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmVsYXN0aWNiZWFuc3RhbGs6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9Ojpzb2x1dGlvbnN0YWNrLyR7IHNvbHV0aW9uU3RhY2tOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBwbGF0Zm9ybSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9kZy9BV1NIb3dUby5pYW0ucG9saWNpZXMuYXJuLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHBsYXRmb3JtTmFtZVdpdGhWZXJzaW9uIC0gSWRlbnRpZmllciBmb3IgdGhlIHBsYXRmb3JtTmFtZVdpdGhWZXJzaW9uLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25QbGF0Zm9ybShwbGF0Zm9ybU5hbWVXaXRoVmVyc2lvbjogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmVsYXN0aWNiZWFuc3RhbGs6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OjpwbGF0Zm9ybS8keyBwbGF0Zm9ybU5hbWVXaXRoVmVyc2lvbiB9YCk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY3Rpb25zIGJhc2VkIG9uIHRoZSBwcmVzZW5jZSBvZiB0YWcga2V5LXZhbHVlIHBhaXJzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9kZy9BV1NIb3dUby5pYW0ucG9saWNpZXMuYWN0aW9ucy5odG1sI0FXU0hvd1RvLmlhbS5wb2xpY2llcy5jb25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9BZGRUYWdzKClcbiAgICogLSAudG9DcmVhdGVBcHBsaWNhdGlvbigpXG4gICAqIC0gLnRvQ3JlYXRlQXBwbGljYXRpb25WZXJzaW9uKClcbiAgICogLSAudG9DcmVhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKiAtIC50b0NyZWF0ZUVudmlyb25tZW50KClcbiAgICogLSAudG9DcmVhdGVQbGF0Zm9ybVZlcnNpb24oKVxuICAgKiAtIC50b1VwZGF0ZVRhZ3NGb3JSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjdGlvbnMgYmFzZWQgb24gdGFnIGtleS12YWx1ZSBwYWlycyBhdHRhY2hlZCB0byB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2RnL0FXU0hvd1RvLmlhbS5wb2xpY2llcy5hY3Rpb25zLmh0bWwjQVdTSG93VG8uaWFtLnBvbGljaWVzLmNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBhcHBsaWNhdGlvblxuICAgKiAtIGFwcGxpY2F0aW9udmVyc2lvblxuICAgKiAtIGNvbmZpZ3VyYXRpb250ZW1wbGF0ZVxuICAgKiAtIGVudmlyb25tZW50XG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWN0aW9ucyBiYXNlZCBvbiB0aGUgcHJlc2VuY2Ugb2YgdGFnIGtleXMgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2RnL0FXU0hvd1RvLmlhbS5wb2xpY2llcy5hY3Rpb25zLmh0bWwjQVdTSG93VG8uaWFtLnBvbGljaWVzLmNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0FkZFRhZ3MoKVxuICAgKiAtIC50b0NyZWF0ZUFwcGxpY2F0aW9uKClcbiAgICogLSAudG9DcmVhdGVBcHBsaWNhdGlvblZlcnNpb24oKVxuICAgKiAtIC50b0NyZWF0ZUNvbmZpZ3VyYXRpb25UZW1wbGF0ZSgpXG4gICAqIC0gLnRvQ3JlYXRlRW52aXJvbm1lbnQoKVxuICAgKiAtIC50b0NyZWF0ZVBsYXRmb3JtVmVyc2lvbigpXG4gICAqIC0gLnRvUmVtb3ZlVGFncygpXG4gICAqIC0gLnRvVXBkYXRlVGFnc0ZvclJlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGFuIGFwcGxpY2F0aW9uIGFzIGEgZGVwZW5kZW5jeSBvciBhIGNvbnN0cmFpbnQgb24gYW4gaW5wdXQgcGFyYW1ldGVyXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9kZy9BV1NIb3dUby5pYW0ucG9saWNpZXMuYWN0aW9ucy5odG1sI0FXU0hvd1RvLmlhbS5wb2xpY2llcy5jb25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKiAtIC50b1VwZGF0ZUNvbmZpZ3VyYXRpb25UZW1wbGF0ZSgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW2FybiBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfQVJOKS4gKipEZWZhdWx0OioqIGBBcm5MaWtlYFxuICAgKi9cbiAgcHVibGljIGlmRnJvbUFwcGxpY2F0aW9uKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBGcm9tQXBwbGljYXRpb25gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ0Fybkxpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhbiBhcHBsaWNhdGlvbiB2ZXJzaW9uIGFzIGEgZGVwZW5kZW5jeSBvciBhIGNvbnN0cmFpbnQgb24gYW4gaW5wdXQgcGFyYW1ldGVyXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9kZy9BV1NIb3dUby5pYW0ucG9saWNpZXMuYWN0aW9ucy5odG1sI0FXU0hvd1RvLmlhbS5wb2xpY2llcy5jb25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKiAtIC50b0NyZWF0ZUVudmlyb25tZW50KClcbiAgICogLSAudG9VcGRhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKiAtIC50b1VwZGF0ZUVudmlyb25tZW50KClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkxpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZGcm9tQXBwbGljYXRpb25WZXJzaW9uKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBGcm9tQXBwbGljYXRpb25WZXJzaW9uYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdBcm5MaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgYSBjb25maWd1cmF0aW9uIHRlbXBsYXRlIGFzIGEgZGVwZW5kZW5jeSBvciBhIGNvbnN0cmFpbnQgb24gYW4gaW5wdXQgcGFyYW1ldGVyXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9kZy9BV1NIb3dUby5pYW0ucG9saWNpZXMuYWN0aW9ucy5odG1sI0FXU0hvd1RvLmlhbS5wb2xpY2llcy5jb25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKiAtIC50b0NyZWF0ZUVudmlyb25tZW50KClcbiAgICogLSAudG9VcGRhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKiAtIC50b1VwZGF0ZUVudmlyb25tZW50KClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkxpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZGcm9tQ29uZmlndXJhdGlvblRlbXBsYXRlKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBGcm9tQ29uZmlndXJhdGlvblRlbXBsYXRlYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdBcm5MaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgYW4gZW52aXJvbm1lbnQgYXMgYSBkZXBlbmRlbmN5IG9yIGEgY29uc3RyYWludCBvbiBhbiBpbnB1dCBwYXJhbWV0ZXJcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNiZWFuc3RhbGsvbGF0ZXN0L2RnL0FXU0hvd1RvLmlhbS5wb2xpY2llcy5hY3Rpb25zLmh0bWwjQVdTSG93VG8uaWFtLnBvbGljaWVzLmNvbmRpdGlvbnNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUNvbmZpZ3VyYXRpb25UZW1wbGF0ZSgpXG4gICAqIC0gLnRvU3dhcEVudmlyb25tZW50Q05BTUVzKClcbiAgICogLSAudG9VcGRhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFthcm4gb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX0FSTikuICoqRGVmYXVsdDoqKiBgQXJuTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkZyb21FbnZpcm9ubWVudCh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgRnJvbUVudmlyb25tZW50YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdBcm5MaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgYSBwbGF0Zm9ybSBhcyBhIGRlcGVuZGVuY3kgb3IgYSBjb25zdHJhaW50IG9uIGFuIGlucHV0IHBhcmFtZXRlclxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2JlYW5zdGFsay9sYXRlc3QvZGcvQVdTSG93VG8uaWFtLnBvbGljaWVzLmFjdGlvbnMuaHRtbCNBV1NIb3dUby5pYW0ucG9saWNpZXMuY29uZGl0aW9uc1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlQ29uZmlndXJhdGlvblRlbXBsYXRlKClcbiAgICogLSAudG9DcmVhdGVFbnZpcm9ubWVudCgpXG4gICAqIC0gLnRvVXBkYXRlQ29uZmlndXJhdGlvblRlbXBsYXRlKClcbiAgICogLSAudG9VcGRhdGVFbnZpcm9ubWVudCgpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW2FybiBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfQVJOKS4gKipEZWZhdWx0OioqIGBBcm5MaWtlYFxuICAgKi9cbiAgcHVibGljIGlmRnJvbVBsYXRmb3JtKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBGcm9tUGxhdGZvcm1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ0Fybkxpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhIHNvbHV0aW9uIHN0YWNrIGFzIGEgZGVwZW5kZW5jeSBvciBhIGNvbnN0cmFpbnQgb24gYW4gaW5wdXQgcGFyYW1ldGVyXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9kZy9BV1NIb3dUby5pYW0ucG9saWNpZXMuYWN0aW9ucy5odG1sI0FXU0hvd1RvLmlhbS5wb2xpY2llcy5jb25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKiAtIC50b0NyZWF0ZUVudmlyb25tZW50KClcbiAgICogLSAudG9VcGRhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKiAtIC50b1VwZGF0ZUVudmlyb25tZW50KClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbYXJuIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19BUk4pLiAqKkRlZmF1bHQ6KiogYEFybkxpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZGcm9tU29sdXRpb25TdGFjayh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgRnJvbVNvbHV0aW9uU3RhY2tgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ0Fybkxpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0aGUgYXBwbGljYXRpb24gdGhhdCBjb250YWlucyB0aGUgcmVzb3VyY2UgdGhhdCB0aGUgYWN0aW9uIG9wZXJhdGVzIG9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljYmVhbnN0YWxrL2xhdGVzdC9kZy9BV1NIb3dUby5pYW0ucG9saWNpZXMuYWN0aW9ucy5odG1sI0FXU0hvd1RvLmlhbS5wb2xpY2llcy5jb25kaXRpb25zXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9BYm9ydEVudmlyb25tZW50VXBkYXRlKClcbiAgICogLSAudG9BcHBseUVudmlyb25tZW50TWFuYWdlZEFjdGlvbigpXG4gICAqIC0gLnRvQ29tcG9zZUVudmlyb25tZW50cygpXG4gICAqIC0gLnRvQ3JlYXRlQXBwbGljYXRpb25WZXJzaW9uKClcbiAgICogLSAudG9DcmVhdGVDb25maWd1cmF0aW9uVGVtcGxhdGUoKVxuICAgKiAtIC50b0NyZWF0ZUVudmlyb25tZW50KClcbiAgICogLSAudG9EZWxldGVBcHBsaWNhdGlvblZlcnNpb24oKVxuICAgKiAtIC50b0RlbGV0ZUNvbmZpZ3VyYXRpb25UZW1wbGF0ZSgpXG4gICAqIC0gLnRvRGVsZXRlRW52aXJvbm1lbnRDb25maWd1cmF0aW9uKClcbiAgICogLSAudG9EZXNjcmliZUFwcGxpY2F0aW9uVmVyc2lvbnMoKVxuICAgKiAtIC50b0Rlc2NyaWJlQ29uZmlndXJhdGlvbk9wdGlvbnMoKVxuICAgKiAtIC50b0Rlc2NyaWJlQ29uZmlndXJhdGlvblNldHRpbmdzKClcbiAgICogLSAudG9EZXNjcmliZUVudmlyb25tZW50TWFuYWdlZEFjdGlvbkhpc3RvcnkoKVxuICAgKiAtIC50b0Rlc2NyaWJlRW52aXJvbm1lbnRNYW5hZ2VkQWN0aW9ucygpXG4gICAqIC0gLnRvRGVzY3JpYmVFbnZpcm9ubWVudFJlc291cmNlcygpXG4gICAqIC0gLnRvRGVzY3JpYmVFbnZpcm9ubWVudHMoKVxuICAgKiAtIC50b0Rlc2NyaWJlRXZlbnRzKClcbiAgICogLSAudG9SZWJ1aWxkRW52aXJvbm1lbnQoKVxuICAgKiAtIC50b1JlcXVlc3RFbnZpcm9ubWVudEluZm8oKVxuICAgKiAtIC50b1Jlc3RhcnRBcHBTZXJ2ZXIoKVxuICAgKiAtIC50b1JldHJpZXZlRW52aXJvbm1lbnRJbmZvKClcbiAgICogLSAudG9Td2FwRW52aXJvbm1lbnRDTkFNRXMoKVxuICAgKiAtIC50b1Rlcm1pbmF0ZUVudmlyb25tZW50KClcbiAgICogLSAudG9VcGRhdGVBcHBsaWNhdGlvblZlcnNpb24oKVxuICAgKiAtIC50b1VwZGF0ZUNvbmZpZ3VyYXRpb25UZW1wbGF0ZSgpXG4gICAqIC0gLnRvVXBkYXRlRW52aXJvbm1lbnQoKVxuICAgKiAtIC50b1ZhbGlkYXRlQ29uZmlndXJhdGlvblNldHRpbmdzKClcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBhcHBsaWNhdGlvbnZlcnNpb25cbiAgICogLSBjb25maWd1cmF0aW9udGVtcGxhdGVcbiAgICogLSBlbnZpcm9ubWVudFxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFthcm4gb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX0FSTikuICoqRGVmYXVsdDoqKiBgQXJuTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkluQXBwbGljYXRpb24odmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYEluQXBwbGljYXRpb25gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ0Fybkxpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2VsYXN0aWNiZWFuc3RhbGtdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c2VsYXN0aWNiZWFuc3RhbGsuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19