"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ecr = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ecr](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticcontainerregistry.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ecr extends shared_1.PolicyStatement {
    /**
     * Grants permission to check the availability of multiple image layers in a specified registry and repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_BatchCheckLayerAvailability.html
     */
    toBatchCheckLayerAvailability() {
        return this.to('BatchCheckLayerAvailability');
    }
    /**
     * Grants permission to delete a list of specified images within a specified repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_BatchDeleteImage.html
     */
    toBatchDeleteImage() {
        return this.to('BatchDeleteImage');
    }
    /**
     * Grants permission to get detailed information for specified images within a specified repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_BatchGetImage.html
     */
    toBatchGetImage() {
        return this.to('BatchGetImage');
    }
    /**
     * Grants permission to retrieve repository scanning configuration for a list of repositories
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_BatchGetRepositoryScanningConfiguration.html
     */
    toBatchGetRepositoryScanningConfiguration() {
        return this.to('BatchGetRepositoryScanningConfiguration');
    }
    /**
     * Grants permission to retrieve the image from the upstream registry and import it to your private registry
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/userguide/pull-through-cache.html
     */
    toBatchImportUpstreamImage() {
        return this.to('BatchImportUpstreamImage');
    }
    /**
     * Grants permission to inform Amazon ECR that the image layer upload for a specified registry, repository name, and upload ID, has completed
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_CompleteLayerUpload.html
     */
    toCompleteLayerUpload() {
        return this.to('CompleteLayerUpload');
    }
    /**
     * Grants permission to create new pull-through cache rule
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_CreatePullThroughCacheRule.html
     */
    toCreatePullThroughCacheRule() {
        return this.to('CreatePullThroughCacheRule');
    }
    /**
     * Grants permission to create an image repository
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ecr:TagResource
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_CreateRepository.html
     */
    toCreateRepository() {
        return this.to('CreateRepository');
    }
    /**
     * Grants permission to create the repository creation template
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ecr:CreateRepository
     * - ecr:PutLifecyclePolicy
     * - ecr:SetRepositoryPolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_CreateRepositoryCreationTemplate.html
     */
    toCreateRepositoryCreationTemplate() {
        return this.to('CreateRepositoryCreationTemplate');
    }
    /**
     * Grants permission to delete the specified lifecycle policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DeleteLifecyclePolicy.html
     */
    toDeleteLifecyclePolicy() {
        return this.to('DeleteLifecyclePolicy');
    }
    /**
     * Grants permission to delete the pull-through cache rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DeletePullThroughCacheRule.html
     */
    toDeletePullThroughCacheRule() {
        return this.to('DeletePullThroughCacheRule');
    }
    /**
     * Grants permission to delete the registry policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DeleteRegistryPolicy.html
     */
    toDeleteRegistryPolicy() {
        return this.to('DeleteRegistryPolicy');
    }
    /**
     * Grants permission to delete an existing image repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DeleteRepository.html
     */
    toDeleteRepository() {
        return this.to('DeleteRepository');
    }
    /**
     * Grants permission to delete the repository creation template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DeleteRepositoryCreationTemplate.html
     */
    toDeleteRepositoryCreationTemplate() {
        return this.to('DeleteRepositoryCreationTemplate');
    }
    /**
     * Grants permission to delete the repository policy from a specified repository
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DeleteRepositoryPolicy.html
     */
    toDeleteRepositoryPolicy() {
        return this.to('DeleteRepositoryPolicy');
    }
    /**
     * Grants permission to retrieve replication status about an image in a registry, including failure reason if replication fails
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DescribeImageReplicationStatus.html
     */
    toDescribeImageReplicationStatus() {
        return this.to('DescribeImageReplicationStatus');
    }
    /**
     * Grants permission to describe the image scan findings for the specified image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DescribeImageScanFindings.html
     */
    toDescribeImageScanFindings() {
        return this.to('DescribeImageScanFindings');
    }
    /**
     * Grants permission to get metadata about the images in a repository, including image size, image tags, and creation date
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DescribeImages.html
     */
    toDescribeImages() {
        return this.to('DescribeImages');
    }
    /**
     * Grants permission to describe the pull-through cache rules
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DescribePullThroughCacheRules.html
     */
    toDescribePullThroughCacheRules() {
        return this.to('DescribePullThroughCacheRules');
    }
    /**
     * Grants permission to describe the registry settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DescribeRegistry.html
     */
    toDescribeRegistry() {
        return this.to('DescribeRegistry');
    }
    /**
     * Grants permission to describe image repositories in a registry
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DescribeRepositories.html
     */
    toDescribeRepositories() {
        return this.to('DescribeRepositories');
    }
    /**
     * Grants permission to describe the repository creation template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_DescribeRepositoryCreationTemplates.html
     */
    toDescribeRepositoryCreationTemplates() {
        return this.to('DescribeRepositoryCreationTemplates');
    }
    /**
     * Grants permission to retrieve account settings
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAccountSetting()
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_GetAccountSetting.html
     */
    toGetAccountSetting() {
        return this.to('GetAccountSetting');
    }
    /**
     * Grants permission to retrieve a token that is valid for a specified registry for 12 hours
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_GetAuthorizationToken.html
     */
    toGetAuthorizationToken() {
        return this.to('GetAuthorizationToken');
    }
    /**
     * Grants permission to retrieve the download URL corresponding to an image layer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_GetDownloadUrlForLayer.html
     */
    toGetDownloadUrlForLayer() {
        return this.to('GetDownloadUrlForLayer');
    }
    /**
     * Grants permission to retrieve the status about an image copy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/userguide/pull-through-cache.html
     */
    toGetImageCopyStatus() {
        return this.to('GetImageCopyStatus');
    }
    /**
     * Grants permission to retrieve the specified lifecycle policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_GetLifecyclePolicy.html
     */
    toGetLifecyclePolicy() {
        return this.to('GetLifecyclePolicy');
    }
    /**
     * Grants permission to retrieve the results of the specified lifecycle policy preview request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_GetLifecyclePolicyPreview.html
     */
    toGetLifecyclePolicyPreview() {
        return this.to('GetLifecyclePolicyPreview');
    }
    /**
     * Grants permission to retrieve the registry policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_GetRegistryPolicy.html
     */
    toGetRegistryPolicy() {
        return this.to('GetRegistryPolicy');
    }
    /**
     * Grants permission to retrieve registry scanning configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_GetRegistryScanningConfiguration.html
     */
    toGetRegistryScanningConfiguration() {
        return this.to('GetRegistryScanningConfiguration');
    }
    /**
     * Grants permission to retrieve the repository policy for a specified repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_GetRepositoryPolicy.html
     */
    toGetRepositoryPolicy() {
        return this.to('GetRepositoryPolicy');
    }
    /**
     * Grants permission to notify Amazon ECR that you intend to upload an image layer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_InitiateLayerUpload.html
     */
    toInitiateLayerUpload() {
        return this.to('InitiateLayerUpload');
    }
    /**
     * Grants permission to list all the image IDs for a given repository
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_ListImages.html
     */
    toListImages() {
        return this.to('ListImages');
    }
    /**
     * Grants permission to list the tags for an Amazon ECR resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to update account settings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAccountSetting()
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_PutAccountSetting.html
     */
    toPutAccountSetting() {
        return this.to('PutAccountSetting');
    }
    /**
     * Grants permission to create or update the image manifest associated with an image
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_PutImage.html
     */
    toPutImage() {
        return this.to('PutImage');
    }
    /**
     * Grants permission to update the image scanning configuration for a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_PutImageScanningConfiguration.html
     */
    toPutImageScanningConfiguration() {
        return this.to('PutImageScanningConfiguration');
    }
    /**
     * Grants permission to update the image tag mutability settings for a repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_PutImageTagMutability.html
     */
    toPutImageTagMutability() {
        return this.to('PutImageTagMutability');
    }
    /**
     * Grants permission to create or update a lifecycle policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_PutLifecyclePolicy.html
     */
    toPutLifecyclePolicy() {
        return this.to('PutLifecyclePolicy');
    }
    /**
     * Grants permission to update the registry policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_PutRegistryPolicy.html
     */
    toPutRegistryPolicy() {
        return this.to('PutRegistryPolicy');
    }
    /**
     * Grants permission to update registry scanning configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_PutRegistryScanningConfiguration.html
     */
    toPutRegistryScanningConfiguration() {
        return this.to('PutRegistryScanningConfiguration');
    }
    /**
     * Grants permission to update the replication configuration for the registry
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_PutReplicationConfiguration.html
     */
    toPutReplicationConfiguration() {
        return this.to('PutReplicationConfiguration');
    }
    /**
     * Grants permission to replicate images to the destination registry
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/userguide/registry-permissions.html
     */
    toReplicateImage() {
        return this.to('ReplicateImage');
    }
    /**
     * Grants permission to apply a repository policy on a specified repository to control access permissions
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_SetRepositoryPolicy.html
     */
    toSetRepositoryPolicy() {
        return this.to('SetRepositoryPolicy');
    }
    /**
     * Grants permission to start an image scan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_StartImageScan.html
     */
    toStartImageScan() {
        return this.to('StartImageScan');
    }
    /**
     * Grants permission to start a preview of the specified lifecycle policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_StartLifecyclePolicyPreview.html
     */
    toStartLifecyclePolicyPreview() {
        return this.to('StartLifecyclePolicyPreview');
    }
    /**
     * Grants permission to tag an Amazon ECR resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag an Amazon ECR resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the pull-through cache rule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_UpdatePullThroughCacheRule.html
     */
    toUpdatePullThroughCacheRule() {
        return this.to('UpdatePullThroughCacheRule');
    }
    /**
     * Grants permission to update the repository creation template
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ecr:CreateRepository
     * - ecr:PutLifecyclePolicy
     * - ecr:SetRepositoryPolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_UpdateRepositoryCreationTemplate.html
     */
    toUpdateRepositoryCreationTemplate() {
        return this.to('UpdateRepositoryCreationTemplate');
    }
    /**
     * Grants permission to upload an image layer part to Amazon ECR
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_UploadLayerPart.html
     */
    toUploadLayerPart() {
        return this.to('UploadLayerPart');
    }
    /**
     * Grants permission to validate the pull-through cache rule
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_ValidatePullThroughCacheRule.html
     */
    toValidatePullThroughCacheRule() {
        return this.to('ValidatePullThroughCacheRule');
    }
    /**
     * Adds a resource of type repository to the statement
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/userguide/Repositories.html
     *
     * @param repositoryName - Identifier for the repositoryName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onRepository(repositoryName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ecr:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:repository/${repositoryName}`);
    }
    /**
     * Filters access by the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateRepository()
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - repository
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateRepository()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ECR account setting name
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toGetAccountSetting()
     * - .toPutAccountSetting()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccountSetting(value, operator) {
        return this.if(`AccountSetting`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/AmazonECR/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to resource types:
     * - repository
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [ecr](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticcontainerregistry.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ecr';
        this.accessLevelList = {
            Read: [
                'BatchCheckLayerAvailability',
                'BatchGetImage',
                'BatchGetRepositoryScanningConfiguration',
                'DescribeImageReplicationStatus',
                'DescribeImageScanFindings',
                'DescribeRegistry',
                'DescribeRepositories',
                'DescribeRepositoryCreationTemplates',
                'GetAccountSetting',
                'GetAuthorizationToken',
                'GetDownloadUrlForLayer',
                'GetImageCopyStatus',
                'GetLifecyclePolicy',
                'GetLifecyclePolicyPreview',
                'GetRegistryPolicy',
                'GetRegistryScanningConfiguration',
                'GetRepositoryPolicy',
                'ListTagsForResource',
                'ValidatePullThroughCacheRule'
            ],
            Write: [
                'BatchDeleteImage',
                'BatchImportUpstreamImage',
                'CompleteLayerUpload',
                'CreatePullThroughCacheRule',
                'CreateRepository',
                'CreateRepositoryCreationTemplate',
                'DeleteLifecyclePolicy',
                'DeletePullThroughCacheRule',
                'DeleteRepository',
                'DeleteRepositoryCreationTemplate',
                'InitiateLayerUpload',
                'PutAccountSetting',
                'PutImage',
                'PutImageScanningConfiguration',
                'PutImageTagMutability',
                'PutLifecyclePolicy',
                'PutRegistryScanningConfiguration',
                'PutReplicationConfiguration',
                'ReplicateImage',
                'StartImageScan',
                'StartLifecyclePolicyPreview',
                'UpdatePullThroughCacheRule',
                'UpdateRepositoryCreationTemplate',
                'UploadLayerPart'
            ],
            'Permissions management': [
                'DeleteRegistryPolicy',
                'DeleteRepositoryPolicy',
                'PutRegistryPolicy',
                'SetRepositoryPolicy'
            ],
            List: [
                'DescribeImages',
                'DescribePullThroughCacheRules',
                'ListImages'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Ecr = Ecr;
//# sourceMappingURL=data:application/json;base64,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