"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticloadbalancing = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [elasticloadbalancing](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselasticloadbalancing.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticloadbalancing extends shared_1.PolicyStatement {
    /**
     * Grants permission to add the specified tags to the specified load balancer. Each load balancer can have a maximum of 10 tags
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifCreateAction()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_AddTags.html
     */
    toAddTags() {
        return this.to('AddTags');
    }
    /**
     * Grants permission to associate one or more security groups with your load balancer in a virtual private cloud (VPC)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifSecurityGroup()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_ApplySecurityGroupsToLoadBalancer.html
     */
    toApplySecurityGroupsToLoadBalancer() {
        return this.to('ApplySecurityGroupsToLoadBalancer');
    }
    /**
     * Grants permission to add one or more subnets to the set of configured subnets for the specified load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifSubnet()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_AttachLoadBalancerToSubnets.html
     */
    toAttachLoadBalancerToSubnets() {
        return this.to('AttachLoadBalancerToSubnets');
    }
    /**
     * Grants permission to specify the health check settings to use when evaluating the health state of your back-end instances
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_ConfigureHealthCheck.html
     */
    toConfigureHealthCheck() {
        return this.to('ConfigureHealthCheck');
    }
    /**
     * Grants permission to generate a stickiness policy with sticky session lifetimes that follow that of an application-generated cookie
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_CreateAppCookieStickinessPolicy.html
     */
    toCreateAppCookieStickinessPolicy() {
        return this.to('CreateAppCookieStickinessPolicy');
    }
    /**
     * Grants permission to generate a stickiness policy with sticky session lifetimes controlled by the lifetime of the browser (user-agent) or a specified expiration period
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_CreateLBCookieStickinessPolicy.html
     */
    toCreateLBCookieStickinessPolicy() {
        return this.to('CreateLBCookieStickinessPolicy');
    }
    /**
     * Grants permission to create a load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifSecurityGroup()
     * - .ifSubnet()
     * - .ifScheme()
     * - .ifListenerProtocol()
     *
     * Dependent actions:
     * - elasticloadbalancing:AddTags
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_CreateLoadBalancer.html
     */
    toCreateLoadBalancer() {
        return this.to('CreateLoadBalancer');
    }
    /**
     * Grants permission to create one or more listeners for the specified load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifListenerProtocol()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_CreateLoadBalancerListeners.html
     */
    toCreateLoadBalancerListeners() {
        return this.to('CreateLoadBalancerListeners');
    }
    /**
     * Grants permission to create a policy with the specified attributes for the specified load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifSecurityPolicy()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_CreateLoadBalancerPolicy.html
     */
    toCreateLoadBalancerPolicy() {
        return this.to('CreateLoadBalancerPolicy');
    }
    /**
     * Grants permission to delete the specified load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DeleteLoadBalancer.html
     */
    toDeleteLoadBalancer() {
        return this.to('DeleteLoadBalancer');
    }
    /**
     * Grants permission to delete the specified listeners from the specified load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DeleteLoadBalancerListeners.html
     */
    toDeleteLoadBalancerListeners() {
        return this.to('DeleteLoadBalancerListeners');
    }
    /**
     * Grants permission to delete the specified policy from the specified load balancer. This policy must not be enabled for any listeners
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DeleteLoadBalancerPolicy.html
     */
    toDeleteLoadBalancerPolicy() {
        return this.to('DeleteLoadBalancerPolicy');
    }
    /**
     * Grants permission to deregister the specified instances from the specified load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DeregisterInstancesFromLoadBalancer.html
     */
    toDeregisterInstancesFromLoadBalancer() {
        return this.to('DeregisterInstancesFromLoadBalancer');
    }
    /**
     * Grants permission to describe the state of the specified instances with respect to the specified load balancer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DescribeInstanceHealth.html
     */
    toDescribeInstanceHealth() {
        return this.to('DescribeInstanceHealth');
    }
    /**
     * Grants permission to describe the attributes for the specified load balancer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DescribeLoadBalancerAttributes.html
     */
    toDescribeLoadBalancerAttributes() {
        return this.to('DescribeLoadBalancerAttributes');
    }
    /**
     * Grants permission to describe the specified policies
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DescribeLoadBalancerPolicies.html
     */
    toDescribeLoadBalancerPolicies() {
        return this.to('DescribeLoadBalancerPolicies');
    }
    /**
     * Grants permission to describe the specified load balancer policy types
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DescribeLoadBalancerPolicyTypes.html
     */
    toDescribeLoadBalancerPolicyTypes() {
        return this.to('DescribeLoadBalancerPolicyTypes');
    }
    /**
     * Grants permission to describe the specified the load balancers. If no load balancers are specified, the call describes all of your load balancers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DescribeLoadBalancers.html
     */
    toDescribeLoadBalancers() {
        return this.to('DescribeLoadBalancers');
    }
    /**
     * Grants permission to describe the tags associated with the specified load balancers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DescribeTags.html
     */
    toDescribeTags() {
        return this.to('DescribeTags');
    }
    /**
     * Grants permission to remove the specified subnets from the set of configured subnets for the load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DetachLoadBalancerFromSubnets.html
     */
    toDetachLoadBalancerFromSubnets() {
        return this.to('DetachLoadBalancerFromSubnets');
    }
    /**
     * Grants permission to remove the specified Availability Zones from the set of Availability Zones for the specified load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_DisableAvailabilityZonesForLoadBalancer.html
     */
    toDisableAvailabilityZonesForLoadBalancer() {
        return this.to('DisableAvailabilityZonesForLoadBalancer');
    }
    /**
     * Grants permission to add the specified Availability Zones to the set of Availability Zones for the specified load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_EnableAvailabilityZonesForLoadBalancer.html
     */
    toEnableAvailabilityZonesForLoadBalancer() {
        return this.to('EnableAvailabilityZonesForLoadBalancer');
    }
    /**
     * Grants permission to modify the attributes of the specified load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_ModifyLoadBalancerAttributes.html
     */
    toModifyLoadBalancerAttributes() {
        return this.to('ModifyLoadBalancerAttributes');
    }
    /**
     * Grants permission to add the specified instances to the specified load balancer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_RegisterInstancesWithLoadBalancer.html
     */
    toRegisterInstancesWithLoadBalancer() {
        return this.to('RegisterInstancesWithLoadBalancer');
    }
    /**
     * Grants permission to remove one or more tags from the specified load balancer
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_RemoveTags.html
     */
    toRemoveTags() {
        return this.to('RemoveTags');
    }
    /**
     * Grants permission to set the certificate that terminates the specified listener's SSL connections
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_SetLoadBalancerListenerSSLCertificate.html
     */
    toSetLoadBalancerListenerSSLCertificate() {
        return this.to('SetLoadBalancerListenerSSLCertificate');
    }
    /**
     * Grants permission to replace the set of policies associated with the specified port on which the back-end server is listening with a new set of policies
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_SetLoadBalancerPoliciesForBackendServer.html
     */
    toSetLoadBalancerPoliciesForBackendServer() {
        return this.to('SetLoadBalancerPoliciesForBackendServer');
    }
    /**
     * Grants permission to replace the current set of policies for the specified load balancer port with the specified set of policies
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     * - .ifSecurityPolicy()
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/2012-06-01/APIReference/API_SetLoadBalancerPoliciesOfListener.html
     */
    toSetLoadBalancerPoliciesOfListener() {
        return this.to('SetLoadBalancerPoliciesOfListener');
    }
    /**
     * Adds a resource of type loadbalancer to the statement
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/what-is-load-balancing.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onLoadbalancer(loadBalancerName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticloadbalancing:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:loadbalancer/${loadBalancerName}`);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateLoadBalancer()
     * - .toRemoveTags()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toApplySecurityGroupsToLoadBalancer()
     * - .toAttachLoadBalancerToSubnets()
     * - .toConfigureHealthCheck()
     * - .toCreateAppCookieStickinessPolicy()
     * - .toCreateLBCookieStickinessPolicy()
     * - .toCreateLoadBalancer()
     * - .toCreateLoadBalancerListeners()
     * - .toCreateLoadBalancerPolicy()
     * - .toDeleteLoadBalancer()
     * - .toDeleteLoadBalancerListeners()
     * - .toDeleteLoadBalancerPolicy()
     * - .toDeregisterInstancesFromLoadBalancer()
     * - .toDetachLoadBalancerFromSubnets()
     * - .toDisableAvailabilityZonesForLoadBalancer()
     * - .toEnableAvailabilityZonesForLoadBalancer()
     * - .toModifyLoadBalancerAttributes()
     * - .toRegisterInstancesWithLoadBalancer()
     * - .toRemoveTags()
     * - .toSetLoadBalancerListenerSSLCertificate()
     * - .toSetLoadBalancerPoliciesForBackendServer()
     * - .toSetLoadBalancerPoliciesOfListener()
     *
     * Applies to resource types:
     * - loadbalancer
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateLoadBalancer()
     * - .toRemoveTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the name of a resource-creating API action
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/tagging-resources-during-creation.html
     *
     * Applies to actions:
     * - .toAddTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreateAction(value, operator) {
        return this.if(`CreateAction`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the listener protocols that are allowed in the request
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/security_iam_service-with-iam.html#listenerprotocol-condition
     *
     * Applies to actions:
     * - .toCreateLoadBalancer()
     * - .toCreateLoadBalancerListeners()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifListenerProtocol(value, operator) {
        return this.if(`ListenerProtocol`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the preface string for a tag key and value pair that are attached to a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagExists(value, operator) {
        return this.if(`ResourceTag/`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the preface string for a tag key and value pair that are attached to a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toApplySecurityGroupsToLoadBalancer()
     * - .toAttachLoadBalancerToSubnets()
     * - .toConfigureHealthCheck()
     * - .toCreateAppCookieStickinessPolicy()
     * - .toCreateLBCookieStickinessPolicy()
     * - .toCreateLoadBalancer()
     * - .toCreateLoadBalancerListeners()
     * - .toCreateLoadBalancerPolicy()
     * - .toDeleteLoadBalancer()
     * - .toDeleteLoadBalancerListeners()
     * - .toDeleteLoadBalancerPolicy()
     * - .toDeregisterInstancesFromLoadBalancer()
     * - .toDetachLoadBalancerFromSubnets()
     * - .toDisableAvailabilityZonesForLoadBalancer()
     * - .toEnableAvailabilityZonesForLoadBalancer()
     * - .toModifyLoadBalancerAttributes()
     * - .toRegisterInstancesWithLoadBalancer()
     * - .toRemoveTags()
     * - .toSetLoadBalancerListenerSSLCertificate()
     * - .toSetLoadBalancerPoliciesForBackendServer()
     * - .toSetLoadBalancerPoliciesOfListener()
     *
     * Applies to resource types:
     * - loadbalancer
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTagTagKey(tagKey, value, operator) {
        return this.if(`ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the load balancer scheme that are allowed in the request
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/security_iam_service-with-iam.html#scheme-condition
     *
     * Applies to actions:
     * - .toCreateLoadBalancer()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifScheme(value, operator) {
        return this.if(`Scheme`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the security-group IDs that are allowed in the request
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/security_iam_service-with-iam.html#securitygroup-condition
     *
     * Applies to actions:
     * - .toApplySecurityGroupsToLoadBalancer()
     * - .toCreateLoadBalancer()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecurityGroup(value, operator) {
        return this.if(`SecurityGroup`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the SSL Security Policies that are allowed in the request
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/security_iam_service-with-iam.html#securitypolicy-condition
     *
     * Applies to actions:
     * - .toCreateLoadBalancerPolicy()
     * - .toSetLoadBalancerPoliciesOfListener()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSecurityPolicy(value, operator) {
        return this.if(`SecurityPolicy`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the subnet IDs that are allowed in the request
     *
     * https://docs.aws.amazon.com/elasticloadbalancing/latest/userguide/security_iam_service-with-iam.html#subnet-condition
     *
     * Applies to actions:
     * - .toAttachLoadBalancerToSubnets()
     * - .toCreateLoadBalancer()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSubnet(value, operator) {
        return this.if(`Subnet`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [elasticloadbalancing](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselasticloadbalancing.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'elasticloadbalancing';
        this.accessLevelList = {
            Tagging: [
                'AddTags',
                'RemoveTags'
            ],
            Write: [
                'ApplySecurityGroupsToLoadBalancer',
                'AttachLoadBalancerToSubnets',
                'ConfigureHealthCheck',
                'CreateAppCookieStickinessPolicy',
                'CreateLBCookieStickinessPolicy',
                'CreateLoadBalancer',
                'CreateLoadBalancerListeners',
                'CreateLoadBalancerPolicy',
                'DeleteLoadBalancer',
                'DeleteLoadBalancerListeners',
                'DeleteLoadBalancerPolicy',
                'DeregisterInstancesFromLoadBalancer',
                'DetachLoadBalancerFromSubnets',
                'DisableAvailabilityZonesForLoadBalancer',
                'EnableAvailabilityZonesForLoadBalancer',
                'ModifyLoadBalancerAttributes',
                'RegisterInstancesWithLoadBalancer',
                'SetLoadBalancerListenerSSLCertificate',
                'SetLoadBalancerPoliciesForBackendServer',
                'SetLoadBalancerPoliciesOfListener'
            ],
            Read: [
                'DescribeInstanceHealth',
                'DescribeLoadBalancerAttributes',
                'DescribeLoadBalancerPolicies',
                'DescribeLoadBalancerPolicyTypes',
                'DescribeTags'
            ],
            List: [
                'DescribeLoadBalancers'
            ]
        };
    }
}
exports.Elasticloadbalancing = Elasticloadbalancing;
//# sourceMappingURL=data:application/json;base64,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