"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticmapreduce = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [elasticmapreduce](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticmapreduce.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticmapreduce extends shared_1.PolicyStatement {
    /**
     * Grants permission to view all event logs in a persistent application history server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio-debug.html
     */
    toAccessAllEventLogs() {
        return this.to('AccessAllEventLogs');
    }
    /**
     * Grants permission to add an instance fleet to a running cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddInstanceFleet.html
     */
    toAddInstanceFleet() {
        return this.to('AddInstanceFleet');
    }
    /**
     * Grants permission to add instance groups to a running cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddInstanceGroups.html
     */
    toAddInstanceGroups() {
        return this.to('AddInstanceGroups');
    }
    /**
     * Grants permission to add new steps to a running cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifExecutionRoleArn()
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddJobFlowSteps.html
     */
    toAddJobFlowSteps() {
        return this.to('AddJobFlowSteps');
    }
    /**
     * Grants permission to add tags to an Amazon EMR resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_AddTags.html
     */
    toAddTags() {
        return this.to('AddTags');
    }
    /**
     * Grants permission to attach an EMR notebook to a compute engine
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toAttachEditor() {
        return this.to('AttachEditor');
    }
    /**
     * Grants permission to cancel a pending step or steps in a running cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_CancelSteps.html
     */
    toCancelSteps() {
        return this.to('CancelSteps');
    }
    /**
     * Grants permission to create an EMR notebook
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-create.html
     */
    toCreateEditor() {
        return this.to('CreateEditor');
    }
    /**
     * Grants permission to create a persistent application history server
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio-debug.html
     */
    toCreatePersistentAppUI() {
        return this.to('CreatePersistentAppUI');
    }
    /**
     * Grants permission to create an EMR notebook repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toCreateRepository() {
        return this.to('CreateRepository');
    }
    /**
     * Grants permission to create a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_CreateSecurityConfiguration.html
     */
    toCreateSecurityConfiguration() {
        return this.to('CreateSecurityConfiguration');
    }
    /**
     * Grants permission to create an EMR Studio
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toCreateStudio() {
        return this.to('CreateStudio');
    }
    /**
     * Grants permission to launch an EMR Studio using IAM authentication mode
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toCreateStudioPresignedUrl() {
        return this.to('CreateStudioPresignedUrl');
    }
    /**
     * Grants permission to create an EMR Studio session mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toCreateStudioSessionMapping() {
        return this.to('CreateStudioSessionMapping');
    }
    /**
     * Grants permission to delete an EMR notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-deleting
     */
    toDeleteEditor() {
        return this.to('DeleteEditor');
    }
    /**
     * Grants permission to delete an EMR notebook repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toDeleteRepository() {
        return this.to('DeleteRepository');
    }
    /**
     * Grants permission to delete a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DeleteSecurityConfiguration.html
     */
    toDeleteSecurityConfiguration() {
        return this.to('DeleteSecurityConfiguration');
    }
    /**
     * Grants permission to delete an EMR Studio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toDeleteStudio() {
        return this.to('DeleteStudio');
    }
    /**
     * Grants permission to delete an EMR Studio session mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toDeleteStudioSessionMapping() {
        return this.to('DeleteStudioSessionMapping');
    }
    /**
     * Grants permission to block an identity from opening a collaborative workspace
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toDeleteWorkspaceAccess() {
        return this.to('DeleteWorkspaceAccess');
    }
    /**
     * Grants permission to get details about a cluster, including status, hardware and software configuration, VPC settings, and so on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeCluster.html
     */
    toDescribeCluster() {
        return this.to('DescribeCluster');
    }
    /**
     * Grants permission to view information about a notebook, including status, user, role, tags, location, and more
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toDescribeEditor() {
        return this.to('DescribeEditor');
    }
    /**
     * Grants permission to describe details of clusters (job flows). This API is deprecated and will eventually be removed. We recommend you use ListClusters, DescribeCluster, ListSteps, ListInstanceGroups and ListBootstrapActions instead
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeJobFlows.html
     */
    toDescribeJobFlows() {
        return this.to('DescribeJobFlows');
    }
    /**
     * Grants permission to view information about a notebook execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-headless.html
     */
    toDescribeNotebookExecution() {
        return this.to('DescribeNotebookExecution');
    }
    /**
     * Grants permission to describe a persistent application history server
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio-debug.html
     */
    toDescribePersistentAppUI() {
        return this.to('DescribePersistentAppUI');
    }
    /**
     * Grants permission to view information about an EMR release, such as which applications are supported
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeReleaseLabel.html
     */
    toDescribeReleaseLabel() {
        return this.to('DescribeReleaseLabel');
    }
    /**
     * Grants permission to describe an EMR notebook repository
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toDescribeRepository() {
        return this.to('DescribeRepository');
    }
    /**
     * Grants permission to get details of a security configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeSecurityConfiguration.html
     */
    toDescribeSecurityConfiguration() {
        return this.to('DescribeSecurityConfiguration');
    }
    /**
     * Grants permission to get details about a cluster step
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_DescribeStep.html
     */
    toDescribeStep() {
        return this.to('DescribeStep');
    }
    /**
     * Grants permission to view information about an EMR Studio
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toDescribeStudio() {
        return this.to('DescribeStudio');
    }
    /**
     * Grants permission to detach an EMR notebook from a compute engine
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toDetachEditor() {
        return this.to('DetachEditor');
    }
    /**
     * Grants permission to retrieve the auto-termination policy associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_GetAutoTerminationPolicy.html
     */
    toGetAutoTerminationPolicy() {
        return this.to('GetAutoTerminationPolicy');
    }
    /**
     * Grants permission to retrieve the EMR block public access configuration for the AWS account in the Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_GetBlockPublicAccessConfiguration.html
     */
    toGetBlockPublicAccessConfiguration() {
        return this.to('GetBlockPublicAccessConfiguration');
    }
    /**
     * Grants permission to retrieve HTTP basic credentials associated with a given execution IAM Role for a fine-grained access control enabled EMR Cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifExecutionRoleArn()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-steps-runtime-roles.html
     */
    toGetClusterSessionCredentials() {
        return this.to('GetClusterSessionCredentials');
    }
    /**
     * Grants permission to retrieve the managed scaling policy associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_GetManagedScalingPolicy.html
     */
    toGetManagedScalingPolicy() {
        return this.to('GetManagedScalingPolicy');
    }
    /**
     * Grants permission to get a presigned URL for an application history server running on the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio-debug.html
     */
    toGetOnClusterAppUIPresignedURL() {
        return this.to('GetOnClusterAppUIPresignedURL');
    }
    /**
     * Grants permission to get a presigned URL for a persistent application history server
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifExecutionRoleArn()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio-debug.html
     */
    toGetPersistentAppUIPresignedURL() {
        return this.to('GetPersistentAppUIPresignedURL');
    }
    /**
     * Grants permission to view information about an EMR Studio session mapping
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toGetStudioSessionMapping() {
        return this.to('GetStudioSessionMapping');
    }
    /**
     * Grants permission to link an EMR notebook repository to EMR notebooks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toLinkRepository() {
        return this.to('LinkRepository');
    }
    /**
     * Grants permission to get details about the bootstrap actions associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListBootstrapActions.html
     */
    toListBootstrapActions() {
        return this.to('ListBootstrapActions');
    }
    /**
     * Grants permission to get the status of accessible clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListClusters.html
     */
    toListClusters() {
        return this.to('ListClusters');
    }
    /**
     * Grants permission to list summary information for accessible EMR notebooks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toListEditors() {
        return this.to('ListEditors');
    }
    /**
     * Grants permission to get details of instance fleets in a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListInstanceFleets.html
     */
    toListInstanceFleets() {
        return this.to('ListInstanceFleets');
    }
    /**
     * Grants permission to get details of instance groups in a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListInstanceGroups.html
     */
    toListInstanceGroups() {
        return this.to('ListInstanceGroups');
    }
    /**
     * Grants permission to get details about the Amazon EC2 instances in a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListInstances.html
     */
    toListInstances() {
        return this.to('ListInstances');
    }
    /**
     * Grants permission to list summary information for notebook executions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-headless.html
     */
    toListNotebookExecutions() {
        return this.to('ListNotebookExecutions');
    }
    /**
     * Grants permission to list and filter the available EMR releases in the current region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListReleaseLabels.html
     */
    toListReleaseLabels() {
        return this.to('ListReleaseLabels');
    }
    /**
     * Grants permission to list existing EMR notebook repositories
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toListRepositories() {
        return this.to('ListRepositories');
    }
    /**
     * Grants permission to list available security configurations in this account by name, along with creation dates and times
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListSecurityConfigurations.html
     */
    toListSecurityConfigurations() {
        return this.to('ListSecurityConfigurations');
    }
    /**
     * Grants permission to list steps associated with a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListSteps.html
     */
    toListSteps() {
        return this.to('ListSteps');
    }
    /**
     * Grants permission to list summary information about EMR Studio session mappings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toListStudioSessionMappings() {
        return this.to('ListStudioSessionMappings');
    }
    /**
     * Grants permission to list summary information about EMR Studios
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toListStudios() {
        return this.to('ListStudios');
    }
    /**
     * Grants permission to list the Amazon EC2 instance types that an Amazon EMR release supports
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ListSupportedInstanceTypes.html
     */
    toListSupportedInstanceTypes() {
        return this.to('ListSupportedInstanceTypes');
    }
    /**
     * Grants permission to list identities that are granted access to a workspace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toListWorkspaceAccessIdentities() {
        return this.to('ListWorkspaceAccessIdentities');
    }
    /**
     * Grants permission to change cluster settings such as number of steps that can be executed concurrently for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyCluster.html
     */
    toModifyCluster() {
        return this.to('ModifyCluster');
    }
    /**
     * Grants permission to change the target On-Demand and target Spot capacities for a instance fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceFleet.html
     */
    toModifyInstanceFleet() {
        return this.to('ModifyInstanceFleet');
    }
    /**
     * Grants permission to change the number and configuration of EC2 instances for an instance group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_ModifyInstanceGroups.html
     */
    toModifyInstanceGroups() {
        return this.to('ModifyInstanceGroups');
    }
    /**
     * Grants permission to launch the Jupyter notebook editor for an EMR notebook from within the console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toOpenEditorInConsole() {
        return this.to('OpenEditorInConsole');
    }
    /**
     * Grants permission to create or update an automatic scaling policy for a core instance group or task instance group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutAutoScalingPolicy.html
     */
    toPutAutoScalingPolicy() {
        return this.to('PutAutoScalingPolicy');
    }
    /**
     * Grants permission to create or update the auto-termination policy associated with a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutAutoTerminationPolicy.html
     */
    toPutAutoTerminationPolicy() {
        return this.to('PutAutoTerminationPolicy');
    }
    /**
     * Grants permission to create or update the EMR block public access configuration for the AWS account in the Region
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutBlockPublicAccessConfiguration.html
     */
    toPutBlockPublicAccessConfiguration() {
        return this.to('PutBlockPublicAccessConfiguration');
    }
    /**
     * Grants permission to create or update the managed scaling policy associated with a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_PutManagedScalingPolicy.html
     */
    toPutManagedScalingPolicy() {
        return this.to('PutManagedScalingPolicy');
    }
    /**
     * Grants permission to allow an identity to open a collaborative workspace
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toPutWorkspaceAccess() {
        return this.to('PutWorkspaceAccess');
    }
    /**
     * Grants permission to remove an automatic scaling policy from an instance group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveAutoScalingPolicy.html
     */
    toRemoveAutoScalingPolicy() {
        return this.to('RemoveAutoScalingPolicy');
    }
    /**
     * Grants permission to remove the auto-termination policy associated with a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveAutoTerminationPolicy.html
     */
    toRemoveAutoTerminationPolicy() {
        return this.to('RemoveAutoTerminationPolicy');
    }
    /**
     * Grants permission to remove the managed scaling policy associated with a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveManagedScalingPolicy.html
     */
    toRemoveManagedScalingPolicy() {
        return this.to('RemoveManagedScalingPolicy');
    }
    /**
     * Grants permission to remove tags from an Amazon EMR resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RemoveTags.html
     */
    toRemoveTags() {
        return this.to('RemoveTags');
    }
    /**
     * Grants permission to create and launch a cluster (job flow)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_RunJobFlow.html
     */
    toRunJobFlow() {
        return this.to('RunJobFlow');
    }
    /**
     * Grants permission to add and remove auto terminate after step execution for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_SetKeepJobFlowAliveWhenNoSteps.html
     */
    toSetKeepJobFlowAliveWhenNoSteps() {
        return this.to('SetKeepJobFlowAliveWhenNoSteps');
    }
    /**
     * Grants permission to add and remove termination protection for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_SetTerminationProtection.html
     */
    toSetTerminationProtection() {
        return this.to('SetTerminationProtection');
    }
    /**
     * Grants permission to enable or disable unhealthy node replacement for a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_SetUnhealthyNodeReplacement.html
     */
    toSetUnhealthyNodeReplacement() {
        return this.to('SetUnhealthyNodeReplacement');
    }
    /**
     * Grants permission to set whether all AWS Identity and Access Management (IAM) users in the AWS account can view a cluster. This API is deprecated and your cluster may be visible to all users in your account. To restrict cluster access using an IAM policy, see AWS Identity and Access Management for Amazon EMR (https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_SetVisibleToAllUsers.html
     */
    toSetVisibleToAllUsers() {
        return this.to('SetVisibleToAllUsers');
    }
    /**
     * Grants permission to start an EMR notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toStartEditor() {
        return this.to('StartEditor');
    }
    /**
     * Grants permission to start an EMR notebook execution
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifRequestTag()
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-headless.html
     */
    toStartNotebookExecution() {
        return this.to('StartNotebookExecution');
    }
    /**
     * Grants permission to shut down an EMR notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html
     */
    toStopEditor() {
        return this.to('StopEditor');
    }
    /**
     * Grants permission to stop notebook execution
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-headless.html
     */
    toStopNotebookExecution() {
        return this.to('StopNotebookExecution');
    }
    /**
     * Grants permission to terminate a cluster (job flow)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/APIReference/API_TerminateJobFlows.html
     */
    toTerminateJobFlows() {
        return this.to('TerminateJobFlows');
    }
    /**
     * Grants permission to unlink an EMR notebook repository from EMR notebooks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toUnlinkRepository() {
        return this.to('UnlinkRepository');
    }
    /**
     * Grants permission to update an EMR notebook
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-working-with.html
     */
    toUpdateEditor() {
        return this.to('UpdateEditor');
    }
    /**
     * Grants permission to update an EMR notebook repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html#emr-managed-notebooks-editor
     */
    toUpdateRepository() {
        return this.to('UpdateRepository');
    }
    /**
     * Grants permission to update information about an EMR Studio
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toUpdateStudio() {
        return this.to('UpdateStudio');
    }
    /**
     * Grants permission to update an EMR Studio session mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     */
    toUpdateStudioSessionMapping() {
        return this.to('UpdateStudioSessionMapping');
    }
    /**
     * Grants permission to use the EMR console to view events from all clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticmapreduce.html
     */
    toViewEventsFromAllClustersInConsole() {
        return this.to('ViewEventsFromAllClustersInConsole');
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-overview.html
     *
     * @param clusterId - Identifier for the clusterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onCluster(clusterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticmapreduce:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster/${clusterId}`);
    }
    /**
     * Adds a resource of type editor to the statement
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks.html
     *
     * @param editorId - Identifier for the editorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onEditor(editorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticmapreduce:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:editor/${editorId}`);
    }
    /**
     * Adds a resource of type notebook-execution to the statement
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-headless.html
     *
     * @param notebookExecutionId - Identifier for the notebookExecutionId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onNotebookExecution(notebookExecutionId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticmapreduce:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:notebook-execution/${notebookExecutionId}`);
    }
    /**
     * Adds a resource of type studio to the statement
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-studio.html
     *
     * @param studioId - Identifier for the studioId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifResourceTag()
     */
    onStudio(studioId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticmapreduce:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:studio/${studioId}`);
    }
    /**
     * Filters access by whether the tag and value pair is provided with the action
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateEditor()
     * - .toCreateStudio()
     * - .toRunJobFlow()
     * - .toStartNotebookExecution()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag and value pair associated with an Amazon EMR resource
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to resource types:
     * - cluster
     * - editor
     * - notebook-execution
     * - studio
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the tag keys are provided with the action regardless of tag value
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateEditor()
     * - .toCreateStudio()
     * - .toRemoveTags()
     * - .toRunJobFlow()
     * - .toStartNotebookExecution()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the execution role ARN is provided with the action
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-security
     *
     * Applies to actions:
     * - .toAddJobFlowSteps()
     * - .toGetClusterSessionCredentials()
     * - .toGetPersistentAppUIPresignedURL()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifExecutionRoleArn(value, operator) {
        return this.if(`ExecutionRoleArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by whether the tag and value pair is provided with the action
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to actions:
     * - .toAddTags()
     * - .toCreateEditor()
     * - .toCreateStudio()
     * - .toRunJobFlow()
     * - .toStartNotebookExecution()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestTag(tagKey, value, operator) {
        return this.if(`RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag and value pair associated with an Amazon EMR resource
     *
     * https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-plan-access-iam.html#emr-fine-grained-cluster-access
     *
     * Applies to resource types:
     * - cluster
     * - editor
     * - notebook-execution
     * - studio
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTag(tagKey, value, operator) {
        return this.if(`ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [elasticmapreduce](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticmapreduce.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'elasticmapreduce';
        this.accessLevelList = {
            Write: [
                'AccessAllEventLogs',
                'AddInstanceFleet',
                'AddInstanceGroups',
                'AddJobFlowSteps',
                'AttachEditor',
                'CancelSteps',
                'CreateEditor',
                'CreatePersistentAppUI',
                'CreateRepository',
                'CreateSecurityConfiguration',
                'CreateStudio',
                'CreateStudioPresignedUrl',
                'CreateStudioSessionMapping',
                'DeleteEditor',
                'DeleteRepository',
                'DeleteSecurityConfiguration',
                'DeleteStudio',
                'DeleteStudioSessionMapping',
                'DetachEditor',
                'GetClusterSessionCredentials',
                'GetOnClusterAppUIPresignedURL',
                'GetPersistentAppUIPresignedURL',
                'LinkRepository',
                'ModifyCluster',
                'ModifyInstanceFleet',
                'ModifyInstanceGroups',
                'OpenEditorInConsole',
                'PutAutoScalingPolicy',
                'PutAutoTerminationPolicy',
                'PutManagedScalingPolicy',
                'RemoveAutoScalingPolicy',
                'RemoveAutoTerminationPolicy',
                'RemoveManagedScalingPolicy',
                'RunJobFlow',
                'SetKeepJobFlowAliveWhenNoSteps',
                'SetTerminationProtection',
                'SetUnhealthyNodeReplacement',
                'SetVisibleToAllUsers',
                'StartEditor',
                'StartNotebookExecution',
                'StopEditor',
                'StopNotebookExecution',
                'TerminateJobFlows',
                'UnlinkRepository',
                'UpdateEditor',
                'UpdateRepository',
                'UpdateStudio',
                'UpdateStudioSessionMapping'
            ],
            Tagging: [
                'AddTags',
                'RemoveTags'
            ],
            'Permissions management': [
                'DeleteWorkspaceAccess',
                'PutBlockPublicAccessConfiguration',
                'PutWorkspaceAccess'
            ],
            Read: [
                'DescribeCluster',
                'DescribeEditor',
                'DescribeJobFlows',
                'DescribeNotebookExecution',
                'DescribePersistentAppUI',
                'DescribeReleaseLabel',
                'DescribeRepository',
                'DescribeSecurityConfiguration',
                'DescribeStep',
                'DescribeStudio',
                'GetAutoTerminationPolicy',
                'GetBlockPublicAccessConfiguration',
                'GetManagedScalingPolicy',
                'GetStudioSessionMapping',
                'ListBootstrapActions',
                'ListInstanceFleets',
                'ListInstanceGroups',
                'ListInstances',
                'ListSteps'
            ],
            List: [
                'ListClusters',
                'ListEditors',
                'ListNotebookExecutions',
                'ListReleaseLabels',
                'ListRepositories',
                'ListSecurityConfigurations',
                'ListStudioSessionMappings',
                'ListStudios',
                'ListSupportedInstanceTypes',
                'ListWorkspaceAccessIdentities',
                'ViewEventsFromAllClustersInConsole'
            ]
        };
    }
}
exports.Elasticmapreduce = Elasticmapreduce;
//# sourceMappingURL=data:application/json;base64,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