"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elastictranscoder = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [elastictranscoder](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelastictranscoder.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elastictranscoder extends shared_1.PolicyStatement {
    /**
     * Cancel a job that Elastic Transcoder has not begun to process
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/cancel-job.html
     */
    toCancelJob() {
        return this.to('CancelJob');
    }
    /**
     * Create a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-job.html
     */
    toCreateJob() {
        return this.to('CreateJob');
    }
    /**
     * Create a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-pipeline.html
     */
    toCreatePipeline() {
        return this.to('CreatePipeline');
    }
    /**
     * Create a preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/create-preset.html
     */
    toCreatePreset() {
        return this.to('CreatePreset');
    }
    /**
     * Delete a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-pipeline.html
     */
    toDeletePipeline() {
        return this.to('DeletePipeline');
    }
    /**
     * Delete a preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/delete-preset.html
     */
    toDeletePreset() {
        return this.to('DeletePreset');
    }
    /**
     * Get a list of the jobs that you assigned to a pipeline
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-pipeline.html
     */
    toListJobsByPipeline() {
        return this.to('ListJobsByPipeline');
    }
    /**
     * Get information about all of the jobs associated with the current AWS account that have a specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-jobs-by-status.html
     */
    toListJobsByStatus() {
        return this.to('ListJobsByStatus');
    }
    /**
     * Get a list of the pipelines associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-pipelines.html
     */
    toListPipelines() {
        return this.to('ListPipelines');
    }
    /**
     * Get a list of all presets associated with the current AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/list-presets.html
     */
    toListPresets() {
        return this.to('ListPresets');
    }
    /**
     * Get detailed information about a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-job.html
     */
    toReadJob() {
        return this.to('ReadJob');
    }
    /**
     * Get detailed information about a pipeline
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-pipeline.html
     */
    toReadPipeline() {
        return this.to('ReadPipeline');
    }
    /**
     * Get detailed information about a preset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/get-preset.html
     */
    toReadPreset() {
        return this.to('ReadPreset');
    }
    /**
     * Test the settings for a pipeline to ensure that Elastic Transcoder can create and process jobs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/test-pipeline-role.html
     */
    toTestRole() {
        return this.to('TestRole');
    }
    /**
     * Update settings for a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline.html
     */
    toUpdatePipeline() {
        return this.to('UpdatePipeline');
    }
    /**
     * Update only Amazon Simple Notification Service (Amazon SNS) notifications for a pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-notifications.html
     */
    toUpdatePipelineNotifications() {
        return this.to('UpdatePipelineNotifications');
    }
    /**
     * Pause or reactivate a pipeline, so the pipeline stops or restarts processing jobs, update the status for the pipeline
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/update-pipeline-status.html
     */
    toUpdatePipelineStatus() {
        return this.to('UpdatePipelineStatus');
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elastictranscoder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:job/${jobId}`);
    }
    /**
     * Adds a resource of type pipeline to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-pipelines.html
     *
     * @param pipelineId - Identifier for the pipelineId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onPipeline(pipelineId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elastictranscoder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:pipeline/${pipelineId}`);
    }
    /**
     * Adds a resource of type preset to the statement
     *
     * https://docs.aws.amazon.com/elastictranscoder/latest/developerguide/operations-presets.html
     *
     * @param presetId - Identifier for the presetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onPreset(presetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elastictranscoder:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:preset/${presetId}`);
    }
    /**
     * Statement provider for service [elastictranscoder](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelastictranscoder.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'elastictranscoder';
        this.accessLevelList = {
            Write: [
                'CancelJob',
                'CreateJob',
                'CreatePipeline',
                'CreatePreset',
                'DeletePipeline',
                'DeletePreset',
                'TestRole',
                'UpdatePipeline',
                'UpdatePipelineNotifications',
                'UpdatePipelineStatus'
            ],
            List: [
                'ListJobsByPipeline',
                'ListJobsByStatus',
                'ListPipelines',
                'ListPresets'
            ],
            Read: [
                'ReadJob',
                'ReadPipeline',
                'ReadPreset'
            ]
        };
    }
}
exports.Elastictranscoder = Elastictranscoder;
//# sourceMappingURL=data:application/json;base64,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