import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [evs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticvmwareservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Evs extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to associate an Elastic IP address (EIP) with a public VLAN in an Amazon EVS environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_AssociateEipToVlan.html
     */
    toAssociateEipToVlan(): this;
    /**
     * Grants permission to create an Amazon EVS environment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_CreateEnvironment.html
     */
    toCreateEnvironment(): this;
    /**
     * Grants permission to add host to an Amazon EVS environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_CreateEnvironmentHost.html
     */
    toCreateEnvironmentHost(): this;
    /**
     * Grants permission to delete an Amazon EVS environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_DeleteEnvironment.html
     */
    toDeleteEnvironment(): this;
    /**
     * Grants permission to delete a host from an Amazon EVS environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_DeleteEnvironmentHost.html
     */
    toDeleteEnvironmentHost(): this;
    /**
     * Grants permission to disassociate an Elastic IP address (EIP) from a public VLAN in an Amazon EVS environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_DisassociateEipFromVlan.html
     */
    toDisassociateEipFromVlan(): this;
    /**
     * Grants permission to get an Amazon EVS environment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_GetEnvironment.html
     */
    toGetEnvironment(): this;
    /**
     * Grants permission to retrieve a list of hosts associated with an Amazon EVS environment
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_ListEnvironmentHosts.html
     */
    toListEnvironmentHosts(): this;
    /**
     * Grants permission to retrieve a list of Amazon EVS environment VLANs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_ListEnvironmentVlans.html
     */
    toListEnvironmentVlans(): this;
    /**
     * Grants permission to retrieve a list of Amazon EVS environments in an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_ListEnvironments.html
     */
    toListEnvironments(): this;
    /**
     * Grants permission to list the tags on a specified resource ARN
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to tag a specified resource ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to remove tags from a specified resource ARN
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/evs/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type environment to the statement
     *
     * https://docs.aws.amazon.com/evs/latest/userguide/concepts.html#concepts-evs-virt-env
     *
     * @param environmentIdentifier - Identifier for the environmentIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEnvironment(environmentIdentifier: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/evs/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateEnvironment()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/evs/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - environment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/evs/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies
     *
     * Applies to actions:
     * - .toCreateEnvironment()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [evs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticvmwareservice.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
