"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mediaconvert = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [mediaconvert](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediaconvert.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mediaconvert extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate an AWS Certificate Manager (ACM) Amazon Resource Name (ARN) with AWS Elemental MediaConvert
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html
     */
    toAssociateCertificate() {
        return this.to('AssociateCertificate');
    }
    /**
     * Grants permission to cancel an AWS Elemental MediaConvert job that is waiting in queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toCancelJob() {
        return this.to('CancelJob');
    }
    /**
     * Grants permission to create and submit an AWS Elemental MediaConvert job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifHttpInputsAllowed()
     * - .ifHttpsInputsAllowed()
     * - .ifS3InputsAllowed()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     */
    toCreateJob() {
        return this.to('CreateJob');
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toCreateJobTemplate() {
        return this.to('CreateJobTemplate');
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     */
    toCreatePreset() {
        return this.to('CreatePreset');
    }
    /**
     * Grants permission to create an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     */
    toCreateQueue() {
        return this.to('CreateQueue');
    }
    /**
     * Grants permission to share an AWS Elemental MediaConvert job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/resourceshares.html
     */
    toCreateResourceShare() {
        return this.to('CreateResourceShare');
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toDeleteJobTemplate() {
        return this.to('DeleteJobTemplate');
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/policy.html
     */
    toDeletePolicy() {
        return this.to('DeletePolicy');
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toDeletePreset() {
        return this.to('DeletePreset');
    }
    /**
     * Grants permission to delete an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toDeleteQueue() {
        return this.to('DeleteQueue');
    }
    /**
     * Grants permission to subscribe to the AWS Elemental MediaConvert service, by sending a request for an account-specific endpoint. All transcoding requests must be sent to the endpoint that the service returns
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/endpoints.html
     */
    toDescribeEndpoints() {
        return this.to('DescribeEndpoints');
    }
    /**
     * Grants permission to remove an association between the Amazon Resource Name (ARN) of an AWS Certificate Manager (ACM) certificate and an AWS Elemental MediaConvert resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates-arn.html
     */
    toDisassociateCertificate() {
        return this.to('DisassociateCertificate');
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs-id.html
     */
    toGetJob() {
        return this.to('GetJob');
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toGetJobTemplate() {
        return this.to('GetJobTemplate');
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/policy.html
     */
    toGetPolicy() {
        return this.to('GetPolicy');
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert output preset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toGetPreset() {
        return this.to('GetPreset');
    }
    /**
     * Grants permission to get an AWS Elemental MediaConvert job queue
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toGetQueue() {
        return this.to('GetQueue');
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert job templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html
     */
    toListJobTemplates() {
        return this.to('ListJobTemplates');
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     */
    toListJobs() {
        return this.to('ListJobs');
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert output presets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     */
    toListPresets() {
        return this.to('ListPresets');
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert job queues
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     */
    toListQueues() {
        return this.to('ListQueues');
    }
    /**
     * Grants permission to retrieve the tags for a MediaConvert queue, preset, or job template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list AWS Elemental MediaConvert job engine versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/versions.html
     */
    toListVersions() {
        return this.to('ListVersions');
    }
    /**
     * Grants permission to probe a file
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/probe.html
     */
    toProbe() {
        return this.to('Probe');
    }
    /**
     * Grants permission to put an AWS Elemental MediaConvert policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/policy.html
     */
    toPutPolicy() {
        return this.to('PutPolicy');
    }
    /**
     * Grants permission to search AWS Elemental MediaConvert jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/search.html
     */
    toSearchJobs() {
        return this.to('SearchJobs');
    }
    /**
     * Grants permission to add tags to a MediaConvert queue, preset, or job template
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a MediaConvert queue, preset, or job template
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags-arn.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert custom job template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates-name.html
     */
    toUpdateJobTemplate() {
        return this.to('UpdateJobTemplate');
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert custom output preset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets-name.html
     */
    toUpdatePreset() {
        return this.to('UpdatePreset');
    }
    /**
     * Grants permission to update an AWS Elemental MediaConvert job queue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues-name.html
     */
    toUpdateQueue() {
        return this.to('UpdateQueue');
    }
    /**
     * Adds a resource of type Job to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJob(jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconvert:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:jobs/${jobId}`);
    }
    /**
     * Adds a resource of type Queue to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/queues.html
     *
     * @param queueName - Identifier for the queueName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQueue(queueName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconvert:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:queues/${queueName}`);
    }
    /**
     * Adds a resource of type Preset to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/presets.html
     *
     * @param presetName - Identifier for the presetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPreset(presetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconvert:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:presets/${presetName}`);
    }
    /**
     * Adds a resource of type JobTemplate to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/jobtemplates.html
     *
     * @param jobTemplateName - Identifier for the jobTemplateName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJobTemplate(jobTemplateName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconvert:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:jobTemplates/${jobTemplateName}`);
    }
    /**
     * Adds a resource of type CertificateAssociation to the statement
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/certificates.html
     *
     * @param certificateArn - Identifier for the certificateArn.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCertificateAssociation(certificateArn, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediaconvert:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:certificates/${certificateArn}`);
    }
    /**
     * Filters access by tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to actions:
     * - .toCreateJob()
     * - .toCreateJobTemplate()
     * - .toCreatePreset()
     * - .toCreateQueue()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to actions:
     * - .toCreatePreset()
     * - .toCreateQueue()
     *
     * Applies to resource types:
     * - Job
     * - Queue
     * - Preset
     * - JobTemplate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys in the request
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/tags.html
     *
     * Applies to actions:
     * - .toCreateJob()
     * - .toCreateJobTemplate()
     * - .toCreatePreset()
     * - .toCreateQueue()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by an HTTP input policy present in the account
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/input-policies.html
     *
     * Applies to actions:
     * - .toCreateJob()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifHttpInputsAllowed(value) {
        return this.if(`HttpInputsAllowed`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by an HTTPS input policy present in the account
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/input-policies.html
     *
     * Applies to actions:
     * - .toCreateJob()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifHttpsInputsAllowed(value) {
        return this.if(`HttpsInputsAllowed`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by an S3 input policy present in the account
     *
     * https://docs.aws.amazon.com/mediaconvert/latest/apireference/input-policies.html
     *
     * Applies to actions:
     * - .toCreateJob()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifS3InputsAllowed(value) {
        return this.if(`S3InputsAllowed`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Statement provider for service [mediaconvert](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediaconvert.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mediaconvert';
        this.accessLevelList = {
            Write: [
                'AssociateCertificate',
                'CancelJob',
                'CreateJob',
                'CreateJobTemplate',
                'CreatePreset',
                'CreateQueue',
                'CreateResourceShare',
                'DeleteJobTemplate',
                'DeletePolicy',
                'DeletePreset',
                'DeleteQueue',
                'DisassociateCertificate',
                'PutPolicy',
                'UpdateJobTemplate',
                'UpdatePreset',
                'UpdateQueue'
            ],
            List: [
                'DescribeEndpoints',
                'ListJobTemplates',
                'ListJobs',
                'ListPresets',
                'ListQueues',
                'ListVersions',
                'SearchJobs'
            ],
            Read: [
                'GetJob',
                'GetJobTemplate',
                'GetPolicy',
                'GetPreset',
                'GetQueue',
                'ListTagsForResource',
                'Probe'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Mediaconvert = Mediaconvert;
//# sourceMappingURL=data:application/json;base64,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