"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Medialive = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [medialive](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmedialive.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Medialive extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept an input device transfer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toAcceptInputDeviceTransfer() {
        return this.to('AcceptInputDeviceTransfer');
    }
    /**
     * Grants permission to delete channels, inputs, input security groups, and multiplexes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    toBatchDelete() {
        return this.to('BatchDelete');
    }
    /**
     * Grants permission to start channels and multiplexes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toBatchStart() {
        return this.to('BatchStart');
    }
    /**
     * Grants permission to stop channels and multiplexes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toBatchStop() {
        return this.to('BatchStop');
    }
    /**
     * Grants permission to add and remove actions from a channel's schedule
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/submitting-batch-command.html
     */
    toBatchUpdateSchedule() {
        return this.to('BatchUpdateSchedule');
    }
    /**
     * Grants permission to cancel an input device transfer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toCancelInputDeviceTransfer() {
        return this.to('CancelInputDeviceTransfer');
    }
    /**
     * Grants permission to claim an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toClaimDevice() {
        return this.to('ClaimDevice');
    }
    /**
     * Grants permission to create a channel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/creating-channel-scratch.html
     */
    toCreateChannel() {
        return this.to('CreateChannel');
    }
    /**
     * Grants permission to create a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateChannelPlacementGroup() {
        return this.to('CreateChannelPlacementGroup');
    }
    /**
     * Grants permission to create a cloudwatch alarm template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     */
    toCreateCloudWatchAlarmTemplate() {
        return this.to('CreateCloudWatchAlarmTemplate');
    }
    /**
     * Grants permission to create a cloudwatch alarm template group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     */
    toCreateCloudWatchAlarmTemplateGroup() {
        return this.to('CreateCloudWatchAlarmTemplateGroup');
    }
    /**
     * Grants permission to create a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateCluster() {
        return this.to('CreateCluster');
    }
    /**
     * Grants permission to create a eventbridge rule template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     */
    toCreateEventBridgeRuleTemplate() {
        return this.to('CreateEventBridgeRuleTemplate');
    }
    /**
     * Grants permission to create a eventbridge rule template group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     */
    toCreateEventBridgeRuleTemplateGroup() {
        return this.to('CreateEventBridgeRuleTemplateGroup');
    }
    /**
     * Grants permission to create an input
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/creating-input.html
     */
    toCreateInput() {
        return this.to('CreateInput');
    }
    /**
     * Grants permission to create an input security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/working-with-input-security-groups.html
     */
    toCreateInputSecurityGroup() {
        return this.to('CreateInputSecurityGroup');
    }
    /**
     * Grants permission to create a multiplex
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/multiplex-create.html
     */
    toCreateMultiplex() {
        return this.to('CreateMultiplex');
    }
    /**
     * Grants permission to create a multiplex program
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/multiplex-create.html
     */
    toCreateMultiplexProgram() {
        return this.to('CreateMultiplexProgram');
    }
    /**
     * Grants permission to create a network
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateNetwork() {
        return this.to('CreateNetwork');
    }
    /**
     * Grants permission to create a node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateNode() {
        return this.to('CreateNode');
    }
    /**
     * Grants permission to create a node registration script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateNodeRegistrationScript() {
        return this.to('CreateNodeRegistrationScript');
    }
    /**
     * Grants permission to create a partner input
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/input-create-cdi-partners.html
     */
    toCreatePartnerInput() {
        return this.to('CreatePartnerInput');
    }
    /**
     * Grants permission to create a SDI source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toCreateSdiSource() {
        return this.to('CreateSdiSource');
    }
    /**
     * Grants permission to create a signal map
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-create.html
     */
    toCreateSignalMap() {
        return this.to('CreateSignalMap');
    }
    /**
     * Grants permission to create tags for channels, inputs, input security groups, multiplexes, reservations, nodes, networks, clusters, channel placement groups, signal maps, SDI sources, template groups, and templates
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/tagging.html
     */
    toCreateTags() {
        return this.to('CreateTags');
    }
    /**
     * Grants permission to delete a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    toDeleteChannel() {
        return this.to('DeleteChannel');
    }
    /**
     * Grants permission to delete a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toDeleteChannelPlacementGroup() {
        return this.to('DeleteChannelPlacementGroup');
    }
    /**
     * Grants permission to delete a cloudwatch alarm template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toDeleteCloudWatchAlarmTemplate() {
        return this.to('DeleteCloudWatchAlarmTemplate');
    }
    /**
     * Grants permission to delete a cloudwatch alarm template group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toDeleteCloudWatchAlarmTemplateGroup() {
        return this.to('DeleteCloudWatchAlarmTemplateGroup');
    }
    /**
     * Grants permission to delete a cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toDeleteCluster() {
        return this.to('DeleteCluster');
    }
    /**
     * Grants permission to delete a eventbridge rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toDeleteEventBridgeRuleTemplate() {
        return this.to('DeleteEventBridgeRuleTemplate');
    }
    /**
     * Grants permission to delete a eventbridge rule template group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toDeleteEventBridgeRuleTemplateGroup() {
        return this.to('DeleteEventBridgeRuleTemplateGroup');
    }
    /**
     * Grants permission to delete an input
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-input.html
     */
    toDeleteInput() {
        return this.to('DeleteInput');
    }
    /**
     * Grants permission to delete an input security group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-input-security-group.html
     */
    toDeleteInputSecurityGroup() {
        return this.to('DeleteInputSecurityGroup');
    }
    /**
     * Grants permission to delete a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-multiplex-program.html
     */
    toDeleteMultiplex() {
        return this.to('DeleteMultiplex');
    }
    /**
     * Grants permission to delete a multiplex program
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/delete-multiplex-program.html
     */
    toDeleteMultiplexProgram() {
        return this.to('DeleteMultiplexProgram');
    }
    /**
     * Grants permission to delete a network
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toDeleteNetwork() {
        return this.to('DeleteNetwork');
    }
    /**
     * Grants permission to delete a node
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toDeleteNode() {
        return this.to('DeleteNode');
    }
    /**
     * Grants permission to delete an expired reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/deleting-reservations.html
     */
    toDeleteReservation() {
        return this.to('DeleteReservation');
    }
    /**
     * Grants permission to delete all schedule actions for a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/schedule-using-console-delete.html
     */
    toDeleteSchedule() {
        return this.to('DeleteSchedule');
    }
    /**
     * Grants permission to delete a SDI source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toDeleteSdiSource() {
        return this.to('DeleteSdiSource');
    }
    /**
     * Grants permission to delete a signal map
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-delete.html
     */
    toDeleteSignalMap() {
        return this.to('DeleteSignalMap');
    }
    /**
     * Grants permission to delete tags from channels, inputs, input security groups, multiplexes, reservations, nodes, clusters, networks, channel placement groups, SDI source, signal maps, template groups, and templates
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/tagging.html
     */
    toDeleteTags() {
        return this.to('DeleteTags');
    }
    /**
     * Grants permission to view the account configuration of the customer
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toDescribeAccountConfiguration() {
        return this.to('DescribeAccountConfiguration');
    }
    /**
     * Grants permission to get details about a channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/viewing-channel-configuration.html
     */
    toDescribeChannel() {
        return this.to('DescribeChannel');
    }
    /**
     * Grants permission to describe a channel placement group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-create.html
     */
    toDescribeChannelPlacementGroup() {
        return this.to('DescribeChannelPlacementGroup');
    }
    /**
     * Grants permission to describe a cluster
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-create.html
     */
    toDescribeCluster() {
        return this.to('DescribeCluster');
    }
    /**
     * Grants permission to describe an input
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html
     */
    toDescribeInput() {
        return this.to('DescribeInput');
    }
    /**
     * Grants permission to describe an input device
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/device-edit.html
     */
    toDescribeInputDevice() {
        return this.to('DescribeInputDevice');
    }
    /**
     * Grants permission to describe an input device thumbnail
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/device-edit.html
     */
    toDescribeInputDeviceThumbnail() {
        return this.to('DescribeInputDeviceThumbnail');
    }
    /**
     * Grants permission to describe an input security group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html
     */
    toDescribeInputSecurityGroup() {
        return this.to('DescribeInputSecurityGroup');
    }
    /**
     * Grants permission to describe a multiplex
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-multiplex-program-channel.html
     */
    toDescribeMultiplex() {
        return this.to('DescribeMultiplex');
    }
    /**
     * Grants permission to describe a multiplex program
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitoring-multiplex-console.html
     */
    toDescribeMultiplexProgram() {
        return this.to('DescribeMultiplexProgram');
    }
    /**
     * Grants permission to describe a network
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-networks.html
     */
    toDescribeNetwork() {
        return this.to('DescribeNetwork');
    }
    /**
     * Grants permission to describe a node
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-nodes-create.html
     */
    toDescribeNode() {
        return this.to('DescribeNode');
    }
    /**
     * Grants permission to get details about a reservation offering
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html
     */
    toDescribeOffering() {
        return this.to('DescribeOffering');
    }
    /**
     * Grants permission to get details about a reservation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/view-reservations.html
     */
    toDescribeReservation() {
        return this.to('DescribeReservation');
    }
    /**
     * Grants permission to view a list of actions scheduled on a channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/schedule-using-console-view.html
     */
    toDescribeSchedule() {
        return this.to('DescribeSchedule');
    }
    /**
     * Grants permission to describe a SDI source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-create.html
     */
    toDescribeSdiSource() {
        return this.to('DescribeSdiSource');
    }
    /**
     * Grants permission to view the thumbnails for a channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toDescribeThumbnails() {
        return this.to('DescribeThumbnails');
    }
    /**
     * Grants permission to get a cloudwatch alarm template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toGetCloudWatchAlarmTemplate() {
        return this.to('GetCloudWatchAlarmTemplate');
    }
    /**
     * Grants permission to get a cloudwatch alarm template group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toGetCloudWatchAlarmTemplateGroup() {
        return this.to('GetCloudWatchAlarmTemplateGroup');
    }
    /**
     * Grants permission to get a eventbridge rule template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toGetEventBridgeRuleTemplate() {
        return this.to('GetEventBridgeRuleTemplate');
    }
    /**
     * Grants permission to get a eventbridge rule template group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toGetEventBridgeRuleTemplateGroup() {
        return this.to('GetEventBridgeRuleTemplateGroup');
    }
    /**
     * Grants permission to get a signal map
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-view.html
     */
    toGetSignalMap() {
        return this.to('GetSignalMap');
    }
    /**
     * Grants permission to list channel placement groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toListChannelPlacementGroups() {
        return this.to('ListChannelPlacementGroups');
    }
    /**
     * Grants permission to list channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/viewing-channel-configuration.html
     */
    toListChannels() {
        return this.to('ListChannels');
    }
    /**
     * Grants permission to list cloudwatch alarm template groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toListCloudWatchAlarmTemplateGroups() {
        return this.to('ListCloudWatchAlarmTemplateGroups');
    }
    /**
     * Grants permission to list cloudwatch alarm templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms.html
     */
    toListCloudWatchAlarmTemplates() {
        return this.to('ListCloudWatchAlarmTemplates');
    }
    /**
     * Grants permission to list clusters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toListClusters() {
        return this.to('ListClusters');
    }
    /**
     * Grants permission to list eventbridge rule template groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toListEventBridgeRuleTemplateGroups() {
        return this.to('ListEventBridgeRuleTemplateGroups');
    }
    /**
     * Grants permission to list eventbridge rule templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications.html
     */
    toListEventBridgeRuleTemplates() {
        return this.to('ListEventBridgeRuleTemplates');
    }
    /**
     * Grants permission to list input device transfers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toListInputDeviceTransfers() {
        return this.to('ListInputDeviceTransfers');
    }
    /**
     * Grants permission to list input devices
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/device-edit.html
     */
    toListInputDevices() {
        return this.to('ListInputDevices');
    }
    /**
     * Grants permission to list input security groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html
     */
    toListInputSecurityGroups() {
        return this.to('ListInputSecurityGroups');
    }
    /**
     * Grants permission to list inputs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html
     */
    toListInputs() {
        return this.to('ListInputs');
    }
    /**
     * Grants permission to list multiplex programs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitoring-multiplex-console.html
     */
    toListMultiplexPrograms() {
        return this.to('ListMultiplexPrograms');
    }
    /**
     * Grants permission to list multiplexes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-multiplex-program-channel.html
     */
    toListMultiplexes() {
        return this.to('ListMultiplexes');
    }
    /**
     * Grants permission to list networks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toListNetworks() {
        return this.to('ListNetworks');
    }
    /**
     * Grants permission to list nodes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toListNodes() {
        return this.to('ListNodes');
    }
    /**
     * Grants permission to list reservation offerings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html
     */
    toListOfferings() {
        return this.to('ListOfferings');
    }
    /**
     * Grants permission to list reservations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/view-reservations.html
     */
    toListReservations() {
        return this.to('ListReservations');
    }
    /**
     * Grants permission to list SDI sources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toListSdiSources() {
        return this.to('ListSdiSources');
    }
    /**
     * Grants permission to list signal maps
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-view.html
     */
    toListSignalMaps() {
        return this.to('ListSignalMaps');
    }
    /**
     * Grants permission to list tags for channels, inputs, input security groups, multiplexes, reservations, nodes, clusters, networks, channel placement groups, SDI sources, signal maps, template groups, and templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/tagging.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list available versions of MediaLive
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/medialive-versions.html
     */
    toListVersions() {
        return this.to('ListVersions');
    }
    /**
     * Grants permission to the node to poll the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/about-emla.html
     */
    toPollAnywhere() {
        return this.to('PollAnywhere');
    }
    /**
     * Grants permission to purchase a reservation offering
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/purchasing-reservations.html
     */
    toPurchaseOffering() {
        return this.to('PurchaseOffering');
    }
    /**
     * Grants permission to reboot an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toRebootInputDevice() {
        return this.to('RebootInputDevice');
    }
    /**
     * Grants permission to reject an input device transfer
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toRejectInputDeviceTransfer() {
        return this.to('RejectInputDeviceTransfer');
    }
    /**
     * Grants permission to restart pipelines on a running channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/maintenance-user-initiated.html
     */
    toRestartChannelPipelines() {
        return this.to('RestartChannelPipelines');
    }
    /**
     * Grants permission to start a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toStartChannel() {
        return this.to('StartChannel');
    }
    /**
     * Grants permission to start deletion of a signal map's monitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-delete.html
     */
    toStartDeleteMonitorDeployment() {
        return this.to('StartDeleteMonitorDeployment');
    }
    /**
     * Grants permission to start an input device attached to a MediaConnect flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toStartInputDevice() {
        return this.to('StartInputDevice');
    }
    /**
     * Grants permission to start a maintenance window for an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toStartInputDeviceMaintenanceWindow() {
        return this.to('StartInputDeviceMaintenanceWindow');
    }
    /**
     * Grants permission to start a signal map monitor deployment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-deploy.html
     */
    toStartMonitorDeployment() {
        return this.to('StartMonitorDeployment');
    }
    /**
     * Grants permission to start a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/start-multiplex.html
     */
    toStartMultiplex() {
        return this.to('StartMultiplex');
    }
    /**
     * Grants permission to start a signal map update
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-update.html
     */
    toStartUpdateSignalMap() {
        return this.to('StartUpdateSignalMap');
    }
    /**
     * Grants permission to stop a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toStopChannel() {
        return this.to('StopChannel');
    }
    /**
     * Grants permission to stop an input device attached to a MediaConnect flow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toStopInputDevice() {
        return this.to('StopInputDevice');
    }
    /**
     * Grants permission to stop a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/stop-multiplex.title.html
     */
    toStopMultiplex() {
        return this.to('StopMultiplex');
    }
    /**
     * Grants permission to the node to submit state changes to the cluster
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/about-emla.html
     */
    toSubmitAnywhereStateChange() {
        return this.to('SubmitAnywhereStateChange');
    }
    /**
     * Grants permission to transfer an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     */
    toTransferInputDevice() {
        return this.to('TransferInputDevice');
    }
    /**
     * Grants permission to update a customer's account configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/starting-stopping-deleting-a-channel.html
     */
    toUpdateAccountConfiguration() {
        return this.to('UpdateAccountConfiguration');
    }
    /**
     * Grants permission to update a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    toUpdateChannel() {
        return this.to('UpdateChannel');
    }
    /**
     * Grants permission to update the class of a channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/editing-deleting-channel.html
     */
    toUpdateChannelClass() {
        return this.to('UpdateChannelClass');
    }
    /**
     * Grants permission to update a node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toUpdateChannelPlacementGroup() {
        return this.to('UpdateChannelPlacementGroup');
    }
    /**
     * Grants permission to update a cloudwatch alarm template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     */
    toUpdateCloudWatchAlarmTemplate() {
        return this.to('UpdateCloudWatchAlarmTemplate');
    }
    /**
     * Grants permission to update a cloudwatch alarm template group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     */
    toUpdateCloudWatchAlarmTemplateGroup() {
        return this.to('UpdateCloudWatchAlarmTemplateGroup');
    }
    /**
     * Grants permission to update a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-create.html
     */
    toUpdateCluster() {
        return this.to('UpdateCluster');
    }
    /**
     * Grants permission to update a eventbridge rule template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     */
    toUpdateEventBridgeRuleTemplate() {
        return this.to('UpdateEventBridgeRuleTemplate');
    }
    /**
     * Grants permission to update a eventbridge rule template group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     */
    toUpdateEventBridgeRuleTemplateGroup() {
        return this.to('UpdateEventBridgeRuleTemplateGroup');
    }
    /**
     * Grants permission to update an input
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input.html
     */
    toUpdateInput() {
        return this.to('UpdateInput');
    }
    /**
     * Grants permission to update an input device
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/device-edit.html
     */
    toUpdateInputDevice() {
        return this.to('UpdateInputDevice');
    }
    /**
     * Grants permission to update an input security group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-input-security-group.html
     */
    toUpdateInputSecurityGroup() {
        return this.to('UpdateInputSecurityGroup');
    }
    /**
     * Grants permission to update a multiplex
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-multiplex-program-channel.html
     */
    toUpdateMultiplex() {
        return this.to('UpdateMultiplex');
    }
    /**
     * Grants permission to update a multiplex program
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/edit-multiplex-program-channel.html
     */
    toUpdateMultiplexProgram() {
        return this.to('UpdateMultiplexProgram');
    }
    /**
     * Grants permission to update the state of a node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-deploy-identify-network-requirements.html
     */
    toUpdateNetwork() {
        return this.to('UpdateNetwork');
    }
    /**
     * Grants permission to update a node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/emla-setup-cl-nodes-create.html
     */
    toUpdateNode() {
        return this.to('UpdateNode');
    }
    /**
     * Grants permission to update the state of a node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toUpdateNodeState() {
        return this.to('UpdateNodeState');
    }
    /**
     * Grants permission to update a reservation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/reservations.html
     */
    toUpdateReservation() {
        return this.to('UpdateReservation');
    }
    /**
     * Grants permission to update the state of a sdi source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     */
    toUpdateSdiSource() {
        return this.to('UpdateSdiSource');
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/container-channel.html
     *
     * @param channelId - Identifier for the channelId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(channelId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channel:${channelId}`);
    }
    /**
     * Adds a resource of type input to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/creating-input.html
     *
     * @param inputId - Identifier for the inputId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInput(inputId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:input:${inputId}`);
    }
    /**
     * Adds a resource of type input-device to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-devices.html
     *
     * @param deviceId - Identifier for the deviceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onInputDevice(deviceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:inputDevice:${deviceId}`);
    }
    /**
     * Adds a resource of type input-security-group to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/working-with-input-security-groups.html
     *
     * @param inputSecurityGroupId - Identifier for the inputSecurityGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInputSecurityGroup(inputSecurityGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:inputSecurityGroup:${inputSecurityGroupId}`);
    }
    /**
     * Adds a resource of type multiplex to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/eml-multiplex.html
     *
     * @param multiplexId - Identifier for the multiplexId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMultiplex(multiplexId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:multiplex:${multiplexId}`);
    }
    /**
     * Adds a resource of type reservation to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/reservations.html
     *
     * @param reservationId - Identifier for the reservationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReservation(reservationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:reservation:${reservationId}`);
    }
    /**
     * Adds a resource of type offering to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/input-output-reservations.html
     *
     * @param offeringId - Identifier for the offeringId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOffering(offeringId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:offering:${offeringId}`);
    }
    /**
     * Adds a resource of type signal-map to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-signal-maps-create.html
     *
     * @param signalMapId - Identifier for the signalMapId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSignalMap(signalMapId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:signal-map:${signalMapId}`);
    }
    /**
     * Adds a resource of type cloudwatch-alarm-template-group to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     *
     * @param cloudWatchAlarmTemplateGroupId - Identifier for the cloudWatchAlarmTemplateGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCloudwatchAlarmTemplateGroup(cloudWatchAlarmTemplateGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cloudwatch-alarm-template-group:${cloudWatchAlarmTemplateGroupId}`);
    }
    /**
     * Adds a resource of type cloudwatch-alarm-template to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-alarms-templates-create.html
     *
     * @param cloudWatchAlarmTemplateId - Identifier for the cloudWatchAlarmTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCloudwatchAlarmTemplate(cloudWatchAlarmTemplateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cloudwatch-alarm-template:${cloudWatchAlarmTemplateId}`);
    }
    /**
     * Adds a resource of type eventbridge-rule-template-group to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     *
     * @param eventBridgeRuleTemplateGroupId - Identifier for the eventBridgeRuleTemplateGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventbridgeRuleTemplateGroup(eventBridgeRuleTemplateGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:eventbridge-rule-template-group:${eventBridgeRuleTemplateGroupId}`);
    }
    /**
     * Adds a resource of type eventbridge-rule-template to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/monitor-with-workflow-monitor-configure-notifications-template-create.html
     *
     * @param eventBridgeRuleTemplateId - Identifier for the eventBridgeRuleTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventbridgeRuleTemplate(eventBridgeRuleTemplateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:eventbridge-rule-template:${eventBridgeRuleTemplateId}`);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     *
     * @param clusterId - Identifier for the clusterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster:${clusterId}`);
    }
    /**
     * Adds a resource of type node to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     *
     * @param clusterId - Identifier for the clusterId.
     * @param nodeId - Identifier for the nodeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNode(clusterId, nodeId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:node:${clusterId}/${nodeId}`);
    }
    /**
     * Adds a resource of type network to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     *
     * @param networkId - Identifier for the networkId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onNetwork(networkId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:network:${networkId}`);
    }
    /**
     * Adds a resource of type channel-placement-group to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     *
     * @param clusterId - Identifier for the clusterId.
     * @param channelPlacementGroupId - Identifier for the channelPlacementGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannelPlacementGroup(clusterId, channelPlacementGroupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:channelPlacementGroup:${clusterId}/${channelPlacementGroupId}`);
    }
    /**
     * Adds a resource of type sdi-source to the statement
     *
     * https://docs.aws.amazon.com/medialive/latest/ug/setup-emla.html
     *
     * @param sdiSourceId - Identifier for the sdiSourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSdiSource(sdiSourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:medialive:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sdiSource:${sdiSourceId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/medialive/latest/ugtagging.html
     *
     * Applies to actions:
     * - .toCreateChannel()
     * - .toCreateChannelPlacementGroup()
     * - .toCreateCloudWatchAlarmTemplate()
     * - .toCreateCloudWatchAlarmTemplateGroup()
     * - .toCreateCluster()
     * - .toCreateEventBridgeRuleTemplate()
     * - .toCreateEventBridgeRuleTemplateGroup()
     * - .toCreateInput()
     * - .toCreateInputSecurityGroup()
     * - .toCreateMultiplex()
     * - .toCreateNetwork()
     * - .toCreateNode()
     * - .toCreatePartnerInput()
     * - .toCreateSdiSource()
     * - .toCreateSignalMap()
     * - .toCreateTags()
     * - .toPurchaseOffering()
     * - .toUpdateChannelPlacementGroup()
     * - .toUpdateCluster()
     * - .toUpdateInputSecurityGroup()
     * - .toUpdateNetwork()
     * - .toUpdateNode()
     * - .toUpdateNodeState()
     * - .toUpdateSdiSource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/medialive/latest/ugtagging.html
     *
     * Applies to resource types:
     * - channel
     * - input
     * - input-security-group
     * - multiplex
     * - reservation
     * - signal-map
     * - cloudwatch-alarm-template-group
     * - cloudwatch-alarm-template
     * - eventbridge-rule-template-group
     * - eventbridge-rule-template
     * - cluster
     * - node
     * - network
     * - channel-placement-group
     * - sdi-source
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/medialive/latest/ugtagging.html
     *
     * Applies to actions:
     * - .toCreateChannel()
     * - .toCreateChannelPlacementGroup()
     * - .toCreateCloudWatchAlarmTemplate()
     * - .toCreateCloudWatchAlarmTemplateGroup()
     * - .toCreateCluster()
     * - .toCreateEventBridgeRuleTemplate()
     * - .toCreateEventBridgeRuleTemplateGroup()
     * - .toCreateInput()
     * - .toCreateInputSecurityGroup()
     * - .toCreateMultiplex()
     * - .toCreateNetwork()
     * - .toCreateNode()
     * - .toCreatePartnerInput()
     * - .toCreateSdiSource()
     * - .toCreateSignalMap()
     * - .toCreateTags()
     * - .toDeleteTags()
     * - .toPurchaseOffering()
     * - .toUpdateChannelPlacementGroup()
     * - .toUpdateCluster()
     * - .toUpdateInputSecurityGroup()
     * - .toUpdateNetwork()
     * - .toUpdateNode()
     * - .toUpdateNodeState()
     * - .toUpdateSdiSource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [medialive](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmedialive.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'medialive';
        this.accessLevelList = {
            Write: [
                'AcceptInputDeviceTransfer',
                'BatchDelete',
                'BatchStart',
                'BatchStop',
                'BatchUpdateSchedule',
                'CancelInputDeviceTransfer',
                'ClaimDevice',
                'CreateChannel',
                'CreateChannelPlacementGroup',
                'CreateCloudWatchAlarmTemplate',
                'CreateCloudWatchAlarmTemplateGroup',
                'CreateCluster',
                'CreateEventBridgeRuleTemplate',
                'CreateEventBridgeRuleTemplateGroup',
                'CreateInput',
                'CreateInputSecurityGroup',
                'CreateMultiplex',
                'CreateMultiplexProgram',
                'CreateNetwork',
                'CreateNode',
                'CreateNodeRegistrationScript',
                'CreatePartnerInput',
                'CreateSdiSource',
                'CreateSignalMap',
                'DeleteChannel',
                'DeleteChannelPlacementGroup',
                'DeleteCloudWatchAlarmTemplate',
                'DeleteCloudWatchAlarmTemplateGroup',
                'DeleteCluster',
                'DeleteEventBridgeRuleTemplate',
                'DeleteEventBridgeRuleTemplateGroup',
                'DeleteInput',
                'DeleteInputSecurityGroup',
                'DeleteMultiplex',
                'DeleteMultiplexProgram',
                'DeleteNetwork',
                'DeleteNode',
                'DeleteReservation',
                'DeleteSchedule',
                'DeleteSdiSource',
                'DeleteSignalMap',
                'PollAnywhere',
                'PurchaseOffering',
                'RebootInputDevice',
                'RejectInputDeviceTransfer',
                'RestartChannelPipelines',
                'StartChannel',
                'StartDeleteMonitorDeployment',
                'StartInputDevice',
                'StartInputDeviceMaintenanceWindow',
                'StartMonitorDeployment',
                'StartMultiplex',
                'StartUpdateSignalMap',
                'StopChannel',
                'StopInputDevice',
                'StopMultiplex',
                'SubmitAnywhereStateChange',
                'TransferInputDevice',
                'UpdateAccountConfiguration',
                'UpdateChannel',
                'UpdateChannelClass',
                'UpdateChannelPlacementGroup',
                'UpdateCloudWatchAlarmTemplate',
                'UpdateCloudWatchAlarmTemplateGroup',
                'UpdateCluster',
                'UpdateEventBridgeRuleTemplate',
                'UpdateEventBridgeRuleTemplateGroup',
                'UpdateInput',
                'UpdateInputDevice',
                'UpdateInputSecurityGroup',
                'UpdateMultiplex',
                'UpdateMultiplexProgram',
                'UpdateNetwork',
                'UpdateNode',
                'UpdateNodeState',
                'UpdateReservation',
                'UpdateSdiSource'
            ],
            Tagging: [
                'CreateTags',
                'DeleteTags'
            ],
            Read: [
                'DescribeAccountConfiguration',
                'DescribeChannel',
                'DescribeChannelPlacementGroup',
                'DescribeCluster',
                'DescribeInput',
                'DescribeInputDevice',
                'DescribeInputDeviceThumbnail',
                'DescribeInputSecurityGroup',
                'DescribeMultiplex',
                'DescribeMultiplexProgram',
                'DescribeNetwork',
                'DescribeNode',
                'DescribeOffering',
                'DescribeReservation',
                'DescribeSchedule',
                'DescribeSdiSource',
                'DescribeThumbnails',
                'GetCloudWatchAlarmTemplate',
                'GetCloudWatchAlarmTemplateGroup',
                'GetEventBridgeRuleTemplate',
                'GetEventBridgeRuleTemplateGroup',
                'GetSignalMap'
            ],
            List: [
                'ListChannelPlacementGroups',
                'ListChannels',
                'ListCloudWatchAlarmTemplateGroups',
                'ListCloudWatchAlarmTemplates',
                'ListClusters',
                'ListEventBridgeRuleTemplateGroups',
                'ListEventBridgeRuleTemplates',
                'ListInputDeviceTransfers',
                'ListInputDevices',
                'ListInputSecurityGroups',
                'ListInputs',
                'ListMultiplexPrograms',
                'ListMultiplexes',
                'ListNetworks',
                'ListNodes',
                'ListOfferings',
                'ListReservations',
                'ListSdiSources',
                'ListSignalMaps',
                'ListTagsForResource',
                'ListVersions'
            ]
        };
    }
}
exports.Medialive = Medialive;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZWxlbWVudGFsbWVkaWFsaXZlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZWxlbWVudGFsbWVkaWFsaXZlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUF5RDtBQUd6RDs7OztHQUlHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsd0JBQWU7SUFHNUM7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG9DQUFvQztRQUN6QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksb0NBQW9DO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwrQkFBK0I7UUFDcEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9DQUFvQztRQUN6QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0NBQW9DLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLCtCQUErQjtRQUNwQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNEJBQTRCO1FBQ2pDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUNBQWlDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDRCQUE0QixDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlDQUFpQztRQUN0QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1DQUFtQztRQUN4QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksOEJBQThCO1FBQ25DLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw4QkFBOEI7UUFDbkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDhCQUE4QixDQUFDLENBQUM7SUFDakQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0JBQXNCO1FBQzNCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQ0FBb0M7UUFDekMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQ0FBb0M7UUFDekMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUF1SUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksU0FBUyxDQUFDLFNBQWlCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsWUFBYSxTQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxPQUFPLENBQUMsT0FBZSxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ25GLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFVBQVcsT0FBUSxFQUFFLENBQUMsQ0FBQztJQUNuSyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksYUFBYSxDQUFDLFFBQWdCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDMUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZ0JBQWlCLFFBQVMsRUFBRSxDQUFDLENBQUM7SUFDMUssQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLG9CQUFvQixDQUFDLG9CQUE0QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzdHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLHVCQUF3QixvQkFBcUIsRUFBRSxDQUFDLENBQUM7SUFDN0wsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFdBQVcsQ0FBQyxXQUFtQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGNBQWUsV0FBWSxFQUFFLENBQUMsQ0FBQztJQUMzSyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksYUFBYSxDQUFDLGFBQXFCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDL0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZ0JBQWlCLGFBQWMsRUFBRSxDQUFDLENBQUM7SUFDL0ssQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLFVBQVUsQ0FBQyxVQUFrQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3pGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGFBQWMsVUFBVyxFQUFFLENBQUMsQ0FBQztJQUN6SyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksV0FBVyxDQUFDLFdBQW1CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZUFBZ0IsV0FBWSxFQUFFLENBQUMsQ0FBQztJQUM1SyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksOEJBQThCLENBQUMsOEJBQXNDLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakksT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsb0NBQXFDLDhCQUErQixFQUFFLENBQUMsQ0FBQztJQUNwTixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0kseUJBQXlCLENBQUMseUJBQWlDLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsOEJBQStCLHlCQUEwQixFQUFFLENBQUMsQ0FBQztJQUN6TSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksOEJBQThCLENBQUMsOEJBQXNDLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDakksT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsb0NBQXFDLDhCQUErQixFQUFFLENBQUMsQ0FBQztJQUNwTixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0kseUJBQXlCLENBQUMseUJBQWlDLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsOEJBQStCLHlCQUEwQixFQUFFLENBQUMsQ0FBQztJQUN6TSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksU0FBUyxDQUFDLFNBQWlCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdkYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsWUFBYSxTQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksTUFBTSxDQUFDLFNBQWlCLEVBQUUsTUFBYyxFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3BHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFNBQVUsU0FBVSxJQUFLLE1BQU8sRUFBRSxDQUFDLENBQUM7SUFDaEwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFNBQVMsQ0FBQyxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3ZGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFlBQWEsU0FBVSxFQUFFLENBQUMsQ0FBQztJQUN2SyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLHVCQUF1QixDQUFDLFNBQWlCLEVBQUUsdUJBQStCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDdEksT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsMEJBQTJCLFNBQVUsSUFBSyx1QkFBd0IsRUFBRSxDQUFDLENBQUM7SUFDbE4sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLFdBQVcsQ0FBQyxXQUFtQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzNGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGNBQWUsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGNBQWUsV0FBWSxFQUFFLENBQUMsQ0FBQztJQUMzSyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FrQ0c7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQXlCRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FrQ0c7SUFDSSxZQUFZLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUN4RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakUsQ0FBQztJQUVEOzs7T0FHRztJQUNILFlBQVksS0FBZ0M7UUFDMUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBdDdEUixrQkFBYSxHQUFHLFdBQVcsQ0FBQztRQW02Q3pCLG9CQUFlLEdBQW9CO1lBQzNDLEtBQUssRUFBRTtnQkFDTCwyQkFBMkI7Z0JBQzNCLGFBQWE7Z0JBQ2IsWUFBWTtnQkFDWixXQUFXO2dCQUNYLHFCQUFxQjtnQkFDckIsMkJBQTJCO2dCQUMzQixhQUFhO2dCQUNiLGVBQWU7Z0JBQ2YsNkJBQTZCO2dCQUM3QiwrQkFBK0I7Z0JBQy9CLG9DQUFvQztnQkFDcEMsZUFBZTtnQkFDZiwrQkFBK0I7Z0JBQy9CLG9DQUFvQztnQkFDcEMsYUFBYTtnQkFDYiwwQkFBMEI7Z0JBQzFCLGlCQUFpQjtnQkFDakIsd0JBQXdCO2dCQUN4QixlQUFlO2dCQUNmLFlBQVk7Z0JBQ1osOEJBQThCO2dCQUM5QixvQkFBb0I7Z0JBQ3BCLGlCQUFpQjtnQkFDakIsaUJBQWlCO2dCQUNqQixlQUFlO2dCQUNmLDZCQUE2QjtnQkFDN0IsK0JBQStCO2dCQUMvQixvQ0FBb0M7Z0JBQ3BDLGVBQWU7Z0JBQ2YsK0JBQStCO2dCQUMvQixvQ0FBb0M7Z0JBQ3BDLGFBQWE7Z0JBQ2IsMEJBQTBCO2dCQUMxQixpQkFBaUI7Z0JBQ2pCLHdCQUF3QjtnQkFDeEIsZUFBZTtnQkFDZixZQUFZO2dCQUNaLG1CQUFtQjtnQkFDbkIsZ0JBQWdCO2dCQUNoQixpQkFBaUI7Z0JBQ2pCLGlCQUFpQjtnQkFDakIsY0FBYztnQkFDZCxrQkFBa0I7Z0JBQ2xCLG1CQUFtQjtnQkFDbkIsMkJBQTJCO2dCQUMzQix5QkFBeUI7Z0JBQ3pCLGNBQWM7Z0JBQ2QsOEJBQThCO2dCQUM5QixrQkFBa0I7Z0JBQ2xCLG1DQUFtQztnQkFDbkMsd0JBQXdCO2dCQUN4QixnQkFBZ0I7Z0JBQ2hCLHNCQUFzQjtnQkFDdEIsYUFBYTtnQkFDYixpQkFBaUI7Z0JBQ2pCLGVBQWU7Z0JBQ2YsMkJBQTJCO2dCQUMzQixxQkFBcUI7Z0JBQ3JCLDRCQUE0QjtnQkFDNUIsZUFBZTtnQkFDZixvQkFBb0I7Z0JBQ3BCLDZCQUE2QjtnQkFDN0IsK0JBQStCO2dCQUMvQixvQ0FBb0M7Z0JBQ3BDLGVBQWU7Z0JBQ2YsK0JBQStCO2dCQUMvQixvQ0FBb0M7Z0JBQ3BDLGFBQWE7Z0JBQ2IsbUJBQW1CO2dCQUNuQiwwQkFBMEI7Z0JBQzFCLGlCQUFpQjtnQkFDakIsd0JBQXdCO2dCQUN4QixlQUFlO2dCQUNmLFlBQVk7Z0JBQ1osaUJBQWlCO2dCQUNqQixtQkFBbUI7Z0JBQ25CLGlCQUFpQjthQUNsQjtZQUNELE9BQU8sRUFBRTtnQkFDUCxZQUFZO2dCQUNaLFlBQVk7YUFDYjtZQUNELElBQUksRUFBRTtnQkFDSiw4QkFBOEI7Z0JBQzlCLGlCQUFpQjtnQkFDakIsK0JBQStCO2dCQUMvQixpQkFBaUI7Z0JBQ2pCLGVBQWU7Z0JBQ2YscUJBQXFCO2dCQUNyQiw4QkFBOEI7Z0JBQzlCLDRCQUE0QjtnQkFDNUIsbUJBQW1CO2dCQUNuQiwwQkFBMEI7Z0JBQzFCLGlCQUFpQjtnQkFDakIsY0FBYztnQkFDZCxrQkFBa0I7Z0JBQ2xCLHFCQUFxQjtnQkFDckIsa0JBQWtCO2dCQUNsQixtQkFBbUI7Z0JBQ25CLG9CQUFvQjtnQkFDcEIsNEJBQTRCO2dCQUM1QixpQ0FBaUM7Z0JBQ2pDLDRCQUE0QjtnQkFDNUIsaUNBQWlDO2dCQUNqQyxjQUFjO2FBQ2Y7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osNEJBQTRCO2dCQUM1QixjQUFjO2dCQUNkLG1DQUFtQztnQkFDbkMsOEJBQThCO2dCQUM5QixjQUFjO2dCQUNkLG1DQUFtQztnQkFDbkMsOEJBQThCO2dCQUM5QiwwQkFBMEI7Z0JBQzFCLGtCQUFrQjtnQkFDbEIseUJBQXlCO2dCQUN6QixZQUFZO2dCQUNaLHVCQUF1QjtnQkFDdkIsaUJBQWlCO2dCQUNqQixjQUFjO2dCQUNkLFdBQVc7Z0JBQ1gsZUFBZTtnQkFDZixrQkFBa0I7Z0JBQ2xCLGdCQUFnQjtnQkFDaEIsZ0JBQWdCO2dCQUNoQixxQkFBcUI7Z0JBQ3JCLGNBQWM7YUFDZjtTQUNGLENBQUM7SUFpWkYsQ0FBQztDQUNGO0FBejdERCw4QkF5N0RDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFttZWRpYWxpdmVdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c2VsZW1lbnRhbG1lZGlhbGl2ZS5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIE1lZGlhbGl2ZSBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ21lZGlhbGl2ZSc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFjY2VwdCBhbiBpbnB1dCBkZXZpY2UgdHJhbnNmZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lbWwtZGV2aWNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9BY2NlcHRJbnB1dERldmljZVRyYW5zZmVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBY2NlcHRJbnB1dERldmljZVRyYW5zZmVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGNoYW5uZWxzLCBpbnB1dHMsIGlucHV0IHNlY3VyaXR5IGdyb3VwcywgYW5kIG11bHRpcGxleGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvZWRpdGluZy1kZWxldGluZy1jaGFubmVsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0JhdGNoRGVsZXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdCYXRjaERlbGV0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IGNoYW5uZWxzIGFuZCBtdWx0aXBsZXhlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3N0YXJ0aW5nLXN0b3BwaW5nLWRlbGV0aW5nLWEtY2hhbm5lbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9CYXRjaFN0YXJ0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdCYXRjaFN0YXJ0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCBjaGFubmVscyBhbmQgbXVsdGlwbGV4ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zdGFydGluZy1zdG9wcGluZy1kZWxldGluZy1hLWNoYW5uZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQmF0Y2hTdG9wKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdCYXRjaFN0b3AnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBhZGQgYW5kIHJlbW92ZSBhY3Rpb25zIGZyb20gYSBjaGFubmVsJ3Mgc2NoZWR1bGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zdWJtaXR0aW5nLWJhdGNoLWNvbW1hbmQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQmF0Y2hVcGRhdGVTY2hlZHVsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hVcGRhdGVTY2hlZHVsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNhbmNlbCBhbiBpbnB1dCBkZXZpY2UgdHJhbnNmZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lbWwtZGV2aWNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DYW5jZWxJbnB1dERldmljZVRyYW5zZmVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDYW5jZWxJbnB1dERldmljZVRyYW5zZmVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY2xhaW0gYW4gaW5wdXQgZGV2aWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvZW1sLWRldmljZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ2xhaW1EZXZpY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NsYWltRGV2aWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgY2hhbm5lbFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2NyZWF0aW5nLWNoYW5uZWwtc2NyYXRjaC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVDaGFubmVsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVDaGFubmVsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3NldHVwLWVtbGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ2hhbm5lbFBsYWNlbWVudEdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVDaGFubmVsUGxhY2VtZW50R3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBjbG91ZHdhdGNoIGFsYXJtIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvbW9uaXRvci13aXRoLXdvcmtmbG93LW1vbml0b3ItY29uZmlndXJlLWFsYXJtcy10ZW1wbGF0ZXMtY3JlYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUNsb3VkV2F0Y2hBbGFybVRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGNsb3Vkd2F0Y2ggYWxhcm0gdGVtcGxhdGUgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtYWxhcm1zLXRlbXBsYXRlcy1jcmVhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQ2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zZXR1cC1lbWxhLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUNsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBldmVudGJyaWRnZSBydWxlIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvbW9uaXRvci13aXRoLXdvcmtmbG93LW1vbml0b3ItY29uZmlndXJlLW5vdGlmaWNhdGlvbnMtdGVtcGxhdGUtY3JlYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVFdmVudEJyaWRnZVJ1bGVUZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGV2ZW50YnJpZGdlIHJ1bGUgdGVtcGxhdGUgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtbm90aWZpY2F0aW9ucy10ZW1wbGF0ZS1jcmVhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGVHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlRXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGVHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBpbnB1dFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2NyZWF0aW5nLWlucHV0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUlucHV0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVJbnB1dCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBpbnB1dCBzZWN1cml0eSBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3dvcmtpbmctd2l0aC1pbnB1dC1zZWN1cml0eS1ncm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSW5wdXRTZWN1cml0eUdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVJbnB1dFNlY3VyaXR5R3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBtdWx0aXBsZXhcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tdWx0aXBsZXgtY3JlYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU11bHRpcGxleCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlTXVsdGlwbGV4Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbXVsdGlwbGV4IHByb2dyYW1cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tdWx0aXBsZXgtY3JlYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU11bHRpcGxleFByb2dyYW0oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZU11bHRpcGxleFByb2dyYW0nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBuZXR3b3JrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvc2V0dXAtZW1sYS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVOZXR3b3JrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVOZXR3b3JrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbm9kZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3NldHVwLWVtbGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTm9kZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlTm9kZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIG5vZGUgcmVnaXN0cmF0aW9uIHNjcmlwdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3NldHVwLWVtbGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTm9kZVJlZ2lzdHJhdGlvblNjcmlwdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlTm9kZVJlZ2lzdHJhdGlvblNjcmlwdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHBhcnRuZXIgaW5wdXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9pbnB1dC1jcmVhdGUtY2RpLXBhcnRuZXJzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVBhcnRuZXJJbnB1dCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlUGFydG5lcklucHV0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgU0RJIHNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3NldHVwLWVtbGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU2RpU291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVTZGlTb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBzaWduYWwgbWFwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvbW9uaXRvci13aXRoLXdvcmtmbG93LW1vbml0b3ItY29uZmlndXJlLXNpZ25hbC1tYXBzLWNyZWF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVTaWduYWxNYXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVNpZ25hbE1hcCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSB0YWdzIGZvciBjaGFubmVscywgaW5wdXRzLCBpbnB1dCBzZWN1cml0eSBncm91cHMsIG11bHRpcGxleGVzLCByZXNlcnZhdGlvbnMsIG5vZGVzLCBuZXR3b3JrcywgY2x1c3RlcnMsIGNoYW5uZWwgcGxhY2VtZW50IGdyb3Vwcywgc2lnbmFsIG1hcHMsIFNESSBzb3VyY2VzLCB0ZW1wbGF0ZSBncm91cHMsIGFuZCB0ZW1wbGF0ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3RhZ2dpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlVGFncygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlVGFncycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGNoYW5uZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lZGl0aW5nLWRlbGV0aW5nLWNoYW5uZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ2hhbm5lbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQ2hhbm5lbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zZXR1cC1lbWxhLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUNoYW5uZWxQbGFjZW1lbnRHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQ2hhbm5lbFBsYWNlbWVudEdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgY2xvdWR3YXRjaCBhbGFybSB0ZW1wbGF0ZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1hbGFybXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ2xvdWRXYXRjaEFsYXJtVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUNsb3VkV2F0Y2hBbGFybVRlbXBsYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgY2xvdWR3YXRjaCBhbGFybSB0ZW1wbGF0ZSBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1hbGFybXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQ2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zZXR1cC1lbWxhLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUNsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBldmVudGJyaWRnZSBydWxlIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvbW9uaXRvci13aXRoLXdvcmtmbG93LW1vbml0b3ItY29uZmlndXJlLW5vdGlmaWNhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgZXZlbnRicmlkZ2UgcnVsZSB0ZW1wbGF0ZSBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1ub3RpZmljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gaW5wdXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9kZWxldGUtaW5wdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlSW5wdXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUlucHV0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIGlucHV0IHNlY3VyaXR5IGdyb3VwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvZGVsZXRlLWlucHV0LXNlY3VyaXR5LWdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUlucHV0U2VjdXJpdHlHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlSW5wdXRTZWN1cml0eUdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgbXVsdGlwbGV4XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvZGVsZXRlLW11bHRpcGxleC1wcm9ncmFtLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU11bHRpcGxleCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlTXVsdGlwbGV4Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgbXVsdGlwbGV4IHByb2dyYW1cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9kZWxldGUtbXVsdGlwbGV4LXByb2dyYW0uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlTXVsdGlwbGV4UHJvZ3JhbSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlTXVsdGlwbGV4UHJvZ3JhbScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIG5ldHdvcmtcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zZXR1cC1lbWxhLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU5ldHdvcmsoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZU5ldHdvcmsnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBub2RlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvc2V0dXAtZW1sYS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVOb2RlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVOb2RlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFuIGV4cGlyZWQgcmVzZXJ2YXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9kZWxldGluZy1yZXNlcnZhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUmVzZXJ2YXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVJlc2VydmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGFsbCBzY2hlZHVsZSBhY3Rpb25zIGZvciBhIGNoYW5uZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zY2hlZHVsZS11c2luZy1jb25zb2xlLWRlbGV0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTY2hlZHVsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlU2NoZWR1bGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBTREkgc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvc2V0dXAtZW1sYS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTZGlTb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVNkaVNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHNpZ25hbCBtYXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtc2lnbmFsLW1hcHMtZGVsZXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVNpZ25hbE1hcCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlU2lnbmFsTWFwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIHRhZ3MgZnJvbSBjaGFubmVscywgaW5wdXRzLCBpbnB1dCBzZWN1cml0eSBncm91cHMsIG11bHRpcGxleGVzLCByZXNlcnZhdGlvbnMsIG5vZGVzLCBjbHVzdGVycywgbmV0d29ya3MsIGNoYW5uZWwgcGxhY2VtZW50IGdyb3VwcywgU0RJIHNvdXJjZSwgc2lnbmFsIG1hcHMsIHRlbXBsYXRlIGdyb3VwcywgYW5kIHRlbXBsYXRlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvdGFnZ2luZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVUYWdzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVUYWdzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyB0aGUgYWNjb3VudCBjb25maWd1cmF0aW9uIG9mIHRoZSBjdXN0b21lclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvc3RhcnRpbmctc3RvcHBpbmctZGVsZXRpbmctYS1jaGFubmVsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQWNjb3VudENvbmZpZ3VyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlQWNjb3VudENvbmZpZ3VyYXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgZGV0YWlscyBhYm91dCBhIGNoYW5uZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3ZpZXdpbmctY2hhbm5lbC1jb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlQ2hhbm5lbCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVDaGFubmVsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYSBjaGFubmVsIHBsYWNlbWVudCBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvZW1sYS1zZXR1cC1jbC1jcmVhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVDaGFubmVsUGxhY2VtZW50R3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlQ2hhbm5lbFBsYWNlbWVudEdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYSBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lbWxhLXNldHVwLWNsLWNyZWF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlQ2x1c3RlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGFuIGlucHV0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lZGl0LWlucHV0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlSW5wdXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlSW5wdXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBhbiBpbnB1dCBkZXZpY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2RldmljZS1lZGl0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlSW5wdXREZXZpY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlSW5wdXREZXZpY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBhbiBpbnB1dCBkZXZpY2UgdGh1bWJuYWlsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9kZXZpY2UtZWRpdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUlucHV0RGV2aWNlVGh1bWJuYWlsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZUlucHV0RGV2aWNlVGh1bWJuYWlsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYW4gaW5wdXQgc2VjdXJpdHkgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VkaXQtaW5wdXQtc2VjdXJpdHktZ3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVJbnB1dFNlY3VyaXR5R3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlSW5wdXRTZWN1cml0eUdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYSBtdWx0aXBsZXhcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VkaXQtbXVsdGlwbGV4LXByb2dyYW0tY2hhbm5lbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZU11bHRpcGxleCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVNdWx0aXBsZXgnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBhIG11bHRpcGxleCBwcm9ncmFtXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yaW5nLW11bHRpcGxleC1jb25zb2xlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTXVsdGlwbGV4UHJvZ3JhbSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVNdWx0aXBsZXhQcm9ncmFtJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYSBuZXR3b3JrXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lbWxhLXNldHVwLWNsLW5ldHdvcmtzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTmV0d29yaygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVOZXR3b3JrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVzY3JpYmUgYSBub2RlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lbWxhLXNldHVwLWNsLW5vZGVzLWNyZWF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZU5vZGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlTm9kZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBkZXRhaWxzIGFib3V0IGEgcmVzZXJ2YXRpb24gb2ZmZXJpbmdcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3B1cmNoYXNpbmctcmVzZXJ2YXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlT2ZmZXJpbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlT2ZmZXJpbmcnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgZGV0YWlscyBhYm91dCBhIHJlc2VydmF0aW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy92aWV3LXJlc2VydmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVJlc2VydmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVJlc2VydmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyBhIGxpc3Qgb2YgYWN0aW9ucyBzY2hlZHVsZWQgb24gYSBjaGFubmVsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zY2hlZHVsZS11c2luZy1jb25zb2xlLXZpZXcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTY2hlZHVsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVTY2hlZHVsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlc2NyaWJlIGEgU0RJIHNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvZW1sYS1zZXR1cC1jbC1jcmVhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVTZGlTb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlU2RpU291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdmlldyB0aGUgdGh1bWJuYWlscyBmb3IgYSBjaGFubmVsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zdGFydGluZy1zdG9wcGluZy1kZWxldGluZy1hLWNoYW5uZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVUaHVtYm5haWxzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZVRodW1ibmFpbHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBjbG91ZHdhdGNoIGFsYXJtIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtYWxhcm1zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldENsb3VkV2F0Y2hBbGFybVRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGNsb3Vkd2F0Y2ggYWxhcm0gdGVtcGxhdGUgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1hbGFybXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0Q2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0Q2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGV2ZW50YnJpZGdlIHJ1bGUgdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1ub3RpZmljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRFdmVudEJyaWRnZVJ1bGVUZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGV2ZW50YnJpZGdlIHJ1bGUgdGVtcGxhdGUgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1ub3RpZmljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldEV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlR3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldEV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBzaWduYWwgbWFwXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtc2lnbmFsLW1hcHMtdmlldy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTaWduYWxNYXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFNpZ25hbE1hcCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgY2hhbm5lbCBwbGFjZW1lbnQgZ3JvdXBzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zZXR1cC1lbWxhLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RDaGFubmVsUGxhY2VtZW50R3JvdXBzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0Q2hhbm5lbFBsYWNlbWVudEdyb3VwcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgY2hhbm5lbHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3ZpZXdpbmctY2hhbm5lbC1jb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RDaGFubmVscygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdENoYW5uZWxzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBjbG91ZHdhdGNoIGFsYXJtIHRlbXBsYXRlIGdyb3Vwc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvbW9uaXRvci13aXRoLXdvcmtmbG93LW1vbml0b3ItY29uZmlndXJlLWFsYXJtcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0Q2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZUdyb3VwcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgY2xvdWR3YXRjaCBhbGFybSB0ZW1wbGF0ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1hbGFybXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdENsb3VkV2F0Y2hBbGFybVRlbXBsYXRlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdENsb3VkV2F0Y2hBbGFybVRlbXBsYXRlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgY2x1c3RlcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3NldHVwLWVtbGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdENsdXN0ZXJzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0Q2x1c3RlcnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGV2ZW50YnJpZGdlIHJ1bGUgdGVtcGxhdGUgZ3JvdXBzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtbm90aWZpY2F0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGVHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RFdmVudEJyaWRnZVJ1bGVUZW1wbGF0ZUdyb3VwcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgZXZlbnRicmlkZ2UgcnVsZSB0ZW1wbGF0ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1ub3RpZmljYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RFdmVudEJyaWRnZVJ1bGVUZW1wbGF0ZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RFdmVudEJyaWRnZVJ1bGVUZW1wbGF0ZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGlucHV0IGRldmljZSB0cmFuc2ZlcnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VtbC1kZXZpY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbnB1dERldmljZVRyYW5zZmVycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdElucHV0RGV2aWNlVHJhbnNmZXJzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBpbnB1dCBkZXZpY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9kZXZpY2UtZWRpdC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SW5wdXREZXZpY2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0SW5wdXREZXZpY2VzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBpbnB1dCBzZWN1cml0eSBncm91cHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VkaXQtaW5wdXQtc2VjdXJpdHktZ3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdElucHV0U2VjdXJpdHlHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RJbnB1dFNlY3VyaXR5R3JvdXBzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBpbnB1dHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VkaXQtaW5wdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdElucHV0cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdElucHV0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgbXVsdGlwbGV4IHByb2dyYW1zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yaW5nLW11bHRpcGxleC1jb25zb2xlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RNdWx0aXBsZXhQcm9ncmFtcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdE11bHRpcGxleFByb2dyYW1zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBtdWx0aXBsZXhlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvZWRpdC1tdWx0aXBsZXgtcHJvZ3JhbS1jaGFubmVsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RNdWx0aXBsZXhlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdE11bHRpcGxleGVzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBuZXR3b3Jrc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvc2V0dXAtZW1sYS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TmV0d29ya3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3ROZXR3b3JrcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3Qgbm9kZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3NldHVwLWVtbGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdE5vZGVzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0Tm9kZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHJlc2VydmF0aW9uIG9mZmVyaW5nc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvcHVyY2hhc2luZy1yZXNlcnZhdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdE9mZmVyaW5ncygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdE9mZmVyaW5ncycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgcmVzZXJ2YXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy92aWV3LXJlc2VydmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UmVzZXJ2YXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0UmVzZXJ2YXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBTREkgc291cmNlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvc2V0dXAtZW1sYS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U2RpU291cmNlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFNkaVNvdXJjZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHNpZ25hbCBtYXBzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtc2lnbmFsLW1hcHMtdmlldy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U2lnbmFsTWFwcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFNpZ25hbE1hcHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRhZ3MgZm9yIGNoYW5uZWxzLCBpbnB1dHMsIGlucHV0IHNlY3VyaXR5IGdyb3VwcywgbXVsdGlwbGV4ZXMsIHJlc2VydmF0aW9ucywgbm9kZXMsIGNsdXN0ZXJzLCBuZXR3b3JrcywgY2hhbm5lbCBwbGFjZW1lbnQgZ3JvdXBzLCBTREkgc291cmNlcywgc2lnbmFsIG1hcHMsIHRlbXBsYXRlIGdyb3VwcywgYW5kIHRlbXBsYXRlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvdGFnZ2luZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhdmFpbGFibGUgdmVyc2lvbnMgb2YgTWVkaWFMaXZlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tZWRpYWxpdmUtdmVyc2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFZlcnNpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0VmVyc2lvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB0aGUgbm9kZSB0byBwb2xsIHRoZSBjbHVzdGVyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvYWJvdXQtZW1sYS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Qb2xsQW55d2hlcmUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1BvbGxBbnl3aGVyZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHB1cmNoYXNlIGEgcmVzZXJ2YXRpb24gb2ZmZXJpbmdcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9wdXJjaGFzaW5nLXJlc2VydmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXJjaGFzZU9mZmVyaW5nKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQdXJjaGFzZU9mZmVyaW5nJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVib290IGFuIGlucHV0IGRldmljZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VtbC1kZXZpY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1JlYm9vdElucHV0RGV2aWNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZWJvb3RJbnB1dERldmljZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJlamVjdCBhbiBpbnB1dCBkZXZpY2UgdHJhbnNmZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lbWwtZGV2aWNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZWplY3RJbnB1dERldmljZVRyYW5zZmVyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdSZWplY3RJbnB1dERldmljZVRyYW5zZmVyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmVzdGFydCBwaXBlbGluZXMgb24gYSBydW5uaW5nIGNoYW5uZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tYWludGVuYW5jZS11c2VyLWluaXRpYXRlZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZXN0YXJ0Q2hhbm5lbFBpcGVsaW5lcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUmVzdGFydENoYW5uZWxQaXBlbGluZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhIGNoYW5uZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zdGFydGluZy1zdG9wcGluZy1kZWxldGluZy1hLWNoYW5uZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRDaGFubmVsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydENoYW5uZWwnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBkZWxldGlvbiBvZiBhIHNpZ25hbCBtYXAncyBtb25pdG9yXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvbW9uaXRvci13aXRoLXdvcmtmbG93LW1vbml0b3ItY29uZmlndXJlLXNpZ25hbC1tYXBzLWRlbGV0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydERlbGV0ZU1vbml0b3JEZXBsb3ltZW50KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydERlbGV0ZU1vbml0b3JEZXBsb3ltZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgYW4gaW5wdXQgZGV2aWNlIGF0dGFjaGVkIHRvIGEgTWVkaWFDb25uZWN0IGZsb3dcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lbWwtZGV2aWNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydElucHV0RGV2aWNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydElucHV0RGV2aWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgYSBtYWludGVuYW5jZSB3aW5kb3cgZm9yIGFuIGlucHV0IGRldmljZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VtbC1kZXZpY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0SW5wdXREZXZpY2VNYWludGVuYW5jZVdpbmRvdygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRJbnB1dERldmljZU1haW50ZW5hbmNlV2luZG93Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RhcnQgYSBzaWduYWwgbWFwIG1vbml0b3IgZGVwbG95bWVudFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1kZXBsb3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRNb25pdG9yRGVwbG95bWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRNb25pdG9yRGVwbG95bWVudCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IGEgbXVsdGlwbGV4XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvc3RhcnQtbXVsdGlwbGV4Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0TXVsdGlwbGV4KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydE11bHRpcGxleCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IGEgc2lnbmFsIG1hcCB1cGRhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtc2lnbmFsLW1hcHMtdXBkYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0VXBkYXRlU2lnbmFsTWFwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdGFydFVwZGF0ZVNpZ25hbE1hcCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0b3AgYSBjaGFubmVsXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvc3RhcnRpbmctc3RvcHBpbmctZGVsZXRpbmctYS1jaGFubmVsLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0b3BDaGFubmVsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdG9wQ2hhbm5lbCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0b3AgYW4gaW5wdXQgZGV2aWNlIGF0dGFjaGVkIHRvIGEgTWVkaWFDb25uZWN0IGZsb3dcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lbWwtZGV2aWNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wSW5wdXREZXZpY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N0b3BJbnB1dERldmljZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0b3AgYSBtdWx0aXBsZXhcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zdG9wLW11bHRpcGxleC50aXRsZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdG9wTXVsdGlwbGV4KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdG9wTXVsdGlwbGV4Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdGhlIG5vZGUgdG8gc3VibWl0IHN0YXRlIGNoYW5nZXMgdG8gdGhlIGNsdXN0ZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9hYm91dC1lbWxhLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N1Ym1pdEFueXdoZXJlU3RhdGVDaGFuZ2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1N1Ym1pdEFueXdoZXJlU3RhdGVDaGFuZ2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB0cmFuc2ZlciBhbiBpbnB1dCBkZXZpY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lbWwtZGV2aWNlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UcmFuc2ZlcklucHV0RGV2aWNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdUcmFuc2ZlcklucHV0RGV2aWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgY3VzdG9tZXIncyBhY2NvdW50IGNvbmZpZ3VyYXRpb25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zdGFydGluZy1zdG9wcGluZy1kZWxldGluZy1hLWNoYW5uZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQWNjb3VudENvbmZpZ3VyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUFjY291bnRDb25maWd1cmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgY2hhbm5lbFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VkaXRpbmctZGVsZXRpbmctY2hhbm5lbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDaGFubmVsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVDaGFubmVsJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBjbGFzcyBvZiBhIGNoYW5uZWxcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lZGl0aW5nLWRlbGV0aW5nLWNoYW5uZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQ2hhbm5lbENsYXNzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVDaGFubmVsQ2xhc3MnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBub2RlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvc2V0dXAtZW1sYS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDaGFubmVsUGxhY2VtZW50R3JvdXAoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUNoYW5uZWxQbGFjZW1lbnRHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIGNsb3Vkd2F0Y2ggYWxhcm0gdGVtcGxhdGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtYWxhcm1zLXRlbXBsYXRlcy1jcmVhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQ2xvdWRXYXRjaEFsYXJtVGVtcGxhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUNsb3VkV2F0Y2hBbGFybVRlbXBsYXRlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgY2xvdWR3YXRjaCBhbGFybSB0ZW1wbGF0ZSBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1hbGFybXMtdGVtcGxhdGVzLWNyZWF0ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZUdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZUdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgY2x1c3RlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VtbGEtc2V0dXAtY2wtY3JlYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUNsdXN0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUNsdXN0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBldmVudGJyaWRnZSBydWxlIHRlbXBsYXRlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvbW9uaXRvci13aXRoLXdvcmtmbG93LW1vbml0b3ItY29uZmlndXJlLW5vdGlmaWNhdGlvbnMtdGVtcGxhdGUtY3JlYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVFdmVudEJyaWRnZVJ1bGVUZW1wbGF0ZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIGV2ZW50YnJpZGdlIHJ1bGUgdGVtcGxhdGUgZ3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtbm90aWZpY2F0aW9ucy10ZW1wbGF0ZS1jcmVhdGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGVHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlRXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGVHcm91cCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBpbnB1dFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VkaXQtaW5wdXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlSW5wdXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZUlucHV0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGFuIGlucHV0IGRldmljZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2RldmljZS1lZGl0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUlucHV0RGV2aWNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVJbnB1dERldmljZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBpbnB1dCBzZWN1cml0eSBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VkaXQtaW5wdXQtc2VjdXJpdHktZ3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlSW5wdXRTZWN1cml0eUdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVJbnB1dFNlY3VyaXR5R3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBtdWx0aXBsZXhcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lZGl0LW11bHRpcGxleC1wcm9ncmFtLWNoYW5uZWwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlTXVsdGlwbGV4KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVNdWx0aXBsZXgnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBtdWx0aXBsZXggcHJvZ3JhbVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VkaXQtbXVsdGlwbGV4LXByb2dyYW0tY2hhbm5lbC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVNdWx0aXBsZXhQcm9ncmFtKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVNdWx0aXBsZXhQcm9ncmFtJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIHRoZSBzdGF0ZSBvZiBhIG5vZGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9lbWxhLWRlcGxveS1pZGVudGlmeS1uZXR3b3JrLXJlcXVpcmVtZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVOZXR3b3JrKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVOZXR3b3JrJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgbm9kZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VtbGEtc2V0dXAtY2wtbm9kZXMtY3JlYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZU5vZGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZU5vZGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIHN0YXRlIG9mIGEgbm9kZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3NldHVwLWVtbGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlTm9kZVN0YXRlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVOb2RlU3RhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSByZXNlcnZhdGlvblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3Jlc2VydmF0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVSZXNlcnZhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlUmVzZXJ2YXRpb24nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgdGhlIHN0YXRlIG9mIGEgc2RpIHNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3NldHVwLWVtbGEuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlU2RpU291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVTZGlTb3VyY2UnKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBXcml0ZTogW1xuICAgICAgJ0FjY2VwdElucHV0RGV2aWNlVHJhbnNmZXInLFxuICAgICAgJ0JhdGNoRGVsZXRlJyxcbiAgICAgICdCYXRjaFN0YXJ0JyxcbiAgICAgICdCYXRjaFN0b3AnLFxuICAgICAgJ0JhdGNoVXBkYXRlU2NoZWR1bGUnLFxuICAgICAgJ0NhbmNlbElucHV0RGV2aWNlVHJhbnNmZXInLFxuICAgICAgJ0NsYWltRGV2aWNlJyxcbiAgICAgICdDcmVhdGVDaGFubmVsJyxcbiAgICAgICdDcmVhdGVDaGFubmVsUGxhY2VtZW50R3JvdXAnLFxuICAgICAgJ0NyZWF0ZUNsb3VkV2F0Y2hBbGFybVRlbXBsYXRlJyxcbiAgICAgICdDcmVhdGVDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZUdyb3VwJyxcbiAgICAgICdDcmVhdGVDbHVzdGVyJyxcbiAgICAgICdDcmVhdGVFdmVudEJyaWRnZVJ1bGVUZW1wbGF0ZScsXG4gICAgICAnQ3JlYXRlRXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGVHcm91cCcsXG4gICAgICAnQ3JlYXRlSW5wdXQnLFxuICAgICAgJ0NyZWF0ZUlucHV0U2VjdXJpdHlHcm91cCcsXG4gICAgICAnQ3JlYXRlTXVsdGlwbGV4JyxcbiAgICAgICdDcmVhdGVNdWx0aXBsZXhQcm9ncmFtJyxcbiAgICAgICdDcmVhdGVOZXR3b3JrJyxcbiAgICAgICdDcmVhdGVOb2RlJyxcbiAgICAgICdDcmVhdGVOb2RlUmVnaXN0cmF0aW9uU2NyaXB0JyxcbiAgICAgICdDcmVhdGVQYXJ0bmVySW5wdXQnLFxuICAgICAgJ0NyZWF0ZVNkaVNvdXJjZScsXG4gICAgICAnQ3JlYXRlU2lnbmFsTWFwJyxcbiAgICAgICdEZWxldGVDaGFubmVsJyxcbiAgICAgICdEZWxldGVDaGFubmVsUGxhY2VtZW50R3JvdXAnLFxuICAgICAgJ0RlbGV0ZUNsb3VkV2F0Y2hBbGFybVRlbXBsYXRlJyxcbiAgICAgICdEZWxldGVDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZUdyb3VwJyxcbiAgICAgICdEZWxldGVDbHVzdGVyJyxcbiAgICAgICdEZWxldGVFdmVudEJyaWRnZVJ1bGVUZW1wbGF0ZScsXG4gICAgICAnRGVsZXRlRXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGVHcm91cCcsXG4gICAgICAnRGVsZXRlSW5wdXQnLFxuICAgICAgJ0RlbGV0ZUlucHV0U2VjdXJpdHlHcm91cCcsXG4gICAgICAnRGVsZXRlTXVsdGlwbGV4JyxcbiAgICAgICdEZWxldGVNdWx0aXBsZXhQcm9ncmFtJyxcbiAgICAgICdEZWxldGVOZXR3b3JrJyxcbiAgICAgICdEZWxldGVOb2RlJyxcbiAgICAgICdEZWxldGVSZXNlcnZhdGlvbicsXG4gICAgICAnRGVsZXRlU2NoZWR1bGUnLFxuICAgICAgJ0RlbGV0ZVNkaVNvdXJjZScsXG4gICAgICAnRGVsZXRlU2lnbmFsTWFwJyxcbiAgICAgICdQb2xsQW55d2hlcmUnLFxuICAgICAgJ1B1cmNoYXNlT2ZmZXJpbmcnLFxuICAgICAgJ1JlYm9vdElucHV0RGV2aWNlJyxcbiAgICAgICdSZWplY3RJbnB1dERldmljZVRyYW5zZmVyJyxcbiAgICAgICdSZXN0YXJ0Q2hhbm5lbFBpcGVsaW5lcycsXG4gICAgICAnU3RhcnRDaGFubmVsJyxcbiAgICAgICdTdGFydERlbGV0ZU1vbml0b3JEZXBsb3ltZW50JyxcbiAgICAgICdTdGFydElucHV0RGV2aWNlJyxcbiAgICAgICdTdGFydElucHV0RGV2aWNlTWFpbnRlbmFuY2VXaW5kb3cnLFxuICAgICAgJ1N0YXJ0TW9uaXRvckRlcGxveW1lbnQnLFxuICAgICAgJ1N0YXJ0TXVsdGlwbGV4JyxcbiAgICAgICdTdGFydFVwZGF0ZVNpZ25hbE1hcCcsXG4gICAgICAnU3RvcENoYW5uZWwnLFxuICAgICAgJ1N0b3BJbnB1dERldmljZScsXG4gICAgICAnU3RvcE11bHRpcGxleCcsXG4gICAgICAnU3VibWl0QW55d2hlcmVTdGF0ZUNoYW5nZScsXG4gICAgICAnVHJhbnNmZXJJbnB1dERldmljZScsXG4gICAgICAnVXBkYXRlQWNjb3VudENvbmZpZ3VyYXRpb24nLFxuICAgICAgJ1VwZGF0ZUNoYW5uZWwnLFxuICAgICAgJ1VwZGF0ZUNoYW5uZWxDbGFzcycsXG4gICAgICAnVXBkYXRlQ2hhbm5lbFBsYWNlbWVudEdyb3VwJyxcbiAgICAgICdVcGRhdGVDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZScsXG4gICAgICAnVXBkYXRlQ2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cCcsXG4gICAgICAnVXBkYXRlQ2x1c3RlcicsXG4gICAgICAnVXBkYXRlRXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGUnLFxuICAgICAgJ1VwZGF0ZUV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlR3JvdXAnLFxuICAgICAgJ1VwZGF0ZUlucHV0JyxcbiAgICAgICdVcGRhdGVJbnB1dERldmljZScsXG4gICAgICAnVXBkYXRlSW5wdXRTZWN1cml0eUdyb3VwJyxcbiAgICAgICdVcGRhdGVNdWx0aXBsZXgnLFxuICAgICAgJ1VwZGF0ZU11bHRpcGxleFByb2dyYW0nLFxuICAgICAgJ1VwZGF0ZU5ldHdvcmsnLFxuICAgICAgJ1VwZGF0ZU5vZGUnLFxuICAgICAgJ1VwZGF0ZU5vZGVTdGF0ZScsXG4gICAgICAnVXBkYXRlUmVzZXJ2YXRpb24nLFxuICAgICAgJ1VwZGF0ZVNkaVNvdXJjZSdcbiAgICBdLFxuICAgIFRhZ2dpbmc6IFtcbiAgICAgICdDcmVhdGVUYWdzJyxcbiAgICAgICdEZWxldGVUYWdzJ1xuICAgIF0sXG4gICAgUmVhZDogW1xuICAgICAgJ0Rlc2NyaWJlQWNjb3VudENvbmZpZ3VyYXRpb24nLFxuICAgICAgJ0Rlc2NyaWJlQ2hhbm5lbCcsXG4gICAgICAnRGVzY3JpYmVDaGFubmVsUGxhY2VtZW50R3JvdXAnLFxuICAgICAgJ0Rlc2NyaWJlQ2x1c3RlcicsXG4gICAgICAnRGVzY3JpYmVJbnB1dCcsXG4gICAgICAnRGVzY3JpYmVJbnB1dERldmljZScsXG4gICAgICAnRGVzY3JpYmVJbnB1dERldmljZVRodW1ibmFpbCcsXG4gICAgICAnRGVzY3JpYmVJbnB1dFNlY3VyaXR5R3JvdXAnLFxuICAgICAgJ0Rlc2NyaWJlTXVsdGlwbGV4JyxcbiAgICAgICdEZXNjcmliZU11bHRpcGxleFByb2dyYW0nLFxuICAgICAgJ0Rlc2NyaWJlTmV0d29yaycsXG4gICAgICAnRGVzY3JpYmVOb2RlJyxcbiAgICAgICdEZXNjcmliZU9mZmVyaW5nJyxcbiAgICAgICdEZXNjcmliZVJlc2VydmF0aW9uJyxcbiAgICAgICdEZXNjcmliZVNjaGVkdWxlJyxcbiAgICAgICdEZXNjcmliZVNkaVNvdXJjZScsXG4gICAgICAnRGVzY3JpYmVUaHVtYm5haWxzJyxcbiAgICAgICdHZXRDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZScsXG4gICAgICAnR2V0Q2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cCcsXG4gICAgICAnR2V0RXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGUnLFxuICAgICAgJ0dldEV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlR3JvdXAnLFxuICAgICAgJ0dldFNpZ25hbE1hcCdcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0Q2hhbm5lbFBsYWNlbWVudEdyb3VwcycsXG4gICAgICAnTGlzdENoYW5uZWxzJyxcbiAgICAgICdMaXN0Q2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cHMnLFxuICAgICAgJ0xpc3RDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZXMnLFxuICAgICAgJ0xpc3RDbHVzdGVycycsXG4gICAgICAnTGlzdEV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlR3JvdXBzJyxcbiAgICAgICdMaXN0RXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGVzJyxcbiAgICAgICdMaXN0SW5wdXREZXZpY2VUcmFuc2ZlcnMnLFxuICAgICAgJ0xpc3RJbnB1dERldmljZXMnLFxuICAgICAgJ0xpc3RJbnB1dFNlY3VyaXR5R3JvdXBzJyxcbiAgICAgICdMaXN0SW5wdXRzJyxcbiAgICAgICdMaXN0TXVsdGlwbGV4UHJvZ3JhbXMnLFxuICAgICAgJ0xpc3RNdWx0aXBsZXhlcycsXG4gICAgICAnTGlzdE5ldHdvcmtzJyxcbiAgICAgICdMaXN0Tm9kZXMnLFxuICAgICAgJ0xpc3RPZmZlcmluZ3MnLFxuICAgICAgJ0xpc3RSZXNlcnZhdGlvbnMnLFxuICAgICAgJ0xpc3RTZGlTb3VyY2VzJyxcbiAgICAgICdMaXN0U2lnbmFsTWFwcycsXG4gICAgICAnTGlzdFRhZ3NGb3JSZXNvdXJjZScsXG4gICAgICAnTGlzdFZlcnNpb25zJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY2hhbm5lbCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2NvbnRhaW5lci1jaGFubmVsLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGNoYW5uZWxJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBjaGFubmVsSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkNoYW5uZWwoY2hhbm5lbElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVkaWFsaXZlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpjaGFubmVsOiR7IGNoYW5uZWxJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgaW5wdXQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9jcmVhdGluZy1pbnB1dC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBpbnB1dElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlucHV0SWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbklucHV0KGlucHV0SWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTptZWRpYWxpdmU6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmlucHV0OiR7IGlucHV0SWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGlucHV0LWRldmljZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL2VtbC1kZXZpY2VzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGRldmljZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGRldmljZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbklucHV0RGV2aWNlKGRldmljZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVkaWFsaXZlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTppbnB1dERldmljZTokeyBkZXZpY2VJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgaW5wdXQtc2VjdXJpdHktZ3JvdXAgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy93b3JraW5nLXdpdGgtaW5wdXQtc2VjdXJpdHktZ3JvdXBzLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGlucHV0U2VjdXJpdHlHcm91cElkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlucHV0U2VjdXJpdHlHcm91cElkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25JbnB1dFNlY3VyaXR5R3JvdXAoaW5wdXRTZWN1cml0eUdyb3VwSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTptZWRpYWxpdmU6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmlucHV0U2VjdXJpdHlHcm91cDokeyBpbnB1dFNlY3VyaXR5R3JvdXBJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgbXVsdGlwbGV4IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvZW1sLW11bHRpcGxleC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBtdWx0aXBsZXhJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBtdWx0aXBsZXhJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uTXVsdGlwbGV4KG11bHRpcGxleElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVkaWFsaXZlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTptdWx0aXBsZXg6JHsgbXVsdGlwbGV4SWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHJlc2VydmF0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvcmVzZXJ2YXRpb25zLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc2VydmF0aW9uSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzZXJ2YXRpb25JZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUmVzZXJ2YXRpb24ocmVzZXJ2YXRpb25JZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9Om1lZGlhbGl2ZTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06cmVzZXJ2YXRpb246JHsgcmVzZXJ2YXRpb25JZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgb2ZmZXJpbmcgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9pbnB1dC1vdXRwdXQtcmVzZXJ2YXRpb25zLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIG9mZmVyaW5nSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgb2ZmZXJpbmdJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25PZmZlcmluZyhvZmZlcmluZ0lkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVkaWFsaXZlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpvZmZlcmluZzokeyBvZmZlcmluZ0lkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBzaWduYWwtbWFwIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvbW9uaXRvci13aXRoLXdvcmtmbG93LW1vbml0b3ItY29uZmlndXJlLXNpZ25hbC1tYXBzLWNyZWF0ZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzaWduYWxNYXBJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBzaWduYWxNYXBJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU2lnbmFsTWFwKHNpZ25hbE1hcElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVkaWFsaXZlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpzaWduYWwtbWFwOiR7IHNpZ25hbE1hcElkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBjbG91ZHdhdGNoLWFsYXJtLXRlbXBsYXRlLWdyb3VwIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvbW9uaXRvci13aXRoLXdvcmtmbG93LW1vbml0b3ItY29uZmlndXJlLWFsYXJtcy10ZW1wbGF0ZXMtY3JlYXRlLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGNsb3VkV2F0Y2hBbGFybVRlbXBsYXRlR3JvdXBJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBjbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZUdyb3VwSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkNsb3Vkd2F0Y2hBbGFybVRlbXBsYXRlR3JvdXAoY2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cElkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVkaWFsaXZlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpjbG91ZHdhdGNoLWFsYXJtLXRlbXBsYXRlLWdyb3VwOiR7IGNsb3VkV2F0Y2hBbGFybVRlbXBsYXRlR3JvdXBJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY2xvdWR3YXRjaC1hbGFybS10ZW1wbGF0ZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1hbGFybXMtdGVtcGxhdGVzLWNyZWF0ZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBjbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGNsb3VkV2F0Y2hBbGFybVRlbXBsYXRlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkNsb3Vkd2F0Y2hBbGFybVRlbXBsYXRlKGNsb3VkV2F0Y2hBbGFybVRlbXBsYXRlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTptZWRpYWxpdmU6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmNsb3Vkd2F0Y2gtYWxhcm0tdGVtcGxhdGU6JHsgY2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZXZlbnRicmlkZ2UtcnVsZS10ZW1wbGF0ZS1ncm91cCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL21vbml0b3Itd2l0aC13b3JrZmxvdy1tb25pdG9yLWNvbmZpZ3VyZS1ub3RpZmljYXRpb25zLXRlbXBsYXRlLWNyZWF0ZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBldmVudEJyaWRnZVJ1bGVUZW1wbGF0ZUdyb3VwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGVHcm91cElkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25FdmVudGJyaWRnZVJ1bGVUZW1wbGF0ZUdyb3VwKGV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlR3JvdXBJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9Om1lZGlhbGl2ZTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06ZXZlbnRicmlkZ2UtcnVsZS10ZW1wbGF0ZS1ncm91cDokeyBldmVudEJyaWRnZVJ1bGVUZW1wbGF0ZUdyb3VwSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGV2ZW50YnJpZGdlLXJ1bGUtdGVtcGxhdGUgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9tb25pdG9yLXdpdGgtd29ya2Zsb3ctbW9uaXRvci1jb25maWd1cmUtbm90aWZpY2F0aW9ucy10ZW1wbGF0ZS1jcmVhdGUuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gZXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGVJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBldmVudEJyaWRnZVJ1bGVUZW1wbGF0ZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25FdmVudGJyaWRnZVJ1bGVUZW1wbGF0ZShldmVudEJyaWRnZVJ1bGVUZW1wbGF0ZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVkaWFsaXZlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpldmVudGJyaWRnZS1ydWxlLXRlbXBsYXRlOiR7IGV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNsdXN0ZXIgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zZXR1cC1lbWxhLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGNsdXN0ZXJJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBjbHVzdGVySWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkNsdXN0ZXIoY2x1c3RlcklkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVkaWFsaXZlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpjbHVzdGVyOiR7IGNsdXN0ZXJJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgbm9kZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9tZWRpYWxpdmUvbGF0ZXN0L3VnL3NldHVwLWVtbGEuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gY2x1c3RlcklkIC0gSWRlbnRpZmllciBmb3IgdGhlIGNsdXN0ZXJJZC5cbiAgICogQHBhcmFtIG5vZGVJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBub2RlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbk5vZGUoY2x1c3RlcklkOiBzdHJpbmcsIG5vZGVJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9Om1lZGlhbGl2ZTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06bm9kZTokeyBjbHVzdGVySWQgfS8keyBub2RlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIG5ldHdvcmsgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zZXR1cC1lbWxhLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIG5ldHdvcmtJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBuZXR3b3JrSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbk5ldHdvcmsobmV0d29ya0lkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVkaWFsaXZlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpuZXR3b3JrOiR7IG5ldHdvcmtJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgY2hhbm5lbC1wbGFjZW1lbnQtZ3JvdXAgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Zy9zZXR1cC1lbWxhLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGNsdXN0ZXJJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBjbHVzdGVySWQuXG4gICAqIEBwYXJhbSBjaGFubmVsUGxhY2VtZW50R3JvdXBJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBjaGFubmVsUGxhY2VtZW50R3JvdXBJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uQ2hhbm5lbFBsYWNlbWVudEdyb3VwKGNsdXN0ZXJJZDogc3RyaW5nLCBjaGFubmVsUGxhY2VtZW50R3JvdXBJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9Om1lZGlhbGl2ZTokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06Y2hhbm5lbFBsYWNlbWVudEdyb3VwOiR7IGNsdXN0ZXJJZCB9LyR7IGNoYW5uZWxQbGFjZW1lbnRHcm91cElkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBzZGktc291cmNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWcvc2V0dXAtZW1sYS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzZGlTb3VyY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBzZGlTb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uU2RpU291cmNlKHNkaVNvdXJjZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bWVkaWFsaXZlOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpzZGlTb3VyY2U6JHsgc2RpU291cmNlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Z3RhZ2dpbmcuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlQ2hhbm5lbCgpXG4gICAqIC0gLnRvQ3JlYXRlQ2hhbm5lbFBsYWNlbWVudEdyb3VwKClcbiAgICogLSAudG9DcmVhdGVDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZSgpXG4gICAqIC0gLnRvQ3JlYXRlQ2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cCgpXG4gICAqIC0gLnRvQ3JlYXRlQ2x1c3RlcigpXG4gICAqIC0gLnRvQ3JlYXRlRXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGUoKVxuICAgKiAtIC50b0NyZWF0ZUV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlR3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZUlucHV0KClcbiAgICogLSAudG9DcmVhdGVJbnB1dFNlY3VyaXR5R3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZU11bHRpcGxleCgpXG4gICAqIC0gLnRvQ3JlYXRlTmV0d29yaygpXG4gICAqIC0gLnRvQ3JlYXRlTm9kZSgpXG4gICAqIC0gLnRvQ3JlYXRlUGFydG5lcklucHV0KClcbiAgICogLSAudG9DcmVhdGVTZGlTb3VyY2UoKVxuICAgKiAtIC50b0NyZWF0ZVNpZ25hbE1hcCgpXG4gICAqIC0gLnRvQ3JlYXRlVGFncygpXG4gICAqIC0gLnRvUHVyY2hhc2VPZmZlcmluZygpXG4gICAqIC0gLnRvVXBkYXRlQ2hhbm5lbFBsYWNlbWVudEdyb3VwKClcbiAgICogLSAudG9VcGRhdGVDbHVzdGVyKClcbiAgICogLSAudG9VcGRhdGVJbnB1dFNlY3VyaXR5R3JvdXAoKVxuICAgKiAtIC50b1VwZGF0ZU5ldHdvcmsoKVxuICAgKiAtIC50b1VwZGF0ZU5vZGUoKVxuICAgKiAtIC50b1VwZGF0ZU5vZGVTdGF0ZSgpXG4gICAqIC0gLnRvVXBkYXRlU2RpU291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL21lZGlhbGl2ZS9sYXRlc3QvdWd0YWdnaW5nLmh0bWxcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBjaGFubmVsXG4gICAqIC0gaW5wdXRcbiAgICogLSBpbnB1dC1zZWN1cml0eS1ncm91cFxuICAgKiAtIG11bHRpcGxleFxuICAgKiAtIHJlc2VydmF0aW9uXG4gICAqIC0gc2lnbmFsLW1hcFxuICAgKiAtIGNsb3Vkd2F0Y2gtYWxhcm0tdGVtcGxhdGUtZ3JvdXBcbiAgICogLSBjbG91ZHdhdGNoLWFsYXJtLXRlbXBsYXRlXG4gICAqIC0gZXZlbnRicmlkZ2UtcnVsZS10ZW1wbGF0ZS1ncm91cFxuICAgKiAtIGV2ZW50YnJpZGdlLXJ1bGUtdGVtcGxhdGVcbiAgICogLSBjbHVzdGVyXG4gICAqIC0gbm9kZVxuICAgKiAtIG5ldHdvcmtcbiAgICogLSBjaGFubmVsLXBsYWNlbWVudC1ncm91cFxuICAgKiAtIHNkaS1zb3VyY2VcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZyBrZXlzIHRoYXQgYXJlIHBhc3NlZCBpbiB0aGUgcmVxdWVzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vbWVkaWFsaXZlL2xhdGVzdC91Z3RhZ2dpbmcuaHRtbFxuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlQ2hhbm5lbCgpXG4gICAqIC0gLnRvQ3JlYXRlQ2hhbm5lbFBsYWNlbWVudEdyb3VwKClcbiAgICogLSAudG9DcmVhdGVDbG91ZFdhdGNoQWxhcm1UZW1wbGF0ZSgpXG4gICAqIC0gLnRvQ3JlYXRlQ2xvdWRXYXRjaEFsYXJtVGVtcGxhdGVHcm91cCgpXG4gICAqIC0gLnRvQ3JlYXRlQ2x1c3RlcigpXG4gICAqIC0gLnRvQ3JlYXRlRXZlbnRCcmlkZ2VSdWxlVGVtcGxhdGUoKVxuICAgKiAtIC50b0NyZWF0ZUV2ZW50QnJpZGdlUnVsZVRlbXBsYXRlR3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZUlucHV0KClcbiAgICogLSAudG9DcmVhdGVJbnB1dFNlY3VyaXR5R3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZU11bHRpcGxleCgpXG4gICAqIC0gLnRvQ3JlYXRlTmV0d29yaygpXG4gICAqIC0gLnRvQ3JlYXRlTm9kZSgpXG4gICAqIC0gLnRvQ3JlYXRlUGFydG5lcklucHV0KClcbiAgICogLSAudG9DcmVhdGVTZGlTb3VyY2UoKVxuICAgKiAtIC50b0NyZWF0ZVNpZ25hbE1hcCgpXG4gICAqIC0gLnRvQ3JlYXRlVGFncygpXG4gICAqIC0gLnRvRGVsZXRlVGFncygpXG4gICAqIC0gLnRvUHVyY2hhc2VPZmZlcmluZygpXG4gICAqIC0gLnRvVXBkYXRlQ2hhbm5lbFBsYWNlbWVudEdyb3VwKClcbiAgICogLSAudG9VcGRhdGVDbHVzdGVyKClcbiAgICogLSAudG9VcGRhdGVJbnB1dFNlY3VyaXR5R3JvdXAoKVxuICAgKiAtIC50b1VwZGF0ZU5ldHdvcmsoKVxuICAgKiAtIC50b1VwZGF0ZU5vZGUoKVxuICAgKiAtIC50b1VwZGF0ZU5vZGVTdGF0ZSgpXG4gICAqIC0gLnRvVXBkYXRlU2RpU291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbbWVkaWFsaXZlXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NlbGVtZW50YWxtZWRpYWxpdmUuaHRtbCkuXG4gICAqXG4gICAqL1xuICBjb25zdHJ1Y3Rvcihwcm9wcz86IGlhbS5Qb2xpY3lTdGF0ZW1lbnRQcm9wcykge1xuICAgIHN1cGVyKHByb3BzKTtcbiAgfVxufVxuIl19