"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MediapackageVod = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [mediapackage-vod](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediapackagevod.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class MediapackageVod extends shared_1.PolicyStatement {
    /**
     * Grants permission to configure egress access logs for a PackagingGroup
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id-configure_logs.html#packaging_groups-id-configure_logsput
     */
    toConfigureLogs() {
        return this.to('ConfigureLogs');
    }
    /**
     * Grants permission to create an asset in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets.html#assetspost
     */
    toCreateAsset() {
        return this.to('CreateAsset');
    }
    /**
     * Grants permission to create a packaging configuration in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations.html#packaging_configurationspost
     */
    toCreatePackagingConfiguration() {
        return this.to('CreatePackagingConfiguration');
    }
    /**
     * Grants permission to create a packaging group in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups.html#packaging_groupspost
     */
    toCreatePackagingGroup() {
        return this.to('CreatePackagingGroup');
    }
    /**
     * Grants permission to delete an asset in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets-id.html#assets-iddelete
     */
    toDeleteAsset() {
        return this.to('DeleteAsset');
    }
    /**
     * Grants permission to delete a packaging configuration in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations-id.html#packaging_configurations-iddelete
     */
    toDeletePackagingConfiguration() {
        return this.to('DeletePackagingConfiguration');
    }
    /**
     * Grants permission to delete a packaging group in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-iddelete
     */
    toDeletePackagingGroup() {
        return this.to('DeletePackagingGroup');
    }
    /**
     * Grants permission to view the details of an asset in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets-id.html#assets-idget
     */
    toDescribeAsset() {
        return this.to('DescribeAsset');
    }
    /**
     * Grants permission to view the details of a packaging configuration in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations-id.html#packaging_configurations-idget
     */
    toDescribePackagingConfiguration() {
        return this.to('DescribePackagingConfiguration');
    }
    /**
     * Grants permission to view the details of a packaging group in AWS Elemental MediaPackage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-idget
     */
    toDescribePackagingGroup() {
        return this.to('DescribePackagingGroup');
    }
    /**
     * Grants permission to view a list of assets in AWS Elemental MediaPackage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/assets.html#assetsget
     */
    toListAssets() {
        return this.to('ListAssets');
    }
    /**
     * Grants permission to view a list of packaging configurations in AWS Elemental MediaPackage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_configurations.html#packaging_configurationsget
     */
    toListPackagingConfigurations() {
        return this.to('ListPackagingConfigurations');
    }
    /**
     * Grants permission to view a list of packaging groups in AWS Elemental MediaPackage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups.html#packaging_groupsget
     */
    toListPackagingGroups() {
        return this.to('ListPackagingGroups');
    }
    /**
     * Grants permission to list the tags assigned to a PackagingGroup, PackagingConfiguration, or Asset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arnget
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to assign tags to a PackagingGroup, PackagingConfiguration, or Asset
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arnpost
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to delete tags from a PackagingGroup, PackagingConfiguration, or Asset
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/tags-resource-arn.html#tags-resource-arndelete
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a packaging group in AWS Elemental MediaPackage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/mediapackage-vod/latest/apireference/packaging_groups-id.html#packaging_groups-idput
     */
    toUpdatePackagingGroup() {
        return this.to('UpdatePackagingGroup');
    }
    /**
     * Adds a resource of type assets to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/asset.html
     *
     * @param assetIdentifier - Identifier for the assetIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssets(assetIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackage-vod:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:assets/${assetIdentifier}`);
    }
    /**
     * Adds a resource of type packaging-configurations to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/pkg-cfig.html
     *
     * @param packagingConfigurationIdentifier - Identifier for the packagingConfigurationIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPackagingConfigurations(packagingConfigurationIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackage-vod:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:packaging-configurations/${packagingConfigurationIdentifier}`);
    }
    /**
     * Adds a resource of type packaging-groups to the statement
     *
     * https://docs.aws.amazon.com/mediapackage/latest/ug/pkg-group.html
     *
     * @param packagingGroupIdentifier - Identifier for the packagingGroupIdentifier.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPackagingGroups(packagingGroupIdentifier, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mediapackage-vod:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:packaging-groups/${packagingGroupIdentifier}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAsset()
     * - .toCreatePackagingConfiguration()
     * - .toCreatePackagingGroup()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - assets
     * - packaging-configurations
     * - packaging-groups
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAsset()
     * - .toCreatePackagingConfiguration()
     * - .toCreatePackagingGroup()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [mediapackage-vod](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awselementalmediapackagevod.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mediapackage-vod';
        this.accessLevelList = {
            Write: [
                'ConfigureLogs',
                'CreateAsset',
                'CreatePackagingConfiguration',
                'CreatePackagingGroup',
                'DeleteAsset',
                'DeletePackagingConfiguration',
                'DeletePackagingGroup',
                'UpdatePackagingGroup'
            ],
            Read: [
                'DescribeAsset',
                'DescribePackagingConfiguration',
                'DescribePackagingGroup',
                'ListTagsForResource'
            ],
            List: [
                'ListAssets',
                'ListPackagingConfigurations',
                'ListPackagingGroups'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.MediapackageVod = MediapackageVod;
//# sourceMappingURL=data:application/json;base64,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