"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SocialMessaging = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [social-messaging](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsendusermessagingsocial.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class SocialMessaging extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a WhatsApp Business Account with your AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_AssociateWhatsAppBusinessAccount.html
     */
    toAssociateWhatsAppBusinessAccount() {
        return this.to('AssociateWhatsAppBusinessAccount');
    }
    /**
     * Grants permission to create a WhatsApp message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_CreateWhatsAppMessageTemplate.html
     */
    toCreateWhatsAppMessageTemplate() {
        return this.to('CreateWhatsAppMessageTemplate');
    }
    /**
     * Grants permission to create a WhatsApp message template from Meta's template library
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_CreateWhatsAppMessageTemplateFromLibrary.html
     */
    toCreateWhatsAppMessageTemplateFromLibrary() {
        return this.to('CreateWhatsAppMessageTemplateFromLibrary');
    }
    /**
     * Grants permission to create media for WhatsApp message templates
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_CreateWhatsAppMessageTemplateMedia.html
     */
    toCreateWhatsAppMessageTemplateMedia() {
        return this.to('CreateWhatsAppMessageTemplateMedia');
    }
    /**
     * Grants permission to delete a media object from WhatsApp
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_DeleteWhatsAppMessageMedia.html
     */
    toDeleteWhatsAppMessageMedia() {
        return this.to('DeleteWhatsAppMessageMedia');
    }
    /**
     * Grants permission to delete a WhatsApp message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_DeleteWhatsAppMessageTemplate.html
     */
    toDeleteWhatsAppMessageTemplate() {
        return this.to('DeleteWhatsAppMessageTemplate');
    }
    /**
     * Grants permission to disassociate a WhatsApp Business Account from your AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_DisassociateWhatsAppBusinessAccount.html
     */
    toDisassociateWhatsAppBusinessAccount() {
        return this.to('DisassociateWhatsAppBusinessAccount');
    }
    /**
     * Grants permission to view the details of a WhatsApp Business Account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_GetLinkedWhatsAppBusinessAccount.html
     */
    toGetLinkedWhatsAppBusinessAccount() {
        return this.to('GetLinkedWhatsAppBusinessAccount');
    }
    /**
     * Grants permission to view the details of a phone number
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_GetLinkedWhatsAppBusinessAccountPhoneNumber.html
     */
    toGetLinkedWhatsAppBusinessAccountPhoneNumber() {
        return this.to('GetLinkedWhatsAppBusinessAccountPhoneNumber');
    }
    /**
     * Grants permission to get a media object from WhatsApp
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_GetWhatsAppMessageMedia.html
     */
    toGetWhatsAppMessageMedia() {
        return this.to('GetWhatsAppMessageMedia');
    }
    /**
     * Grants permission to get details of a WhatsApp message template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_GetWhatsAppMessageTemplate.html
     */
    toGetWhatsAppMessageTemplate() {
        return this.to('GetWhatsAppMessageTemplate');
    }
    /**
     * Grants permission to view all of your WhatsApp Business Accounts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_ListLinkedWhatsAppBusinessAccounts.html
     */
    toListLinkedWhatsAppBusinessAccounts() {
        return this.to('ListLinkedWhatsAppBusinessAccounts');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list WhatsApp message templates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_ListWhatsAppMessageTemplates.html
     */
    toListWhatsAppMessageTemplates() {
        return this.to('ListWhatsAppMessageTemplates');
    }
    /**
     * Grants permission to list available templates from Meta's template library
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_ListWhatsAppTemplateLibrary.html
     */
    toListWhatsAppTemplateLibrary() {
        return this.to('ListWhatsAppTemplateLibrary');
    }
    /**
     * Grants permission to upload a media object to WhatsApp
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_PostWhatsAppMessageMedia.html
     */
    toPostWhatsAppMessageMedia() {
        return this.to('PostWhatsAppMessageMedia');
    }
    /**
     * Grants permission to update a WhatsApp Business Accounts event destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_PutWhatsAppBusinessAccountEventDestinations.html
     */
    toPutWhatsAppBusinessAccountEventDestinations() {
        return this.to('PutWhatsAppBusinessAccountEventDestinations');
    }
    /**
     * Grants permission to send a message through WhatsApp
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_SendWhatsAppMessage.html
     */
    toSendWhatsAppMessage() {
        return this.to('SendWhatsAppMessage');
    }
    /**
     * Grants permission to add a tag to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a WhatsApp message template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_UpdateWhatsAppMessageTemplate.html
     */
    toUpdateWhatsAppMessageTemplate() {
        return this.to('UpdateWhatsAppMessageTemplate');
    }
    /**
     * Adds a resource of type phone-number-id to the statement
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_WhatsAppPhoneNumberDetail.html
     *
     * @param originationPhoneNumberId - Identifier for the originationPhoneNumberId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPhoneNumberId(originationPhoneNumberId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:social-messaging:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:phone-number-id/${originationPhoneNumberId}`);
    }
    /**
     * Adds a resource of type waba to the statement
     *
     * https://docs.aws.amazon.com/social-messaging/latest/APIReference/API_LinkedWhatsAppBusinessAccountSummary.html
     *
     * @param wabaId - Identifier for the wabaId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWaba(wabaId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:social-messaging:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:waba/${wabaId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAssociateWhatsAppBusinessAccount()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateWhatsAppBusinessAccount()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - phone-number-id
     * - waba
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAssociateWhatsAppBusinessAccount()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [social-messaging](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsendusermessagingsocial.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'social-messaging';
        this.accessLevelList = {
            Write: [
                'AssociateWhatsAppBusinessAccount',
                'CreateWhatsAppMessageTemplate',
                'CreateWhatsAppMessageTemplateFromLibrary',
                'CreateWhatsAppMessageTemplateMedia',
                'DeleteWhatsAppMessageMedia',
                'DeleteWhatsAppMessageTemplate',
                'DisassociateWhatsAppBusinessAccount',
                'GetWhatsAppMessageMedia',
                'PostWhatsAppMessageMedia',
                'PutWhatsAppBusinessAccountEventDestinations',
                'SendWhatsAppMessage',
                'UpdateWhatsAppMessageTemplate'
            ],
            Read: [
                'GetLinkedWhatsAppBusinessAccount',
                'GetLinkedWhatsAppBusinessAccountPhoneNumber',
                'GetWhatsAppMessageTemplate',
                'ListTagsForResource'
            ],
            List: [
                'ListLinkedWhatsAppBusinessAccounts',
                'ListWhatsAppMessageTemplates',
                'ListWhatsAppTemplateLibrary'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.SocialMessaging = SocialMessaging;
//# sourceMappingURL=data:application/json;base64,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