"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Entityresolution = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [entityresolution](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsentityresolution.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Entityresolution extends shared_1.PolicyStatement {
    /**
     * Grants permission to give an AWS service or another account permission to use an AWS Entity Resolution resources
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_AddPolicyStatement.html
     */
    toAddPolicyStatement() {
        return this.to('AddPolicyStatement');
    }
    /**
     * Grants permission to batch delete unique Id
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_BatchDeleteUniqueId.html
     */
    toBatchDeleteUniqueId() {
        return this.to('BatchDeleteUniqueId');
    }
    /**
     * Grants permission to create a idmapping workflow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_CreateIdMappingWorkflow.html
     */
    toCreateIdMappingWorkflow() {
        return this.to('CreateIdMappingWorkflow');
    }
    /**
     * Grants permission to create a IdNamespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_CreateIdNamespace.html
     */
    toCreateIdNamespace() {
        return this.to('CreateIdNamespace');
    }
    /**
     * Grants permission to create a matching workflow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_CreateMatchingWorkflow.html
     */
    toCreateMatchingWorkflow() {
        return this.to('CreateMatchingWorkflow');
    }
    /**
     * Grants permission to create a schema mapping
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_CreateSchemaMapping.html
     */
    toCreateSchemaMapping() {
        return this.to('CreateSchemaMapping');
    }
    /**
     * Grants permission to delete a idmapping workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_DeleteIdMappingWorkflow.html
     */
    toDeleteIdMappingWorkflow() {
        return this.to('DeleteIdMappingWorkflow');
    }
    /**
     * Grants permission to delete a IdNamespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_DeleteIdNamespace.html
     */
    toDeleteIdNamespace() {
        return this.to('DeleteIdNamespace');
    }
    /**
     * Grants permission to delete a matching workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_DeleteMatchingWorkflow.html
     */
    toDeleteMatchingWorkflow() {
        return this.to('DeleteMatchingWorkflow');
    }
    /**
     * Grants permission to delete permission given to an AWS service or another account permission to use an AWS Entity Resolution resources
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_DeletePolicyStatement.html
     */
    toDeletePolicyStatement() {
        return this.to('DeletePolicyStatement');
    }
    /**
     * Grants permission to delete a schema mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_DeleteSchemaMapping.html
     */
    toDeleteSchemaMapping() {
        return this.to('DeleteSchemaMapping');
    }
    /**
     * Grants permission to generate match Id
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GenerateMatchId.html
     */
    toGenerateMatchId() {
        return this.to('GenerateMatchId');
    }
    /**
     * Grants permission to get a idmapping job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetIdMappingJob.html
     */
    toGetIdMappingJob() {
        return this.to('GetIdMappingJob');
    }
    /**
     * Grants permission to get a idmapping workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetIdMappingWorkflow.html
     */
    toGetIdMappingWorkflow() {
        return this.to('GetIdMappingWorkflow');
    }
    /**
     * Grants permission to get a IdNamespace
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetIdNamespace.html
     */
    toGetIdNamespace() {
        return this.to('GetIdNamespace');
    }
    /**
     * Grants permission to get match Id
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetMatchId.html
     */
    toGetMatchId() {
        return this.to('GetMatchId');
    }
    /**
     * Grants permission to get a matching job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetMatchingJob.html
     */
    toGetMatchingJob() {
        return this.to('GetMatchingJob');
    }
    /**
     * Grants permission to get a matching workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetMatchingWorkflow.html
     */
    toGetMatchingWorkflow() {
        return this.to('GetMatchingWorkflow');
    }
    /**
     * Grants permission to get a resource policy for an AWS Entity Resolution resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetPolicy.html
     */
    toGetPolicy() {
        return this.to('GetPolicy');
    }
    /**
     * Grants permission to get provider service
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetProviderService.html
     */
    toGetProviderService() {
        return this.to('GetProviderService');
    }
    /**
     * Grants permission to get a schema mapping
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_GetSchemaMapping.html
     */
    toGetSchemaMapping() {
        return this.to('GetSchemaMapping');
    }
    /**
     * Grants permission to list idmapping jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListIdMappingJobs.html
     */
    toListIdMappingJobs() {
        return this.to('ListIdMappingJobs');
    }
    /**
     * Grants permission to list idmapping workflows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListIdMappingWorkflows.html
     */
    toListIdMappingWorkflows() {
        return this.to('ListIdMappingWorkflows');
    }
    /**
     * Grants permission to list IdNamespaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListIdNamespaces.html
     */
    toListIdNamespaces() {
        return this.to('ListIdNamespaces');
    }
    /**
     * Grants permission to list matching jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListMatchingJobs.html
     */
    toListMatchingJobs() {
        return this.to('ListMatchingJobs');
    }
    /**
     * Grants permission to list matching workflows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListMatchingWorkflows.html
     */
    toListMatchingWorkflows() {
        return this.to('ListMatchingWorkflows');
    }
    /**
     * Grants permission to list provider service
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListProviderServices.html
     */
    toListProviderServices() {
        return this.to('ListProviderServices');
    }
    /**
     * Grants permission to list schema mappings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListSchemaMappings.html
     */
    toListSchemaMappings() {
        return this.to('ListSchemaMappings');
    }
    /**
     * Grants permission to List tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to put a resource policy for an AWS Entity Resolution resources
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_PutPolicy.html
     */
    toPutPolicy() {
        return this.to('PutPolicy');
    }
    /**
     * Grants permission to start a idmapping job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_StartIdMappingJob.html
     */
    toStartIdMappingJob() {
        return this.to('StartIdMappingJob');
    }
    /**
     * Grants permission to start a matching job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_StartMatchingJob.html
     */
    toStartMatchingJob() {
        return this.to('StartMatchingJob');
    }
    /**
     * Grants permission to adds tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a idmapping workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UpdateIdMappingWorkflow.html
     */
    toUpdateIdMappingWorkflow() {
        return this.to('UpdateIdMappingWorkflow');
    }
    /**
     * Grants permission to update a IdNamespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UpdateIdNamespace.html
     */
    toUpdateIdNamespace() {
        return this.to('UpdateIdNamespace');
    }
    /**
     * Grants permission to update a matching workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UpdateMatchingWorkflow.html
     */
    toUpdateMatchingWorkflow() {
        return this.to('UpdateMatchingWorkflow');
    }
    /**
     * Grants permission to update a schema mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UpdateSchemaMapping.html
     */
    toUpdateSchemaMapping() {
        return this.to('UpdateSchemaMapping');
    }
    /**
     * Grants permission to give an AWS service or another account permission to use IdNamespace within a workflow
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UseIdNamespace.html
     */
    toUseIdNamespace() {
        return this.to('UseIdNamespace');
    }
    /**
     * Grants permission to give an AWS service or another account permission to use workflow within a IdNamespace
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/entityresolution/latest/apireference/API_UseWorkflow.html
     */
    toUseWorkflow() {
        return this.to('UseWorkflow');
    }
    /**
     * Adds a resource of type MatchingWorkflow to the statement
     *
     * https://docs.aws.amazon.com/entityresolution/latest/userguide/
     *
     * @param workflowName - Identifier for the workflowName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMatchingWorkflow(workflowName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:entityresolution:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:matchingworkflow/${workflowName}`);
    }
    /**
     * Adds a resource of type SchemaMapping to the statement
     *
     * https://docs.aws.amazon.com/entityresolution/latest/userguide/
     *
     * @param schemaName - Identifier for the schemaName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSchemaMapping(schemaName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:entityresolution:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:schemamapping/${schemaName}`);
    }
    /**
     * Adds a resource of type IdMappingWorkflow to the statement
     *
     * https://docs.aws.amazon.com/entityresolution/latest/userguide/
     *
     * @param workflowName - Identifier for the workflowName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdMappingWorkflow(workflowName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:entityresolution:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:idmappingworkflow/${workflowName}`);
    }
    /**
     * Adds a resource of type ProviderService to the statement
     *
     * https://docs.aws.amazon.com/entityresolution/latest/userguide/
     *
     * @param providerName - Identifier for the providerName.
     * @param providerServiceName - Identifier for the providerServiceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProviderService(providerName, providerServiceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:entityresolution:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:providerservice/${providerName}/${providerServiceName}`);
    }
    /**
     * Adds a resource of type IdNamespace to the statement
     *
     * https://docs.aws.amazon.com/entityresolution/latest/userguide/
     *
     * @param idNamespaceName - Identifier for the idNamespaceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdNamespace(idNamespaceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:entityresolution:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:idnamespace/${idNamespaceName}`);
    }
    /**
     * Filters access by a key that is present in the request the user makes to the entity resolution service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateIdMappingWorkflow()
     * - .toCreateIdNamespace()
     * - .toCreateMatchingWorkflow()
     * - .toCreateSchemaMapping()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to resource types:
     * - MatchingWorkflow
     * - SchemaMapping
     * - IdMappingWorkflow
     * - ProviderService
     * - IdNamespace
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the list of all the tag key names present in the request the user makes to the entity resolution service
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_iam-permissions.html#iam-contextkeys
     *
     * Applies to actions:
     * - .toCreateIdMappingWorkflow()
     * - .toCreateIdNamespace()
     * - .toCreateMatchingWorkflow()
     * - .toCreateSchemaMapping()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [entityresolution](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsentityresolution.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'entityresolution';
        this.accessLevelList = {
            'Permissions management': [
                'AddPolicyStatement',
                'DeletePolicyStatement',
                'PutPolicy',
                'UseIdNamespace',
                'UseWorkflow'
            ],
            Write: [
                'BatchDeleteUniqueId',
                'CreateIdMappingWorkflow',
                'CreateIdNamespace',
                'CreateMatchingWorkflow',
                'CreateSchemaMapping',
                'DeleteIdMappingWorkflow',
                'DeleteIdNamespace',
                'DeleteMatchingWorkflow',
                'DeleteSchemaMapping',
                'GenerateMatchId',
                'StartIdMappingJob',
                'StartMatchingJob',
                'UpdateIdMappingWorkflow',
                'UpdateIdNamespace',
                'UpdateMatchingWorkflow',
                'UpdateSchemaMapping'
            ],
            Read: [
                'GetIdMappingJob',
                'GetIdMappingWorkflow',
                'GetIdNamespace',
                'GetMatchId',
                'GetMatchingJob',
                'GetMatchingWorkflow',
                'GetPolicy',
                'GetProviderService',
                'GetSchemaMapping',
                'ListTagsForResource'
            ],
            List: [
                'ListIdMappingJobs',
                'ListIdMappingWorkflows',
                'ListIdNamespaces',
                'ListMatchingJobs',
                'ListMatchingWorkflows',
                'ListProviderServices',
                'ListSchemaMappings'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Entityresolution = Entityresolution;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZW50aXR5cmVzb2x1dGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImVudGl0eXJlc29sdXRpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EseUNBQXlEO0FBR3pEOzs7O0dBSUc7QUFDSCxNQUFhLGdCQUFpQixTQUFRLHdCQUFlO0lBR25EOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7SUFDNUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksbUJBQW1CO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztJQUNyQyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztJQUMzQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUF1REQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksa0JBQWtCLENBQUMsWUFBb0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNuRyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixxQkFBc0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLHFCQUFzQixZQUFhLEVBQUUsQ0FBQyxDQUFDO0lBQzFMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxlQUFlLENBQUMsVUFBa0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM5RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixxQkFBc0IsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGtCQUFtQixVQUFXLEVBQUUsQ0FBQyxDQUFDO0lBQ3JMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxtQkFBbUIsQ0FBQyxZQUFvQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3BHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLHFCQUFzQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsc0JBQXVCLFlBQWEsRUFBRSxDQUFDLENBQUM7SUFDM0wsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxpQkFBaUIsQ0FBQyxZQUFvQixFQUFFLG1CQUEyQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9ILE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLHFCQUFzQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsb0JBQXFCLFlBQWEsSUFBSyxtQkFBb0IsRUFBRSxDQUFDLENBQUM7SUFDbE4sQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGFBQWEsQ0FBQyxlQUF1QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ2pHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLHFCQUFzQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsZ0JBQWlCLGVBQWdCLEVBQUUsQ0FBQyxDQUFDO0lBQ3hMLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxnQkFBZ0IsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUM1RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW9CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLFlBQVksQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3hFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsWUFBWSxLQUFnQztRQUMxQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUE3cEJSLGtCQUFhLEdBQUcsa0JBQWtCLENBQUM7UUFpZGhDLG9CQUFlLEdBQW9CO1lBQzNDLHdCQUF3QixFQUFFO2dCQUN4QixvQkFBb0I7Z0JBQ3BCLHVCQUF1QjtnQkFDdkIsV0FBVztnQkFDWCxnQkFBZ0I7Z0JBQ2hCLGFBQWE7YUFDZDtZQUNELEtBQUssRUFBRTtnQkFDTCxxQkFBcUI7Z0JBQ3JCLHlCQUF5QjtnQkFDekIsbUJBQW1CO2dCQUNuQix3QkFBd0I7Z0JBQ3hCLHFCQUFxQjtnQkFDckIseUJBQXlCO2dCQUN6QixtQkFBbUI7Z0JBQ25CLHdCQUF3QjtnQkFDeEIscUJBQXFCO2dCQUNyQixpQkFBaUI7Z0JBQ2pCLG1CQUFtQjtnQkFDbkIsa0JBQWtCO2dCQUNsQix5QkFBeUI7Z0JBQ3pCLG1CQUFtQjtnQkFDbkIsd0JBQXdCO2dCQUN4QixxQkFBcUI7YUFDdEI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osaUJBQWlCO2dCQUNqQixzQkFBc0I7Z0JBQ3RCLGdCQUFnQjtnQkFDaEIsWUFBWTtnQkFDWixnQkFBZ0I7Z0JBQ2hCLHFCQUFxQjtnQkFDckIsV0FBVztnQkFDWCxvQkFBb0I7Z0JBQ3BCLGtCQUFrQjtnQkFDbEIscUJBQXFCO2FBQ3RCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLG1CQUFtQjtnQkFDbkIsd0JBQXdCO2dCQUN4QixrQkFBa0I7Z0JBQ2xCLGtCQUFrQjtnQkFDbEIsdUJBQXVCO2dCQUN2QixzQkFBc0I7Z0JBQ3RCLG9CQUFvQjthQUNyQjtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBMEpGLENBQUM7Q0FDRjtBQWhxQkQsNENBZ3FCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbZW50aXR5cmVzb2x1dGlvbl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzZW50aXR5cmVzb2x1dGlvbi5odG1sKS5cbiAqXG4gKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIEVudGl0eXJlc29sdXRpb24gZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdlbnRpdHlyZXNvbHV0aW9uJztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2l2ZSBhbiBBV1Mgc2VydmljZSBvciBhbm90aGVyIGFjY291bnQgcGVybWlzc2lvbiB0byB1c2UgYW4gQVdTIEVudGl0eSBSZXNvbHV0aW9uIHJlc291cmNlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfQWRkUG9saWN5U3RhdGVtZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0FkZFBvbGljeVN0YXRlbWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQWRkUG9saWN5U3RhdGVtZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYmF0Y2ggZGVsZXRlIHVuaXF1ZSBJZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0JhdGNoRGVsZXRlVW5pcXVlSWQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQmF0Y2hEZWxldGVVbmlxdWVJZCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQmF0Y2hEZWxldGVVbmlxdWVJZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIGlkbWFwcGluZyB3b3JrZmxvd1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0NyZWF0ZUlkTWFwcGluZ1dvcmtmbG93Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUlkTWFwcGluZ1dvcmtmbG93KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVJZE1hcHBpbmdXb3JrZmxvdycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIElkTmFtZXNwYWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfQ3JlYXRlSWROYW1lc3BhY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSWROYW1lc3BhY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZUlkTmFtZXNwYWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgbWF0Y2hpbmcgd29ya2Zsb3dcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9DcmVhdGVNYXRjaGluZ1dvcmtmbG93Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZU1hdGNoaW5nV29ya2Zsb3coKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZU1hdGNoaW5nV29ya2Zsb3cnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBzY2hlbWEgbWFwcGluZ1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0NyZWF0ZVNjaGVtYU1hcHBpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU2NoZW1hTWFwcGluZygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU2NoZW1hTWFwcGluZycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIGlkbWFwcGluZyB3b3JrZmxvd1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0RlbGV0ZUlkTWFwcGluZ1dvcmtmbG93Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUlkTWFwcGluZ1dvcmtmbG93KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVJZE1hcHBpbmdXb3JrZmxvdycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIElkTmFtZXNwYWNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlSWROYW1lc3BhY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlSWROYW1lc3BhY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUlkTmFtZXNwYWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgbWF0Y2hpbmcgd29ya2Zsb3dcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVNYXRjaGluZ1dvcmtmbG93Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU1hdGNoaW5nV29ya2Zsb3coKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZU1hdGNoaW5nV29ya2Zsb3cnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgcGVybWlzc2lvbiBnaXZlbiB0byBhbiBBV1Mgc2VydmljZSBvciBhbm90aGVyIGFjY291bnQgcGVybWlzc2lvbiB0byB1c2UgYW4gQVdTIEVudGl0eSBSZXNvbHV0aW9uIHJlc291cmNlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfRGVsZXRlUG9saWN5U3RhdGVtZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVBvbGljeVN0YXRlbWVudCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlUG9saWN5U3RhdGVtZW50Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgc2NoZW1hIG1hcHBpbmdcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9EZWxldGVTY2hlbWFNYXBwaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVNjaGVtYU1hcHBpbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVNjaGVtYU1hcHBpbmcnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZW5lcmF0ZSBtYXRjaCBJZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dlbmVyYXRlTWF0Y2hJZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZW5lcmF0ZU1hdGNoSWQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dlbmVyYXRlTWF0Y2hJZCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGlkbWFwcGluZyBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dldElkTWFwcGluZ0pvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJZE1hcHBpbmdKb2IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldElkTWFwcGluZ0pvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBhIGlkbWFwcGluZyB3b3JrZmxvd1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfR2V0SWRNYXBwaW5nV29ya2Zsb3cuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SWRNYXBwaW5nV29ya2Zsb3coKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldElkTWFwcGluZ1dvcmtmbG93Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgSWROYW1lc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dldElkTmFtZXNwYWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldElkTmFtZXNwYWNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRJZE5hbWVzcGFjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBtYXRjaCBJZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfR2V0TWF0Y2hJZC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNYXRjaElkKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRNYXRjaElkJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgbWF0Y2hpbmcgam9iXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9HZXRNYXRjaGluZ0pvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNYXRjaGluZ0pvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0TWF0Y2hpbmdKb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgYSBtYXRjaGluZyB3b3JrZmxvd1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfR2V0TWF0Y2hpbmdXb3JrZmxvdy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNYXRjaGluZ1dvcmtmbG93KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRNYXRjaGluZ1dvcmtmbG93Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgcmVzb3VyY2UgcG9saWN5IGZvciBhbiBBV1MgRW50aXR5IFJlc29sdXRpb24gcmVzb3VyY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9HZXRQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRQb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZXQgcHJvdmlkZXIgc2VydmljZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfR2V0UHJvdmlkZXJTZXJ2aWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFByb3ZpZGVyU2VydmljZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0UHJvdmlkZXJTZXJ2aWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2V0IGEgc2NoZW1hIG1hcHBpbmdcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0dldFNjaGVtYU1hcHBpbmcuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U2NoZW1hTWFwcGluZygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0U2NoZW1hTWFwcGluZycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgaWRtYXBwaW5nIGpvYnNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0xpc3RJZE1hcHBpbmdKb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJZE1hcHBpbmdKb2JzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0SWRNYXBwaW5nSm9icycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgaWRtYXBwaW5nIHdvcmtmbG93c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfTGlzdElkTWFwcGluZ1dvcmtmbG93cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SWRNYXBwaW5nV29ya2Zsb3dzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0SWRNYXBwaW5nV29ya2Zsb3dzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBJZE5hbWVzcGFjZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0xpc3RJZE5hbWVzcGFjZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdElkTmFtZXNwYWNlcygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdElkTmFtZXNwYWNlcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgbWF0Y2hpbmcgam9ic1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfTGlzdE1hdGNoaW5nSm9icy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TWF0Y2hpbmdKb2JzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0TWF0Y2hpbmdKb2JzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBtYXRjaGluZyB3b3JrZmxvd3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0xpc3RNYXRjaGluZ1dvcmtmbG93cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TWF0Y2hpbmdXb3JrZmxvd3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RNYXRjaGluZ1dvcmtmbG93cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGxpc3QgcHJvdmlkZXIgc2VydmljZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfTGlzdFByb3ZpZGVyU2VydmljZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFByb3ZpZGVyU2VydmljZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RQcm92aWRlclNlcnZpY2VzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBzY2hlbWEgbWFwcGluZ3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX0xpc3RTY2hlbWFNYXBwaW5ncy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U2NoZW1hTWFwcGluZ3MoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RTY2hlbWFNYXBwaW5ncycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIExpc3QgdGFncyBmb3IgYSByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfTGlzdFRhZ3NGb3JSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcHV0IGEgcmVzb3VyY2UgcG9saWN5IGZvciBhbiBBV1MgRW50aXR5IFJlc29sdXRpb24gcmVzb3VyY2VzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9QdXRQb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0UG9saWN5KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQdXRQb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhIGlkbWFwcGluZyBqb2JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9TdGFydElkTWFwcGluZ0pvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9TdGFydElkTWFwcGluZ0pvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRJZE1hcHBpbmdKb2InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBzdGFydCBhIG1hdGNoaW5nIGpvYlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1N0YXJ0TWF0Y2hpbmdKb2IuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvU3RhcnRNYXRjaGluZ0pvYigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRNYXRjaGluZ0pvYicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFkZHMgdGFncyB0byBhIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9UYWdSZXNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UYWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1bnRhZyBhIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VudGFnUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBpZG1hcHBpbmcgd29ya2Zsb3dcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9VcGRhdGVJZE1hcHBpbmdXb3JrZmxvdy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVJZE1hcHBpbmdXb3JrZmxvdygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVXBkYXRlSWRNYXBwaW5nV29ya2Zsb3cnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBJZE5hbWVzcGFjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1VwZGF0ZUlkTmFtZXNwYWNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUlkTmFtZXNwYWNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVJZE5hbWVzcGFjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIG1hdGNoaW5nIHdvcmtmbG93XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfVXBkYXRlTWF0Y2hpbmdXb3JrZmxvdy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVNYXRjaGluZ1dvcmtmbG93KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVNYXRjaGluZ1dvcmtmbG93Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgc2NoZW1hIG1hcHBpbmdcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9VcGRhdGVTY2hlbWFNYXBwaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVNjaGVtYU1hcHBpbmcoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVNjaGVtYU1hcHBpbmcnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnaXZlIGFuIEFXUyBzZXJ2aWNlIG9yIGFub3RoZXIgYWNjb3VudCBwZXJtaXNzaW9uIHRvIHVzZSBJZE5hbWVzcGFjZSB3aXRoaW4gYSB3b3JrZmxvd1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L2FwaXJlZmVyZW5jZS9BUElfVXNlSWROYW1lc3BhY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXNlSWROYW1lc3BhY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VzZUlkTmFtZXNwYWNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZ2l2ZSBhbiBBV1Mgc2VydmljZSBvciBhbm90aGVyIGFjY291bnQgcGVybWlzc2lvbiB0byB1c2Ugd29ya2Zsb3cgd2l0aGluIGEgSWROYW1lc3BhY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC9hcGlyZWZlcmVuY2UvQVBJX1VzZVdvcmtmbG93Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VzZVdvcmtmbG93KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVc2VXb3JrZmxvdycpO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgICdQZXJtaXNzaW9ucyBtYW5hZ2VtZW50JzogW1xuICAgICAgJ0FkZFBvbGljeVN0YXRlbWVudCcsXG4gICAgICAnRGVsZXRlUG9saWN5U3RhdGVtZW50JyxcbiAgICAgICdQdXRQb2xpY3knLFxuICAgICAgJ1VzZUlkTmFtZXNwYWNlJyxcbiAgICAgICdVc2VXb3JrZmxvdydcbiAgICBdLFxuICAgIFdyaXRlOiBbXG4gICAgICAnQmF0Y2hEZWxldGVVbmlxdWVJZCcsXG4gICAgICAnQ3JlYXRlSWRNYXBwaW5nV29ya2Zsb3cnLFxuICAgICAgJ0NyZWF0ZUlkTmFtZXNwYWNlJyxcbiAgICAgICdDcmVhdGVNYXRjaGluZ1dvcmtmbG93JyxcbiAgICAgICdDcmVhdGVTY2hlbWFNYXBwaW5nJyxcbiAgICAgICdEZWxldGVJZE1hcHBpbmdXb3JrZmxvdycsXG4gICAgICAnRGVsZXRlSWROYW1lc3BhY2UnLFxuICAgICAgJ0RlbGV0ZU1hdGNoaW5nV29ya2Zsb3cnLFxuICAgICAgJ0RlbGV0ZVNjaGVtYU1hcHBpbmcnLFxuICAgICAgJ0dlbmVyYXRlTWF0Y2hJZCcsXG4gICAgICAnU3RhcnRJZE1hcHBpbmdKb2InLFxuICAgICAgJ1N0YXJ0TWF0Y2hpbmdKb2InLFxuICAgICAgJ1VwZGF0ZUlkTWFwcGluZ1dvcmtmbG93JyxcbiAgICAgICdVcGRhdGVJZE5hbWVzcGFjZScsXG4gICAgICAnVXBkYXRlTWF0Y2hpbmdXb3JrZmxvdycsXG4gICAgICAnVXBkYXRlU2NoZW1hTWFwcGluZydcbiAgICBdLFxuICAgIFJlYWQ6IFtcbiAgICAgICdHZXRJZE1hcHBpbmdKb2InLFxuICAgICAgJ0dldElkTWFwcGluZ1dvcmtmbG93JyxcbiAgICAgICdHZXRJZE5hbWVzcGFjZScsXG4gICAgICAnR2V0TWF0Y2hJZCcsXG4gICAgICAnR2V0TWF0Y2hpbmdKb2InLFxuICAgICAgJ0dldE1hdGNoaW5nV29ya2Zsb3cnLFxuICAgICAgJ0dldFBvbGljeScsXG4gICAgICAnR2V0UHJvdmlkZXJTZXJ2aWNlJyxcbiAgICAgICdHZXRTY2hlbWFNYXBwaW5nJyxcbiAgICAgICdMaXN0VGFnc0ZvclJlc291cmNlJ1xuICAgIF0sXG4gICAgTGlzdDogW1xuICAgICAgJ0xpc3RJZE1hcHBpbmdKb2JzJyxcbiAgICAgICdMaXN0SWRNYXBwaW5nV29ya2Zsb3dzJyxcbiAgICAgICdMaXN0SWROYW1lc3BhY2VzJyxcbiAgICAgICdMaXN0TWF0Y2hpbmdKb2JzJyxcbiAgICAgICdMaXN0TWF0Y2hpbmdXb3JrZmxvd3MnLFxuICAgICAgJ0xpc3RQcm92aWRlclNlcnZpY2VzJyxcbiAgICAgICdMaXN0U2NoZW1hTWFwcGluZ3MnXG4gICAgXSxcbiAgICBUYWdnaW5nOiBbXG4gICAgICAnVGFnUmVzb3VyY2UnLFxuICAgICAgJ1VudGFnUmVzb3VyY2UnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBNYXRjaGluZ1dvcmtmbG93IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L3VzZXJndWlkZS9cbiAgICpcbiAgICogQHBhcmFtIHdvcmtmbG93TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSB3b3JrZmxvd05hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbk1hdGNoaW5nV29ya2Zsb3cod29ya2Zsb3dOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06ZW50aXR5cmVzb2x1dGlvbjokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06bWF0Y2hpbmd3b3JrZmxvdy8keyB3b3JrZmxvd05hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIFNjaGVtYU1hcHBpbmcgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW50aXR5cmVzb2x1dGlvbi9sYXRlc3QvdXNlcmd1aWRlL1xuICAgKlxuICAgKiBAcGFyYW0gc2NoZW1hTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzY2hlbWFOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25TY2hlbWFNYXBwaW5nKHNjaGVtYU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTplbnRpdHlyZXNvbHV0aW9uOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpzY2hlbWFtYXBwaW5nLyR7IHNjaGVtYU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIElkTWFwcGluZ1dvcmtmbG93IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L3VzZXJndWlkZS9cbiAgICpcbiAgICogQHBhcmFtIHdvcmtmbG93TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSB3b3JrZmxvd05hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbklkTWFwcGluZ1dvcmtmbG93KHdvcmtmbG93TmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmVudGl0eXJlc29sdXRpb246JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmlkbWFwcGluZ3dvcmtmbG93LyR7IHdvcmtmbG93TmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgUHJvdmlkZXJTZXJ2aWNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VudGl0eXJlc29sdXRpb24vbGF0ZXN0L3VzZXJndWlkZS9cbiAgICpcbiAgICogQHBhcmFtIHByb3ZpZGVyTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBwcm92aWRlck5hbWUuXG4gICAqIEBwYXJhbSBwcm92aWRlclNlcnZpY2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHByb3ZpZGVyU2VydmljZU5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblByb3ZpZGVyU2VydmljZShwcm92aWRlck5hbWU6IHN0cmluZywgcHJvdmlkZXJTZXJ2aWNlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmVudGl0eXJlc29sdXRpb246JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnByb3ZpZGVyc2VydmljZS8keyBwcm92aWRlck5hbWUgfS8keyBwcm92aWRlclNlcnZpY2VOYW1lIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBJZE5hbWVzcGFjZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbnRpdHlyZXNvbHV0aW9uL2xhdGVzdC91c2VyZ3VpZGUvXG4gICAqXG4gICAqIEBwYXJhbSBpZE5hbWVzcGFjZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgaWROYW1lc3BhY2VOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25JZE5hbWVzcGFjZShpZE5hbWVzcGFjZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTplbnRpdHlyZXNvbHV0aW9uOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTppZG5hbWVzcGFjZS8keyBpZE5hbWVzcGFjZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IGEga2V5IHRoYXQgaXMgcHJlc2VudCBpbiB0aGUgcmVxdWVzdCB0aGUgdXNlciBtYWtlcyB0byB0aGUgZW50aXR5IHJlc29sdXRpb24gc2VydmljZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX2lhbS1wZXJtaXNzaW9ucy5odG1sI2lhbS1jb250ZXh0a2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvQ3JlYXRlSWRNYXBwaW5nV29ya2Zsb3coKVxuICAgKiAtIC50b0NyZWF0ZUlkTmFtZXNwYWNlKClcbiAgICogLSAudG9DcmVhdGVNYXRjaGluZ1dvcmtmbG93KClcbiAgICogLSAudG9DcmVhdGVTY2hlbWFNYXBwaW5nKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBhIHRhZyBrZXkgYW5kIHZhbHVlIHBhaXJcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9pYW0tcGVybWlzc2lvbnMuaHRtbCNpYW0tY29udGV4dGtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBNYXRjaGluZ1dvcmtmbG93XG4gICAqIC0gU2NoZW1hTWFwcGluZ1xuICAgKiAtIElkTWFwcGluZ1dvcmtmbG93XG4gICAqIC0gUHJvdmlkZXJTZXJ2aWNlXG4gICAqIC0gSWROYW1lc3BhY2VcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIGxpc3Qgb2YgYWxsIHRoZSB0YWcga2V5IG5hbWVzIHByZXNlbnQgaW4gdGhlIHJlcXVlc3QgdGhlIHVzZXIgbWFrZXMgdG8gdGhlIGVudGl0eSByZXNvbHV0aW9uIHNlcnZpY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9pYW0tcGVybWlzc2lvbnMuaHRtbCNpYW0tY29udGV4dGtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUlkTWFwcGluZ1dvcmtmbG93KClcbiAgICogLSAudG9DcmVhdGVJZE5hbWVzcGFjZSgpXG4gICAqIC0gLnRvQ3JlYXRlTWF0Y2hpbmdXb3JrZmxvdygpXG4gICAqIC0gLnRvQ3JlYXRlU2NoZW1hTWFwcGluZygpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKiAtIC50b1VudGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1RhZ0tleXModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFtlbnRpdHlyZXNvbHV0aW9uXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3NlbnRpdHlyZXNvbHV0aW9uLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==