"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Events = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [events](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoneventbridge.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Events extends shared_1.PolicyStatement {
    /**
     * Grants permission to activate partner event sources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ActivateEventSource.html
     */
    toActivateEventSource() {
        return this.to('ActivateEventSource');
    }
    /**
     * Grants permission to configure vended log delivery for EventBridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-event-bus-logs.html
     */
    toAllowVendedLogDeliveryForResource() {
        return this.to('AllowVendedLogDeliveryForResource');
    }
    /**
     * Grants permission to cancel a replay
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CancelReplay.html
     */
    toCancelReplay() {
        return this.to('CancelReplay');
    }
    /**
     * Grants permission to create a new api destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateApiDestination.html
     */
    toCreateApiDestination() {
        return this.to('CreateApiDestination');
    }
    /**
     * Grants permission to create a new archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateArchive.html
     */
    toCreateArchive() {
        return this.to('CreateArchive');
    }
    /**
     * Grants permission to create a new connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateConnection.html
     */
    toCreateConnection() {
        return this.to('CreateConnection');
    }
    /**
     * Grants permission to create an endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEventBusArn()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEndpoint.html
     */
    toCreateEndpoint() {
        return this.to('CreateEndpoint');
    }
    /**
     * Grants permission to create event buses
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEventBus.html
     */
    toCreateEventBus() {
        return this.to('CreateEventBus');
    }
    /**
     * Grants permission to create partner event sources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreatePartnerEventSource.html
     */
    toCreatePartnerEventSource() {
        return this.to('CreatePartnerEventSource');
    }
    /**
     * Grants permission to deactivate event sources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeactivateEventSource.html
     */
    toDeactivateEventSource() {
        return this.to('DeactivateEventSource');
    }
    /**
     * Grants permission to deauthorize a connection, deleting its stored authorization secrets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeauthorizeConnection.html
     */
    toDeauthorizeConnection() {
        return this.to('DeauthorizeConnection');
    }
    /**
     * Grants permission to delete an api destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteApiDestination.html
     */
    toDeleteApiDestination() {
        return this.to('DeleteApiDestination');
    }
    /**
     * Grants permission to delete an archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteArchive.html
     */
    toDeleteArchive() {
        return this.to('DeleteArchive');
    }
    /**
     * Grants permission to delete a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteConnection.html
     */
    toDeleteConnection() {
        return this.to('DeleteConnection');
    }
    /**
     * Grants permission to delete an endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        return this.to('DeleteEndpoint');
    }
    /**
     * Grants permission to delete event buses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteEventBus.html
     */
    toDeleteEventBus() {
        return this.to('DeleteEventBus');
    }
    /**
     * Grants permission to delete partner event sources
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource.html
     */
    toDeletePartnerEventSource() {
        return this.to('DeletePartnerEventSource');
    }
    /**
     * Grants permission to delete rules
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCreatorAccount()
     * - .ifManagedBy()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteRule.html
     */
    toDeleteRule() {
        return this.to('DeleteRule');
    }
    /**
     * Grants permission to retrieve details about an api destination
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeApiDestination.html
     */
    toDescribeApiDestination() {
        return this.to('DescribeApiDestination');
    }
    /**
     * Grants permission to retrieve details about an archive
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeArchive.html
     */
    toDescribeArchive() {
        return this.to('DescribeArchive');
    }
    /**
     * Grants permission to retrieve details about a conection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeConnection.html
     */
    toDescribeConnection() {
        return this.to('DescribeConnection');
    }
    /**
     * Grants permission to retrieve details about an endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEndpoint.html
     */
    toDescribeEndpoint() {
        return this.to('DescribeEndpoint');
    }
    /**
     * Grants permission to retrieve details about event buses
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventBus.html
     */
    toDescribeEventBus() {
        return this.to('DescribeEventBus');
    }
    /**
     * Grants permission to retrieve details about event sources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventSource.html
     */
    toDescribeEventSource() {
        return this.to('DescribeEventSource');
    }
    /**
     * Grants permission to retrieve details about partner event sources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribePartnerEventSource.html
     */
    toDescribePartnerEventSource() {
        return this.to('DescribePartnerEventSource');
    }
    /**
     * Grants permission to retrieve the details of a replay
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeReplay.html
     */
    toDescribeReplay() {
        return this.to('DescribeReplay');
    }
    /**
     * Grants permission to retrieve details about rules
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCreatorAccount()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeRule.html
     */
    toDescribeRule() {
        return this.to('DescribeRule');
    }
    /**
     * Grants permission to disable rules
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCreatorAccount()
     * - .ifManagedBy()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DisableRule.html
     */
    toDisableRule() {
        return this.to('DisableRule');
    }
    /**
     * Grants permission to enable rules
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCreatorAccount()
     * - .ifManagedBy()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_EnableRule.html
     */
    toEnableRule() {
        return this.to('EnableRule');
    }
    /**
     * Grants permission to invoke an api destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-identity-based-access-control-eventbridge.html
     */
    toInvokeApiDestination() {
        return this.to('InvokeApiDestination');
    }
    /**
     * Grants permission to retrieve a list of api destinations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListApiDestinations.html
     */
    toListApiDestinations() {
        return this.to('ListApiDestinations');
    }
    /**
     * Grants permission to retrieve a list of archives
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListArchives.html
     */
    toListArchives() {
        return this.to('ListArchives');
    }
    /**
     * Grants permission to retrieve a list of connections
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListConnections.html
     */
    toListConnections() {
        return this.to('ListConnections');
    }
    /**
     * Grants permission to retrieve a list of endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEndpoints.html
     */
    toListEndpoints() {
        return this.to('ListEndpoints');
    }
    /**
     * Grants permission to retrieve a list of the event buses in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventBuses.html
     */
    toListEventBuses() {
        return this.to('ListEventBuses');
    }
    /**
     * Grants permission to to retrieve a list of event sources shared with this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventSources.html
     */
    toListEventSources() {
        return this.to('ListEventSources');
    }
    /**
     * Grants permission to retrieve a list of AWS account IDs associated with an event source
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSourceAccounts.html
     */
    toListPartnerEventSourceAccounts() {
        return this.to('ListPartnerEventSourceAccounts');
    }
    /**
     * Grants permission to retrieve a list partner event sources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSources.html
     */
    toListPartnerEventSources() {
        return this.to('ListPartnerEventSources');
    }
    /**
     * Grants permission to retrieve a list of replays
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListReplays.html
     */
    toListReplays() {
        return this.to('ListReplays');
    }
    /**
     * Grants permission to retrieve a list of the names of the rules associated with a target
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRuleNamesByTarget.html
     */
    toListRuleNamesByTarget() {
        return this.to('ListRuleNamesByTarget');
    }
    /**
     * Grants permission to retrieve a list of the Amazon EventBridge rules in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRules.html
     */
    toListRules() {
        return this.to('ListRules');
    }
    /**
     * Grants permission to retrieve a list of tags associated with an Amazon EventBridge resource
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCreatorAccount()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve a list of targets defined for a rule
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCreatorAccount()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTargetsByRule.html
     */
    toListTargetsByRule() {
        return this.to('ListTargetsByRule');
    }
    /**
     * Grants permission to send custom events to Amazon EventBridge
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDetailType()
     * - .ifSource()
     * - .ifEventBusInvocation()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html
     */
    toPutEvents() {
        return this.to('PutEvents');
    }
    /**
     * Grants permission to sends custom events to Amazon EventBridge
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPartnerEvents.html
     */
    toPutPartnerEvents() {
        return this.to('PutPartnerEvents');
    }
    /**
     * Grants permission to use the PutPermission action to grants permission to another AWS account to put events to your default event bus
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPermission.html
     */
    toPutPermission() {
        return this.to('PutPermission');
    }
    /**
     * Grants permission to create or updates rules
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDetailUserIdentityPrincipalId()
     * - .ifDetailType()
     * - .ifSource()
     * - .ifDetailService()
     * - .ifDetailEventTypeCode()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCreatorAccount()
     * - .ifManagedBy()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutRule.html
     */
    toPutRule() {
        return this.to('PutRule');
    }
    /**
     * Grants permission to add targets to a rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetArn()
     * - .ifCreatorAccount()
     * - .ifManagedBy()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutTargets.html
     */
    toPutTargets() {
        return this.to('PutTargets');
    }
    /**
     * Grants permission to revoke the permission of another AWS account to put events to your default event bus
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemovePermission.html
     */
    toRemovePermission() {
        return this.to('RemovePermission');
    }
    /**
     * Grants permission to removes targets from a rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCreatorAccount()
     * - .ifManagedBy()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemoveTargets.html
     */
    toRemoveTargets() {
        return this.to('RemoveTargets');
    }
    /**
     * Grants permission to retrieve credentials from a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-api-destinations.html
     */
    toRetrieveConnectionCredentials() {
        return this.to('RetrieveConnectionCredentials');
    }
    /**
     * Grants permission to start a replay of an archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_StartReplay.html
     */
    toStartReplay() {
        return this.to('StartReplay');
    }
    /**
     * Grants permission to add a tag to an Amazon EventBridge resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifCreatorAccount()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to test whether an event pattern matches the provided event
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TestEventPattern.html
     */
    toTestEventPattern() {
        return this.to('TestEventPattern');
    }
    /**
     * Grants permission to remove a tag from an Amazon EventBridge resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifCreatorAccount()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an api destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UpdateApiDestination.html
     */
    toUpdateApiDestination() {
        return this.to('UpdateApiDestination');
    }
    /**
     * Grants permission to update an archive
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UpdateArchive.html
     */
    toUpdateArchive() {
        return this.to('UpdateArchive');
    }
    /**
     * Grants permission to update a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UpdateConnection.html
     */
    toUpdateConnection() {
        return this.to('UpdateConnection');
    }
    /**
     * Grants permission to update an endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEventBusArn()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UpdateEndpoint.html
     */
    toUpdateEndpoint() {
        return this.to('UpdateEndpoint');
    }
    /**
     * Grants permission to update event buses
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UpdateEventBus.html
     */
    toUpdateEventBus() {
        return this.to('UpdateEventBus');
    }
    /**
     * Adds a resource of type event-source to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param eventSourceName - Identifier for the eventSourceName.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEventSource(eventSourceName, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}::event-source/${eventSourceName}`);
    }
    /**
     * Adds a resource of type event-bus to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param eventBusName - Identifier for the eventBusName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventBus(eventBusName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:event-bus/${eventBusName}`);
    }
    /**
     * Adds a resource of type rule-on-default-event-bus to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRuleOnDefaultEventBus(ruleName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:rule/${ruleName}`);
    }
    /**
     * Adds a resource of type rule-on-custom-event-bus to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param eventBusName - Identifier for the eventBusName.
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRuleOnCustomEventBus(eventBusName, ruleName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:rule/${eventBusName}/${ruleName}`);
    }
    /**
     * Adds a resource of type archive to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param archiveName - Identifier for the archiveName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onArchive(archiveName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:archive/${archiveName}`);
    }
    /**
     * Adds a resource of type replay to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param replayName - Identifier for the replayName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onReplay(replayName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:replay/${replayName}`);
    }
    /**
     * Adds a resource of type connection to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param connectionName - Identifier for the connectionName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onConnection(connectionName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:connection/${connectionName}`);
    }
    /**
     * Adds a resource of type api-destination to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param apiDestinationName - Identifier for the apiDestinationName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApiDestination(apiDestinationName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:api-destination/${apiDestinationName}`);
    }
    /**
     * Adds a resource of type endpoint to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param endpointName - Identifier for the endpointName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onEndpoint(endpointName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:endpoint/${endpointName}`);
    }
    /**
     * Adds a resource of type create-snapshot to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onCreateSnapshot(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:target/create-snapshot`);
    }
    /**
     * Adds a resource of type reboot-instance to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onRebootInstance(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:target/reboot-instance`);
    }
    /**
     * Adds a resource of type stop-instance to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onStopInstance(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:target/stop-instance`);
    }
    /**
     * Adds a resource of type terminate-instance to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-manage-iam-access.html#eventbridge-arn-format
     *
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTerminateInstance(account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:events:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:target/terminate-instance`);
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/kms-alias.html
     *
     * @param alias - Identifier for the alias.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAlias(alias, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:alias/${alias}`);
    }
    /**
     * Adds a resource of type key to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onKey(keyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:key/${keyId}`);
    }
    /**
     * Filters access by the allowed set of values for each of the tags to event bus and rule actions
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateEventBus()
     * - .toPutRule()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag-value associated with the resource to event bus and rule actions
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - event-bus
     * - rule-on-default-event-bus
     * - rule-on-custom-event-bus
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags in the request to event bus and rule actions
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateEventBus()
     * - .toPutRule()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of the event buses that can be associated with an endpoint to CreateEndpoint and UpdateEndpoint actions
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limiting-access-to-event-buses
     *
     * Applies to actions:
     * - .toCreateEndpoint()
     * - .toUpdateEndpoint()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifEventBusArn(value, operator) {
        return this.if(`EventBusArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by AWS services. If a rule is created by an AWS service on your behalf, the value is the principal name of the service that created the rule
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html
     *
     * Applies to actions:
     * - .toDeleteRule()
     * - .toDisableRule()
     * - .toEnableRule()
     * - .toPutRule()
     * - .toPutTargets()
     * - .toRemoveTargets()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifManagedBy(value, operator) {
        return this.if(`ManagedBy`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ARN of a target that can be put to a rule to PutTargets actions. TargetARN doesn't include DeadLetterConfigArn
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limiting-access-to-targets
     *
     * Applies to actions:
     * - .toPutTargets()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTargetArn(value, operator) {
        return this.if(`TargetArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by the account the rule was created in to rule actions
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-creator-account
     *
     * Applies to actions:
     * - .toDeleteRule()
     * - .toDescribeRule()
     * - .toDisableRule()
     * - .toEnableRule()
     * - .toListTagsForResource()
     * - .toListTargetsByRule()
     * - .toPutRule()
     * - .toPutTargets()
     * - .toRemoveTargets()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCreatorAccount(value, operator) {
        return this.if(`creatorAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the literal string of the detail-type of the event to PutEvents and PutRule actions
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-pattern-detail-type
     *
     * Applies to actions:
     * - .toPutEvents()
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailType(value, operator) {
        return this.if(`detail-type`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the literal string for the detail.eventTypeCode field of the event to PutRule actions
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limit-rule-by-type-code
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailEventTypeCode(value, operator) {
        return this.if(`detail.eventTypeCode`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the literal string for the detail.service field of the event to PutRule actions
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limit-rule-by-service
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailService(value, operator) {
        return this.if(`detail.service`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the literal string for the detail.useridentity.principalid field of the event to PutRule actions
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#consume-specific-events
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailUserIdentityPrincipalId(value, operator) {
        return this.if(`detail.userIdentity.principalId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the event was generated via API or cross-account bus invocation to PutEvents actions
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-bus-invocation
     *
     * Applies to actions:
     * - .toPutEvents()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEventBusInvocation(value, operator) {
        return this.if(`eventBusInvocation`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the AWS service or AWS partner event source that generated the event to PutEvents and PutRule actions. Matches the literal string of the source field of the event
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-limit-access-control
     *
     * Applies to actions:
     * - .toPutEvents()
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSource(value, operator) {
        return this.if(`source`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [events](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoneventbridge.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'events';
        this.accessLevelList = {
            Write: [
                'ActivateEventSource',
                'AllowVendedLogDeliveryForResource',
                'CancelReplay',
                'CreateApiDestination',
                'CreateArchive',
                'CreateConnection',
                'CreateEndpoint',
                'CreateEventBus',
                'CreatePartnerEventSource',
                'DeactivateEventSource',
                'DeauthorizeConnection',
                'DeleteApiDestination',
                'DeleteArchive',
                'DeleteConnection',
                'DeleteEndpoint',
                'DeleteEventBus',
                'DeletePartnerEventSource',
                'DeleteRule',
                'DisableRule',
                'EnableRule',
                'InvokeApiDestination',
                'PutEvents',
                'PutPartnerEvents',
                'PutRule',
                'PutTargets',
                'RemoveTargets',
                'RetrieveConnectionCredentials',
                'StartReplay',
                'UpdateApiDestination',
                'UpdateArchive',
                'UpdateConnection',
                'UpdateEndpoint',
                'UpdateEventBus'
            ],
            Read: [
                'DescribeApiDestination',
                'DescribeArchive',
                'DescribeConnection',
                'DescribeEndpoint',
                'DescribeEventBus',
                'DescribeEventSource',
                'DescribePartnerEventSource',
                'DescribeReplay',
                'DescribeRule',
                'TestEventPattern'
            ],
            List: [
                'ListApiDestinations',
                'ListArchives',
                'ListConnections',
                'ListEndpoints',
                'ListEventBuses',
                'ListEventSources',
                'ListPartnerEventSourceAccounts',
                'ListPartnerEventSources',
                'ListReplays',
                'ListRuleNamesByTarget',
                'ListRules',
                'ListTagsForResource',
                'ListTargetsByRule'
            ],
            'Permissions management': [
                'PutPermission',
                'RemovePermission'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Events = Events;
//# sourceMappingURL=data:application/json;base64,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