"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Frauddetector = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [frauddetector](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Frauddetector extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a batch of variables
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchCreateVariable.html
     */
    toBatchCreateVariable() {
        return this.to('BatchCreateVariable');
    }
    /**
     * Grants permission to get a batch of variables
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchGetVariable.html
     */
    toBatchGetVariable() {
        return this.to('BatchGetVariable');
    }
    /**
     * Grants permission to cancel the specified batch import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CancelBatchImportJob.html
     */
    toCancelBatchImportJob() {
        return this.to('CancelBatchImportJob');
    }
    /**
     * Grants permission to cancel the specified batch prediction job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CancelBatchPredictionJob.html
     */
    toCancelBatchPredictionJob() {
        return this.to('CancelBatchPredictionJob');
    }
    /**
     * Grants permission to create a batch import job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateBatchImportJob.html
     */
    toCreateBatchImportJob() {
        return this.to('CreateBatchImportJob');
    }
    /**
     * Grants permission to create a batch prediction job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateBatchPredictionJob.html
     */
    toCreateBatchPredictionJob() {
        return this.to('CreateBatchPredictionJob');
    }
    /**
     * Grants permission to create a detector version. The detector version starts in a DRAFT status
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateDetectorVersion.html
     */
    toCreateDetectorVersion() {
        return this.to('CreateDetectorVersion');
    }
    /**
     * Grants permission to create a list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateList.html
     */
    toCreateList() {
        return this.to('CreateList');
    }
    /**
     * Grants permission to create a model using the specified model type
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModel.html
     */
    toCreateModel() {
        return this.to('CreateModel');
    }
    /**
     * Grants permission to create a version of the model using the specified model type and model id
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModelVersion.html
     */
    toCreateModelVersion() {
        return this.to('CreateModelVersion');
    }
    /**
     * Grants permission to create a rule for use with the specified detector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateRule.html
     */
    toCreateRule() {
        return this.to('CreateRule');
    }
    /**
     * Grants permission to create a variable
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateVariable.html
     */
    toCreateVariable() {
        return this.to('CreateVariable');
    }
    /**
     * Grants permission to delete a batch import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteBatchImportJob.html
     */
    toDeleteBatchImportJob() {
        return this.to('DeleteBatchImportJob');
    }
    /**
     * Grants permission to delete a batch prediction job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteBatchPredictionJob.html
     */
    toDeleteBatchPredictionJob() {
        return this.to('DeleteBatchPredictionJob');
    }
    /**
     * Grants permission to delete the detector. Before deleting a detector, you must first delete all detector versions and rule versions associated with the detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetector.html
     */
    toDeleteDetector() {
        return this.to('DeleteDetector');
    }
    /**
     * Grants permission to delete the detector version. You cannot delete detector versions that are in ACTIVE status
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetectorVersion.html
     */
    toDeleteDetectorVersion() {
        return this.to('DeleteDetectorVersion');
    }
    /**
     * Grants permission to delete an entity type. You cannot delete an entity type that is included in an event type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEntityType.html
     */
    toDeleteEntityType() {
        return this.to('DeleteEntityType');
    }
    /**
     * Grants permission to deletes the specified event
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEvent.html
     */
    toDeleteEvent() {
        return this.to('DeleteEvent');
    }
    /**
     * Grants permission to delete an event type. You cannot delete an event type that is used in a detector or a model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEventType.html
     */
    toDeleteEventType() {
        return this.to('DeleteEventType');
    }
    /**
     * Grants permission to delete events for the specified event type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEventsByEventType.html
     */
    toDeleteEventsByEventType() {
        return this.to('DeleteEventsByEventType');
    }
    /**
     * Grants permission to remove a SageMaker model from Amazon Fraud Detector. You can remove an Amazon SageMaker model if it is not associated with a detector version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteExternalModel.html
     */
    toDeleteExternalModel() {
        return this.to('DeleteExternalModel');
    }
    /**
     * Grants permission to delete a label. You cannot delete labels that are included in an event type in Amazon Fraud Detector. You cannot delete a label assigned to an event ID. You must first delete the relevant event ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteLabel.html
     */
    toDeleteLabel() {
        return this.to('DeleteLabel');
    }
    /**
     * Grants permission to delete a list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteList.html
     */
    toDeleteList() {
        return this.to('DeleteList');
    }
    /**
     * Grants permission to delete a model. You can delete models and model versions in Amazon Fraud Detector, provided that they are not associated with a detector version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteModel.html
     */
    toDeleteModel() {
        return this.to('DeleteModel');
    }
    /**
     * Grants permission to delete a model version. You can delete models and model versions in Amazon Fraud Detector, provided that they are not associated with a detector version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteModelVersion.html
     */
    toDeleteModelVersion() {
        return this.to('DeleteModelVersion');
    }
    /**
     * Grants permission to delete an outcome. You cannot delete an outcome that is used in a rule version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteOutcome.html
     */
    toDeleteOutcome() {
        return this.to('DeleteOutcome');
    }
    /**
     * Grants permission to delete the rule. You cannot delete a rule if it is used by an ACTIVE or INACTIVE detector version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteRule.html
     */
    toDeleteRule() {
        return this.to('DeleteRule');
    }
    /**
     * Grants permission to delete a variable. You cannot delete variables that are included in an event type in Amazon Fraud Detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteVariable.html
     */
    toDeleteVariable() {
        return this.to('DeleteVariable');
    }
    /**
     * Grants permission to get all versions for a specified detector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeDetector.html
     */
    toDescribeDetector() {
        return this.to('DescribeDetector');
    }
    /**
     * Grants permission to get all of the model versions for the specified model type or for the specified model type and model ID. You can also get details for a single, specified model version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeModelVersions.html
     */
    toDescribeModelVersions() {
        return this.to('DescribeModelVersions');
    }
    /**
     * Grants permission to get the data validation report of a specific batch import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/ug/prepare-storage-event-data.html#smart-data-validation
     */
    toGetBatchImportJobValidationReport() {
        return this.to('GetBatchImportJobValidationReport');
    }
    /**
     * Grants permission to get all batch import jobs or a specific job if you specify a job ID
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetBatchImportJobs.html
     */
    toGetBatchImportJobs() {
        return this.to('GetBatchImportJobs');
    }
    /**
     * Grants permission to get all batch prediction jobs or a specific job if you specify a job ID. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 50 records per page. If you provide a maxResults, the value must be between 1 and 50. To get the next page results, provide the pagination token from the GetBatchPredictionJobsResponse as part of your request. A null pagination token fetches the records from the beginning
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetBatchPredictionJobs.html
     */
    toGetBatchPredictionJobs() {
        return this.to('GetBatchPredictionJobs');
    }
    /**
     * Grants permission to get a specific event type DeleteEventsByEventType API execution status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDeleteEventsByEventTypeStatus.html
     */
    toGetDeleteEventsByEventTypeStatus() {
        return this.to('GetDeleteEventsByEventTypeStatus');
    }
    /**
     * Grants permission to get a particular detector version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectorVersion.html
     */
    toGetDetectorVersion() {
        return this.to('GetDetectorVersion');
    }
    /**
     * Grants permission to get all detectors or a single detector if a detectorId is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetDetectorsResponse as part of your request. A null pagination token fetches the records from the beginning
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectors.html
     */
    toGetDetectors() {
        return this.to('GetDetectors');
    }
    /**
     * Grants permission to get all entity types or a specific entity type if a name is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEntityTypesResponse as part of your request. A null pagination token fetches the records from the beginning
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEntityTypes.html
     */
    toGetEntityTypes() {
        return this.to('GetEntityTypes');
    }
    /**
     * Grants permission to get the details of the specified event
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEvent.html
     */
    toGetEvent() {
        return this.to('GetEvent');
    }
    /**
     * Grants permission to evaluate an event against a detector version. If a version ID is not provided, the detector's (ACTIVE) version is used
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventPrediction.html
     */
    toGetEventPrediction() {
        return this.to('GetEventPrediction');
    }
    /**
     * Grants permission to get more details of a particular prediction
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventPredictionMetadata.html
     */
    toGetEventPredictionMetadata() {
        return this.to('GetEventPredictionMetadata');
    }
    /**
     * Grants permission to get all event types or a specific event type if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEventTypesResponse as part of your request. A null pagination token fetches the records from the beginning
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventTypes.html
     */
    toGetEventTypes() {
        return this.to('GetEventTypes');
    }
    /**
     * Grants permission to get the details for one or more Amazon SageMaker models that have been imported into the service. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetExternalModelsResult as part of your request. A null pagination token fetches the records from the beginning
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetExternalModels.html
     */
    toGetExternalModels() {
        return this.to('GetExternalModels');
    }
    /**
     * Grants permission to get the encryption key if a Key Management Service (KMS) customer master key (CMK) has been specified to be used to encrypt content in Amazon Fraud Detector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetKMSEncryptionKey.html
     */
    toGetKMSEncryptionKey() {
        return this.to('GetKMSEncryptionKey');
    }
    /**
     * Grants permission to get all labels or a specific label if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 50 records per page. If you provide a maxResults, the value must be between 10 and 50. To get the next page results, provide the pagination token from the GetGetLabelsResponse as part of your request. A null pagination token fetches the records from the beginning
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetLabels.html
     */
    toGetLabels() {
        return this.to('GetLabels');
    }
    /**
     * Grants permission to get elements of a list
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetListElements.html
     */
    toGetListElements() {
        return this.to('GetListElements');
    }
    /**
     * Grants permission to get metadata about lists
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetListsMetadata.html
     */
    toGetListsMetadata() {
        return this.to('GetListsMetadata');
    }
    /**
     * Grants permission to get the details of the specified model version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModelVersion.html
     */
    toGetModelVersion() {
        return this.to('GetModelVersion');
    }
    /**
     * Grants permission to get one or more models. Gets all models for the AWS account if no model type and no model id provided. Gets all models for the AWS account and model type, if the model type is specified but model id is not provided. Gets a specific model if (model type, model id) tuple is specified
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModels.html
     */
    toGetModels() {
        return this.to('GetModels');
    }
    /**
     * Grants permission to get one or more outcomes. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 100 records per page. If you provide a maxResults, the value must be between 50 and 100. To get the next page results, provide the pagination token from the GetOutcomesResult as part of your request. A null pagination token fetches the records from the beginning
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetOutcomes.html
     */
    toGetOutcomes() {
        return this.to('GetOutcomes');
    }
    /**
     * Grants permission to get all rules for a detector (paginated) if ruleId and ruleVersion are not specified. Gets all rules for the detector and the ruleId if present (paginated). Gets a specific rule if both the ruleId and the ruleVersion are specified
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetRules.html
     */
    toGetRules() {
        return this.to('GetRules');
    }
    /**
     * Grants permission to get all of the variables or the specific variable. This is a paginated API. Providing null maxSizePerPage results in retrieving maximum of 100 records per page. If you provide maxSizePerPage the value must be between 50 and 100. To get the next page result, a provide a pagination token from GetVariablesResult as part of your request. Null pagination token fetches the records from the beginning
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetVariables.html
     */
    toGetVariables() {
        return this.to('GetVariables');
    }
    /**
     * Grants permission to get a list of past predictions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_ListEventPredictions.html
     */
    toListEventPredictions() {
        return this.to('ListEventPredictions');
    }
    /**
     * Grants permission to list all tags associated with the resource. This is a paginated API. To get the next page results, provide the pagination token from the response as part of your request. A null pagination token fetches the records from the beginning
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to create or update a detector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutDetector.html
     */
    toPutDetector() {
        return this.to('PutDetector');
    }
    /**
     * Grants permission to create or update an entity type. An entity represents who is performing the event. As part of a fraud prediction, you pass the entity ID to indicate the specific entity who performed the event. An entity type classifies the entity. Example classifications include customer, merchant, or account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEntityType.html
     */
    toPutEntityType() {
        return this.to('PutEntityType');
    }
    /**
     * Grants permission to create or update an event type. An event is a business activity that is evaluated for fraud risk. With Amazon Fraud Detector, you generate fraud predictions for events. An event type defines the structure for an event sent to Amazon Fraud Detector. This includes the variables sent as part of the event, the entity performing the event (such as a customer), and the labels that classify the event. Example event types include online payment transactions, account registrations, and authentications
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEventType.html
     */
    toPutEventType() {
        return this.to('PutEventType');
    }
    /**
     * Grants permission to create or update an Amazon SageMaker model endpoint. You can also use this action to update the configuration of the model endpoint, including the IAM role and/or the mapped variables
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutExternalModel.html
     */
    toPutExternalModel() {
        return this.to('PutExternalModel');
    }
    /**
     * Grants permission to specify the Key Management Service (KMS) customer master key (CMK) to be used to encrypt content in Amazon Fraud Detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutKMSEncryptionKey.html
     */
    toPutKMSEncryptionKey() {
        return this.to('PutKMSEncryptionKey');
    }
    /**
     * Grants permission to create or update label. A label classifies an event as fraudulent or legitimate. Labels are associated with event types and used to train supervised machine learning models in Amazon Fraud Detector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutLabel.html
     */
    toPutLabel() {
        return this.to('PutLabel');
    }
    /**
     * Grants permission to create or update an outcome
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutOutcome.html
     */
    toPutOutcome() {
        return this.to('PutOutcome');
    }
    /**
     * Grants permission to send event
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_SendEvent.html
     */
    toSendEvent() {
        return this.to('SendEvent');
    }
    /**
     * Grants permission to assign tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a detector version. The detector version attributes that you can update include models, external model endpoints, rules, rule execution mode, and description. You can only update a DRAFT detector version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersion.html
     */
    toUpdateDetectorVersion() {
        return this.to('UpdateDetectorVersion');
    }
    /**
     * Grants permission to update the detector version's description. You can update the metadata for any detector version (DRAFT, ACTIVE, or INACTIVE)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionMetadata.html
     */
    toUpdateDetectorVersionMetadata() {
        return this.to('UpdateDetectorVersionMetadata');
    }
    /**
     * Grants permission to update the detector version's status. You can perform the following promotions or demotions using UpdateDetectorVersionStatus: DRAFT to ACTIVE, ACTIVE to INACTIVE, and INACTIVE to ACTIVE
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionStatus.html
     */
    toUpdateDetectorVersionStatus() {
        return this.to('UpdateDetectorVersionStatus');
    }
    /**
     * Grants permission to update an existing event record's label value
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateEventLabel.html
     */
    toUpdateEventLabel() {
        return this.to('UpdateEventLabel');
    }
    /**
     * Grants permission to update a list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateList.html
     */
    toUpdateList() {
        return this.to('UpdateList');
    }
    /**
     * Grants permission to update a model. You can update the description attribute using this action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModel.html
     */
    toUpdateModel() {
        return this.to('UpdateModel');
    }
    /**
     * Grants permission to update a model version. Updating a model version retrains an existing model version using updated training data and produces a new minor version of the model. You can update the training data set location and data access role attributes using this action. This action creates and trains a new minor version of the model, for example version 1.01, 1.02, 1.03
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersion.html
     */
    toUpdateModelVersion() {
        return this.to('UpdateModelVersion');
    }
    /**
     * Grants permission to update the status of a model version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersionStatus.html
     */
    toUpdateModelVersionStatus() {
        return this.to('UpdateModelVersionStatus');
    }
    /**
     * Grants permission to update a rule's metadata. The description attribute can be updated
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleMetadata.html
     */
    toUpdateRuleMetadata() {
        return this.to('UpdateRuleMetadata');
    }
    /**
     * Grants permission to update a rule version resulting in a new rule version. Updates a rule version resulting in a new rule version (version 1, 2, 3 ...)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleVersion.html
     */
    toUpdateRuleVersion() {
        return this.to('UpdateRuleVersion');
    }
    /**
     * Grants permission to update a variable
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateVariable.html
     */
    toUpdateVariable() {
        return this.to('UpdateVariable');
    }
    /**
     * Adds a resource of type batch-prediction to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBatchPrediction(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:batch-prediction/${resourcePath}`);
    }
    /**
     * Adds a resource of type detector to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetector(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${resourcePath}`);
    }
    /**
     * Adds a resource of type detector-version to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetectorVersion(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector-version/${resourcePath}`);
    }
    /**
     * Adds a resource of type entity-type to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityType(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:entity-type/${resourcePath}`);
    }
    /**
     * Adds a resource of type external-model to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExternalModel(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:external-model/${resourcePath}`);
    }
    /**
     * Adds a resource of type event-type to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventType(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:event-type/${resourcePath}`);
    }
    /**
     * Adds a resource of type label to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLabel(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:label/${resourcePath}`);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModel(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model/${resourcePath}`);
    }
    /**
     * Adds a resource of type model-version to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelVersion(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model-version/${resourcePath}`);
    }
    /**
     * Adds a resource of type outcome to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOutcome(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:outcome/${resourcePath}`);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:rule/${resourcePath}`);
    }
    /**
     * Adds a resource of type variable to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVariable(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:variable/${resourcePath}`);
    }
    /**
     * Adds a resource of type batch-import to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBatchImport(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:batch-import/${resourcePath}`);
    }
    /**
     * Adds a resource of type list to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html#amazonfrauddetector-resources-for-iam-policies
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onList(resourcePath, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:frauddetector:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:list/${resourcePath}`);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toBatchCreateVariable()
     * - .toCreateBatchImportJob()
     * - .toCreateBatchPredictionJob()
     * - .toCreateDetectorVersion()
     * - .toCreateList()
     * - .toCreateModel()
     * - .toCreateModelVersion()
     * - .toCreateRule()
     * - .toCreateVariable()
     * - .toPutDetector()
     * - .toPutEntityType()
     * - .toPutEventType()
     * - .toPutExternalModel()
     * - .toPutLabel()
     * - .toPutOutcome()
     * - .toSendEvent()
     * - .toTagResource()
     * - .toUpdateEventLabel()
     * - .toUpdateModelVersion()
     * - .toUpdateRuleVersion()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteList()
     * - .toGetListElements()
     * - .toGetListsMetadata()
     * - .toUpdateList()
     *
     * Applies to resource types:
     * - batch-prediction
     * - detector
     * - detector-version
     * - entity-type
     * - external-model
     * - event-type
     * - label
     * - model
     * - model-version
     * - outcome
     * - rule
     * - variable
     * - batch-import
     * - list
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toBatchCreateVariable()
     * - .toCreateBatchImportJob()
     * - .toCreateBatchPredictionJob()
     * - .toCreateDetectorVersion()
     * - .toCreateList()
     * - .toCreateModel()
     * - .toCreateModelVersion()
     * - .toCreateRule()
     * - .toCreateVariable()
     * - .toPutDetector()
     * - .toPutEntityType()
     * - .toPutEventType()
     * - .toPutExternalModel()
     * - .toPutLabel()
     * - .toPutOutcome()
     * - .toSendEvent()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateEventLabel()
     * - .toUpdateModelVersion()
     * - .toUpdateRuleVersion()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [frauddetector](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfrauddetector.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'frauddetector';
        this.accessLevelList = {
            Write: [
                'BatchCreateVariable',
                'CancelBatchImportJob',
                'CancelBatchPredictionJob',
                'CreateBatchImportJob',
                'CreateBatchPredictionJob',
                'CreateDetectorVersion',
                'CreateList',
                'CreateModel',
                'CreateModelVersion',
                'CreateRule',
                'CreateVariable',
                'DeleteBatchImportJob',
                'DeleteBatchPredictionJob',
                'DeleteDetector',
                'DeleteDetectorVersion',
                'DeleteEntityType',
                'DeleteEvent',
                'DeleteEventType',
                'DeleteEventsByEventType',
                'DeleteExternalModel',
                'DeleteLabel',
                'DeleteList',
                'DeleteModel',
                'DeleteModelVersion',
                'DeleteOutcome',
                'DeleteRule',
                'DeleteVariable',
                'PutDetector',
                'PutEntityType',
                'PutEventType',
                'PutExternalModel',
                'PutKMSEncryptionKey',
                'PutLabel',
                'PutOutcome',
                'SendEvent',
                'UpdateDetectorVersion',
                'UpdateDetectorVersionMetadata',
                'UpdateDetectorVersionStatus',
                'UpdateEventLabel',
                'UpdateList',
                'UpdateModel',
                'UpdateModelVersion',
                'UpdateModelVersionStatus',
                'UpdateRuleMetadata',
                'UpdateRuleVersion',
                'UpdateVariable'
            ],
            List: [
                'BatchGetVariable',
                'GetBatchImportJobs',
                'GetBatchPredictionJobs',
                'GetDetectors',
                'GetEntityTypes',
                'GetEventTypes',
                'GetExternalModels',
                'GetLabels',
                'GetListsMetadata',
                'GetModels',
                'GetOutcomes',
                'GetRules',
                'GetVariables',
                'ListEventPredictions'
            ],
            Read: [
                'DescribeDetector',
                'DescribeModelVersions',
                'GetBatchImportJobValidationReport',
                'GetDeleteEventsByEventTypeStatus',
                'GetDetectorVersion',
                'GetEvent',
                'GetEventPrediction',
                'GetEventPredictionMetadata',
                'GetKMSEncryptionKey',
                'GetListElements',
                'GetModelVersion',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Frauddetector = Frauddetector;
//# sourceMappingURL=data:application/json;base64,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