"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Fsx = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [fsx](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfsx.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Fsx extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a File Gateway instance with an Amazon FSx for Windows File Server file system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/filegateway/latest/filefsxw/what-is-file-fsxw.html
     */
    toAssociateFileGateway() {
        return this.to('AssociateFileGateway');
    }
    /**
     * Grants permission to associate DNS aliases with an Amazon FSx for Windows File Server file system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_AssociateFileSystemAliases.html
     */
    toAssociateFileSystemAliases() {
        return this.to('AssociateFileSystemAliases');
    }
    /**
     * Grants permission to allow deletion of an FSx for ONTAP SnapLock Enterprise volume that contains WORM (write once, read many) files with active retention periods
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/snaplock-enterprise.html#bypass-enterprise
     */
    toBypassSnaplockEnterpriseRetention() {
        return this.to('BypassSnaplockEnterpriseRetention');
    }
    /**
     * Grants permission to cancel a data repository task
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CancelDataRepositoryTask.html
     */
    toCancelDataRepositoryTask() {
        return this.to('CancelDataRepositoryTask');
    }
    /**
     * Grants permission to copy a backup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopyBackup.html
     */
    toCopyBackup() {
        return this.to('CopyBackup');
    }
    /**
     * Grants permission to update an existing volume by using a snapshot from another Amazon FSx for OpenZFS file system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopySnapshotAndUpdateVolume.html
     */
    toCopySnapshotAndUpdateVolume() {
        return this.to('CopySnapshotAndUpdateVolume');
    }
    /**
     * Grants permission to create and attach a S3 Access Point to a FSx File System
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:CreateAccessPoint
     * - s3:GetAccessPoint
     * - s3:PutAccessPointPolicy
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateAndAttachS3AccessPoint.html
     */
    toCreateAndAttachS3AccessPoint() {
        return this.to('CreateAndAttachS3AccessPoint');
    }
    /**
     * Grants permission to create a new backup of an Amazon FSx file system or an Amazon FSx volume
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateBackup.html
     */
    toCreateBackup() {
        return this.to('CreateBackup');
    }
    /**
     * Grants permission to create a new data respository association for an Amazon FSx for Lustre file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateDataRepositoryAssociation.html
     */
    toCreateDataRepositoryAssociation() {
        return this.to('CreateDataRepositoryAssociation');
    }
    /**
     * Grants permission to create a new data respository task for an Amazon FSx for Lustre file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateDataRepositoryTask.html
     */
    toCreateDataRepositoryTask() {
        return this.to('CreateDataRepositoryTask');
    }
    /**
     * Grants permission to create a new, empty, Amazon file cache
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - ec2:GetSecurityGroupsForVpc
     * - fsx:CreateDataRepositoryAssociation
     * - fsx:TagResource
     * - logs:CreateLogGroup
     * - logs:CreateLogStream
     * - logs:PutLogEvents
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateFileCache.html
     */
    toCreateFileCache() {
        return this.to('CreateFileCache');
    }
    /**
     * Grants permission to create a new, empty, Amazon FSx file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:GetSecurityGroupsForVpc
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateFileSystem.html
     */
    toCreateFileSystem() {
        return this.to('CreateFileSystem');
    }
    /**
     * Grants permission to create a new Amazon FSx file system from an existing backup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:GetSecurityGroupsForVpc
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateFileSystemFromBackup.html
     */
    toCreateFileSystemFromBackup() {
        return this.to('CreateFileSystemFromBackup');
    }
    /**
     * Grants permission to create a new snapshot on a volume
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permission to create a new storage virtual machine in an Amazon FSx for Ontap file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateStorageVirtualMachine.html
     */
    toCreateStorageVirtualMachine() {
        return this.to('CreateStorageVirtualMachine');
    }
    /**
     * Grants permission to create a new volume
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifStorageVirtualMachineId()
     * - .ifParentVolumeId()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateVolume.html
     */
    toCreateVolume() {
        return this.to('CreateVolume');
    }
    /**
     * Grants permission to create a new volume from backup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifStorageVirtualMachineId()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CreateVolumeFromBackup.html
     */
    toCreateVolumeFromBackup() {
        return this.to('CreateVolumeFromBackup');
    }
    /**
     * Grants permission to delete a backup, deleting its contents. After deletion, the backup no longer exists, and its data is no longer available
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteBackup.html
     */
    toDeleteBackup() {
        return this.to('DeleteBackup');
    }
    /**
     * Grants permission to delete a data repository association
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteDataRepositoryAssociation.html
     */
    toDeleteDataRepositoryAssociation() {
        return this.to('DeleteDataRepositoryAssociation');
    }
    /**
     * Grants permission to delete a file cache, deleting its contents
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:DeleteDataRepositoryAssociation
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteFileCache.html
     */
    toDeleteFileCache() {
        return this.to('DeleteFileCache');
    }
    /**
     * Grants permission to delete a file system, deleting its contents and any existing automatic backups of the file system
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - fsx:CreateBackup
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteFileSystem.html
     */
    toDeleteFileSystem() {
        return this.to('DeleteFileSystem');
    }
    /**
     * Required to manage cross-account sharing of FSx volumes through AWS Resource Access Manager (RAM). PutResourcePolicy and GetResourcePolicy are also required
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/on-demand-replication.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a snapshot on a volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        return this.to('DeleteSnapshot');
    }
    /**
     * Grants permission to delete a storage virtual machine, deleting its contents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteStorageVirtualMachine.html
     */
    toDeleteStorageVirtualMachine() {
        return this.to('DeleteStorageVirtualMachine');
    }
    /**
     * Grants permission to delete a volume, deleting its contents and any existing automatic backups of the volume
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifStorageVirtualMachineId()
     * - .ifParentVolumeId()
     *
     * Dependent actions:
     * - fsx:TagResource
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DeleteVolume.html
     */
    toDeleteVolume() {
        return this.to('DeleteVolume');
    }
    /**
     * Grants permission to describe the File Gateway instances associated with an Amazon FSx for Windows File Server file system
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/filegateway/latest/filefsxw/what-is-file-fsxw.html
     */
    toDescribeAssociatedFileGateways() {
        return this.to('DescribeAssociatedFileGateways');
    }
    /**
     * Grants permission to return the descriptions of all backups owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeBackups.html
     */
    toDescribeBackups() {
        return this.to('DescribeBackups');
    }
    /**
     * Grants permission to return the descriptions of all data repository associations owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeDataRepositoryAssociations.html
     */
    toDescribeDataRepositoryAssociations() {
        return this.to('DescribeDataRepositoryAssociations');
    }
    /**
     * Grants permission to return the descriptions of all data repository tasks owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeDataRepositoryTasks.html
     */
    toDescribeDataRepositoryTasks() {
        return this.to('DescribeDataRepositoryTasks');
    }
    /**
     * Grants permission to return the descriptions of all file caches owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileCaches.html
     */
    toDescribeFileCaches() {
        return this.to('DescribeFileCaches');
    }
    /**
     * Grants permission to return the description of all DNS aliases owned by your Amazon FSx for Windows File Server file system
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystemAliases.html
     */
    toDescribeFileSystemAliases() {
        return this.to('DescribeFileSystemAliases');
    }
    /**
     * Grants permission to return the descriptions of all file systems owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html
     */
    toDescribeFileSystems() {
        return this.to('DescribeFileSystems');
    }
    /**
     * Grants permission to return the descriptions of S3 Access Point Attachments
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeS3AccessPointAttachments.html
     */
    toDescribeS3AccessPointAttachments() {
        return this.to('DescribeS3AccessPointAttachments');
    }
    /**
     * Grants permission to return the descriptions of whether FSx route table updates from participant accounts are allowed in your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeSharedVpcConfiguration.html
     */
    toDescribeSharedVpcConfiguration() {
        return this.to('DescribeSharedVpcConfiguration');
    }
    /**
     * Grants permission to return the descriptions of all snapshots owned by your AWS account in the AWS Region of the endpoint you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        return this.to('DescribeSnapshots');
    }
    /**
     * Grants permission to return the descriptions of all storage virtual machines owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeStorageVirtualMachines.html
     */
    toDescribeStorageVirtualMachines() {
        return this.to('DescribeStorageVirtualMachines');
    }
    /**
     * Grants permission to return the descriptions of all volumes owned by your AWS account in the AWS Region of the endpoint that you're calling
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeVolumes.html
     */
    toDescribeVolumes() {
        return this.to('DescribeVolumes');
    }
    /**
     * Grants permission to detach an S3 Access Point from an Amazon FSx File System and delete the S3 Access Point
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:DeleteAccessPoint
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DetachAndDeleteS3AccessPoint.html
     */
    toDetachAndDeleteS3AccessPoint() {
        return this.to('DetachAndDeleteS3AccessPoint');
    }
    /**
     * Grants permission to disassociate a File Gateway instance from an Amazon FSx for Windows File Server file system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/filegateway/latest/filefsxw/what-is-file-fsxw.html
     */
    toDisassociateFileGateway() {
        return this.to('DisassociateFileGateway');
    }
    /**
     * Grants permission to disassociate file system aliases with an Amazon FSx for Windows File Server file system
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_DisassociateFileSystemAliases.html
     */
    toDisassociateFileSystemAliases() {
        return this.to('DisassociateFileSystemAliases');
    }
    /**
     * Required to manage cross-account sharing of FSx volumes through AWS Resource Access Manager (RAM). PutResourcePolicy and DeleteResourcePolicy are also required
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/on-demand-replication.html
     */
    toGetResourcePolicy() {
        return this.to('GetResourcePolicy');
    }
    /**
     * Grants permission to list tags for an Amazon FSx resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to manage backup principal associations through AWS Backup
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopyBackup.html
     */
    toManageBackupPrincipalAssociations() {
        return this.to('ManageBackupPrincipalAssociations');
    }
    /**
     * Required to manage cross-account sharing of FSx volumes through AWS Resource Access Manager (RAM). DeleteResourcePolicy and GetResourcePolicy are also required
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/on-demand-replication.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to release file system NFS V3 locks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_ReleaseFileSystemNfsV3Locks.html
     */
    toReleaseFileSystemNfsV3Locks() {
        return this.to('ReleaseFileSystemNfsV3Locks');
    }
    /**
     * Grants permission to restore volume state from a snapshot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_RestoreVolumeFromSnapshot.html
     */
    toRestoreVolumeFromSnapshot() {
        return this.to('RestoreVolumeFromSnapshot');
    }
    /**
     * Grants permission to start misconfigured state recovery
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_StartMisconfiguredStateRecovery.html
     */
    toStartMisconfiguredStateRecovery() {
        return this.to('StartMisconfiguredStateRecovery');
    }
    /**
     * Grants permission to tag an Amazon FSx resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove a tag from an Amazon FSx resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update data repository association configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateDataRepositoryAssociation.html
     */
    toUpdateDataRepositoryAssociation() {
        return this.to('UpdateDataRepositoryAssociation');
    }
    /**
     * Grants permission to update file cache configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateFileCache.html
     */
    toUpdateFileCache() {
        return this.to('UpdateFileCache');
    }
    /**
     * Grants permission to update file system configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateFileSystem.html
     */
    toUpdateFileSystem() {
        return this.to('UpdateFileSystem');
    }
    /**
     * Grants permission to enable or disable FSx route table updates from participant accounts in your account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateSharedVpcConfiguration.html
     */
    toUpdateSharedVpcConfiguration() {
        return this.to('UpdateSharedVpcConfiguration');
    }
    /**
     * Grants permission to update snapshot configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateSnapshot.html
     */
    toUpdateSnapshot() {
        return this.to('UpdateSnapshot');
    }
    /**
     * Grants permission to update storage virtual machine configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateStorageVirtualMachine.html
     */
    toUpdateStorageVirtualMachine() {
        return this.to('UpdateStorageVirtualMachine');
    }
    /**
     * Grants permission to update volume configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStorageVirtualMachineId()
     * - .ifParentVolumeId()
     *
     * https://docs.aws.amazon.com/fsx/latest/APIReference/API_UpdateVolume.html
     */
    toUpdateVolume() {
        return this.to('UpdateVolume');
    }
    /**
     * Adds a resource of type file-system to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/WindowsGuide/access-control-overview.html#access-control-resources
     *
     * @param fileSystemId - Identifier for the fileSystemId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFileSystem(fileSystemId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:file-system/${fileSystemId}`);
    }
    /**
     * Adds a resource of type file-cache to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/FileCacheGuide/security-iam.html
     *
     * @param fileCacheId - Identifier for the fileCacheId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFileCache(fileCacheId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:file-cache/${fileCacheId}`);
    }
    /**
     * Adds a resource of type backup to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/WindowsGuide/access-control-overview.html#access-control-resources
     *
     * @param backupId - Identifier for the backupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBackup(backupId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:backup/${backupId}`);
    }
    /**
     * Adds a resource of type storage-virtual-machine to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/security-iam.html
     *
     * @param fileSystemId - Identifier for the fileSystemId.
     * @param storageVirtualMachineId - Identifier for the storageVirtualMachineId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStorageVirtualMachine(fileSystemId, storageVirtualMachineId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:storage-virtual-machine/${fileSystemId}/${storageVirtualMachineId}`);
    }
    /**
     * Adds a resource of type task to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/LustreGuide/access-control-overview.html#access-control-resources
     *
     * @param taskId - Identifier for the taskId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTask(taskId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:task/${taskId}`);
    }
    /**
     * Adds a resource of type association to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/LustreGuide/access-control-overview.html#access-control-resources
     *
     * @param fileSystemIdOrFileCacheId - Identifier for the fileSystemIdOrFileCacheId.
     * @param dataRepositoryAssociationId - Identifier for the dataRepositoryAssociationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssociation(fileSystemIdOrFileCacheId, dataRepositoryAssociationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:association/${fileSystemIdOrFileCacheId}/${dataRepositoryAssociationId}`);
    }
    /**
     * Adds a resource of type volume to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/security-iam.html
     *
     * @param fileSystemId - Identifier for the fileSystemId.
     * @param volumeId - Identifier for the volumeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVolume(fileSystemId, volumeId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:volume/${fileSystemId}/${volumeId}`);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/access-control-overview.html#access-control-resources
     *
     * @param volumeId - Identifier for the volumeId.
     * @param snapshotId - Identifier for the snapshotId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshot(volumeId, snapshotId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:fsx:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:snapshot/${volumeId}/${snapshotId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCopyBackup()
     * - .toCreateBackup()
     * - .toCreateDataRepositoryAssociation()
     * - .toCreateDataRepositoryTask()
     * - .toCreateFileCache()
     * - .toCreateFileSystem()
     * - .toCreateFileSystemFromBackup()
     * - .toCreateSnapshot()
     * - .toCreateStorageVirtualMachine()
     * - .toCreateVolume()
     * - .toCreateVolumeFromBackup()
     * - .toDeleteFileCache()
     * - .toDeleteFileSystem()
     * - .toDeleteVolume()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - file-system
     * - file-cache
     * - backup
     * - storage-virtual-machine
     * - task
     * - association
     * - volume
     * - snapshot
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCopyBackup()
     * - .toCreateBackup()
     * - .toCreateDataRepositoryAssociation()
     * - .toCreateDataRepositoryTask()
     * - .toCreateFileCache()
     * - .toCreateFileSystem()
     * - .toCreateFileSystemFromBackup()
     * - .toCreateSnapshot()
     * - .toCreateStorageVirtualMachine()
     * - .toCreateVolume()
     * - .toCreateVolumeFromBackup()
     * - .toDeleteFileCache()
     * - .toDeleteFileSystem()
     * - .toDeleteVolume()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by whether the backup is a destination backup for a CopyBackup operation
     *
     * https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html#copy-backups
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifIsBackupCopyDestination(value) {
        return this.if(`IsBackupCopyDestination`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by whether the backup is a source backup for a CopyBackup operation
     *
     * https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html#copy-backups
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifIsBackupCopySource(value) {
        return this.if(`IsBackupCopySource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by NFS data repositories which support authentication
     *
     * https://docs.aws.amazon.com/fsx/latest/FileCacheGuide/encryption-in-transit.html
     *
     * Applies to actions:
     * - .toCreateFileCache()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifNfsDataRepositoryAuthenticationEnabled(value) {
        return this.if(`NfsDataRepositoryAuthenticationEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by NFS data repositories which support encryption-in-transit
     *
     * https://docs.aws.amazon.com/fsx/latest/FileCacheGuide/encryption-in-transit.html
     *
     * Applies to actions:
     * - .toCreateFileCache()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifNfsDataRepositoryEncryptionInTransitEnabled(value) {
        return this.if(`NfsDataRepositoryEncryptionInTransitEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the containing parent volume for mutating volume operations
     *
     * https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/creating-volumes.html
     *
     * Applies to actions:
     * - .toCreateVolume()
     * - .toDeleteVolume()
     * - .toUpdateVolume()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifParentVolumeId(value, operator) {
        return this.if(`ParentVolumeId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the containing storage virtual machine for a volume for mutating volume operations
     *
     * https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/creating-volumes.html
     *
     * Applies to actions:
     * - .toCreateVolume()
     * - .toCreateVolumeFromBackup()
     * - .toDeleteVolume()
     * - .toUpdateVolume()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStorageVirtualMachineId(value, operator) {
        return this.if(`StorageVirtualMachineId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [fsx](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonfsx.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'fsx';
        this.accessLevelList = {
            Write: [
                'AssociateFileGateway',
                'AssociateFileSystemAliases',
                'CancelDataRepositoryTask',
                'CopyBackup',
                'CopySnapshotAndUpdateVolume',
                'CreateAndAttachS3AccessPoint',
                'CreateBackup',
                'CreateDataRepositoryAssociation',
                'CreateDataRepositoryTask',
                'CreateFileCache',
                'CreateFileSystem',
                'CreateFileSystemFromBackup',
                'CreateSnapshot',
                'CreateStorageVirtualMachine',
                'CreateVolume',
                'CreateVolumeFromBackup',
                'DeleteBackup',
                'DeleteDataRepositoryAssociation',
                'DeleteFileCache',
                'DeleteFileSystem',
                'DeleteSnapshot',
                'DeleteStorageVirtualMachine',
                'DeleteVolume',
                'DetachAndDeleteS3AccessPoint',
                'DisassociateFileGateway',
                'DisassociateFileSystemAliases',
                'ReleaseFileSystemNfsV3Locks',
                'RestoreVolumeFromSnapshot',
                'StartMisconfiguredStateRecovery',
                'UpdateDataRepositoryAssociation',
                'UpdateFileCache',
                'UpdateFileSystem',
                'UpdateSharedVpcConfiguration',
                'UpdateSnapshot',
                'UpdateStorageVirtualMachine',
                'UpdateVolume'
            ],
            'Permissions management': [
                'BypassSnaplockEnterpriseRetention',
                'DeleteResourcePolicy',
                'GetResourcePolicy',
                'ManageBackupPrincipalAssociations',
                'PutResourcePolicy'
            ],
            Read: [
                'DescribeAssociatedFileGateways',
                'DescribeBackups',
                'DescribeDataRepositoryAssociations',
                'DescribeDataRepositoryTasks',
                'DescribeFileCaches',
                'DescribeFileSystemAliases',
                'DescribeFileSystems',
                'DescribeS3AccessPointAttachments',
                'DescribeSharedVpcConfiguration',
                'DescribeSnapshots',
                'DescribeStorageVirtualMachines',
                'DescribeVolumes',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Fsx = Fsx;
//# sourceMappingURL=data:application/json;base64,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