"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Globalaccelerator = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [globalaccelerator](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsglobalaccelerator.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Globalaccelerator extends shared_1.PolicyStatement {
    /**
     * Grants permission to add a virtual private cloud (VPC) subnet endpoint to a custom routing accelerator endpoint group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_AddCustomRoutingEndpoints.html
     */
    toAddCustomRoutingEndpoints() {
        return this.to('AddCustomRoutingEndpoints');
    }
    /**
     * Grants permission to add an endpoint to a standard accelerator endpoint group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - globalaccelerator:UpdateEndpointGroup
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_AddEndpoints.html
     */
    toAddEndpoints() {
        return this.to('AddEndpoints');
    }
    /**
     * Grants permission to advertises an IPv4 address range that is provisioned for use with your accelerator through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_AdvertiseByoipCidr.html
     */
    toAdvertiseByoipCidr() {
        return this.to('AdvertiseByoipCidr');
    }
    /**
     * Grants permission to allows custom routing of user traffic to a private destination IP:PORT in a specific VPC subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_AllowCustomRoutingTraffic.html
     */
    toAllowCustomRoutingTraffic() {
        return this.to('AllowCustomRoutingTraffic');
    }
    /**
     * Grants permission to create a standard accelerator
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateAccelerator.html
     */
    toCreateAccelerator() {
        return this.to('CreateAccelerator');
    }
    /**
     * Grants permission to create a CrossAccountAttachment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateCrossAccountAttachment.html
     */
    toCreateCrossAccountAttachment() {
        return this.to('CreateCrossAccountAttachment');
    }
    /**
     * Grants permission to create a Custom Routing accelerator
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateCustomRoutingAccelerator.html
     */
    toCreateCustomRoutingAccelerator() {
        return this.to('CreateCustomRoutingAccelerator');
    }
    /**
     * Grants permission to create an endpoint group for the specified listener for a custom routing accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateCustomRoutingEndpointGroup.html
     */
    toCreateCustomRoutingEndpointGroup() {
        return this.to('CreateCustomRoutingEndpointGroup');
    }
    /**
     * Grants permission to create a listener to process inbound connections from clients to a custom routing accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateCustomRoutingListener.html
     */
    toCreateCustomRoutingListener() {
        return this.to('CreateCustomRoutingListener');
    }
    /**
     * Grants permission to add an endpoint group to a standard accelerator listener
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateEndpointGroup.html
     */
    toCreateEndpointGroup() {
        return this.to('CreateEndpointGroup');
    }
    /**
     * Grants permission to add a listener to a standard accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CreateListener.html
     */
    toCreateListener() {
        return this.to('CreateListener');
    }
    /**
     * Grants permission to delete a standard accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteAccelerator.html
     */
    toDeleteAccelerator() {
        return this.to('DeleteAccelerator');
    }
    /**
     * Grants permission to delete a CrossAccountAttachment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteCrossAccountAttachment.html
     */
    toDeleteCrossAccountAttachment() {
        return this.to('DeleteCrossAccountAttachment');
    }
    /**
     * Grants permission to delete a custom routing accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteCustomRoutingAccelerator.html
     */
    toDeleteCustomRoutingAccelerator() {
        return this.to('DeleteCustomRoutingAccelerator');
    }
    /**
     * Grants permission to delete an endpoint group from a listener for a custom routing accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteCustomRoutingEndpointGroup.html
     */
    toDeleteCustomRoutingEndpointGroup() {
        return this.to('DeleteCustomRoutingEndpointGroup');
    }
    /**
     * Grants permission to delete a listener for a custom routing accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteCustomRoutingListener.html
     */
    toDeleteCustomRoutingListener() {
        return this.to('DeleteCustomRoutingListener');
    }
    /**
     * Grants permission to delete an endpoint group associated with a standard accelerator listener
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteEndpointGroup.html
     */
    toDeleteEndpointGroup() {
        return this.to('DeleteEndpointGroup');
    }
    /**
     * Grants permission to delete a listener from a standard accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeleteListener.html
     */
    toDeleteListener() {
        return this.to('DeleteListener');
    }
    /**
     * Grants permission to disallows custom routing of user traffic to a private destination IP:PORT in a specific VPC subnet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DenyCustomRoutingTraffic.html
     */
    toDenyCustomRoutingTraffic() {
        return this.to('DenyCustomRoutingTraffic');
    }
    /**
     * Grants permission to releases the specified address range that you provisioned for use with your accelerator through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DeprovisionByoipCidr.html
     */
    toDeprovisionByoipCidr() {
        return this.to('DeprovisionByoipCidr');
    }
    /**
     * Grants permissions to describe a standard accelerator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAccelerator.html
     */
    toDescribeAccelerator() {
        return this.to('DescribeAccelerator');
    }
    /**
     * Grants permission to describe a standard accelerator attributes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeAcceleratorAttributes.html
     */
    toDescribeAcceleratorAttributes() {
        return this.to('DescribeAcceleratorAttributes');
    }
    /**
     * Grants permissions to describe a CrossAccountAttachment
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeCrossAccountAttachment.html
     */
    toDescribeCrossAccountAttachment() {
        return this.to('DescribeCrossAccountAttachment');
    }
    /**
     * Grants permission to describe a custom routing accelerator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeCustomRoutingAccelerator.html
     */
    toDescribeCustomRoutingAccelerator() {
        return this.to('DescribeCustomRoutingAccelerator');
    }
    /**
     * Grants permission to describe the attributes of a custom routing accelerator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeCustomRoutingAcceleratorAttributes.html
     */
    toDescribeCustomRoutingAcceleratorAttributes() {
        return this.to('DescribeCustomRoutingAcceleratorAttributes');
    }
    /**
     * Grants permission to describe an endpoint group for a custom routing accelerator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeCustomRoutingEndpointGroup.html
     */
    toDescribeCustomRoutingEndpointGroup() {
        return this.to('DescribeCustomRoutingEndpointGroup');
    }
    /**
     * Grants permission to describe a listener for a custom routing accelerator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeCustomRoutingListener.html
     */
    toDescribeCustomRoutingListener() {
        return this.to('DescribeCustomRoutingListener');
    }
    /**
     * Grants permission to describe a standard accelerator endpoint group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeEndpointGroup.html
     */
    toDescribeEndpointGroup() {
        return this.to('DescribeEndpointGroup');
    }
    /**
     * Grants permission to describe a standard accelerator listener
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_DescribeListener.html
     */
    toDescribeListener() {
        return this.to('DescribeListener');
    }
    /**
     * Grants permission to list all standard accelerators
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListAccelerators.html
     */
    toListAccelerators() {
        return this.to('ListAccelerators');
    }
    /**
     * Grants permission to list the BYOIP cidrs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListByoipCidrs.html
     */
    toListByoipCidrs() {
        return this.to('ListByoipCidrs');
    }
    /**
     * Grants permission to list all CrossAccountAttachments
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListCrossAccountAttachments.html
     */
    toListCrossAccountAttachments() {
        return this.to('ListCrossAccountAttachments');
    }
    /**
     * Grants permission to list accounts with CrossAccountAttachments listing caller as a principal
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListCrossAccountResourceAccounts.html
     */
    toListCrossAccountResourceAccounts() {
        return this.to('ListCrossAccountResourceAccounts');
    }
    /**
     * Grants permission to list all CrossAccountAttachment resources usable by caller
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListCrossAccountResources.html
     */
    toListCrossAccountResources() {
        return this.to('ListCrossAccountResources');
    }
    /**
     * Grants permission to list the custom routing accelerators for an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListCustomRoutingAccelerators.html
     */
    toListCustomRoutingAccelerators() {
        return this.to('ListCustomRoutingAccelerators');
    }
    /**
     * Grants permission to list the endpoint groups that are associated with a listener for a custom routing accelerator
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListCustomRoutingEndpointGroups.html
     */
    toListCustomRoutingEndpointGroups() {
        return this.to('ListCustomRoutingEndpointGroups');
    }
    /**
     * Grants permission to list the listeners for a custom routing accelerator
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListCustomRoutingListeners.html
     */
    toListCustomRoutingListeners() {
        return this.to('ListCustomRoutingListeners');
    }
    /**
     * Grants permission to list the port mappings for a custom routing accelerator
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListCustomRoutingPortMappings.html
     */
    toListCustomRoutingPortMappings() {
        return this.to('ListCustomRoutingPortMappings');
    }
    /**
     * Grants permission to list the port mappings for a specific endpoint IP address (a destination address) in a subnet
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListCustomRoutingPortMappingsByDestination.html
     */
    toListCustomRoutingPortMappingsByDestination() {
        return this.to('ListCustomRoutingPortMappingsByDestination');
    }
    /**
     * Grants permission to list all endpoint groups associated with a standard accelerator listener
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListEndpointGroups.html
     */
    toListEndpointGroups() {
        return this.to('ListEndpointGroups');
    }
    /**
     * Grants permission to list all listeners associated with a standard accelerator
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListListeners.html
     */
    toListListeners() {
        return this.to('ListListeners');
    }
    /**
     * Grants permission to list tags for a globalaccelerator resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to provisions an address range for use with your accelerator through bring your own IP addresses (BYOIP)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_ProvisionByoipCidr.html
     */
    toProvisionByoipCidr() {
        return this.to('ProvisionByoipCidr');
    }
    /**
     * Grants permission to remove virtual private cloud (VPC) subnet endpoints from a custom routing accelerator endpoint group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_RemoveCustomRoutingEndpoints.html
     */
    toRemoveCustomRoutingEndpoints() {
        return this.to('RemoveCustomRoutingEndpoints');
    }
    /**
     * Grants permission to remove an endpoint from a standard accelerator endpoint group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - globalaccelerator:UpdateEndpointGroup
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_RemoveEndpoints.html
     */
    toRemoveEndpoints() {
        return this.to('RemoveEndpoints');
    }
    /**
     * Grants permission to add tags to a globalaccelerator resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a globalaccelerator resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a standard accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAccelerator.html
     */
    toUpdateAccelerator() {
        return this.to('UpdateAccelerator');
    }
    /**
     * Grants permission to update a standard accelerator attributes
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateAcceleratorAttributes.html
     */
    toUpdateAcceleratorAttributes() {
        return this.to('UpdateAcceleratorAttributes');
    }
    /**
     * Grants permission to update a CrossAccountAttachment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateCrossAccountAttachment.html
     */
    toUpdateCrossAccountAttachment() {
        return this.to('UpdateCrossAccountAttachment');
    }
    /**
     * Grants permission to update a custom routing accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateCustomRoutingAccelerator.html
     */
    toUpdateCustomRoutingAccelerator() {
        return this.to('UpdateCustomRoutingAccelerator');
    }
    /**
     * Grants permission to update the attributes for a custom routing accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateCustomRoutingAcceleratorAttributes.html
     */
    toUpdateCustomRoutingAcceleratorAttributes() {
        return this.to('UpdateCustomRoutingAcceleratorAttributes');
    }
    /**
     * Grants permission to update a listener for a custom routing accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateCustomRoutingListener.html
     */
    toUpdateCustomRoutingListener() {
        return this.to('UpdateCustomRoutingListener');
    }
    /**
     * Grants permission to update an endpoint group on a standard accelerator listener
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateEndpointGroup.html
     */
    toUpdateEndpointGroup() {
        return this.to('UpdateEndpointGroup');
    }
    /**
     * Grants permission to update a listener on a standard accelerator
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_UpdateListener.html
     */
    toUpdateListener() {
        return this.to('UpdateListener');
    }
    /**
     * Grants permission to stops advertising a BYOIP IPv4 address
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_WithdrawByoipCidr.html
     */
    toWithdrawByoipCidr() {
        return this.to('WithdrawByoipCidr');
    }
    /**
     * Adds a resource of type accelerator to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_Accelerator.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccelerator(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:globalaccelerator::${account ?? this.defaultAccount}:accelerator/${resourceId}`);
    }
    /**
     * Adds a resource of type listener to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_Listener.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param listenerId - Identifier for the listenerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onListener(resourceId, listenerId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:globalaccelerator::${account ?? this.defaultAccount}:accelerator/${resourceId}/listener/${listenerId}`);
    }
    /**
     * Adds a resource of type endpointgroup to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_EndpointGroup.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param listenerId - Identifier for the listenerId.
     * @param endpointGroupId - Identifier for the endpointGroupId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEndpointgroup(resourceId, listenerId, endpointGroupId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:globalaccelerator::${account ?? this.defaultAccount}:accelerator/${resourceId}/listener/${listenerId}/endpoint-group/${endpointGroupId}`);
    }
    /**
     * Adds a resource of type attachment to the statement
     *
     * https://docs.aws.amazon.com/global-accelerator/latest/api/API_CrossAccountAttachment.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAttachment(resourceId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:globalaccelerator::${account ?? this.defaultAccount}:attachment/${resourceId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAccelerator()
     * - .toCreateCrossAccountAttachment()
     * - .toCreateCustomRoutingAccelerator()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - accelerator
     * - listener
     * - endpointgroup
     * - attachment
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAccelerator()
     * - .toCreateCrossAccountAttachment()
     * - .toCreateCustomRoutingAccelerator()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [globalaccelerator](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsglobalaccelerator.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'globalaccelerator';
        this.accessLevelList = {
            Write: [
                'AddCustomRoutingEndpoints',
                'AddEndpoints',
                'AdvertiseByoipCidr',
                'AllowCustomRoutingTraffic',
                'CreateAccelerator',
                'CreateCrossAccountAttachment',
                'CreateCustomRoutingAccelerator',
                'CreateCustomRoutingEndpointGroup',
                'CreateCustomRoutingListener',
                'CreateEndpointGroup',
                'CreateListener',
                'DeleteAccelerator',
                'DeleteCrossAccountAttachment',
                'DeleteCustomRoutingAccelerator',
                'DeleteCustomRoutingEndpointGroup',
                'DeleteCustomRoutingListener',
                'DeleteEndpointGroup',
                'DeleteListener',
                'DenyCustomRoutingTraffic',
                'DeprovisionByoipCidr',
                'ProvisionByoipCidr',
                'RemoveCustomRoutingEndpoints',
                'RemoveEndpoints',
                'UpdateAccelerator',
                'UpdateAcceleratorAttributes',
                'UpdateCrossAccountAttachment',
                'UpdateCustomRoutingAccelerator',
                'UpdateCustomRoutingAcceleratorAttributes',
                'UpdateCustomRoutingListener',
                'UpdateEndpointGroup',
                'UpdateListener',
                'WithdrawByoipCidr'
            ],
            Read: [
                'DescribeAccelerator',
                'DescribeAcceleratorAttributes',
                'DescribeCrossAccountAttachment',
                'DescribeCustomRoutingAccelerator',
                'DescribeCustomRoutingAcceleratorAttributes',
                'DescribeCustomRoutingEndpointGroup',
                'DescribeCustomRoutingListener',
                'DescribeEndpointGroup',
                'DescribeListener',
                'ListTagsForResource'
            ],
            List: [
                'ListAccelerators',
                'ListByoipCidrs',
                'ListCrossAccountAttachments',
                'ListCrossAccountResourceAccounts',
                'ListCrossAccountResources',
                'ListCustomRoutingAccelerators',
                'ListCustomRoutingEndpointGroups',
                'ListCustomRoutingListeners',
                'ListCustomRoutingPortMappings',
                'ListCustomRoutingPortMappingsByDestination',
                'ListEndpointGroups',
                'ListListeners'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Globalaccelerator = Globalaccelerator;
//# sourceMappingURL=data:application/json;base64,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