"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Guardduty = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [guardduty](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonguardduty.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Guardduty extends shared_1.PolicyStatement {
    /**
     * Grants permission to accept invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_AcceptAdministratorInvitation.html
     */
    toAcceptAdministratorInvitation() {
        return this.to('AcceptAdministratorInvitation');
    }
    /**
     * Grants permission to accept invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_AcceptInvitation.html
     */
    toAcceptInvitation() {
        return this.to('AcceptInvitation');
    }
    /**
     * Grants permission to archive GuardDuty findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ArchiveFindings.html
     */
    toArchiveFindings() {
        return this.to('ArchiveFindings');
    }
    /**
     * Grants permission to create a detector
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateDetector.html
     */
    toCreateDetector() {
        return this.to('CreateDetector');
    }
    /**
     * Grants permission to create GuardDuty filters. A filters defines finding attributes and conditions used to filter findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateFilter.html
     */
    toCreateFilter() {
        return this.to('CreateFilter');
    }
    /**
     * Grants permission to create an IPSet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:DeleteRolePolicy
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateIPSet.html
     */
    toCreateIPSet() {
        return this.to('CreateIPSet');
    }
    /**
     * Grants permission to create a new Malware Protection plan
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateMalwareProtectionPlan.html
     */
    toCreateMalwareProtectionPlan() {
        return this.to('CreateMalwareProtectionPlan');
    }
    /**
     * Grants permission to create GuardDuty member accounts, where the account used to create a member becomes the GuardDuty administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateMembers.html
     */
    toCreateMembers() {
        return this.to('CreateMembers');
    }
    /**
     * Grants permission to create a publishing destination
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreatePublishingDestination.html
     */
    toCreatePublishingDestination() {
        return this.to('CreatePublishingDestination');
    }
    /**
     * Grants permission to create sample findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateSampleFindings.html
     */
    toCreateSampleFindings() {
        return this.to('CreateSampleFindings');
    }
    /**
     * Grants permission to create GuardDuty ThreatEntitySets, where a ThreatEntitySet consists of known malicious IP addresses and/or domains used by GuardDuty to generate findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateThreatEntitySet.html
     */
    toCreateThreatEntitySet() {
        return this.to('CreateThreatEntitySet');
    }
    /**
     * Grants permission to create GuardDuty ThreatIntelSets, where a ThreatIntelSet consists of known malicious IP addresses used by GuardDuty to generate findings
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateThreatIntelSet.html
     */
    toCreateThreatIntelSet() {
        return this.to('CreateThreatIntelSet');
    }
    /**
     * Grants permission to create a TrustedEntitySet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_CreateTrustedEntitySet.html
     */
    toCreateTrustedEntitySet() {
        return this.to('CreateTrustedEntitySet');
    }
    /**
     * Grants permission to decline invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeclineInvitations.html
     */
    toDeclineInvitations() {
        return this.to('DeclineInvitations');
    }
    /**
     * Grants permission to delete GuardDuty detectors
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteDetector.html
     */
    toDeleteDetector() {
        return this.to('DeleteDetector');
    }
    /**
     * Grants permission to delete GuardDuty filters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteFilter.html
     */
    toDeleteFilter() {
        return this.to('DeleteFilter');
    }
    /**
     * Grants permission to delete GuardDuty IPSets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteIPSet.html
     */
    toDeleteIPSet() {
        return this.to('DeleteIPSet');
    }
    /**
     * Grants permission to delete invitations to become a GuardDuty member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteInvitations.html
     */
    toDeleteInvitations() {
        return this.to('DeleteInvitations');
    }
    /**
     * Grants permission to delete a Malware Protection plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteMalwareProtectionPlan.html
     */
    toDeleteMalwareProtectionPlan() {
        return this.to('DeleteMalwareProtectionPlan');
    }
    /**
     * Grants permission to delete GuardDuty member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteMembers.html
     */
    toDeleteMembers() {
        return this.to('DeleteMembers');
    }
    /**
     * Grants permission to delete a publishing destination
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeletePublishingDestination.html
     */
    toDeletePublishingDestination() {
        return this.to('DeletePublishingDestination');
    }
    /**
     * Grants permission to delete GuardDuty ThreatEntitySets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteThreatEntitySet.html
     */
    toDeleteThreatEntitySet() {
        return this.to('DeleteThreatEntitySet');
    }
    /**
     * Grants permission to delete GuardDuty ThreatIntelSets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteThreatIntelSet.html
     */
    toDeleteThreatIntelSet() {
        return this.to('DeleteThreatIntelSet');
    }
    /**
     * Grants permission to delete GuardDuty TrustedEntitySets
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DeleteTrustedEntitySet.html
     */
    toDeleteTrustedEntitySet() {
        return this.to('DeleteTrustedEntitySet');
    }
    /**
     * Grants permission to retrieve details about malware scans
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribeMalwareScans.html
     */
    toDescribeMalwareScans() {
        return this.to('DescribeMalwareScans');
    }
    /**
     * Grants permission to retrieve details about the delegated administrator associated with a GuardDuty detector
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribeOrganizationConfiguration.html
     */
    toDescribeOrganizationConfiguration() {
        return this.to('DescribeOrganizationConfiguration');
    }
    /**
     * Grants permission to retrieve details about a publishing destination
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DescribePublishingDestination.html
     */
    toDescribePublishingDestination() {
        return this.to('DescribePublishingDestination');
    }
    /**
     * Grants permission to disable the organization delegated administrator for GuardDuty
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisableOrganizationAdminAccount.html
     */
    toDisableOrganizationAdminAccount() {
        return this.to('DisableOrganizationAdminAccount');
    }
    /**
     * Grants permission to disassociate a GuardDuty member account from its GuardDuty administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateFromAdministratorAccount.html
     */
    toDisassociateFromAdministratorAccount() {
        return this.to('DisassociateFromAdministratorAccount');
    }
    /**
     * Grants permission to disassociate a GuardDuty member account from its GuardDuty administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateFromMasterAccount.html
     */
    toDisassociateFromMasterAccount() {
        return this.to('DisassociateFromMasterAccount');
    }
    /**
     * Grants permission to disassociate GuardDuty member accounts from their administrator GuardDuty account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_DisassociateMembers.html
     */
    toDisassociateMembers() {
        return this.to('DisassociateMembers');
    }
    /**
     * Grants permission to enable an organization delegated administrator for GuardDuty
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_EnableOrganizationAdminAccount.html
     */
    toEnableOrganizationAdminAccount() {
        return this.to('EnableOrganizationAdminAccount');
    }
    /**
     * Grants permission to retrieve details of the GuardDuty administrator account associated with a member account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetAdministratorAccount.html
     */
    toGetAdministratorAccount() {
        return this.to('GetAdministratorAccount');
    }
    /**
     * Grants permission to list Amazon GuardDuty coverage statistics for the specified GuardDuty account in a Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetCoverageStatistics.html
     */
    toGetCoverageStatistics() {
        return this.to('GetCoverageStatistics');
    }
    /**
     * Grants permission to retrieve GuardDuty detectors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetDetector.html
     */
    toGetDetector() {
        return this.to('GetDetector');
    }
    /**
     * Grants permission to retrieve GuardDuty filters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFilter.html
     */
    toGetFilter() {
        return this.to('GetFilter');
    }
    /**
     * Grants permission to retrieve GuardDuty findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindings.html
     */
    toGetFindings() {
        return this.to('GetFindings');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty finding statistics
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetFindingsStatistics.html
     */
    toGetFindingsStatistics() {
        return this.to('GetFindingsStatistics');
    }
    /**
     * Grants permission to retrieve GuardDuty IPSets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetIPSet.html
     */
    toGetIPSet() {
        return this.to('GetIPSet');
    }
    /**
     * Grants permission to retrieve the count of all GuardDuty invitations sent to a specified account, which does not include the accepted invitation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetInvitationsCount.html
     */
    toGetInvitationsCount() {
        return this.to('GetInvitationsCount');
    }
    /**
     * Grants permission to retrieve a Malware Protection plan details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMalwareProtectionPlan.html
     */
    toGetMalwareProtectionPlan() {
        return this.to('GetMalwareProtectionPlan');
    }
    /**
     * Grants permission to retrieve the malware scan settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMalwareScanSettings.html
     */
    toGetMalwareScanSettings() {
        return this.to('GetMalwareScanSettings');
    }
    /**
     * Grants permission to retrieve details of the GuardDuty administrator account associated with a member account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMasterAccount.html
     */
    toGetMasterAccount() {
        return this.to('GetMasterAccount');
    }
    /**
     * Grants permission to describe which data sources are enabled for member accounts detectors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMemberDetectors.html
     */
    toGetMemberDetectors() {
        return this.to('GetMemberDetectors');
    }
    /**
     * Grants permission to retrieve the member accounts associated with an administrator account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetMembers.html
     */
    toGetMembers() {
        return this.to('GetMembers');
    }
    /**
     * Grants permission to retrieve GuardDuty protection plan coverage statistics for member accounts in a Region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetOrganizationStatistics.html
     */
    toGetOrganizationStatistics() {
        return this.to('GetOrganizationStatistics');
    }
    /**
     * Grants permission to provide the number of days left for each data source used in the free trial period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetRemainingFreeTrialDays.html
     */
    toGetRemainingFreeTrialDays() {
        return this.to('GetRemainingFreeTrialDays');
    }
    /**
     * Grants permission to retrieve GuardDuty ThreatEntitySets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetThreatEntitySet.html
     */
    toGetThreatEntitySet() {
        return this.to('GetThreatEntitySet');
    }
    /**
     * Grants permission to retrieve GuardDuty ThreatIntelSets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetThreatIntelSet.html
     */
    toGetThreatIntelSet() {
        return this.to('GetThreatIntelSet');
    }
    /**
     * Grants permission to retrieve GuardDuty TrustedEntitySets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetTrustedEntitySet.html
     */
    toGetTrustedEntitySet() {
        return this.to('GetTrustedEntitySet');
    }
    /**
     * Grants permission to list Amazon GuardDuty usage statistics over the last 30 days for the specified detector ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_GetUsageStatistics.html
     */
    toGetUsageStatistics() {
        return this.to('GetUsageStatistics');
    }
    /**
     * Grants permission to invite other AWS accounts to enable GuardDuty and become GuardDuty member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_InviteMembers.html
     */
    toInviteMembers() {
        return this.to('InviteMembers');
    }
    /**
     * Grants permission to list all the resource details for a given account in a Region
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListCoverage.html
     */
    toListCoverage() {
        return this.to('ListCoverage');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty detectors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListDetectors.html
     */
    toListDetectors() {
        return this.to('ListDetectors');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFilters.html
     */
    toListFilters() {
        return this.to('ListFilters');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListFindings.html
     */
    toListFindings() {
        return this.to('ListFindings');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty IPSets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListIPSets.html
     */
    toListIPSets() {
        return this.to('ListIPSets');
    }
    /**
     * Grants permission to retrieve a list of all of the GuardDuty membership invitations that were sent to an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListInvitations.html
     */
    toListInvitations() {
        return this.to('ListInvitations');
    }
    /**
     * Grants permission to retrieve a list of Malware Protection plans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListMalwareProtectionPlans.html
     */
    toListMalwareProtectionPlans() {
        return this.to('ListMalwareProtectionPlans');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty member accounts associated with an administrator account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListMembers.html
     */
    toListMembers() {
        return this.to('ListMembers');
    }
    /**
     * Grants permission to list details about the organization delegated administrator for GuardDuty
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListOrganizationAdminAccounts.html
     */
    toListOrganizationAdminAccounts() {
        return this.to('ListOrganizationAdminAccounts');
    }
    /**
     * Grants permission to retrieve a list of publishing destinations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListPublishingDestinations.html
     */
    toListPublishingDestinations() {
        return this.to('ListPublishingDestinations');
    }
    /**
     * Grants permission to retrieve a list of tags associated with a GuardDuty resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty ThreatEntitySets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListThreatEntitySets.html
     */
    toListThreatEntitySets() {
        return this.to('ListThreatEntitySets');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty ThreatIntelSets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListThreatIntelSets.html
     */
    toListThreatIntelSets() {
        return this.to('ListThreatIntelSets');
    }
    /**
     * Grants permission to retrieve a list of GuardDuty TrustedEntitySets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_ListTrustedEntitySets.html
     */
    toListTrustedEntitySets() {
        return this.to('ListTrustedEntitySets');
    }
    /**
     * Grants permission to send security telemetry for a specific GuardDuty account in a Region
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_SendSecurityTelemetry.html
     */
    toSendSecurityTelemetry() {
        return this.to('SendSecurityTelemetry');
    }
    /**
     * Grants permission to initiate a new malware scan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StartMalwareScan.html
     */
    toStartMalwareScan() {
        return this.to('StartMalwareScan');
    }
    /**
     * Grants permission to a GuardDuty administrator account to monitor findings from GuardDuty member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StartMonitoringMembers.html
     */
    toStartMonitoringMembers() {
        return this.to('StartMonitoringMembers');
    }
    /**
     * Grants permission to disable monitoring findings from member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_StopMonitoringMembers.html
     */
    toStopMonitoringMembers() {
        return this.to('StopMonitoringMembers');
    }
    /**
     * Grants permission to add tags to a GuardDuty resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to unarchive GuardDuty findings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UnarchiveFindings.html
     */
    toUnarchiveFindings() {
        return this.to('UnarchiveFindings');
    }
    /**
     * Grants permission to remove tags from a GuardDuty resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update GuardDuty detectors
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateDetector.html
     */
    toUpdateDetector() {
        return this.to('UpdateDetector');
    }
    /**
     * Grants permission to updates GuardDuty filters
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFilter.html
     */
    toUpdateFilter() {
        return this.to('UpdateFilter');
    }
    /**
     * Grants permission to update findings feedback to mark GuardDuty findings as useful or not useful
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateFindingsFeedback.html
     */
    toUpdateFindingsFeedback() {
        return this.to('UpdateFindingsFeedback');
    }
    /**
     * Grants permission to update GuardDuty IPSets
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:DeleteRolePolicy
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateIPSet.html
     */
    toUpdateIPSet() {
        return this.to('UpdateIPSet');
    }
    /**
     * Grants permission to update the Malware Protection plan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateMalwareProtectionPlan.html
     */
    toUpdateMalwareProtectionPlan() {
        return this.to('UpdateMalwareProtectionPlan');
    }
    /**
     * Grants permission to update the malware scan settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateMalwareScanSettings.html
     */
    toUpdateMalwareScanSettings() {
        return this.to('UpdateMalwareScanSettings');
    }
    /**
     * Grants permission to update which data sources are enabled for member accounts detectors
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateMemberDetectors.html
     */
    toUpdateMemberDetectors() {
        return this.to('UpdateMemberDetectors');
    }
    /**
     * Grants permission to update the delegated administrator configuration associated with a GuardDuty detector
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateOrganizationConfiguration.html
     */
    toUpdateOrganizationConfiguration() {
        return this.to('UpdateOrganizationConfiguration');
    }
    /**
     * Grants permission to update a publishing destination
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdatePublishingDestination.html
     */
    toUpdatePublishingDestination() {
        return this.to('UpdatePublishingDestination');
    }
    /**
     * Grants permission to update GuardDuty ThreatEntitySets
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateThreatEntitySet.html
     */
    toUpdateThreatEntitySet() {
        return this.to('UpdateThreatEntitySet');
    }
    /**
     * Grants permission to updates the GuardDuty ThreatIntelSets
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:DeleteRolePolicy
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateThreatIntelSet.html
     */
    toUpdateThreatIntelSet() {
        return this.to('UpdateThreatIntelSet');
    }
    /**
     * Grants permission to update GuardDuty TrustedEntitySets
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/guardduty/latest/APIReference/API_UpdateTrustedEntitySet.html
     */
    toUpdateTrustedEntitySet() {
        return this.to('UpdateTrustedEntitySet');
    }
    /**
     * Adds a resource of type detector to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetector(detectorId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}`);
    }
    /**
     * Adds a resource of type filter to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param filterName - Identifier for the filterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFilter(detectorId, filterName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}/filter/${filterName}`);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param iPSetId - Identifier for the iPSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIpset(detectorId, iPSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}/ipset/${iPSetId}`);
    }
    /**
     * Adds a resource of type threatintelset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param threatIntelSetId - Identifier for the threatIntelSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onThreatintelset(detectorId, threatIntelSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}/threatintelset/${threatIntelSetId}`);
    }
    /**
     * Adds a resource of type trustedentityset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param trustedEntitySetId - Identifier for the trustedEntitySetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrustedentityset(detectorId, trustedEntitySetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}/trustedentityset/${trustedEntitySetId}`);
    }
    /**
     * Adds a resource of type threatentityset to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param threatEntitySetId - Identifier for the threatEntitySetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onThreatentityset(detectorId, threatEntitySetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}/threatentityset/${threatEntitySetId}`);
    }
    /**
     * Adds a resource of type publishingDestination to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param detectorId - Identifier for the detectorId.
     * @param publishingDestinationId - Identifier for the publishingDestinationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onPublishingDestination(detectorId, publishingDestinationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:detector/${detectorId}/publishingDestination/${publishingDestinationId}`);
    }
    /**
     * Adds a resource of type malwareprotectionplan to the statement
     *
     * https://docs.aws.amazon.com/guardduty/latest/ug/guardduty_managing_access.html#guardduty-resources
     *
     * @param malwareProtectionPlanId - Identifier for the malwareProtectionPlanId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMalwareprotectionplan(malwareProtectionPlanId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:guardduty:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:malware-protection-plan/${malwareProtectionPlanId}`);
    }
    /**
     * Filters access by tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDetector()
     * - .toCreateFilter()
     * - .toCreateIPSet()
     * - .toCreateMalwareProtectionPlan()
     * - .toCreateThreatEntitySet()
     * - .toCreateThreatIntelSet()
     * - .toCreateTrustedEntitySet()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateDetector()
     * - .toCreateIPSet()
     * - .toCreateMalwareProtectionPlan()
     * - .toCreateTrustedEntitySet()
     *
     * Applies to resource types:
     * - detector
     * - filter
     * - ipset
     * - threatintelset
     * - trustedentityset
     * - threatentityset
     * - malwareprotectionplan
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDetector()
     * - .toCreateFilter()
     * - .toCreateIPSet()
     * - .toCreateMalwareProtectionPlan()
     * - .toCreateThreatEntitySet()
     * - .toCreateThreatIntelSet()
     * - .toCreateTrustedEntitySet()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [guardduty](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonguardduty.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'guardduty';
        this.accessLevelList = {
            Write: [
                'AcceptAdministratorInvitation',
                'AcceptInvitation',
                'ArchiveFindings',
                'CreateDetector',
                'CreateFilter',
                'CreateIPSet',
                'CreateMalwareProtectionPlan',
                'CreateMembers',
                'CreatePublishingDestination',
                'CreateSampleFindings',
                'CreateThreatEntitySet',
                'CreateThreatIntelSet',
                'CreateTrustedEntitySet',
                'DeclineInvitations',
                'DeleteDetector',
                'DeleteFilter',
                'DeleteIPSet',
                'DeleteInvitations',
                'DeleteMalwareProtectionPlan',
                'DeleteMembers',
                'DeletePublishingDestination',
                'DeleteThreatEntitySet',
                'DeleteThreatIntelSet',
                'DeleteTrustedEntitySet',
                'DisableOrganizationAdminAccount',
                'DisassociateFromAdministratorAccount',
                'DisassociateFromMasterAccount',
                'DisassociateMembers',
                'EnableOrganizationAdminAccount',
                'InviteMembers',
                'SendSecurityTelemetry',
                'StartMalwareScan',
                'StartMonitoringMembers',
                'StopMonitoringMembers',
                'UnarchiveFindings',
                'UpdateDetector',
                'UpdateFilter',
                'UpdateFindingsFeedback',
                'UpdateIPSet',
                'UpdateMalwareProtectionPlan',
                'UpdateMalwareScanSettings',
                'UpdateMemberDetectors',
                'UpdateOrganizationConfiguration',
                'UpdatePublishingDestination',
                'UpdateThreatEntitySet',
                'UpdateThreatIntelSet',
                'UpdateTrustedEntitySet'
            ],
            Read: [
                'DescribeMalwareScans',
                'DescribeOrganizationConfiguration',
                'DescribePublishingDestination',
                'GetAdministratorAccount',
                'GetCoverageStatistics',
                'GetDetector',
                'GetFilter',
                'GetFindings',
                'GetFindingsStatistics',
                'GetIPSet',
                'GetInvitationsCount',
                'GetMalwareProtectionPlan',
                'GetMalwareScanSettings',
                'GetMasterAccount',
                'GetMemberDetectors',
                'GetMembers',
                'GetOrganizationStatistics',
                'GetRemainingFreeTrialDays',
                'GetThreatEntitySet',
                'GetThreatIntelSet',
                'GetTrustedEntitySet',
                'GetUsageStatistics',
                'ListTagsForResource'
            ],
            List: [
                'ListCoverage',
                'ListDetectors',
                'ListFilters',
                'ListFindings',
                'ListIPSets',
                'ListInvitations',
                'ListMalwareProtectionPlans',
                'ListMembers',
                'ListOrganizationAdminAccounts',
                'ListPublishingDestinations',
                'ListThreatEntitySets',
                'ListThreatIntelSets',
                'ListTrustedEntitySets'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Guardduty = Guardduty;
//# sourceMappingURL=data:application/json;base64,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