import { AccessLevelList } from '../../shared/access-level';
import { PolicyStatement, Operator } from '../../shared';
import { aws_iam as iam } from "aws-cdk-lib";
/**
 * Statement provider for service [healthlake](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awshealthlake.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Healthlake extends PolicyStatement {
    servicePrefix: string;
    /**
     * Grants permission to cancel an on going FHIR Export job with Delete
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/export-datastore-rest.html
     */
    toCancelFHIRExportJobWithDelete(): this;
    /**
     * Grants permission to create a datastore that can ingest and export FHIR data
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_CreateFHIRDatastore.html
     */
    toCreateFHIRDatastore(): this;
    /**
     * Grants permission to create resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toCreateResource(): this;
    /**
     * Grants permission to delete a datastore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_DeleteFHIRDatastore.html
     */
    toDeleteFHIRDatastore(): this;
    /**
     * Grants permission to delete resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toDeleteResource(): this;
    /**
     * Grants permission to get the properties associated with the FHIR datastore, including the datastore ID, datastore ARN, datastore name, datastore status, created at, datastore type version, and datastore endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_DescribeFHIRDatastore.html
     */
    toDescribeFHIRDatastore(): this;
    /**
     * Grants permission to display the properties of a FHIR export job, including the ID, ARN, name, and the status of the datastore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_DescribeFHIRExportJob.html
     */
    toDescribeFHIRExportJob(): this;
    /**
     * Grants permission to display the properties of a FHIR export job, including the ID, ARN, name, and the status of the datastore with Get
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/export-datastore-rest.html
     */
    toDescribeFHIRExportJobWithGet(): this;
    /**
     * Grants permission to display the properties of a FHIR import job, including the ID, ARN, name, and the status of the datastore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_DescribeFHIRImportJob.html
     */
    toDescribeFHIRImportJob(): this;
    /**
     * Grants permission to get the capabilities of a FHIR datastore
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toGetCapabilities(): this;
    /**
     * Grants permission to access exported files from a FHIR Export job initiated with Get
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/export-datastore-rest.html
     */
    toGetExportedFile(): this;
    /**
     * Grants permission to read resource history
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toGetHistoryByResourceId(): this;
    /**
     * Grants permission to list all FHIR datastores that are in the user's account, regardless of datastore status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_ListFHIRDatastores.html
     */
    toListFHIRDatastores(): this;
    /**
     * Grants permission to get a list of export jobs for the specified datastore
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_ListFHIRExportJobs.html
     */
    toListFHIRExportJobs(): this;
    /**
     * Grants permission to get a list of import jobs for the specified datastore
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_ListFHIRImportJobs.html
     */
    toListFHIRImportJobs(): this;
    /**
     * Grants permission to get a list of tags for the specified datastore
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to bundle multiple resource operations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toProcessBundle(): this;
    /**
     * Grants permission to read resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toReadResource(): this;
    /**
     * Grants permission to search all resources related to a patient
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/search-healthlake.html
     */
    toSearchEverything(): this;
    /**
     * Grants permission to search resources with GET method
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/search-healthlake.html
     */
    toSearchWithGet(): this;
    /**
     * Grants permission to search resources with POST method
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/search-healthlake.html
     */
    toSearchWithPost(): this;
    /**
     * Grants permission to begin a FHIR Export job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_StartFHIRExportJob.html
     */
    toStartFHIRExportJob(): this;
    /**
     * Grants permission to begin a FHIR Export job with Get
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/export-datastore-rest.html
     */
    toStartFHIRExportJobWithGet(): this;
    /**
     * Grants permission to begin a FHIR Export job with Post
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/export-datastore-rest.html
     */
    toStartFHIRExportJobWithPost(): this;
    /**
     * Grants permission to begin a FHIR Import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_StartFHIRImportJob.html
     */
    toStartFHIRImportJob(): this;
    /**
     * Grants permission to add tags to a datastore
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to remove tags associated with a datastore
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toUpdateResource(): this;
    /**
     * Grants permission to read version of a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/healthlake/latest/devguide/crud-healthlake.html
     */
    toVersionReadResource(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type datastore to the statement
     *
     * https://docs.aws.amazon.com/healthlake/latest/APIReference/API_DatastoreProperties.html
     *
     * @param datastoreId - Identifier for the datastoreId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDatastore(datastoreId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateFHIRDatastore()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * Applies to resource types:
     * - datastore
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey: string, value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateFHIRDatastore()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value: string | string[], operator?: Operator | string): this;
    /**
     * Statement provider for service [healthlake](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awshealthlake.html).
     *
     */
    constructor(props?: iam.PolicyStatementProps);
}
