"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sso = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [sso](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiamidentitycenter.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sso extends shared_1.PolicyStatement {
    /**
     * Grants permission to connect a directory to be used by AWS IAM Identity Center
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:AuthorizeApplication
     * - identitystore:CreateIdentityStore
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAssociateDirectory() {
        return this.to('AssociateDirectory');
    }
    /**
     * Grants permission to create an association between a directory user or group and a profile
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAssociateProfile() {
        return this.to('AssociateProfile');
    }
    /**
     * Grants permission to attach a customer managed policy reference to a permission set
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_AttachCustomerManagedPolicyReferenceToPermissionSet.html
     */
    toAttachCustomerManagedPolicyReferenceToPermissionSet() {
        return this.to('AttachCustomerManagedPolicyReferenceToPermissionSet');
    }
    /**
     * Grants permission to attach an AWS managed policy to a permission set
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_AttachManagedPolicyToPermissionSet.html
     */
    toAttachManagedPolicyToPermissionSet() {
        return this.to('AttachManagedPolicyToPermissionSet');
    }
    /**
     * Grants permission to assign access to a Principal for a specified AWS account using a specified permission set
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateAccountAssignment.html
     */
    toCreateAccountAssignment() {
        return this.to('CreateAccountAssignment');
    }
    /**
     * Grants permission to create an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        return this.to('CreateApplication');
    }
    /**
     * Grants permission to create an application assignment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateApplicationAssignment.html
     */
    toCreateApplicationAssignment() {
        return this.to('CreateApplicationAssignment');
    }
    /**
     * Grants permission to add an application instance to AWS IAM Identity Center
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateApplicationInstance() {
        return this.to('CreateApplicationInstance');
    }
    /**
     * Grants permission to add a new certificate for an application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateApplicationInstanceCertificate() {
        return this.to('CreateApplicationInstanceCertificate');
    }
    /**
     * Grants permission to create an identity center instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - identitystore:CreateIdentityStore
     * - organizations:DescribeOrganization
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateInstance.html
     */
    toCreateInstance() {
        return this.to('CreateInstance');
    }
    /**
     * Grants permission to enable the instance for ABAC and specify the attributes
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateRole
     * - iam:DeleteRole
     * - iam:DeleteRolePolicy
     * - iam:DetachRolePolicy
     * - iam:GetRole
     * - iam:ListAttachedRolePolicies
     * - iam:ListRolePolicies
     * - iam:PutRolePolicy
     * - iam:UpdateAssumeRolePolicy
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateInstanceAccessControlAttributeConfiguration.html
     */
    toCreateInstanceAccessControlAttributeConfiguration() {
        return this.to('CreateInstanceAccessControlAttributeConfiguration');
    }
    /**
     * Grants permission to add a managed application instance to AWS IAM Identity Center
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateManagedApplicationInstance() {
        return this.to('CreateManagedApplicationInstance');
    }
    /**
     * Grants permission to create a permission set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreatePermissionSet.html
     */
    toCreatePermissionSet() {
        return this.to('CreatePermissionSet');
    }
    /**
     * Grants permission to create a profile for an application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateProfile() {
        return this.to('CreateProfile');
    }
    /**
     * Grants permission to create a federation trust in a target account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateTrust() {
        return this.to('CreateTrust');
    }
    /**
     * Grants permission to create a trusted token issuer for an instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_CreateTrustedTokenIssuer.html
     */
    toCreateTrustedTokenIssuer() {
        return this.to('CreateTrustedTokenIssuer');
    }
    /**
     * Grants permission to delete a Principal's access from a specified AWS account using a specified permission set
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteAccountAssignment.html
     */
    toDeleteAccountAssignment() {
        return this.to('DeleteAccountAssignment');
    }
    /**
     * Grants permission to delete an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        return this.to('DeleteApplication');
    }
    /**
     * Grants permission to delete an access scope to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteApplicationAccessScope.html
     */
    toDeleteApplicationAccessScope() {
        return this.to('DeleteApplicationAccessScope');
    }
    /**
     * Grants permission to delete an application assignment
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteApplicationAssignment.html
     */
    toDeleteApplicationAssignment() {
        return this.to('DeleteApplicationAssignment');
    }
    /**
     * Grants permission to delete an authentication method to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteApplicationAuthenticationMethod.html
     */
    toDeleteApplicationAuthenticationMethod() {
        return this.to('DeleteApplicationAuthenticationMethod');
    }
    /**
     * Grants permission to delete a grant from an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteApplicationGrant.html
     */
    toDeleteApplicationGrant() {
        return this.to('DeleteApplicationGrant');
    }
    /**
     * Grants permission to delete the application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteApplicationInstance() {
        return this.to('DeleteApplicationInstance');
    }
    /**
     * Grants permission to delete an inactive or expired certificate from the application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteApplicationInstanceCertificate() {
        return this.to('DeleteApplicationInstanceCertificate');
    }
    /**
     * Grants permission to delete the inline policy from a specified permission set
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteInlinePolicyFromPermissionSet.html
     */
    toDeleteInlinePolicyFromPermissionSet() {
        return this.to('DeleteInlinePolicyFromPermissionSet');
    }
    /**
     * Grants permission to delete an identity center instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:DeleteIdentityStore
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteInstance.html
     */
    toDeleteInstance() {
        return this.to('DeleteInstance');
    }
    /**
     * Grants permission to disable ABAC and remove the attributes list for the instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteInstanceAccessControlAttributeConfiguration.html
     */
    toDeleteInstanceAccessControlAttributeConfiguration() {
        return this.to('DeleteInstanceAccessControlAttributeConfiguration');
    }
    /**
     * Grants permission to delete the managed application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteManagedApplicationInstance() {
        return this.to('DeleteManagedApplicationInstance');
    }
    /**
     * Grants permission to delete a permission set
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeletePermissionSet.html
     */
    toDeletePermissionSet() {
        return this.to('DeletePermissionSet');
    }
    /**
     * Grants permission to remove permissions boundary from a permission set
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeletePermissionsBoundaryFromPermissionSet.html
     */
    toDeletePermissionsBoundaryFromPermissionSet() {
        return this.to('DeletePermissionsBoundaryFromPermissionSet');
    }
    /**
     * Grants permission to delete the profile for an application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteProfile() {
        return this.to('DeleteProfile');
    }
    /**
     * Grants permission to delete a trusted token issuer for an instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DeleteTrustedTokenIssuer.html
     */
    toDeleteTrustedTokenIssuer() {
        return this.to('DeleteTrustedTokenIssuer');
    }
    /**
     * Grants permission to describe the status of the assignment creation request
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeAccountAssignmentCreationStatus.html
     */
    toDescribeAccountAssignmentCreationStatus() {
        return this.to('DescribeAccountAssignmentCreationStatus');
    }
    /**
     * Grants permission to describe the status of an assignment deletion request
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeAccountAssignmentDeletionStatus.html
     */
    toDescribeAccountAssignmentDeletionStatus() {
        return this.to('DescribeAccountAssignmentDeletionStatus');
    }
    /**
     * Grants permission to obtain information about an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeApplication.html
     */
    toDescribeApplication() {
        return this.to('DescribeApplication');
    }
    /**
     * Grants permission to retrieve an application assignment
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeApplicationAssignment.html
     */
    toDescribeApplicationAssignment() {
        return this.to('DescribeApplicationAssignment');
    }
    /**
     * Grants permission to describe an application provider
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeApplicationProvider.html
     */
    toDescribeApplicationProvider() {
        return this.to('DescribeApplicationProvider');
    }
    /**
     * Grants permission to obtain information about an identity center instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeInstance.html
     */
    toDescribeInstance() {
        return this.to('DescribeInstance');
    }
    /**
     * Grants permission to get the list of attributes used by the instance for ABAC
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeInstanceAccessControlAttributeConfiguration.html
     */
    toDescribeInstanceAccessControlAttributeConfiguration() {
        return this.to('DescribeInstanceAccessControlAttributeConfiguration');
    }
    /**
     * Grants permission to describe a permission set
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribePermissionSet.html
     */
    toDescribePermissionSet() {
        return this.to('DescribePermissionSet');
    }
    /**
     * Grants permission to describe the status for the given Permission Set Provisioning request
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribePermissionSetProvisioningStatus.html
     */
    toDescribePermissionSetProvisioningStatus() {
        return this.to('DescribePermissionSetProvisioningStatus');
    }
    /**
     * Grants permission to obtain the regions where your organization has enabled AWS IAM Identity Center
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeRegisteredRegions() {
        return this.to('DescribeRegisteredRegions');
    }
    /**
     * Grants permission to describe a trusted token issuer for an instance
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DescribeTrustedTokenIssuer.html
     */
    toDescribeTrustedTokenIssuer() {
        return this.to('DescribeTrustedTokenIssuer');
    }
    /**
     * Grants permission to detach a customer managed policy reference from a permission set
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DetachCustomerManagedPolicyReferenceFromPermissionSet.html
     */
    toDetachCustomerManagedPolicyReferenceFromPermissionSet() {
        return this.to('DetachCustomerManagedPolicyReferenceFromPermissionSet');
    }
    /**
     * Grants permission to detach the attached AWS managed policy from the specified permission set
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_DetachManagedPolicyFromPermissionSet.html
     */
    toDetachManagedPolicyFromPermissionSet() {
        return this.to('DetachManagedPolicyFromPermissionSet');
    }
    /**
     * Grants permission to disassociate a directory to be used by AWS IAM Identity Center
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:UnauthorizeApplication
     * - identitystore:DeleteIdentityStore
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisassociateDirectory() {
        return this.to('DisassociateDirectory');
    }
    /**
     * Grants permission to disassociate a directory user or group from a profile
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisassociateProfile() {
        return this.to('DisassociateProfile');
    }
    /**
     * Grants permission to get an access scope to an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetApplicationAccessScope.html
     */
    toGetApplicationAccessScope() {
        return this.to('GetApplicationAccessScope');
    }
    /**
     * Grants permission to read assignment configurations for an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetApplicationAssignmentConfiguration.html
     */
    toGetApplicationAssignmentConfiguration() {
        return this.to('GetApplicationAssignmentConfiguration');
    }
    /**
     * Grants permission to get an authentication method to an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetApplicationAuthenticationMethod.html
     */
    toGetApplicationAuthenticationMethod() {
        return this.to('GetApplicationAuthenticationMethod');
    }
    /**
     * Grants permission to obtain details about a grant belonging to an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetApplicationGrant.html
     */
    toGetApplicationGrant() {
        return this.to('GetApplicationGrant');
    }
    /**
     * Grants permission to retrieve details for an application instance
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetApplicationInstance() {
        return this.to('GetApplicationInstance');
    }
    /**
     * Grants permission to get session configuration for an application
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetApplicationSessionConfiguration.html
     */
    toGetApplicationSessionConfiguration() {
        return this.to('GetApplicationSessionConfiguration');
    }
    /**
     * Grants permission to retrieve application template details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetApplicationTemplate() {
        return this.to('GetApplicationTemplate');
    }
    /**
     * Grants permission to obtain the inline policy assigned to the permission set
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetInlinePolicyForPermissionSet.html
     */
    toGetInlinePolicyForPermissionSet() {
        return this.to('GetInlinePolicyForPermissionSet');
    }
    /**
     * Grants permission to retrieve details for an application instance
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetManagedApplicationInstance() {
        return this.to('GetManagedApplicationInstance');
    }
    /**
     * Grants permission to retrieve Mfa Device Management settings for the directory
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetMfaDeviceManagementForDirectory() {
        return this.to('GetMfaDeviceManagementForDirectory');
    }
    /**
     * Grants permission to retrieve details of a permission set
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetPermissionSet() {
        return this.to('GetPermissionSet');
    }
    /**
     * Grants permission to get permissions boundary for a permission set
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_GetPermissionsBoundaryForPermissionSet.html
     */
    toGetPermissionsBoundaryForPermissionSet() {
        return this.to('GetPermissionsBoundaryForPermissionSet');
    }
    /**
     * Grants permission to retrieve a profile for an application instance
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetProfile() {
        return this.to('GetProfile');
    }
    /**
     * Grants permission to check if AWS IAM Identity Center is enabled
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSSOStatus() {
        return this.to('GetSSOStatus');
    }
    /**
     * Grants permission to retrieve shared configuration for the current SSO instance
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSharedSsoConfiguration() {
        return this.to('GetSharedSsoConfiguration');
    }
    /**
     * Grants permission to retrieve configuration for the current SSO instance
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSsoConfiguration() {
        return this.to('GetSsoConfiguration');
    }
    /**
     * Grants permission to retrieve the federation trust in a target account
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetTrust() {
        return this.to('GetTrust');
    }
    /**
     * Grants permission to update the application instance by uploading an application SAML metadata file provided by the service provider
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toImportApplicationInstanceServiceProviderMetadata() {
        return this.to('ImportApplicationInstanceServiceProviderMetadata');
    }
    /**
     * Grants permission to list the status of the AWS account assignment creation requests for a specified SSO instance
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListAccountAssignmentCreationStatus.html
     */
    toListAccountAssignmentCreationStatus() {
        return this.to('ListAccountAssignmentCreationStatus');
    }
    /**
     * Grants permission to list the status of the AWS account assignment deletion requests for a specified SSO instance
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListAccountAssignmentDeletionStatus.html
     */
    toListAccountAssignmentDeletionStatus() {
        return this.to('ListAccountAssignmentDeletionStatus');
    }
    /**
     * Grants permission to list the assignee of the specified AWS account with the specified permission set
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListAccountAssignments.html
     */
    toListAccountAssignments() {
        return this.to('ListAccountAssignments');
    }
    /**
     * Grants permission to list accounts assigned to user or group
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListAccountAssignmentsForPrincipal.html
     */
    toListAccountAssignmentsForPrincipal() {
        return this.to('ListAccountAssignmentsForPrincipal');
    }
    /**
     * Grants permission to list all the AWS accounts where the specified permission set is provisioned
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListAccountsForProvisionedPermissionSet.html
     */
    toListAccountsForProvisionedPermissionSet() {
        return this.to('ListAccountsForProvisionedPermissionSet');
    }
    /**
     * Grants permission to list access scopes to an application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationAccessScopes.html
     */
    toListApplicationAccessScopes() {
        return this.to('ListApplicationAccessScopes');
    }
    /**
     * Grants permission to list application assignments
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationAssignments.html
     */
    toListApplicationAssignments() {
        return this.to('ListApplicationAssignments');
    }
    /**
     * Grants permission to list applications assigned to user or group
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationAssignmentsForPrincipal.html
     */
    toListApplicationAssignmentsForPrincipal() {
        return this.to('ListApplicationAssignmentsForPrincipal');
    }
    /**
     * Grants permission to list authentication methods to an application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationAuthenticationMethods.html
     */
    toListApplicationAuthenticationMethods() {
        return this.to('ListApplicationAuthenticationMethods');
    }
    /**
     * Grants permission to list grants from an application
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationGrants.html
     */
    toListApplicationGrants() {
        return this.to('ListApplicationGrants');
    }
    /**
     * Grants permission to retrieve all of the certificates for a given application instance
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationInstanceCertificates() {
        return this.to('ListApplicationInstanceCertificates');
    }
    /**
     * Grants permission to retrieve all application instances
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     * - sso:GetApplicationInstance
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationInstances() {
        return this.to('ListApplicationInstances');
    }
    /**
     * Grants permission to list application providers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplicationProviders.html
     */
    toListApplicationProviders() {
        return this.to('ListApplicationProviders');
    }
    /**
     * Grants permission to retrieve all supported application templates
     *
     * Access Level: List
     *
     * Dependent actions:
     * - sso:GetApplicationTemplate
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationTemplates() {
        return this.to('ListApplicationTemplates');
    }
    /**
     * Grants permission to retrieve all applications associated with the instance of IAM Identity Center
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListApplications.html
     */
    toListApplications() {
        return this.to('ListApplications');
    }
    /**
     * Grants permission to list the customer managed policy references that are attached to a permission set
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListCustomerManagedPolicyReferencesInPermissionSet.html
     */
    toListCustomerManagedPolicyReferencesInPermissionSet() {
        return this.to('ListCustomerManagedPolicyReferencesInPermissionSet');
    }
    /**
     * Grants permission to retrieve details about the directory connected to AWS IAM Identity Center
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListDirectoryAssociations() {
        return this.to('ListDirectoryAssociations');
    }
    /**
     * Grants permission to list the SSO Instances that the caller has access to
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListInstances.html
     */
    toListInstances() {
        return this.to('ListInstances');
    }
    /**
     * Grants permission to list the AWS managed policies that are attached to a specified permission set
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListManagedPoliciesInPermissionSet.html
     */
    toListManagedPoliciesInPermissionSet() {
        return this.to('ListManagedPoliciesInPermissionSet');
    }
    /**
     * Grants permission to list the status of the Permission Set Provisioning requests for a specified SSO instance
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListPermissionSetProvisioningStatus.html
     */
    toListPermissionSetProvisioningStatus() {
        return this.to('ListPermissionSetProvisioningStatus');
    }
    /**
     * Grants permission to retrieve all permission sets
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListPermissionSets.html
     */
    toListPermissionSets() {
        return this.to('ListPermissionSets');
    }
    /**
     * Grants permission to list all the permission sets that are provisioned to a specified AWS account
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListPermissionSetsProvisionedToAccount.html
     */
    toListPermissionSetsProvisionedToAccount() {
        return this.to('ListPermissionSetsProvisionedToAccount');
    }
    /**
     * Grants permission to retrieve the directory user or group associated with the profile
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProfileAssociations() {
        return this.to('ListProfileAssociations');
    }
    /**
     * Grants permission to retrieve all profiles for an application instance
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     * - sso:GetProfile
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProfiles() {
        return this.to('ListProfiles');
    }
    /**
     * Grants permission to list the tags that are attached to a specified resource
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list trusted token issuers for an instance
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ListTrustedTokenIssuers.html
     */
    toListTrustedTokenIssuers() {
        return this.to('ListTrustedTokenIssuers');
    }
    /**
     * Grants permission to provision a specified permission set to the specified target
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ProvisionPermissionSet.html
     */
    toProvisionPermissionSet() {
        return this.to('ProvisionPermissionSet');
    }
    /**
     * Grants permission to create/update an access scope to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutApplicationAccessScope.html
     */
    toPutApplicationAccessScope() {
        return this.to('PutApplicationAccessScope');
    }
    /**
     * Grants permission to add assignment configurations to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutApplicationAssignmentConfiguration.html
     */
    toPutApplicationAssignmentConfiguration() {
        return this.to('PutApplicationAssignmentConfiguration');
    }
    /**
     * Grants permission to create/update an authentication method to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutApplicationAuthenticationMethod.html
     */
    toPutApplicationAuthenticationMethod() {
        return this.to('PutApplicationAuthenticationMethod');
    }
    /**
     * Grants permission to create/update a grant to an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutApplicationGrant.html
     */
    toPutApplicationGrant() {
        return this.to('PutApplicationGrant');
    }
    /**
     * Grants permission to put session configuration for an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutApplicationSessionConfiguration.html
     */
    toPutApplicationSessionConfiguration() {
        return this.to('PutApplicationSessionConfiguration');
    }
    /**
     * Grants permission to attach an IAM inline policy to a permission set
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutInlinePolicyToPermissionSet.html
     */
    toPutInlinePolicyToPermissionSet() {
        return this.to('PutInlinePolicyToPermissionSet');
    }
    /**
     * Grants permission to put Mfa Device Management settings for the directory
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutMfaDeviceManagementForDirectory() {
        return this.to('PutMfaDeviceManagementForDirectory');
    }
    /**
     * Grants permission to add permissions boundary to a permission set
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_PutPermissionsBoundaryToPermissionSet.html
     */
    toPutPermissionsBoundaryToPermissionSet() {
        return this.to('PutPermissionsBoundaryToPermissionSet');
    }
    /**
     * Grants permission to add a policy to a permission set
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutPermissionsPolicy() {
        return this.to('PutPermissionsPolicy');
    }
    /**
     * Grants permission to search for groups within the associated directory
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toSearchGroups() {
        return this.to('SearchGroups');
    }
    /**
     * Grants permission to search for users within the associated directory
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toSearchUsers() {
        return this.to('SearchUsers');
    }
    /**
     * Grants permission to initialize AWS IAM Identity Center
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:Encrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     * - organizations:DescribeOrganization
     * - organizations:EnableAWSServiceAccess
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toStartSSO() {
        return this.to('StartSSO');
    }
    /**
     * Grants permission to associate a set of tags with a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to disassociate a set of tags from a specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an application
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifApplicationAccount()
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        return this.to('UpdateApplication');
    }
    /**
     * Grants permission to set a certificate as the active one for this application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceActiveCertificate() {
        return this.to('UpdateApplicationInstanceActiveCertificate');
    }
    /**
     * Grants permission to update display data of an application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceDisplayData() {
        return this.to('UpdateApplicationInstanceDisplayData');
    }
    /**
     * Grants permission to update federation response configuration for the application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceResponseConfiguration() {
        return this.to('UpdateApplicationInstanceResponseConfiguration');
    }
    /**
     * Grants permission to update federation response schema configuration for the application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceResponseSchemaConfiguration() {
        return this.to('UpdateApplicationInstanceResponseSchemaConfiguration');
    }
    /**
     * Grants permission to update security details for the application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceSecurityConfiguration() {
        return this.to('UpdateApplicationInstanceSecurityConfiguration');
    }
    /**
     * Grants permission to update service provider related configuration for the application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceServiceProviderConfiguration() {
        return this.to('UpdateApplicationInstanceServiceProviderConfiguration');
    }
    /**
     * Grants permission to update the status of an application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceStatus() {
        return this.to('UpdateApplicationInstanceStatus');
    }
    /**
     * Grants permission to update an identity center instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - identitystore:UpdateIdentityStore
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:Encrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UpdateInstance.html
     */
    toUpdateInstance() {
        return this.to('UpdateInstance');
    }
    /**
     * Grants permission to update the attributes to use with the instance for ABAC
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UpdateInstanceAccessControlAttributeConfiguration.html
     */
    toUpdateInstanceAccessControlAttributeConfiguration() {
        return this.to('UpdateInstanceAccessControlAttributeConfiguration');
    }
    /**
     * Grants permission to update the status of a managed application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateManagedApplicationInstanceStatus() {
        return this.to('UpdateManagedApplicationInstanceStatus');
    }
    /**
     * Grants permission to update the permission set
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UpdatePermissionSet.html
     */
    toUpdatePermissionSet() {
        return this.to('UpdatePermissionSet');
    }
    /**
     * Grants permission to update the profile for an application instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateProfile() {
        return this.to('UpdateProfile');
    }
    /**
     * Grants permission to update the configuration for the current SSO instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateSSOConfiguration() {
        return this.to('UpdateSSOConfiguration');
    }
    /**
     * Grants permission to update the federation trust in a target account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateTrust() {
        return this.to('UpdateTrust');
    }
    /**
     * Grants permission to update a trusted token issuer for an instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_UpdateTrustedTokenIssuer.html
     */
    toUpdateTrustedTokenIssuer() {
        return this.to('UpdateTrustedTokenIssuer');
    }
    /**
     * Adds a resource of type PermissionSet to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/permissionsetsconcept.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param permissionSetId - Identifier for the permissionSetId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPermissionSet(instanceId, permissionSetId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso:::permissionSet/${instanceId}/${permissionSetId}`);
    }
    /**
     * Adds a resource of type Account to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/manage-your-accounts.html
     *
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccount(accountId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso:::account/${accountId ?? this.defaultAccount}`);
    }
    /**
     * Adds a resource of type Instance to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_InstanceMetadata.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstance(instanceId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso:::instance/${instanceId}`);
    }
    /**
     * Adds a resource of type Application to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_Application.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param applicationId - Identifier for the applicationId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifApplicationAccount()
     */
    onApplication(instanceId, applicationId, accountId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso::${accountId ?? this.defaultAccount}:application/${instanceId}/${applicationId}`);
    }
    /**
     * Adds a resource of type TrustedTokenIssuer to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_TrustedTokenIssuerMetadata.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param trustedTokenIssuerId - Identifier for the trustedTokenIssuerId.
     * @param accountId - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrustedTokenIssuer(instanceId, trustedTokenIssuerId, accountId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso::${accountId ?? this.defaultAccount}:trustedTokenIssuer/${instanceId}/${trustedTokenIssuerId}`);
    }
    /**
     * Adds a resource of type ApplicationProvider to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/APIReference/API_ApplicationProvider.html
     *
     * @param applicationProviderId - Identifier for the applicationProviderId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApplicationProvider(applicationProviderId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sso::aws:applicationProvider/${applicationProviderId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/tagging.html
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateInstance()
     * - .toCreatePermissionSet()
     * - .toCreateTrustedTokenIssuer()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/tagging.html
     *
     * Applies to resource types:
     * - PermissionSet
     * - Instance
     * - Application
     * - TrustedTokenIssuer
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/tagging.html
     *
     * Applies to actions:
     * - .toCreateApplication()
     * - .toCreateInstance()
     * - .toCreatePermissionSet()
     * - .toCreateTrustedTokenIssuer()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the account which creates the application. This condition key is not supported for customer managed SAML applications
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/API_Application.html
     *
     * Applies to actions:
     * - .toCreateApplicationAssignment()
     * - .toDeleteApplication()
     * - .toDeleteApplicationAccessScope()
     * - .toDeleteApplicationAssignment()
     * - .toDeleteApplicationAuthenticationMethod()
     * - .toDeleteApplicationGrant()
     * - .toDescribeApplication()
     * - .toDescribeApplicationAssignment()
     * - .toGetApplicationAccessScope()
     * - .toGetApplicationAssignmentConfiguration()
     * - .toGetApplicationAuthenticationMethod()
     * - .toGetApplicationGrant()
     * - .toGetApplicationSessionConfiguration()
     * - .toListApplicationAccessScopes()
     * - .toListApplicationAssignments()
     * - .toListApplicationAssignmentsForPrincipal()
     * - .toListApplicationAuthenticationMethods()
     * - .toListApplicationGrants()
     * - .toPutApplicationAccessScope()
     * - .toPutApplicationAssignmentConfiguration()
     * - .toPutApplicationAuthenticationMethod()
     * - .toPutApplicationGrant()
     * - .toPutApplicationSessionConfiguration()
     * - .toUpdateApplication()
     *
     * Applies to resource types:
     * - Application
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifApplicationAccount(value, operator) {
        return this.if(`ApplicationAccount`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [sso](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiamidentitycenter.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'sso';
        this.accessLevelList = {
            Write: [
                'AssociateDirectory',
                'AssociateProfile',
                'CreateAccountAssignment',
                'CreateApplication',
                'CreateApplicationAssignment',
                'CreateApplicationInstance',
                'CreateApplicationInstanceCertificate',
                'CreateInstance',
                'CreateInstanceAccessControlAttributeConfiguration',
                'CreateManagedApplicationInstance',
                'CreatePermissionSet',
                'CreateProfile',
                'CreateTrust',
                'CreateTrustedTokenIssuer',
                'DeleteAccountAssignment',
                'DeleteApplication',
                'DeleteApplicationAccessScope',
                'DeleteApplicationAssignment',
                'DeleteApplicationAuthenticationMethod',
                'DeleteApplicationGrant',
                'DeleteApplicationInstance',
                'DeleteApplicationInstanceCertificate',
                'DeleteInlinePolicyFromPermissionSet',
                'DeleteInstance',
                'DeleteInstanceAccessControlAttributeConfiguration',
                'DeleteManagedApplicationInstance',
                'DeletePermissionSet',
                'DeleteProfile',
                'DeleteTrustedTokenIssuer',
                'DisassociateDirectory',
                'DisassociateProfile',
                'ImportApplicationInstanceServiceProviderMetadata',
                'ProvisionPermissionSet',
                'PutApplicationAccessScope',
                'PutApplicationAssignmentConfiguration',
                'PutApplicationAuthenticationMethod',
                'PutApplicationGrant',
                'PutApplicationSessionConfiguration',
                'PutInlinePolicyToPermissionSet',
                'PutMfaDeviceManagementForDirectory',
                'StartSSO',
                'UpdateApplication',
                'UpdateApplicationInstanceActiveCertificate',
                'UpdateApplicationInstanceDisplayData',
                'UpdateApplicationInstanceResponseConfiguration',
                'UpdateApplicationInstanceResponseSchemaConfiguration',
                'UpdateApplicationInstanceSecurityConfiguration',
                'UpdateApplicationInstanceServiceProviderConfiguration',
                'UpdateApplicationInstanceStatus',
                'UpdateInstance',
                'UpdateInstanceAccessControlAttributeConfiguration',
                'UpdateManagedApplicationInstanceStatus',
                'UpdateProfile',
                'UpdateSSOConfiguration',
                'UpdateTrust',
                'UpdateTrustedTokenIssuer'
            ],
            'Permissions management': [
                'AttachCustomerManagedPolicyReferenceToPermissionSet',
                'AttachManagedPolicyToPermissionSet',
                'DeletePermissionsBoundaryFromPermissionSet',
                'DetachCustomerManagedPolicyReferenceFromPermissionSet',
                'DetachManagedPolicyFromPermissionSet',
                'PutPermissionsBoundaryToPermissionSet',
                'PutPermissionsPolicy',
                'UpdatePermissionSet'
            ],
            Read: [
                'DescribeAccountAssignmentCreationStatus',
                'DescribeAccountAssignmentDeletionStatus',
                'DescribeApplication',
                'DescribeApplicationAssignment',
                'DescribeApplicationProvider',
                'DescribeInstance',
                'DescribeInstanceAccessControlAttributeConfiguration',
                'DescribePermissionSet',
                'DescribePermissionSetProvisioningStatus',
                'DescribeRegisteredRegions',
                'DescribeTrustedTokenIssuer',
                'GetApplicationAccessScope',
                'GetApplicationAssignmentConfiguration',
                'GetApplicationAuthenticationMethod',
                'GetApplicationGrant',
                'GetApplicationInstance',
                'GetApplicationSessionConfiguration',
                'GetApplicationTemplate',
                'GetInlinePolicyForPermissionSet',
                'GetManagedApplicationInstance',
                'GetMfaDeviceManagementForDirectory',
                'GetPermissionSet',
                'GetPermissionsBoundaryForPermissionSet',
                'GetProfile',
                'GetSSOStatus',
                'GetSharedSsoConfiguration',
                'GetSsoConfiguration',
                'GetTrust',
                'ListApplicationInstanceCertificates',
                'ListDirectoryAssociations',
                'ListProfileAssociations',
                'ListTagsForResource',
                'SearchGroups',
                'SearchUsers'
            ],
            List: [
                'ListAccountAssignmentCreationStatus',
                'ListAccountAssignmentDeletionStatus',
                'ListAccountAssignments',
                'ListAccountAssignmentsForPrincipal',
                'ListAccountsForProvisionedPermissionSet',
                'ListApplicationAccessScopes',
                'ListApplicationAssignments',
                'ListApplicationAssignmentsForPrincipal',
                'ListApplicationAuthenticationMethods',
                'ListApplicationGrants',
                'ListApplicationInstances',
                'ListApplicationProviders',
                'ListApplicationTemplates',
                'ListApplications',
                'ListCustomerManagedPolicyReferencesInPermissionSet',
                'ListInstances',
                'ListManagedPoliciesInPermissionSet',
                'ListPermissionSetProvisioningStatus',
                'ListPermissionSets',
                'ListPermissionSetsProvisionedToAccount',
                'ListProfiles',
                'ListTrustedTokenIssuers'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Sso = Sso;
//# sourceMappingURL=data:application/json;base64,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