"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Identitystore = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [identitystore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsidentitystore.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Identitystore extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a group in the specified IdentityStore
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_CreateGroup.html
     */
    toCreateGroup() {
        return this.to('CreateGroup');
    }
    /**
     * Grants permission to create a member to a group in the specified IdentityStore
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_CreateGroupMembership.html
     */
    toCreateGroupMembership() {
        return this.to('CreateGroupMembership');
    }
    /**
     * Grants permission to create a new IdentityStore in an AWS account
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:Encrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateIdentityStore() {
        return this.to('CreateIdentityStore');
    }
    /**
     * Grants permission to create a user in the specified IdentityStore
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to delete a group in the specified IdentityStore
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DeleteGroup.html
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to remove a member that is part of a group in the specified IdentityStore
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DeleteGroupMembership.html
     */
    toDeleteGroupMembership() {
        return this.to('DeleteGroupMembership');
    }
    /**
     * Grants permission to delete an IdentityStore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteIdentityStore() {
        return this.to('DeleteIdentityStore');
    }
    /**
     * Grants permission to delete a user in the specified IdentityStore
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to retrieve information about a group in the specified IdentityStore
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DescribeGroup.html
     */
    toDescribeGroup() {
        return this.to('DescribeGroup');
    }
    /**
     * Grants permission to retrieve information about a member that is part of a group in the specified IdentityStore
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DescribeGroupMembership.html
     */
    toDescribeGroupMembership() {
        return this.to('DescribeGroupMembership');
    }
    /**
     * Grants permission to retrieve information about user in the specified IdentityStore
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_DescribeUser.html
     */
    toDescribeUser() {
        return this.to('DescribeUser');
    }
    /**
     * Grants permission to retrieve ID information about group in the specified IdentityStore
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_GetGroupId.html
     */
    toGetGroupId() {
        return this.to('GetGroupId');
    }
    /**
     * Grants permission to retrieve ID information of a member which is part of a group in the specified IdentityStore
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_GetGroupMembershipId.html
     */
    toGetGroupMembershipId() {
        return this.to('GetGroupMembershipId');
    }
    /**
     * Grants permission to retrieves ID information about user in the specified IdentityStore
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_GetUserId.html
     */
    toGetUserId() {
        return this.to('GetUserId');
    }
    /**
     * Grants permission to check if a member is a part of groups in the specified IdentityStore
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_IsMemberInGroups.html
     */
    toIsMemberInGroups() {
        return this.to('IsMemberInGroups');
    }
    /**
     * Grants permission to retrieve all members that are part of a group in the specified IdentityStore
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_ListGroupMemberships.html
     */
    toListGroupMemberships() {
        return this.to('ListGroupMemberships');
    }
    /**
     * Grants permission to list groups of the target member in the specified IdentityStore
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_ListGroupMembershipsForMember.html
     */
    toListGroupMembershipsForMember() {
        return this.to('ListGroupMembershipsForMember');
    }
    /**
     * Grants permission to search for groups within the specified IdentityStore
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_ListGroups.html
     */
    toListGroups() {
        return this.to('ListGroups');
    }
    /**
     * Grants permission to search for users in the specified IdentityStore
     *
     * Access Level: List
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_ListUsers.html
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to update information about a group in the specified IdentityStore
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_UpdateGroup.html
     */
    toUpdateGroup() {
        return this.to('UpdateGroup');
    }
    /**
     * Grants permission to update the configuration of an IdentityStore
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     * - kms:DescribeKey
     * - kms:Encrypt
     * - kms:GenerateDataKeyWithoutPlaintext
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateIdentityStore() {
        return this.to('UpdateIdentityStore');
    }
    /**
     * Grants permission to update user information in the specified IdentityStore
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - kms:Decrypt
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Adds a resource of type Identitystore to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/
     *
     * @param identityStoreId - Identifier for the identityStoreId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onIdentitystore(identityStoreId, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore::${account ?? this.defaultAccount}:identitystore/${identityStoreId}`);
    }
    /**
     * Adds a resource of type User to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_User.html
     *
     * @param userId - Identifier for the userId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onUser(userId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::user/${userId}`);
    }
    /**
     * Adds a resource of type Group to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_Group.html
     *
     * @param groupId - Identifier for the groupId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGroup(groupId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::group/${groupId}`);
    }
    /**
     * Adds a resource of type GroupMembership to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_GroupMembership.html
     *
     * @param membershipId - Identifier for the membershipId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onGroupMembership(membershipId, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::membership/${membershipId}`);
    }
    /**
     * Adds a resource of type AllUsers to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_User.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAllUsers(resourceName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::user/${resourceName}`);
    }
    /**
     * Adds a resource of type AllGroups to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_Group.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAllGroups(resourceName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::group/${resourceName}`);
    }
    /**
     * Adds a resource of type AllGroupMemberships to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/IdentityStoreAPIReference/API_GroupMembership.html
     *
     * @param resourceName - Identifier for the resourceName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAllGroupMemberships(resourceName, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identitystore:::membership/${resourceName}`);
    }
    /**
     * Filters access by IAM Identity Center User ID
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/condition-context-keys-sts-idc.html#condition-keys-identity-store-user-id
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserId(value, operator) {
        return this.if(`UserId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [identitystore](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsidentitystore.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'identitystore';
        this.accessLevelList = {
            Write: [
                'CreateGroup',
                'CreateGroupMembership',
                'CreateIdentityStore',
                'CreateUser',
                'DeleteGroup',
                'DeleteGroupMembership',
                'DeleteIdentityStore',
                'DeleteUser',
                'UpdateGroup',
                'UpdateIdentityStore',
                'UpdateUser'
            ],
            Read: [
                'DescribeGroup',
                'DescribeGroupMembership',
                'DescribeUser',
                'GetGroupId',
                'GetGroupMembershipId',
                'GetUserId',
                'IsMemberInGroups'
            ],
            List: [
                'ListGroupMemberships',
                'ListGroupMembershipsForMember',
                'ListGroups',
                'ListUsers'
            ]
        };
    }
}
exports.Identitystore = Identitystore;
//# sourceMappingURL=data:application/json;base64,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