"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdentitySync = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [identity-sync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsidentitysync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class IdentitySync extends shared_1.PolicyStatement {
    /**
     * Grants permission to configure vended log delivery for a Sync Profile
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/logging-ad-sync-errors.html
     */
    toAllowVendedLogDeliveryForResource() {
        return this.to('AllowVendedLogDeliveryForResource');
    }
    /**
     * Grants permission to create a sync filter on the sync profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toCreateSyncFilter() {
        return this.to('CreateSyncFilter');
    }
    /**
     * Grants permission to create a sync profile for the identity source
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:AuthorizeApplication
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toCreateSyncProfile() {
        return this.to('CreateSyncProfile');
    }
    /**
     * Grants permission to create a sync target for the identity source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toCreateSyncTarget() {
        return this.to('CreateSyncTarget');
    }
    /**
     * Grants permission to delete a sync filter from the sync profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toDeleteSyncFilter() {
        return this.to('DeleteSyncFilter');
    }
    /**
     * Grants permission to delete a sync profile from the source
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ds:UnauthorizeApplication
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toDeleteSyncProfile() {
        return this.to('DeleteSyncProfile');
    }
    /**
     * Grants permission to delete a sync target from the source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toDeleteSyncTarget() {
        return this.to('DeleteSyncTarget');
    }
    /**
     * Grants permission to retrieve a sync profile by using a sync profile name
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toGetSyncProfile() {
        return this.to('GetSyncProfile');
    }
    /**
     * Grants permission to retrieve a sync target from the sync profile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toGetSyncTarget() {
        return this.to('GetSyncTarget');
    }
    /**
     * Grants permission to list the sync filters from the sync profile
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toListSyncFilters() {
        return this.to('ListSyncFilters');
    }
    /**
     * Grants permission to start a sync process or to resume a sync process that was previously paused
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toStartSync() {
        return this.to('StartSync');
    }
    /**
     * Grants permission to stop any planned sync process in the sync schedule from starting
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toStopSync() {
        return this.to('StopSync');
    }
    /**
     * Grants permission to update a sync target on the sync profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     */
    toUpdateSyncTarget() {
        return this.to('UpdateSyncTarget');
    }
    /**
     * Adds a resource of type SyncProfileResource to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     *
     * @param syncProfileName - Identifier for the syncProfileName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSyncProfileResource(syncProfileName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identity-sync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:profile/${syncProfileName}`);
    }
    /**
     * Adds a resource of type SyncTargetResource to the statement
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/provision-users-groups-AD.html
     *
     * @param syncProfileName - Identifier for the syncProfileName.
     * @param syncTargetName - Identifier for the syncTargetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onSyncTargetResource(syncProfileName, syncTargetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:identity-sync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:target/${syncProfileName}/${syncTargetName}`);
    }
    /**
     * Statement provider for service [identity-sync](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsidentitysync.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'identity-sync';
        this.accessLevelList = {
            'Permissions management': [
                'AllowVendedLogDeliveryForResource'
            ],
            Write: [
                'CreateSyncFilter',
                'CreateSyncProfile',
                'CreateSyncTarget',
                'DeleteSyncFilter',
                'DeleteSyncProfile',
                'DeleteSyncTarget',
                'StartSync',
                'StopSync',
                'UpdateSyncTarget'
            ],
            Read: [
                'GetSyncProfile',
                'GetSyncTarget'
            ],
            List: [
                'ListSyncFilters'
            ]
        };
    }
}
exports.IdentitySync = IdentitySync;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaWRlbnRpdHlzeW5jLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaWRlbnRpdHlzeW5jLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHlDQUErQztBQUcvQzs7OztHQUlHO0FBQ0gsTUFBYSxZQUFhLFNBQVEsd0JBQWU7SUFHL0M7Ozs7OztPQU1HO0lBQ0ksbUNBQW1DO1FBQ3hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO0lBQ3RELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxVQUFVO1FBQ2YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7SUFDckMsQ0FBQztJQTBCRDs7Ozs7Ozs7O09BU0c7SUFDSSxxQkFBcUIsQ0FBQyxlQUF1QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3pHLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFRLFNBQVMsSUFBSSxJQUFJLENBQUMsZ0JBQWlCLGtCQUFtQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsWUFBYSxlQUFnQixFQUFFLENBQUMsQ0FBQztJQUNqTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLG9CQUFvQixDQUFDLGVBQXVCLEVBQUUsY0FBc0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUNoSSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixrQkFBbUIsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLFdBQVksZUFBZ0IsSUFBSyxjQUFlLEVBQUUsQ0FBQyxDQUFDO0lBQ3BNLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQWpOUixrQkFBYSxHQUFHLGVBQWUsQ0FBQztRQXVKN0Isb0JBQWUsR0FBb0I7WUFDM0Msd0JBQXdCLEVBQUU7Z0JBQ3hCLG1DQUFtQzthQUNwQztZQUNELEtBQUssRUFBRTtnQkFDTCxrQkFBa0I7Z0JBQ2xCLG1CQUFtQjtnQkFDbkIsa0JBQWtCO2dCQUNsQixrQkFBa0I7Z0JBQ2xCLG1CQUFtQjtnQkFDbkIsa0JBQWtCO2dCQUNsQixXQUFXO2dCQUNYLFVBQVU7Z0JBQ1Ysa0JBQWtCO2FBQ25CO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLGdCQUFnQjtnQkFDaEIsZUFBZTthQUNoQjtZQUNELElBQUksRUFBRTtnQkFDSixpQkFBaUI7YUFDbEI7U0FDRixDQUFDO0lBcUNGLENBQUM7Q0FDRjtBQXBORCxvQ0FvTkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tICcuLi8uLi9zaGFyZWQvYWNjZXNzLWxldmVsJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbaWRlbnRpdHktc3luY10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzaWRlbnRpdHlzeW5jLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgSWRlbnRpdHlTeW5jIGV4dGVuZHMgUG9saWN5U3RhdGVtZW50IHtcbiAgcHVibGljIHNlcnZpY2VQcmVmaXggPSAnaWRlbnRpdHktc3luYyc7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNvbmZpZ3VyZSB2ZW5kZWQgbG9nIGRlbGl2ZXJ5IGZvciBhIFN5bmMgUHJvZmlsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2xvZ2dpbmctYWQtc3luYy1lcnJvcnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQWxsb3dWZW5kZWRMb2dEZWxpdmVyeUZvclJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdBbGxvd1ZlbmRlZExvZ0RlbGl2ZXJ5Rm9yUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBzeW5jIGZpbHRlciBvbiB0aGUgc3luYyBwcm9maWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL3Byb3Zpc2lvbi11c2Vycy1ncm91cHMtQUQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU3luY0ZpbHRlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU3luY0ZpbHRlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHN5bmMgcHJvZmlsZSBmb3IgdGhlIGlkZW50aXR5IHNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGRzOkF1dGhvcml6ZUFwcGxpY2F0aW9uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9wcm92aXNpb24tdXNlcnMtZ3JvdXBzLUFELmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVN5bmNQcm9maWxlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVTeW5jUHJvZmlsZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIHN5bmMgdGFyZ2V0IGZvciB0aGUgaWRlbnRpdHkgc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL3Byb3Zpc2lvbi11c2Vycy1ncm91cHMtQUQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlU3luY1RhcmdldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlU3luY1RhcmdldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHN5bmMgZmlsdGVyIGZyb20gdGhlIHN5bmMgcHJvZmlsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9wcm92aXNpb24tdXNlcnMtZ3JvdXBzLUFELmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVN5bmNGaWx0ZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVN5bmNGaWx0ZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBzeW5jIHByb2ZpbGUgZnJvbSB0aGUgc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gZHM6VW5hdXRob3JpemVBcHBsaWNhdGlvblxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvcHJvdmlzaW9uLXVzZXJzLWdyb3Vwcy1BRC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTeW5jUHJvZmlsZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlU3luY1Byb2ZpbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBzeW5jIHRhcmdldCBmcm9tIHRoZSBzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvcHJvdmlzaW9uLXVzZXJzLWdyb3Vwcy1BRC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVTeW5jVGFyZ2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVTeW5jVGFyZ2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBzeW5jIHByb2ZpbGUgYnkgdXNpbmcgYSBzeW5jIHByb2ZpbGUgbmFtZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL3Byb3Zpc2lvbi11c2Vycy1ncm91cHMtQUQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0U3luY1Byb2ZpbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFN5bmNQcm9maWxlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBzeW5jIHRhcmdldCBmcm9tIHRoZSBzeW5jIHByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9wcm92aXNpb24tdXNlcnMtZ3JvdXBzLUFELmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFN5bmNUYXJnZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFN5bmNUYXJnZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IHRoZSBzeW5jIGZpbHRlcnMgZnJvbSB0aGUgc3luYyBwcm9maWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvcHJvdmlzaW9uLXVzZXJzLWdyb3Vwcy1BRC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0U3luY0ZpbHRlcnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RTeW5jRmlsdGVycycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHN0YXJ0IGEgc3luYyBwcm9jZXNzIG9yIHRvIHJlc3VtZSBhIHN5bmMgcHJvY2VzcyB0aGF0IHdhcyBwcmV2aW91c2x5IHBhdXNlZFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9wcm92aXNpb24tdXNlcnMtZ3JvdXBzLUFELmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0U3luYygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnU3RhcnRTeW5jJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gc3RvcCBhbnkgcGxhbm5lZCBzeW5jIHByb2Nlc3MgaW4gdGhlIHN5bmMgc2NoZWR1bGUgZnJvbSBzdGFydGluZ1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9wcm92aXNpb24tdXNlcnMtZ3JvdXBzLUFELmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0b3BTeW5jKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdTdG9wU3luYycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhIHN5bmMgdGFyZ2V0IG9uIHRoZSBzeW5jIHByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvcHJvdmlzaW9uLXVzZXJzLWdyb3Vwcy1BRC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTeW5jVGFyZ2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVTeW5jVGFyZ2V0Jyk7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgJ1Blcm1pc3Npb25zIG1hbmFnZW1lbnQnOiBbXG4gICAgICAnQWxsb3dWZW5kZWRMb2dEZWxpdmVyeUZvclJlc291cmNlJ1xuICAgIF0sXG4gICAgV3JpdGU6IFtcbiAgICAgICdDcmVhdGVTeW5jRmlsdGVyJyxcbiAgICAgICdDcmVhdGVTeW5jUHJvZmlsZScsXG4gICAgICAnQ3JlYXRlU3luY1RhcmdldCcsXG4gICAgICAnRGVsZXRlU3luY0ZpbHRlcicsXG4gICAgICAnRGVsZXRlU3luY1Byb2ZpbGUnLFxuICAgICAgJ0RlbGV0ZVN5bmNUYXJnZXQnLFxuICAgICAgJ1N0YXJ0U3luYycsXG4gICAgICAnU3RvcFN5bmMnLFxuICAgICAgJ1VwZGF0ZVN5bmNUYXJnZXQnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnR2V0U3luY1Byb2ZpbGUnLFxuICAgICAgJ0dldFN5bmNUYXJnZXQnXG4gICAgXSxcbiAgICBMaXN0OiBbXG4gICAgICAnTGlzdFN5bmNGaWx0ZXJzJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgU3luY1Byb2ZpbGVSZXNvdXJjZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9wcm92aXNpb24tdXNlcnMtZ3JvdXBzLUFELmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHN5bmNQcm9maWxlTmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzeW5jUHJvZmlsZU5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uU3luY1Byb2ZpbGVSZXNvdXJjZShzeW5jUHJvZmlsZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTppZGVudGl0eS1zeW5jOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpwcm9maWxlLyR7IHN5bmNQcm9maWxlTmFtZSB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgU3luY1RhcmdldFJlc291cmNlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL3Byb3Zpc2lvbi11c2Vycy1ncm91cHMtQUQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gc3luY1Byb2ZpbGVOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHN5bmNQcm9maWxlTmFtZS5cbiAgICogQHBhcmFtIHN5bmNUYXJnZXROYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHN5bmNUYXJnZXROYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvblN5bmNUYXJnZXRSZXNvdXJjZShzeW5jUHJvZmlsZU5hbWU6IHN0cmluZywgc3luY1RhcmdldE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTppZGVudGl0eS1zeW5jOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTp0YXJnZXQvJHsgc3luY1Byb2ZpbGVOYW1lIH0vJHsgc3luY1RhcmdldE5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbaWRlbnRpdHktc3luY10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYXdzaWRlbnRpdHlzeW5jLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==