"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Inspector2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [inspector2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoninspector2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Inspector2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate an account with an Amazon Inspector administrator account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_AssociateMember.html
     */
    toAssociateMember() {
        return this.to('AssociateMember');
    }
    /**
     * Grants permission to associate multiple code repositories with an Amazon Inspector code security scan configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchAssociateCodeSecurityScanConfiguration.html
     */
    toBatchAssociateCodeSecurityScanConfiguration() {
        return this.to('BatchAssociateCodeSecurityScanConfiguration');
    }
    /**
     * Grants permission to disassociate multiple code repositories from an Amazon Inspector code security scan configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchDisassociateCodeSecurityScanConfiguration.html
     */
    toBatchDisassociateCodeSecurityScanConfiguration() {
        return this.to('BatchDisassociateCodeSecurityScanConfiguration');
    }
    /**
     * Grants permission to retrieve information about Amazon Inspector accounts for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchGetAccountStatus.html
     */
    toBatchGetAccountStatus() {
        return this.to('BatchGetAccountStatus');
    }
    /**
     * Grants permission to retrieve code snippet information about one or more code vulnerability findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchGetCodeSnippet.html
     */
    toBatchGetCodeSnippet() {
        return this.to('BatchGetCodeSnippet');
    }
    /**
     * Grants permission to let a customer get enhanced vulnerability intelligence details for findings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchGetFindingDetails.html
     */
    toBatchGetFindingDetails() {
        return this.to('BatchGetFindingDetails');
    }
    /**
     * Grants permission to retrieve free trial period eligibility about Amazon Inspector accounts for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchGetFreeTrialInfo.html
     */
    toBatchGetFreeTrialInfo() {
        return this.to('BatchGetFreeTrialInfo');
    }
    /**
     * Grants permission to delegated administrator to retrieve ec2 deep inspection status of member accounts
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchGetMemberEc2DeepInspectionStatus.html
     */
    toBatchGetMemberEc2DeepInspectionStatus() {
        return this.to('BatchGetMemberEc2DeepInspectionStatus');
    }
    /**
     * Grants permission to update ec2 deep inspection status by delegated administrator for its associated member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_BatchUpdateMemberEc2DeepInspectionStatus.html
     */
    toBatchUpdateMemberEc2DeepInspectionStatus() {
        return this.to('BatchUpdateMemberEc2DeepInspectionStatus');
    }
    /**
     * Grants permission to cancel the generation of a findings report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CancelFindingsReport.html
     */
    toCancelFindingsReport() {
        return this.to('CancelFindingsReport');
    }
    /**
     * Grants permission to cancel the generation of an SBOM report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CancelSbomExport.html
     */
    toCancelSbomExport() {
        return this.to('CancelSbomExport');
    }
    /**
     * Grants permission to create and define the settings for a CIS scan configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CreateCisScanConfiguration.html
     */
    toCreateCisScanConfiguration() {
        return this.to('CreateCisScanConfiguration');
    }
    /**
     * Grants permission to create a code security integration with a source code repository provider
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CreateCodeSecurityIntegration.html
     */
    toCreateCodeSecurityIntegration() {
        return this.to('CreateCodeSecurityIntegration');
    }
    /**
     * Grants permission to create a scan configuration for code security scanning
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CreateCodeSecurityScanConfiguration.html
     */
    toCreateCodeSecurityScanConfiguration() {
        return this.to('CreateCodeSecurityScanConfiguration');
    }
    /**
     * Grants permission to create and define the settings for a findings filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CreateFilter.html
     */
    toCreateFilter() {
        return this.to('CreateFilter');
    }
    /**
     * Grants permission to request the generation of a findings report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CreateFindingsReport.html
     */
    toCreateFindingsReport() {
        return this.to('CreateFindingsReport');
    }
    /**
     * Grants permission to request the generation of an SBOM report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_CreateSbomExport.html
     */
    toCreateSbomExport() {
        return this.to('CreateSbomExport');
    }
    /**
     * Grants permission to delete a CIS scan configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DeleteCisScanConfiguration.html
     */
    toDeleteCisScanConfiguration() {
        return this.to('DeleteCisScanConfiguration');
    }
    /**
     * Grants permission to delete a code security integration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DeleteCodeSecurityIntegration.html
     */
    toDeleteCodeSecurityIntegration() {
        return this.to('DeleteCodeSecurityIntegration');
    }
    /**
     * Grants permission to delete a code security scan configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DeleteCodeSecurityScanConfiguration.html
     */
    toDeleteCodeSecurityScanConfiguration() {
        return this.to('DeleteCodeSecurityScanConfiguration');
    }
    /**
     * Grants permission to delete a findings filter
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DeleteFilter.html
     */
    toDeleteFilter() {
        return this.to('DeleteFilter');
    }
    /**
     * Grants permission to retrieve information about the Amazon Inspector configuration settings for an AWS organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DescribeOrganizationConfiguration.html
     */
    toDescribeOrganizationConfiguration() {
        return this.to('DescribeOrganizationConfiguration');
    }
    /**
     * Grants permission to disable an Amazon Inspector account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_Disable.html
     */
    toDisable() {
        return this.to('Disable');
    }
    /**
     * Grants permission to disable an account as the delegated Amazon Inspector administrator account for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DisableDelegatedAdminAccount.html
     */
    toDisableDelegatedAdminAccount() {
        return this.to('DisableDelegatedAdminAccount');
    }
    /**
     * Grants permission to an Amazon Inspector administrator account to disassociate from an Inspector member account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_DisassociateMember.html
     */
    toDisassociateMember() {
        return this.to('DisassociateMember');
    }
    /**
     * Grants permission to enable and specify the configuration settings for a new Amazon Inspector account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_Enable.html
     */
    toEnable() {
        return this.to('Enable');
    }
    /**
     * Grants permission to enable an account as the delegated Amazon Inspector administrator account for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_EnableDelegatedAdminAccount.html
     */
    toEnableDelegatedAdminAccount() {
        return this.to('EnableDelegatedAdminAccount');
    }
    /**
     * Grants permission to retrieve a report containing information about completed CIS scans
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetCisScanReport.html
     */
    toGetCisScanReport() {
        return this.to('GetCisScanReport');
    }
    /**
     * Grants permission to retrieve information about all details pertaining to one CIS scan and one targeted resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetCisScanResultDetails.html
     */
    toGetCisScanResultDetails() {
        return this.to('GetCisScanResultDetails');
    }
    /**
     * Grants permission to get cluster information for a given a continuously scanned amazon Ecr image
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetClustersForImage.html
     */
    toGetClustersForImage() {
        return this.to('GetClustersForImage');
    }
    /**
     * Grants permission to retrieve information about a code security integration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetCodeSecurityIntegration.html
     */
    toGetCodeSecurityIntegration() {
        return this.to('GetCodeSecurityIntegration');
    }
    /**
     * Grants permission to retrieve information about a specific code security scan
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetCodeSecurityScan.html
     */
    toGetCodeSecurityScan() {
        return this.to('GetCodeSecurityScan');
    }
    /**
     * Grants permission to retrieve information about a code security scan configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetCodeSecurityScanConfiguration.html
     */
    toGetCodeSecurityScanConfiguration() {
        return this.to('GetCodeSecurityScanConfiguration');
    }
    /**
     * Grants permission to retrieve information about the Amazon Inspector configuration settings for an AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetConfiguration.html
     */
    toGetConfiguration() {
        return this.to('GetConfiguration');
    }
    /**
     * Grants permission to retrieve information about the Amazon Inspector administrator account for an account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetDelegatedAdminAccount.html
     */
    toGetDelegatedAdminAccount() {
        return this.to('GetDelegatedAdminAccount');
    }
    /**
     * Grants permission to retrieve ec2 deep inspection configuration for standalone accounts, delegated administrator and member account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetEc2DeepInspectionConfiguration.html
     */
    toGetEc2DeepInspectionConfiguration() {
        return this.to('GetEc2DeepInspectionConfiguration');
    }
    /**
     * Grants permission to retrieve information about the KMS key used to encrypt code snippets with
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetEncryptionKey.html
     */
    toGetEncryptionKey() {
        return this.to('GetEncryptionKey');
    }
    /**
     * Grants permission to retrieve status for a requested findings report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetFindingsReportStatus.html
     */
    toGetFindingsReportStatus() {
        return this.to('GetFindingsReportStatus');
    }
    /**
     * Grants permission to retrieve information about an account that's associated with an Amazon Inspector administrator account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetMember.html
     */
    toGetMember() {
        return this.to('GetMember');
    }
    /**
     * Grants permission to retrieve a requested SBOM report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_GetSbomExport.html
     */
    toGetSbomExport() {
        return this.to('GetSbomExport');
    }
    /**
     * Grants permission to retrieve feature configuration permissions associated with an Amazon Inspector account within an organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListAccountPermissions.html
     */
    toListAccountPermissions() {
        return this.to('ListAccountPermissions');
    }
    /**
     * Grants permission to retrieve information about all CIS scan configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCisScanConfigurations.html
     */
    toListCisScanConfigurations() {
        return this.to('ListCisScanConfigurations');
    }
    /**
     * Grants permission to retrieve information about all checks pertaining to one CIS scan
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCisScanResultsAggregatedByChecks.html
     */
    toListCisScanResultsAggregatedByChecks() {
        return this.to('ListCisScanResultsAggregatedByChecks');
    }
    /**
     * Grants permission to retrieve information about all resources pertaining to one CIS scan
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCisScanResultsAggregatedByTargetResource.html
     */
    toListCisScanResultsAggregatedByTargetResource() {
        return this.to('ListCisScanResultsAggregatedByTargetResource');
    }
    /**
     * Grants permission to retrieve information about completed CIS scans
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCisScans.html
     */
    toListCisScans() {
        return this.to('ListCisScans');
    }
    /**
     * Grants permission to list all code security integrations in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCodeSecurityIntegrations.html
     */
    toListCodeSecurityIntegrations() {
        return this.to('ListCodeSecurityIntegrations');
    }
    /**
     * Grants permission to list the associations between code repositories and Amazon Inspector code security scan configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCodeSecurityScanConfigurationAssociations.html
     */
    toListCodeSecurityScanConfigurationAssociations() {
        return this.to('ListCodeSecurityScanConfigurationAssociations');
    }
    /**
     * Grants permission to list all code security scan configurations in your account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCodeSecurityScanConfigurations.html
     */
    toListCodeSecurityScanConfigurations() {
        return this.to('ListCodeSecurityScanConfigurations');
    }
    /**
     * Grants permission to retrieve the types of statistics Amazon Inspector can generate for resources Inspector monitors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCoverage.html
     */
    toListCoverage() {
        return this.to('ListCoverage');
    }
    /**
     * Grants permission to retrieve statistical data and other information about the resources Amazon Inspector monitors
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListCoverageStatistics.html
     */
    toListCoverageStatistics() {
        return this.to('ListCoverageStatistics');
    }
    /**
     * Grants permission to retrieve information about the delegated Amazon Inspector administrator account for an AWS organization
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListDelegatedAdminAccounts.html
     */
    toListDelegatedAdminAccounts() {
        return this.to('ListDelegatedAdminAccounts');
    }
    /**
     * Grants permission to retrieve information about all findings filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListFilters.html
     */
    toListFilters() {
        return this.to('ListFilters');
    }
    /**
     * Grants permission to retrieve statistical data and other information about Amazon Inspector findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListFindingAggregations.html
     */
    toListFindingAggregations() {
        return this.to('ListFindingAggregations');
    }
    /**
     * Grants permission to retrieve a subset of information about one or more findings
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListFindings.html
     */
    toListFindings() {
        return this.to('ListFindings');
    }
    /**
     * Grants permission to retrieve information about the Amazon Inspector member accounts that are associated with an Inspector administrator account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListMembers.html
     */
    toListMembers() {
        return this.to('ListMembers');
    }
    /**
     * Grants permission to retrieve the tags for an Amazon Inspector resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve aggregated usage data for an account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ListUsageTotals.html
     */
    toListUsageTotals() {
        return this.to('ListUsageTotals');
    }
    /**
     * Grants permission to let a customer reset to use an Amazon-owned KMS key to encrypt code snippets with
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_ResetEncryptionKey.html
     */
    toResetEncryptionKey() {
        return this.to('ResetEncryptionKey');
    }
    /**
     * Grants permission to list Amazon Inspector coverage details for a specific vulnerability
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_SearchVulnerabilities.html
     */
    toSearchVulnerabilities() {
        return this.to('SearchVulnerabilities');
    }
    /**
     * Grants permission to send CIS health for a CIS scan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_SendCisSessionHealth.html
     */
    toSendCisSessionHealth() {
        return this.to('SendCisSessionHealth');
    }
    /**
     * Grants permission to send CIS telemetry for a CIS scan
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_SendCisSessionTelemetry.html
     */
    toSendCisSessionTelemetry() {
        return this.to('SendCisSessionTelemetry');
    }
    /**
     * Grants permission to start a CIS scan session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_StartCisSession.html
     */
    toStartCisSession() {
        return this.to('StartCisSession');
    }
    /**
     * Grants permission to initiate a code security scan on a specified repository
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_StartCodeSecurityScan.html
     */
    toStartCodeSecurityScan() {
        return this.to('StartCodeSecurityScan');
    }
    /**
     * Grants permission to stop a CIS scan session
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_StopCisSession.html
     */
    toStopCisSession() {
        return this.to('StopCisSession');
    }
    /**
     * Grants permission to add or update the tags for an Amazon Inspector resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from an Amazon Inspector resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the settings for a CIS scan configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateCisScanConfiguration.html
     */
    toUpdateCisScanConfiguration() {
        return this.to('UpdateCisScanConfiguration');
    }
    /**
     * Grants permission to update an existing code security integration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateCodeSecurityIntegration.html
     */
    toUpdateCodeSecurityIntegration() {
        return this.to('UpdateCodeSecurityIntegration');
    }
    /**
     * Grants permission to update an existing code security scan configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateCodeSecurityScanConfiguration.html
     */
    toUpdateCodeSecurityScanConfiguration() {
        return this.to('UpdateCodeSecurityScanConfiguration');
    }
    /**
     * Grants permission to update information about the Amazon Inspector configuration settings for an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateConfiguration.html
     */
    toUpdateConfiguration() {
        return this.to('UpdateConfiguration');
    }
    /**
     * Grants permission to update ec2 deep inspection configuration by delegated administrator, member and standalone account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateEc2DeepInspectionConfiguration.html
     */
    toUpdateEc2DeepInspectionConfiguration() {
        return this.to('UpdateEc2DeepInspectionConfiguration');
    }
    /**
     * Grants permission to let a customer use a KMS key to encrypt code snippets with
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateEncryptionKey.html
     */
    toUpdateEncryptionKey() {
        return this.to('UpdateEncryptionKey');
    }
    /**
     * Grants permission to update the settings for a findings filter
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateFilter.html
     */
    toUpdateFilter() {
        return this.to('UpdateFilter');
    }
    /**
     * Grants permission to update ec2 deep inspection configuration by delegated administrator for its associated member accounts
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateOrgEc2DeepInspectionConfiguration.html
     */
    toUpdateOrgEc2DeepInspectionConfiguration() {
        return this.to('UpdateOrgEc2DeepInspectionConfiguration');
    }
    /**
     * Grants permission to update Amazon Inspector configuration settings for an AWS organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/inspector/v2/APIReference/API_UpdateOrganizationConfiguration.html
     */
    toUpdateOrganizationConfiguration() {
        return this.to('UpdateOrganizationConfiguration');
    }
    /**
     * Adds a resource of type Filter to the statement
     *
     * https://docs.aws.amazon.com/inspector/latest/user/what-is-inspector.html
     *
     * @param ownerId - Identifier for the ownerId.
     * @param filterId - Identifier for the filterId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFilter(ownerId, filterId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:inspector2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:owner/${ownerId}/filter/${filterId}`);
    }
    /**
     * Adds a resource of type Finding to the statement
     *
     * https://docs.aws.amazon.com/inspector/latest/user/what-is-inspector.html
     *
     * @param findingId - Identifier for the findingId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onFinding(findingId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:inspector2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:finding/${findingId}`);
    }
    /**
     * Adds a resource of type CIS Scan Configuration to the statement
     *
     * https://docs.aws.amazon.com/inspector/latest/user/what-is-inspector.html
     *
     * @param ownerId - Identifier for the ownerId.
     * @param cISScanConfigurationId - Identifier for the cISScanConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCISScanConfiguration(ownerId, cISScanConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:inspector2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:owner/${ownerId}/cis-configuration/${cISScanConfigurationId}`);
    }
    /**
     * Adds a resource of type Code Security Scan Configuration to the statement
     *
     * https://docs.aws.amazon.com/inspector/latest/user/what-is-inspector.html
     *
     * @param ownerId - Identifier for the ownerId.
     * @param codeSecurityScanConfigurationId - Identifier for the codeSecurityScanConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCodeSecurityScanConfiguration(ownerId, codeSecurityScanConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:inspector2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:owner/${ownerId}/codesecurity-configuration/${codeSecurityScanConfigurationId}`);
    }
    /**
     * Adds a resource of type Code Security Integration to the statement
     *
     * https://docs.aws.amazon.com/inspector/latest/user/what-is-inspector.html
     *
     * @param codeSecurityIntegrationId - Identifier for the codeSecurityIntegrationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCodeSecurityIntegration(codeSecurityIntegrationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:inspector2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:codesecurity-integration/${codeSecurityIntegrationId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCisScanConfiguration()
     * - .toCreateCodeSecurityIntegration()
     * - .toCreateCodeSecurityScanConfiguration()
     * - .toCreateFilter()
     * - .toTagResource()
     * - .toUpdateFilter()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateCisScanConfiguration()
     * - .toDeleteCisScanConfiguration()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateCisScanConfiguration()
     * - .toUpdateCodeSecurityIntegration()
     * - .toUpdateCodeSecurityScanConfiguration()
     *
     * Applies to resource types:
     * - Filter
     * - CIS Scan Configuration
     * - Code Security Scan Configuration
     * - Code Security Integration
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCisScanConfiguration()
     * - .toCreateCodeSecurityIntegration()
     * - .toCreateCodeSecurityScanConfiguration()
     * - .toCreateFilter()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateFilter()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [inspector2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazoninspector2.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'inspector2';
        this.accessLevelList = {
            Write: [
                'AssociateMember',
                'BatchAssociateCodeSecurityScanConfiguration',
                'BatchDisassociateCodeSecurityScanConfiguration',
                'BatchUpdateMemberEc2DeepInspectionStatus',
                'CancelFindingsReport',
                'CancelSbomExport',
                'CreateCisScanConfiguration',
                'CreateCodeSecurityIntegration',
                'CreateCodeSecurityScanConfiguration',
                'CreateFilter',
                'CreateFindingsReport',
                'CreateSbomExport',
                'DeleteCisScanConfiguration',
                'DeleteCodeSecurityIntegration',
                'DeleteCodeSecurityScanConfiguration',
                'DeleteFilter',
                'Disable',
                'DisableDelegatedAdminAccount',
                'DisassociateMember',
                'Enable',
                'EnableDelegatedAdminAccount',
                'ResetEncryptionKey',
                'SendCisSessionHealth',
                'SendCisSessionTelemetry',
                'StartCisSession',
                'StartCodeSecurityScan',
                'StopCisSession',
                'UpdateCisScanConfiguration',
                'UpdateCodeSecurityIntegration',
                'UpdateCodeSecurityScanConfiguration',
                'UpdateConfiguration',
                'UpdateEc2DeepInspectionConfiguration',
                'UpdateEncryptionKey',
                'UpdateFilter',
                'UpdateOrgEc2DeepInspectionConfiguration',
                'UpdateOrganizationConfiguration'
            ],
            Read: [
                'BatchGetAccountStatus',
                'BatchGetCodeSnippet',
                'BatchGetFindingDetails',
                'BatchGetFreeTrialInfo',
                'BatchGetMemberEc2DeepInspectionStatus',
                'DescribeOrganizationConfiguration',
                'GetCisScanReport',
                'GetClustersForImage',
                'GetCodeSecurityIntegration',
                'GetCodeSecurityScan',
                'GetCodeSecurityScanConfiguration',
                'GetConfiguration',
                'GetDelegatedAdminAccount',
                'GetEc2DeepInspectionConfiguration',
                'GetEncryptionKey',
                'GetFindingsReportStatus',
                'GetMember',
                'GetSbomExport',
                'ListTagsForResource',
                'SearchVulnerabilities'
            ],
            List: [
                'GetCisScanResultDetails',
                'ListAccountPermissions',
                'ListCisScanConfigurations',
                'ListCisScanResultsAggregatedByChecks',
                'ListCisScanResultsAggregatedByTargetResource',
                'ListCisScans',
                'ListCodeSecurityIntegrations',
                'ListCodeSecurityScanConfigurationAssociations',
                'ListCodeSecurityScanConfigurations',
                'ListCoverage',
                'ListCoverageStatistics',
                'ListDelegatedAdminAccounts',
                'ListFilters',
                'ListFindingAggregations',
                'ListFindings',
                'ListMembers',
                'ListUsageTotals'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Inspector2 = Inspector2;
//# sourceMappingURL=data:application/json;base64,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