"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotfleetwise = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [iotfleetwise](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotfleetwise extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate the given vehicle to a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_AssociateVehicleFleet.html
     */
    toAssociateVehicleFleet() {
        return this.to('AssociateVehicleFleet');
    }
    /**
     * Grants permission to create a campaign
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifDestinationArn()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateCampaign.html
     */
    toCreateCampaign() {
        return this.to('CreateCampaign');
    }
    /**
     * Grants permission to create a decoder manifest for an existing model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateDecoderManifest.html
     */
    toCreateDecoderManifest() {
        return this.to('CreateDecoderManifest');
    }
    /**
     * Grants permission to create a fleet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateFleet.html
     */
    toCreateFleet() {
        return this.to('CreateFleet');
    }
    /**
     * Grants permission to create a model manifest definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateModelManifest.html
     */
    toCreateModelManifest() {
        return this.to('CreateModelManifest');
    }
    /**
     * Grants permission to create a signal catalog
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateSignalCatalog.html
     */
    toCreateSignalCatalog() {
        return this.to('CreateSignalCatalog');
    }
    /**
     * Grants permission to create a state template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateStateTemplate.html
     */
    toCreateStateTemplate() {
        return this.to('CreateStateTemplate');
    }
    /**
     * Grants permission to create a vehicle
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iot:CreateThing
     * - iot:DescribeThing
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_CreateVehicle.html
     */
    toCreateVehicle() {
        return this.to('CreateVehicle');
    }
    /**
     * Grants permission to delete a campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteCampaign.html
     */
    toDeleteCampaign() {
        return this.to('DeleteCampaign');
    }
    /**
     * Grants permission to delete the given decoder manifest
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteDecoderManifest.html
     */
    toDeleteDecoderManifest() {
        return this.to('DeleteDecoderManifest');
    }
    /**
     * Grants permission to delete a fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteFleet.html
     */
    toDeleteFleet() {
        return this.to('DeleteFleet');
    }
    /**
     * Grants permission to delete the given model manifest
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteModelManifest.html
     */
    toDeleteModelManifest() {
        return this.to('DeleteModelManifest');
    }
    /**
     * Grants permission to delete a specific signal catalog
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteSignalCatalog.html
     */
    toDeleteSignalCatalog() {
        return this.to('DeleteSignalCatalog');
    }
    /**
     * Grants permission to delete a state template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteStateTemplate.html
     */
    toDeleteStateTemplate() {
        return this.to('DeleteStateTemplate');
    }
    /**
     * Grants permission to delete a vehicle
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DeleteVehicle.html
     */
    toDeleteVehicle() {
        return this.to('DeleteVehicle');
    }
    /**
     * Grants permission to disassociate a vehicle from an existing fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_DisassociateVehicleFleet.html
     */
    toDisassociateVehicleFleet() {
        return this.to('DisassociateVehicleFleet');
    }
    /**
     * Grants permission to generate the payload for running a command on a vehicle
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifSignals()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/controlling-access.html#generate-command-payload
     */
    toGenerateCommandPayload() {
        return this.to('GenerateCommandPayload');
    }
    /**
     * Grants permission to get summary information for a given campaign
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetCampaign.html
     */
    toGetCampaign() {
        return this.to('GetCampaign');
    }
    /**
     * Grants permission to get summary information for a given decoder manifest definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetDecoderManifest.html
     */
    toGetDecoderManifest() {
        return this.to('GetDecoderManifest');
    }
    /**
     * Grants permission to get KMS-based encryption status for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetEncryptionConfiguration.html
     */
    toGetEncryptionConfiguration() {
        return this.to('GetEncryptionConfiguration');
    }
    /**
     * Grants permission to get summary information for a fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetFleet.html
     */
    toGetFleet() {
        return this.to('GetFleet');
    }
    /**
     * Grants permission to get the logging options for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetLoggingOptions.html
     */
    toGetLoggingOptions() {
        return this.to('GetLoggingOptions');
    }
    /**
     * Grants permission to get summary information for a given model manifest definition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetModelManifest.html
     */
    toGetModelManifest() {
        return this.to('GetModelManifest');
    }
    /**
     * Grants permission to get the account registration status with IoT FleetWise
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetRegisterAccountStatus.html
     */
    toGetRegisterAccountStatus() {
        return this.to('GetRegisterAccountStatus');
    }
    /**
     * Grants permission to get summary information for a specific signal catalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetSignalCatalog.html
     */
    toGetSignalCatalog() {
        return this.to('GetSignalCatalog');
    }
    /**
     * Grants permission to get summary information for a given state template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetStateTemplate.html
     */
    toGetStateTemplate() {
        return this.to('GetStateTemplate');
    }
    /**
     * Grants permission to get summary information for a vehicle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetVehicle.html
     */
    toGetVehicle() {
        return this.to('GetVehicle');
    }
    /**
     * Grants permission to get the status of the campaigns running on a specific vehicle
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_GetVehicleStatus.html
     */
    toGetVehicleStatus() {
        return this.to('GetVehicleStatus');
    }
    /**
     * Grants permission to import an existing decoder manifest
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ImportDecoderManifest.html
     */
    toImportDecoderManifest() {
        return this.to('ImportDecoderManifest');
    }
    /**
     * Grants permission to create a signal catalog by importing existing definitions
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ImportSignalCatalog.html
     */
    toImportSignalCatalog() {
        return this.to('ImportSignalCatalog');
    }
    /**
     * Grants permission to list campaigns
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListCampaigns.html
     */
    toListCampaigns() {
        return this.to('ListCampaigns');
    }
    /**
     * Grants permission to list network interfaces associated to the existing decoder manifest
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListDecoderManifestNetworkInterfaces.html
     */
    toListDecoderManifestNetworkInterfaces() {
        return this.to('ListDecoderManifestNetworkInterfaces');
    }
    /**
     * Grants permission to list decoder manifest signals
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListDecoderManifestSignals.html
     */
    toListDecoderManifestSignals() {
        return this.to('ListDecoderManifestSignals');
    }
    /**
     * Grants permission to list all decoder manifests, with an optional filter on model manifest
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListDecoderManifests.html
     */
    toListDecoderManifests() {
        return this.to('ListDecoderManifests');
    }
    /**
     * Grants permission to list all fleets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListFleets.html
     */
    toListFleets() {
        return this.to('ListFleets');
    }
    /**
     * Grants permission to list all the fleets that the given vehicle is associated with
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListFleetsForVehicle.html
     */
    toListFleetsForVehicle() {
        return this.to('ListFleetsForVehicle');
    }
    /**
     * Grants permission to list all nodes for the given model manifest
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListModelManifestNodes.html
     */
    toListModelManifestNodes() {
        return this.to('ListModelManifestNodes');
    }
    /**
     * Grants permission to list all model manifests, with an optional filter on signal catalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListModelManifests.html
     */
    toListModelManifests() {
        return this.to('ListModelManifests');
    }
    /**
     * Grants permission to list all nodes for a given signal catalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListSignalCatalogNodes.html
     */
    toListSignalCatalogNodes() {
        return this.to('ListSignalCatalogNodes');
    }
    /**
     * Grants permission to list all signal catalogs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListSignalCatalogs.html
     */
    toListSignalCatalogs() {
        return this.to('ListSignalCatalogs');
    }
    /**
     * Grants permission to list state templates
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListStateTemplates.html
     */
    toListStateTemplates() {
        return this.to('ListStateTemplates');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all vehicles, with an optional filter on model manifest
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListVehicles.html
     */
    toListVehicles() {
        return this.to('ListVehicles');
    }
    /**
     * Grants permission to list vehicles in the given fleet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_ListVehiclesInFleet.html
     */
    toListVehiclesInFleet() {
        return this.to('ListVehiclesInFleet');
    }
    /**
     * Grants permission to enable or disable KMS-based encryption for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_PutEncryptionConfiguration.html
     */
    toPutEncryptionConfiguration() {
        return this.to('PutEncryptionConfiguration');
    }
    /**
     * Grants permission to put the logging options for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_PutLoggingOptions.html
     */
    toPutLoggingOptions() {
        return this.to('PutLoggingOptions');
    }
    /**
     * Grants permission to register an AWS account to IoT FleetWise
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_RegisterAccount.html
     */
    toRegisterAccount() {
        return this.to('RegisterAccount');
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the given campaign
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateCampaign.html
     */
    toUpdateCampaign() {
        return this.to('UpdateCampaign');
    }
    /**
     * Grants permission to update a decoder manifest defnition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateDecoderManifest.html
     */
    toUpdateDecoderManifest() {
        return this.to('UpdateDecoderManifest');
    }
    /**
     * Grants permission to update the fleet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateFleet.html
     */
    toUpdateFleet() {
        return this.to('UpdateFleet');
    }
    /**
     * Grants permission to update the given model manifest definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateModelManifest.html
     */
    toUpdateModelManifest() {
        return this.to('UpdateModelManifest');
    }
    /**
     * Grants permission to update a specific signal catalog definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateSignalCatalog.html
     */
    toUpdateSignalCatalog() {
        return this.to('UpdateSignalCatalog');
    }
    /**
     * Grants permission to update the given state template
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateStateTemplate.html
     */
    toUpdateStateTemplate() {
        return this.to('UpdateStateTemplate');
    }
    /**
     * Grants permission to update the vehicle
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUpdateToModelManifestArn()
     * - .ifUpdateToDecoderManifestArn()
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/APIReference/API_UpdateVehicle.html
     */
    toUpdateVehicle() {
        return this.to('UpdateVehicle');
    }
    /**
     * Adds a resource of type campaign to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/campaigns.html
     *
     * @param campaignName - Identifier for the campaignName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCampaign(campaignName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:campaign/${campaignName}`);
    }
    /**
     * Adds a resource of type decodermanifest to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/decoder-manifests.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDecodermanifest(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:decoder-manifest/${name}`);
    }
    /**
     * Adds a resource of type fleet to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/fleets.html
     *
     * @param fleetId - Identifier for the fleetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFleet(fleetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:fleet/${fleetId}`);
    }
    /**
     * Adds a resource of type modelmanifest to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/vehicle-models.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelmanifest(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:model-manifest/${name}`);
    }
    /**
     * Adds a resource of type signalcatalog to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/signal-catalogs.html
     *
     * @param name - Identifier for the name.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSignalcatalog(name, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:signal-catalog/${name}`);
    }
    /**
     * Adds a resource of type vehicle to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/vehicles.html
     *
     * @param vehicleId - Identifier for the vehicleId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVehicle(vehicleId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vehicle/${vehicleId}`);
    }
    /**
     * Adds a resource of type statetemplate to the statement
     *
     * https://docs.aws.amazon.com/iot-fleetwise/latest/developerguide/last-known-state.html
     *
     * @param stateTemplateId - Identifier for the stateTemplateId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStatetemplate(stateTemplateId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotfleetwise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:state-template/${stateTemplateId}`);
    }
    /**
     * Filters access by the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateCampaign()
     * - .toCreateDecoderManifest()
     * - .toCreateFleet()
     * - .toCreateModelManifest()
     * - .toCreateSignalCatalog()
     * - .toCreateStateTemplate()
     * - .toCreateVehicle()
     * - .toImportSignalCatalog()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - campaign
     * - decodermanifest
     * - fleet
     * - modelmanifest
     * - signalcatalog
     * - vehicle
     * - statetemplate
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateCampaign()
     * - .toCreateDecoderManifest()
     * - .toCreateFleet()
     * - .toCreateModelManifest()
     * - .toCreateSignalCatalog()
     * - .toCreateStateTemplate()
     * - .toCreateVehicle()
     * - .toImportSignalCatalog()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by campaign destination ARN, eg. an S3 bucket ARN or a Timestream ARN
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html
     *
     * Applies to actions:
     * - .toCreateCampaign()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifDestinationArn(value, operator) {
        return this.if(`DestinationArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by fully qualified signal names
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html
     *
     * Applies to actions:
     * - .toGenerateCommandPayload()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignals(value, operator) {
        return this.if(`Signals`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of IoT FleetWise Decoder Manifest ARNs
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html
     *
     * Applies to actions:
     * - .toUpdateVehicle()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifUpdateToDecoderManifestArn(value, operator) {
        return this.if(`UpdateToDecoderManifestArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by a list of IoT FleetWise Model Manifest ARNs
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html
     *
     * Applies to actions:
     * - .toUpdateVehicle()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifUpdateToModelManifestArn(value, operator) {
        return this.if(`UpdateToModelManifestArn`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [iotfleetwise](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotfleetwise.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'iotfleetwise';
        this.accessLevelList = {
            Write: [
                'AssociateVehicleFleet',
                'CreateCampaign',
                'CreateDecoderManifest',
                'CreateFleet',
                'CreateModelManifest',
                'CreateSignalCatalog',
                'CreateStateTemplate',
                'CreateVehicle',
                'DeleteCampaign',
                'DeleteDecoderManifest',
                'DeleteFleet',
                'DeleteModelManifest',
                'DeleteSignalCatalog',
                'DeleteStateTemplate',
                'DeleteVehicle',
                'DisassociateVehicleFleet',
                'ImportDecoderManifest',
                'ImportSignalCatalog',
                'PutEncryptionConfiguration',
                'PutLoggingOptions',
                'RegisterAccount',
                'UpdateCampaign',
                'UpdateDecoderManifest',
                'UpdateFleet',
                'UpdateModelManifest',
                'UpdateSignalCatalog',
                'UpdateStateTemplate',
                'UpdateVehicle'
            ],
            'Permissions management': [
                'GenerateCommandPayload'
            ],
            Read: [
                'GetCampaign',
                'GetDecoderManifest',
                'GetEncryptionConfiguration',
                'GetFleet',
                'GetLoggingOptions',
                'GetModelManifest',
                'GetRegisterAccountStatus',
                'GetSignalCatalog',
                'GetStateTemplate',
                'GetVehicle',
                'GetVehicleStatus',
                'ListCampaigns',
                'ListDecoderManifests',
                'ListFleets',
                'ListFleetsForVehicle',
                'ListModelManifests',
                'ListSignalCatalogNodes',
                'ListSignalCatalogs',
                'ListStateTemplates',
                'ListTagsForResource',
                'ListVehicles',
                'ListVehiclesInFleet'
            ],
            List: [
                'ListDecoderManifestNetworkInterfaces',
                'ListDecoderManifestSignals',
                'ListModelManifestNodes'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Iotfleetwise = Iotfleetwise;
//# sourceMappingURL=data:application/json;base64,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