"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Iotsitewise = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [iotsitewise](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotsitewise.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Iotsitewise extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a child asset with a parent asset through a hierarchy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_AssociateAssets.html
     */
    toAssociateAssets() {
        return this.to('AssociateAssets');
    }
    /**
     * Grants permission to associate a time series with an asset property
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_AssociateTimeSeriesToAssetProperty.html
     */
    toAssociateTimeSeriesToAssetProperty() {
        return this.to('AssociateTimeSeriesToAssetProperty');
    }
    /**
     * Grants permission to associate assets to a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchAssociateProjectAssets.html
     */
    toBatchAssociateProjectAssets() {
        return this.to('BatchAssociateProjectAssets');
    }
    /**
     * Grants permission to disassociate assets from a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchDisassociateProjectAssets.html
     */
    toBatchDisassociateProjectAssets() {
        return this.to('BatchDisassociateProjectAssets');
    }
    /**
     * Grants permission to retrieve computed aggregates for multiple asset properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchGetAssetPropertyAggregates.html
     */
    toBatchGetAssetPropertyAggregates() {
        return this.to('BatchGetAssetPropertyAggregates');
    }
    /**
     * Grants permission to retrieve the latest value for multiple asset properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchGetAssetPropertyValue.html
     */
    toBatchGetAssetPropertyValue() {
        return this.to('BatchGetAssetPropertyValue');
    }
    /**
     * Grants permission to retrieve the value history for multiple asset properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchGetAssetPropertyValueHistory.html
     */
    toBatchGetAssetPropertyValueHistory() {
        return this.to('BatchGetAssetPropertyValueHistory');
    }
    /**
     * Grants permission to put property values for asset properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_BatchPutAssetPropertyValue.html
     */
    toBatchPutAssetPropertyValue() {
        return this.to('BatchPutAssetPropertyValue');
    }
    /**
     * Grants permission to create an access policy for a portal or a project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAccessPolicy.html
     */
    toCreateAccessPolicy() {
        return this.to('CreateAccessPolicy');
    }
    /**
     * Grants permission to create an asset from an asset model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAsset.html
     */
    toCreateAsset() {
        return this.to('CreateAsset');
    }
    /**
     * Grants permission to create an asset model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAssetModel.html
     */
    toCreateAssetModel() {
        return this.to('CreateAssetModel');
    }
    /**
     * Grants permission to create an asset model composite model inside an asset model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAssetModelCompositeModel.html
     */
    toCreateAssetModelCompositeModel() {
        return this.to('CreateAssetModelCompositeModel');
    }
    /**
     * Grants permission to create bulk import job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateBulkImportJob.html
     */
    toCreateBulkImportJob() {
        return this.to('CreateBulkImportJob');
    }
    /**
     * Grants permission to create a computation model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateComputationModel.html
     */
    toCreateComputationModel() {
        return this.to('CreateComputationModel');
    }
    /**
     * Grants permission to create a dashboard in a project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateDashboard.html
     */
    toCreateDashboard() {
        return this.to('CreateDashboard');
    }
    /**
     * Grants permission to create a dataset
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateDataset.html
     */
    toCreateDataset() {
        return this.to('CreateDataset');
    }
    /**
     * Grants permission to create a gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateGateway.html
     */
    toCreateGateway() {
        return this.to('CreateGateway');
    }
    /**
     * Grants permission to create a portal
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - sso:CreateManagedApplicationInstance
     * - sso:DescribeRegisteredRegions
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreatePortal.html
     */
    toCreatePortal() {
        return this.to('CreatePortal');
    }
    /**
     * Grants permission to create a project in a portal
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateProject.html
     */
    toCreateProject() {
        return this.to('CreateProject');
    }
    /**
     * Grants permission to delete an access policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAccessPolicy.html
     */
    toDeleteAccessPolicy() {
        return this.to('DeleteAccessPolicy');
    }
    /**
     * Grants permission to delete an asset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAsset.html
     */
    toDeleteAsset() {
        return this.to('DeleteAsset');
    }
    /**
     * Grants permission to delete an asset model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAssetModel.html
     */
    toDeleteAssetModel() {
        return this.to('DeleteAssetModel');
    }
    /**
     * Grants permission to delete an asset model composite model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAssetModelCompositeModel.html
     */
    toDeleteAssetModelCompositeModel() {
        return this.to('DeleteAssetModelCompositeModel');
    }
    /**
     * Grants permission to delete a relationship between asset model and interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteAssetModelInterfaceRelationship.html
     */
    toDeleteAssetModelInterfaceRelationship() {
        return this.to('DeleteAssetModelInterfaceRelationship');
    }
    /**
     * Grants permission to delete a computation model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteComputationModel.html
     */
    toDeleteComputationModel() {
        return this.to('DeleteComputationModel');
    }
    /**
     * Grants permission to delete a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteDashboard.html
     */
    toDeleteDashboard() {
        return this.to('DeleteDashboard');
    }
    /**
     * Grants permission to delete a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteDataset.html
     */
    toDeleteDataset() {
        return this.to('DeleteDataset');
    }
    /**
     * Grants permission to delete a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteGateway.html
     */
    toDeleteGateway() {
        return this.to('DeleteGateway');
    }
    /**
     * Grants permission to delete a portal
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sso:DeleteManagedApplicationInstance
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeletePortal.html
     */
    toDeletePortal() {
        return this.to('DeletePortal');
    }
    /**
     * Grants permission to delete a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteProject.html
     */
    toDeleteProject() {
        return this.to('DeleteProject');
    }
    /**
     * Grants permission to delete a time series
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DeleteTimeSeries.html
     */
    toDeleteTimeSeries() {
        return this.to('DeleteTimeSeries');
    }
    /**
     * Grants permission to describe an access policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAccessPolicy.html
     */
    toDescribeAccessPolicy() {
        return this.to('DescribeAccessPolicy');
    }
    /**
     * Grants permission to describe actions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAction.html
     */
    toDescribeAction() {
        return this.to('DescribeAction');
    }
    /**
     * Grants permission to describe an asset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAsset.html
     */
    toDescribeAsset() {
        return this.to('DescribeAsset');
    }
    /**
     * Grants permission to describe an asset composite model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAssetCompositeModel.html
     */
    toDescribeAssetCompositeModel() {
        return this.to('DescribeAssetCompositeModel');
    }
    /**
     * Grants permission to describe an asset model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAssetModel.html
     */
    toDescribeAssetModel() {
        return this.to('DescribeAssetModel');
    }
    /**
     * Grants permission to describe an asset model composite model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAssetModelCompositeModel.html
     */
    toDescribeAssetModelCompositeModel() {
        return this.to('DescribeAssetModelCompositeModel');
    }
    /**
     * Grants permission to describe a relationship between asset model and interface
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAssetModelInterfaceRelationship.html
     */
    toDescribeAssetModelInterfaceRelationship() {
        return this.to('DescribeAssetModelInterfaceRelationship');
    }
    /**
     * Grants permission to describe an asset property
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeAssetProperty.html
     */
    toDescribeAssetProperty() {
        return this.to('DescribeAssetProperty');
    }
    /**
     * Grants permission to describe bulk import job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeBulkImportJob.html
     */
    toDescribeBulkImportJob() {
        return this.to('DescribeBulkImportJob');
    }
    /**
     * Grants permission to describe a computation model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeComputationModel.html
     */
    toDescribeComputationModel() {
        return this.to('DescribeComputationModel');
    }
    /**
     * Grants permission to describe computation model execution summary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeComputationModelExecutionSummary.html
     */
    toDescribeComputationModelExecutionSummary() {
        return this.to('DescribeComputationModelExecutionSummary');
    }
    /**
     * Grants permission to describe a dashboard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeDashboard.html
     */
    toDescribeDashboard() {
        return this.to('DescribeDashboard');
    }
    /**
     * Grants permission to describe dataset
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeDataset.html
     */
    toDescribeDataset() {
        return this.to('DescribeDataset');
    }
    /**
     * Grants permission to describe the default encryption configuration for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeDefaultEncryptionConfiguration.html
     */
    toDescribeDefaultEncryptionConfiguration() {
        return this.to('DescribeDefaultEncryptionConfiguration');
    }
    /**
     * Grants permission to describe an execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeExecution.html
     */
    toDescribeExecution() {
        return this.to('DescribeExecution');
    }
    /**
     * Grants permission to describe a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeGateway.html
     */
    toDescribeGateway() {
        return this.to('DescribeGateway');
    }
    /**
     * Grants permission to describe a capability configuration for a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeGatewayCapabilityConfiguration.html
     */
    toDescribeGatewayCapabilityConfiguration() {
        return this.to('DescribeGatewayCapabilityConfiguration');
    }
    /**
     * Grants permission to describe logging options for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeLoggingOptions.html
     */
    toDescribeLoggingOptions() {
        return this.to('DescribeLoggingOptions');
    }
    /**
     * Grants permission to describe a portal
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribePortal.html
     */
    toDescribePortal() {
        return this.to('DescribePortal');
    }
    /**
     * Grants permission to describe a project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeProject.html
     */
    toDescribeProject() {
        return this.to('DescribeProject');
    }
    /**
     * Grants permission to describe the storage configuration for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeStorageConfiguration.html
     */
    toDescribeStorageConfiguration() {
        return this.to('DescribeStorageConfiguration');
    }
    /**
     * Grants permission to describe a time series
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeTimeSeries.html
     */
    toDescribeTimeSeries() {
        return this.to('DescribeTimeSeries');
    }
    /**
     * Grants permission to disassociate a child asset from a parent asset by a hierarchy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DisassociateAssets.html
     */
    toDisassociateAssets() {
        return this.to('DisassociateAssets');
    }
    /**
     * Grants permission to disassociate a time series from an asset property
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DisassociateTimeSeriesFromAssetProperty.html
     */
    toDisassociateTimeSeriesFromAssetProperty() {
        return this.to('DisassociateTimeSeriesFromAssetProperty');
    }
    /**
     * Grants permission to allow IoT SiteWise integrate with other services
     *
     * Access Level: Write
     */
    toEnableSiteWiseIntegration() {
        return this.to('EnableSiteWiseIntegration');
    }
    /**
     * Grants permission to execute actions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ExecuteAction.html
     */
    toExecuteAction() {
        return this.to('ExecuteAction');
    }
    /**
     * Grants permission to execute query
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ExecuteQuery.html
     */
    toExecuteQuery() {
        return this.to('ExecuteQuery');
    }
    /**
     * Grants permission to retrieve computed aggregates for an asset property
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_GetAssetPropertyAggregates.html
     */
    toGetAssetPropertyAggregates() {
        return this.to('GetAssetPropertyAggregates');
    }
    /**
     * Grants permission to retrieve the latest value for an asset property
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_GetAssetPropertyValue.html
     */
    toGetAssetPropertyValue() {
        return this.to('GetAssetPropertyValue');
    }
    /**
     * Grants permission to retrieve the value history for an asset property
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_GetAssetPropertyValueHistory.html
     */
    toGetAssetPropertyValueHistory() {
        return this.to('GetAssetPropertyValueHistory');
    }
    /**
     * Grants permission to retrieve interpolated values for an asset property
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_GetInterpolatedAssetPropertyValues.html
     */
    toGetInterpolatedAssetPropertyValues() {
        return this.to('GetInterpolatedAssetPropertyValues');
    }
    /**
     * Grants permission to invoke an assistant
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_InvokeAssistant.html
     */
    toInvokeAssistant() {
        return this.to('InvokeAssistant');
    }
    /**
     * Grants permission to list all access policies for an identity or a resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAccessPolicies.html
     */
    toListAccessPolicies() {
        return this.to('ListAccessPolicies');
    }
    /**
     * Grants permission to list all actions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListActions.html
     */
    toListActions() {
        return this.to('ListActions');
    }
    /**
     * Grants permission to list all asset model composite models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssetModelCompositeModels.html
     */
    toListAssetModelCompositeModels() {
        return this.to('ListAssetModelCompositeModels');
    }
    /**
     * Grants permission to list asset model properties
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssetModelProperties.html
     */
    toListAssetModelProperties() {
        return this.to('ListAssetModelProperties');
    }
    /**
     * Grants permission to list all asset models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssetModels.html
     */
    toListAssetModels() {
        return this.to('ListAssetModels');
    }
    /**
     * Grants permission to list asset properties
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssetProperties.html
     */
    toListAssetProperties() {
        return this.to('ListAssetProperties');
    }
    /**
     * Grants permission to list the asset relationship graph for an asset
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssetRelationships.html
     */
    toListAssetRelationships() {
        return this.to('ListAssetRelationships');
    }
    /**
     * Grants permission to list all assets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssets.html
     */
    toListAssets() {
        return this.to('ListAssets');
    }
    /**
     * Grants permission to list all assets associated with an asset through a hierarchy
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListAssociatedAssets.html
     */
    toListAssociatedAssets() {
        return this.to('ListAssociatedAssets');
    }
    /**
     * Grants permission to list bulk import jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListBulkImportJobs.html
     */
    toListBulkImportJobs() {
        return this.to('ListBulkImportJobs');
    }
    /**
     * Grants permission to list all asset model composition relationships
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListCompositionRelationships.html
     */
    toListCompositionRelationships() {
        return this.to('ListCompositionRelationships');
    }
    /**
     * Grants permission to list computation model data binding usages
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListComputationModelDataBindingUsages.html
     */
    toListComputationModelDataBindingUsages() {
        return this.to('ListComputationModelDataBindingUsages');
    }
    /**
     * Grants permission to list computation model resolve to resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListComputationModelResolveToResources.html
     */
    toListComputationModelResolveToResources() {
        return this.to('ListComputationModelResolveToResources');
    }
    /**
     * Grants permission to list all computation models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListComputationModels.html
     */
    toListComputationModels() {
        return this.to('ListComputationModels');
    }
    /**
     * Grants permission to list all dashboards in a project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListDashboards.html
     */
    toListDashboards() {
        return this.to('ListDashboards');
    }
    /**
     * Grants permission to list all datasets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListDatasets.html
     */
    toListDatasets() {
        return this.to('ListDatasets');
    }
    /**
     * Grants permission to list executions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListExecutions.html
     */
    toListExecutions() {
        return this.to('ListExecutions');
    }
    /**
     * Grants permission to list all gateways
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListGateways.html
     */
    toListGateways() {
        return this.to('ListGateways');
    }
    /**
     * Grants permission to list all asset models that are enforced by an interface
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListInterfaceRelationships.html
     */
    toListInterfaceRelationships() {
        return this.to('ListInterfaceRelationships');
    }
    /**
     * Grants permission to list all portals
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListPortals.html
     */
    toListPortals() {
        return this.to('ListPortals');
    }
    /**
     * Grants permission to list all assets associated with a project
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListProjectAssets.html
     */
    toListProjectAssets() {
        return this.to('ListProjectAssets');
    }
    /**
     * Grants permission to list all projects in a portal
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListProjects.html
     */
    toListProjects() {
        return this.to('ListProjects');
    }
    /**
     * Grants permission to list all tags for a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list time series
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_ListTimeSeries.html
     */
    toListTimeSeries() {
        return this.to('ListTimeSeries');
    }
    /**
     * Grants permission to create a relationship between asset model and interface
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_PutAssetModelInterfaceRelationship.html
     */
    toPutAssetModelInterfaceRelationship() {
        return this.to('PutAssetModelInterfaceRelationship');
    }
    /**
     * Grants permission to set the default encryption configuration for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_PutDefaultEncryptionConfiguration.html
     */
    toPutDefaultEncryptionConfiguration() {
        return this.to('PutDefaultEncryptionConfiguration');
    }
    /**
     * Grants permission to set logging options for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_PutLoggingOptions.html
     */
    toPutLoggingOptions() {
        return this.to('PutLoggingOptions');
    }
    /**
     * Grants permission to configure storage settings for the AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_PutStorageConfiguration.html
     */
    toPutStorageConfiguration() {
        return this.to('PutStorageConfiguration');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an access policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAccessPolicy.html
     */
    toUpdateAccessPolicy() {
        return this.to('UpdateAccessPolicy');
    }
    /**
     * Grants permission to update an asset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAsset.html
     */
    toUpdateAsset() {
        return this.to('UpdateAsset');
    }
    /**
     * Grants permission to update an asset model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAssetModel.html
     */
    toUpdateAssetModel() {
        return this.to('UpdateAssetModel');
    }
    /**
     * Grants permission to update asset model composite model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAssetModelCompositeModel.html
     */
    toUpdateAssetModelCompositeModel() {
        return this.to('UpdateAssetModelCompositeModel');
    }
    /**
     * Grants permission to update an AssetModel property routing
     *
     * Access Level: Write
     */
    toUpdateAssetModelPropertyRouting() {
        return this.to('UpdateAssetModelPropertyRouting');
    }
    /**
     * Grants permission to update an asset property
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateAssetProperty.html
     */
    toUpdateAssetProperty() {
        return this.to('UpdateAssetProperty');
    }
    /**
     * Grants permission to update a computation model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateComputationModel.html
     */
    toUpdateComputationModel() {
        return this.to('UpdateComputationModel');
    }
    /**
     * Grants permission to update a dashboard
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateDashboard.html
     */
    toUpdateDashboard() {
        return this.to('UpdateDashboard');
    }
    /**
     * Grants permission to update a dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateDataset.html
     */
    toUpdateDataset() {
        return this.to('UpdateDataset');
    }
    /**
     * Grants permission to update a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateGateway.html
     */
    toUpdateGateway() {
        return this.to('UpdateGateway');
    }
    /**
     * Grants permission to update a capability configuration for a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateGatewayCapabilityConfiguration.html
     */
    toUpdateGatewayCapabilityConfiguration() {
        return this.to('UpdateGatewayCapabilityConfiguration');
    }
    /**
     * Grants permission to update a portal
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdatePortal.html
     */
    toUpdatePortal() {
        return this.to('UpdatePortal');
    }
    /**
     * Grants permission to update a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_UpdateProject.html
     */
    toUpdateProject() {
        return this.to('UpdateProject');
    }
    /**
     * Adds a resource of type asset to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAsset.html
     *
     * @param assetId - Identifier for the assetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAsset(assetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotsitewise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:asset/${assetId}`);
    }
    /**
     * Adds a resource of type asset-model to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAssetModel.html
     *
     * @param assetModelId - Identifier for the assetModelId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAssetModel(assetModelId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotsitewise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:asset-model/${assetModelId}`);
    }
    /**
     * Adds a resource of type time-series to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_DescribeTimeSeries.html
     *
     * @param timeSeriesId - Identifier for the timeSeriesId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTimeSeries(timeSeriesId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotsitewise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:time-series/${timeSeriesId}`);
    }
    /**
     * Adds a resource of type gateway to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateGateway.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGateway(gatewayId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotsitewise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:gateway/${gatewayId}`);
    }
    /**
     * Adds a resource of type portal to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreatePortal.html
     *
     * @param portalId - Identifier for the portalId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPortal(portalId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotsitewise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:portal/${portalId}`);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateProject.html
     *
     * @param projectId - Identifier for the projectId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(projectId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotsitewise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:project/${projectId}`);
    }
    /**
     * Adds a resource of type dashboard to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateDashboard.html
     *
     * @param dashboardId - Identifier for the dashboardId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDashboard(dashboardId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotsitewise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dashboard/${dashboardId}`);
    }
    /**
     * Adds a resource of type access-policy to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateAccessPolicy.html
     *
     * @param accessPolicyId - Identifier for the accessPolicyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAccessPolicy(accessPolicyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotsitewise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:access-policy/${accessPolicyId}`);
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateDataset.html
     *
     * @param datasetId - Identifier for the datasetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataset(datasetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotsitewise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:dataset/${datasetId}`);
    }
    /**
     * Adds a resource of type computation-model to the statement
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/APIReference/API_CreateComputationModel.html
     *
     * @param computationModelId - Identifier for the computationModelId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onComputationModel(computationModelId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:iotsitewise:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:computation-model/${computationModelId}`);
    }
    /**
     * Filters access by the tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateAccessPolicy()
     * - .toCreateAsset()
     * - .toCreateAssetModel()
     * - .toCreateComputationModel()
     * - .toCreateDashboard()
     * - .toCreateDataset()
     * - .toCreateGateway()
     * - .toCreatePortal()
     * - .toCreateProject()
     * - .toDescribeTimeSeries()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toListTagsForResource()
     *
     * Applies to resource types:
     * - asset
     * - asset-model
     * - time-series
     * - gateway
     * - portal
     * - project
     * - dashboard
     * - access-policy
     * - dataset
     * - computation-model
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateAccessPolicy()
     * - .toCreateAsset()
     * - .toCreateAssetModel()
     * - .toCreateComputationModel()
     * - .toCreateDashboard()
     * - .toCreateDataset()
     * - .toCreateGateway()
     * - .toCreatePortal()
     * - .toCreateProject()
     * - .toDescribeTimeSeries()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by an asset hierarchy path, which is the string of asset IDs in the asset's hierarchy, each separated by a forward slash
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAssetHierarchyPath(value, operator) {
        return this.if(`assetHierarchyPath`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of a child asset being associated whith a parent asset
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChildAssetId(value, operator) {
        return this.if(`childAssetId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of an AWS Single Sign-On group
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGroup(value, operator) {
        return this.if(`group`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of an AWS IAM identity
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIam(value, operator) {
        return this.if(`iam`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by data streams associated with or not associated with asset properties
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifIsAssociatedWithAssetProperty(value, operator) {
        return this.if(`isAssociatedWithAssetProperty`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of a portal
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPortal(value, operator) {
        return this.if(`portal`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of a project
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProject(value, operator) {
        return this.if(`project`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the property alias
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPropertyAlias(value, operator) {
        return this.if(`propertyAlias`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of an asset property
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPropertyId(value, operator) {
        return this.if(`propertyId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the ID of an AWS Single Sign-On user
     *
     * https://docs.aws.amazon.com/iot-sitewise/latest/userguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUser(value, operator) {
        return this.if(`user`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [iotsitewise](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsiotsitewise.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'iotsitewise';
        this.accessLevelList = {
            Write: [
                'AssociateAssets',
                'AssociateTimeSeriesToAssetProperty',
                'BatchAssociateProjectAssets',
                'BatchDisassociateProjectAssets',
                'BatchPutAssetPropertyValue',
                'CreateAccessPolicy',
                'CreateAsset',
                'CreateAssetModel',
                'CreateAssetModelCompositeModel',
                'CreateBulkImportJob',
                'CreateComputationModel',
                'CreateDashboard',
                'CreateDataset',
                'CreateGateway',
                'CreatePortal',
                'CreateProject',
                'DeleteAccessPolicy',
                'DeleteAsset',
                'DeleteAssetModel',
                'DeleteAssetModelCompositeModel',
                'DeleteAssetModelInterfaceRelationship',
                'DeleteComputationModel',
                'DeleteDashboard',
                'DeleteDataset',
                'DeleteGateway',
                'DeletePortal',
                'DeleteProject',
                'DeleteTimeSeries',
                'DisassociateAssets',
                'DisassociateTimeSeriesFromAssetProperty',
                'EnableSiteWiseIntegration',
                'ExecuteAction',
                'PutAssetModelInterfaceRelationship',
                'PutDefaultEncryptionConfiguration',
                'PutLoggingOptions',
                'PutStorageConfiguration',
                'UpdateAccessPolicy',
                'UpdateAsset',
                'UpdateAssetModel',
                'UpdateAssetModelCompositeModel',
                'UpdateAssetModelPropertyRouting',
                'UpdateAssetProperty',
                'UpdateComputationModel',
                'UpdateDashboard',
                'UpdateDataset',
                'UpdateGateway',
                'UpdateGatewayCapabilityConfiguration',
                'UpdatePortal',
                'UpdateProject'
            ],
            Read: [
                'BatchGetAssetPropertyAggregates',
                'BatchGetAssetPropertyValue',
                'BatchGetAssetPropertyValueHistory',
                'DescribeAccessPolicy',
                'DescribeAction',
                'DescribeAsset',
                'DescribeAssetCompositeModel',
                'DescribeAssetModel',
                'DescribeAssetModelCompositeModel',
                'DescribeAssetModelInterfaceRelationship',
                'DescribeAssetProperty',
                'DescribeBulkImportJob',
                'DescribeComputationModel',
                'DescribeComputationModelExecutionSummary',
                'DescribeDashboard',
                'DescribeDataset',
                'DescribeDefaultEncryptionConfiguration',
                'DescribeExecution',
                'DescribeGateway',
                'DescribeGatewayCapabilityConfiguration',
                'DescribeLoggingOptions',
                'DescribePortal',
                'DescribeProject',
                'DescribeStorageConfiguration',
                'DescribeTimeSeries',
                'ExecuteQuery',
                'GetAssetPropertyAggregates',
                'GetAssetPropertyValue',
                'GetAssetPropertyValueHistory',
                'GetInterpolatedAssetPropertyValues',
                'InvokeAssistant',
                'ListTagsForResource'
            ],
            List: [
                'ListAccessPolicies',
                'ListActions',
                'ListAssetModelCompositeModels',
                'ListAssetModelProperties',
                'ListAssetModels',
                'ListAssetProperties',
                'ListAssetRelationships',
                'ListAssets',
                'ListAssociatedAssets',
                'ListBulkImportJobs',
                'ListCompositionRelationships',
                'ListComputationModelDataBindingUsages',
                'ListComputationModelResolveToResources',
                'ListComputationModels',
                'ListDashboards',
                'ListDatasets',
                'ListExecutions',
                'ListGateways',
                'ListInterfaceRelationships',
                'ListPortals',
                'ListProjectAssets',
                'ListProjects',
                'ListTimeSeries'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Iotsitewise = Iotsitewise;
//# sourceMappingURL=data:application/json;base64,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