"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kendra = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [kendra](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkendra.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kendra extends shared_1.PolicyStatement {
    /**
     * Grants permission to put principal mapping in index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_PutPrincipalMapping.html
     */
    toAssociateEntitiesToExperience() {
        return this.to('AssociateEntitiesToExperience');
    }
    /**
     * Defines the specific permissions of users or groups in your AWS SSO identity source with access to your Amazon Kendra experience
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_AssociatePersonasToEntities.html
     */
    toAssociatePersonasToEntities() {
        return this.to('AssociatePersonasToEntities');
    }
    /**
     * Grants permission to batch delete document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_BatchDeleteDocument.html
     */
    toBatchDeleteDocument() {
        return this.to('BatchDeleteDocument');
    }
    /**
     * Grants permission to delete a featured results set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteFeaturedResults.html
     */
    toBatchDeleteFeaturedResultsSet() {
        return this.to('BatchDeleteFeaturedResultsSet');
    }
    /**
     * Grants permission to do batch get document status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_BatchGetDocumentStatus.html
     */
    toBatchGetDocumentStatus() {
        return this.to('BatchGetDocumentStatus');
    }
    /**
     * Grants permission to batch put document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_BatchPutDocument.html
     */
    toBatchPutDocument() {
        return this.to('BatchPutDocument');
    }
    /**
     * Grants permission to clear out the suggestions for a given index, generated so far
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ClearQuerySuggestions.html
     */
    toClearQuerySuggestions() {
        return this.to('ClearQuerySuggestions');
    }
    /**
     * Grants permission to create an access control configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateAccessControlConfiguration.html
     */
    toCreateAccessControlConfiguration() {
        return this.to('CreateAccessControlConfiguration');
    }
    /**
     * Grants permission to create a data source
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateDataSource.html
     */
    toCreateDataSource() {
        return this.to('CreateDataSource');
    }
    /**
     * Creates an Amazon Kendra experience such as a search application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateExperience.html
     */
    toCreateExperience() {
        return this.to('CreateExperience');
    }
    /**
     * Grants permission to create an Faq
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateFaq.html
     */
    toCreateFaq() {
        return this.to('CreateFaq');
    }
    /**
     * Grants permission to create a featured results set
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateFeaturedResults.html
     */
    toCreateFeaturedResultsSet() {
        return this.to('CreateFeaturedResultsSet');
    }
    /**
     * Grants permission to create an Index
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateIndex.html
     */
    toCreateIndex() {
        return this.to('CreateIndex');
    }
    /**
     * Grants permission to create a QuerySuggestions BlockList
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateQuerySuggestionsBlockList.html
     */
    toCreateQuerySuggestionsBlockList() {
        return this.to('CreateQuerySuggestionsBlockList');
    }
    /**
     * Grants permission to create a Thesaurus
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateThesaurus.html
     */
    toCreateThesaurus() {
        return this.to('CreateThesaurus');
    }
    /**
     * Grants permission to delete an access control configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteAccessControlConfiguration.html
     */
    toDeleteAccessControlConfiguration() {
        return this.to('DeleteAccessControlConfiguration');
    }
    /**
     * Grants permission to delete a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        return this.to('DeleteDataSource');
    }
    /**
     * Deletes your Amazon Kendra experience such as a search application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteExperience.html
     */
    toDeleteExperience() {
        return this.to('DeleteExperience');
    }
    /**
     * Grants permission to delete an Faq
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteFaq.html
     */
    toDeleteFaq() {
        return this.to('DeleteFaq');
    }
    /**
     * Grants permission to delete an Index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteIndex.html
     */
    toDeleteIndex() {
        return this.to('DeleteIndex');
    }
    /**
     * Grants permission to delete principal mapping from index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeletePrincipalMapping.html
     */
    toDeletePrincipalMapping() {
        return this.to('DeletePrincipalMapping');
    }
    /**
     * Grants permission to delete a QuerySuggestions BlockList
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteQuerySuggestionsBlockList.html
     */
    toDeleteQuerySuggestionsBlockList() {
        return this.to('DeleteQuerySuggestionsBlockList');
    }
    /**
     * Grants permission to delete a Thesaurus
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DeleteThesaurus.html
     */
    toDeleteThesaurus() {
        return this.to('DeleteThesaurus');
    }
    /**
     * Grants permission to describe an access control configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeAccessControlConfiguration.html
     */
    toDescribeAccessControlConfiguration() {
        return this.to('DescribeAccessControlConfiguration');
    }
    /**
     * Grants permission to describe a data source
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeDataSource.html
     */
    toDescribeDataSource() {
        return this.to('DescribeDataSource');
    }
    /**
     * Gets information about your Amazon Kendra experience such as a search application
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeExperience.html
     */
    toDescribeExperience() {
        return this.to('DescribeExperience');
    }
    /**
     * Grants permission to describe an Faq
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeFaq.html
     */
    toDescribeFaq() {
        return this.to('DescribeFaq');
    }
    /**
     * Grants permission to describe a featured results set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeThesaurus.html
     */
    toDescribeFeaturedResultsSet() {
        return this.to('DescribeFeaturedResultsSet');
    }
    /**
     * Grants permission to describe an Index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeIndex.html
     */
    toDescribeIndex() {
        return this.to('DescribeIndex');
    }
    /**
     * Grants permission to describe principal mapping from index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribePrincipalMapping.html
     */
    toDescribePrincipalMapping() {
        return this.to('DescribePrincipalMapping');
    }
    /**
     * Grants permission to describe a QuerySuggestions BlockList
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeQuerySuggestionsBlockList.html
     */
    toDescribeQuerySuggestionsBlockList() {
        return this.to('DescribeQuerySuggestionsBlockList');
    }
    /**
     * Grants permission to describe the query suggestions configuration for an index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeQuerySuggestionsConfig.html
     */
    toDescribeQuerySuggestionsConfig() {
        return this.to('DescribeQuerySuggestionsConfig');
    }
    /**
     * Grants permission to describe a Thesaurus
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DescribeThesaurus.html
     */
    toDescribeThesaurus() {
        return this.to('DescribeThesaurus');
    }
    /**
     * Prevents users or groups in your AWS SSO identity source from accessing your Amazon Kendra experience
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DisassociateEntitiesFromExperience.html
     */
    toDisassociateEntitiesFromExperience() {
        return this.to('DisassociateEntitiesFromExperience');
    }
    /**
     * Removes the specific permissions of users or groups in your AWS SSO identity source with access to your Amazon Kendra experience
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_DisassociatePersonasFromEntities.html
     */
    toDisassociatePersonasFromEntities() {
        return this.to('DisassociatePersonasFromEntities');
    }
    /**
     * Grants permission to get suggestions for a query prefix
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_GetQuerySuggestions.html
     */
    toGetQuerySuggestions() {
        return this.to('GetQuerySuggestions');
    }
    /**
     * Retrieves search metrics data
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_GetSnapshots.html
     */
    toGetSnapshots() {
        return this.to('GetSnapshots');
    }
    /**
     * Grants permission to list the access control configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListAccessControlConfigurations.html
     */
    toListAccessControlConfigurations() {
        return this.to('ListAccessControlConfigurations');
    }
    /**
     * Grants permission to get Data Source sync job history
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListDataSourceSyncJobs.html
     */
    toListDataSourceSyncJobs() {
        return this.to('ListDataSourceSyncJobs');
    }
    /**
     * Grants permission to list the data sources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListDataSources.html
     */
    toListDataSources() {
        return this.to('ListDataSources');
    }
    /**
     * Lists specific permissions of users and groups with access to your Amazon Kendra experience
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListEntityPersonas.html
     */
    toListEntityPersonas() {
        return this.to('ListEntityPersonas');
    }
    /**
     * Lists users or groups in your AWS SSO identity source that are granted access to your Amazon Kendra experience
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListExperienceEntities.html
     */
    toListExperienceEntities() {
        return this.to('ListExperienceEntities');
    }
    /**
     * Lists one or more Amazon Kendra experiences. You can create an Amazon Kendra experience such as a search application
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListExperiences.html
     */
    toListExperiences() {
        return this.to('ListExperiences');
    }
    /**
     * Grants permission to list the Faqs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListFaqs.html
     */
    toListFaqs() {
        return this.to('ListFaqs');
    }
    /**
     * Grants permission to list the featured results sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListFeaturedResults.html
     */
    toListFeaturedResultsSets() {
        return this.to('ListFeaturedResultsSets');
    }
    /**
     * Grants permission to list groups that are older than an ordering id
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListGroupsOlderThanOrderingId.html
     */
    toListGroupsOlderThanOrderingId() {
        return this.to('ListGroupsOlderThanOrderingId');
    }
    /**
     * Grants permission to list the indexes
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListIndices.html
     */
    toListIndices() {
        return this.to('ListIndices');
    }
    /**
     * Grants permission to list the QuerySuggestions BlockLists
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListQuerySuggestionsBlockLists.html
     */
    toListQuerySuggestionsBlockLists() {
        return this.to('ListQuerySuggestionsBlockLists');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list the Thesauri
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_ListThesauri.html
     */
    toListThesauri() {
        return this.to('ListThesauri');
    }
    /**
     * Grants permission to put principal mapping in index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_PutPrincipalMapping.html
     */
    toPutPrincipalMapping() {
        return this.to('PutPrincipalMapping');
    }
    /**
     * Grants permission to query documents and faqs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_Query.html
     */
    toQuery() {
        return this.to('Query');
    }
    /**
     * Grants permission to retrieve relevant content from an index
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_Retrieve.html
     */
    toRetrieve() {
        return this.to('Retrieve');
    }
    /**
     * Grants permission to start Data Source sync job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_StartDataSourceSyncJob.html
     */
    toStartDataSourceSyncJob() {
        return this.to('StartDataSourceSyncJob');
    }
    /**
     * Grants permission to stop Data Source sync job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_StopDataSourceSyncJob.html
     */
    toStopDataSourceSyncJob() {
        return this.to('StopDataSourceSyncJob');
    }
    /**
     * Grants permission to send feedback about a query results
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_SubmitFeedback.html
     */
    toSubmitFeedback() {
        return this.to('SubmitFeedback');
    }
    /**
     * Grants permission to tag a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove the tag with the given key from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an access control configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateAccessControlConfiguration.html
     */
    toUpdateAccessControlConfiguration() {
        return this.to('UpdateAccessControlConfiguration');
    }
    /**
     * Grants permission to update a data source
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        return this.to('UpdateDataSource');
    }
    /**
     * Updates your Amazon Kendra experience such as a search application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateExperience.html
     */
    toUpdateExperience() {
        return this.to('UpdateExperience');
    }
    /**
     * Grants permission to update a featured results set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateFeaturedResults.html
     */
    toUpdateFeaturedResultsSet() {
        return this.to('UpdateFeaturedResultsSet');
    }
    /**
     * Grants permission to update an Index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateIndex.html
     */
    toUpdateIndex() {
        return this.to('UpdateIndex');
    }
    /**
     * Grants permission to update a QuerySuggestions BlockList
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateQuerySuggestionsBlockList.html
     */
    toUpdateQuerySuggestionsBlockList() {
        return this.to('UpdateQuerySuggestionsBlockList');
    }
    /**
     * Grants permission to update the query suggestions configuration for an index
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateQuerySuggestionsConfig.html
     */
    toUpdateQuerySuggestionsConfig() {
        return this.to('UpdateQuerySuggestionsConfig');
    }
    /**
     * Grants permission to update a thesaurus
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_UpdateThesaurus.html
     */
    toUpdateThesaurus() {
        return this.to('UpdateThesaurus');
    }
    /**
     * Adds a resource of type index to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/index.html
     *
     * @param indexId - Identifier for the indexId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIndex(indexId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kendra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:index/${indexId}`);
    }
    /**
     * Adds a resource of type data-source to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/data-source.html
     *
     * @param indexId - Identifier for the indexId.
     * @param dataSourceId - Identifier for the dataSourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDataSource(indexId, dataSourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kendra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:index/${indexId}/data-source/${dataSourceId}`);
    }
    /**
     * Adds a resource of type faq to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/faq.html
     *
     * @param indexId - Identifier for the indexId.
     * @param faqId - Identifier for the faqId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFaq(indexId, faqId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kendra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:index/${indexId}/faq/${faqId}`);
    }
    /**
     * Adds a resource of type experience to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/deploying-search-experience-no-code.html
     *
     * @param indexId - Identifier for the indexId.
     * @param experienceId - Identifier for the experienceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onExperience(indexId, experienceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kendra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:index/${indexId}/experience/${experienceId}`);
    }
    /**
     * Adds a resource of type thesaurus to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/thesaurus.html
     *
     * @param indexId - Identifier for the indexId.
     * @param thesaurusId - Identifier for the thesaurusId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onThesaurus(indexId, thesaurusId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kendra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:index/${indexId}/thesaurus/${thesaurusId}`);
    }
    /**
     * Adds a resource of type query-suggestions-block-list to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/query-suggestions-block-list.html
     *
     * @param indexId - Identifier for the indexId.
     * @param querySuggestionsBlockListId - Identifier for the querySuggestionsBlockListId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQuerySuggestionsBlockList(indexId, querySuggestionsBlockListId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kendra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:index/${indexId}/query-suggestions-block-list/${querySuggestionsBlockListId}`);
    }
    /**
     * Adds a resource of type featured-results-set to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/featured-results.html
     *
     * @param indexId - Identifier for the indexId.
     * @param featuredResultsSetId - Identifier for the featuredResultsSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFeaturedResultsSet(indexId, featuredResultsSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kendra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:index/${indexId}/featured-results-set/${featuredResultsSetId}`);
    }
    /**
     * Adds a resource of type access-control-configuration to the statement
     *
     * https://docs.aws.amazon.com/kendra/latest/dg/API_CreateAccessControlConfiguration.html
     *
     * @param indexId - Identifier for the indexId.
     * @param accessControlConfigurationId - Identifier for the accessControlConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAccessControlConfiguration(indexId, accessControlConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kendra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:index/${indexId}/access-control-configuration/${accessControlConfigurationId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateDataSource()
     * - .toCreateFaq()
     * - .toCreateFeaturedResultsSet()
     * - .toCreateIndex()
     * - .toCreateQuerySuggestionsBlockList()
     * - .toCreateThesaurus()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - index
     * - data-source
     * - faq
     * - thesaurus
     * - query-suggestions-block-list
     * - featured-results-set
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateDataSource()
     * - .toCreateFaq()
     * - .toCreateFeaturedResultsSet()
     * - .toCreateIndex()
     * - .toCreateQuerySuggestionsBlockList()
     * - .toCreateThesaurus()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [kendra](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkendra.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'kendra';
        this.accessLevelList = {
            Write: [
                'AssociateEntitiesToExperience',
                'AssociatePersonasToEntities',
                'BatchDeleteDocument',
                'BatchDeleteFeaturedResultsSet',
                'BatchPutDocument',
                'ClearQuerySuggestions',
                'CreateAccessControlConfiguration',
                'CreateDataSource',
                'CreateExperience',
                'CreateFaq',
                'CreateFeaturedResultsSet',
                'CreateIndex',
                'CreateQuerySuggestionsBlockList',
                'CreateThesaurus',
                'DeleteAccessControlConfiguration',
                'DeleteDataSource',
                'DeleteExperience',
                'DeleteFaq',
                'DeleteIndex',
                'DeletePrincipalMapping',
                'DeleteQuerySuggestionsBlockList',
                'DeleteThesaurus',
                'DisassociateEntitiesFromExperience',
                'DisassociatePersonasFromEntities',
                'PutPrincipalMapping',
                'StartDataSourceSyncJob',
                'StopDataSourceSyncJob',
                'SubmitFeedback',
                'UpdateAccessControlConfiguration',
                'UpdateDataSource',
                'UpdateExperience',
                'UpdateFeaturedResultsSet',
                'UpdateIndex',
                'UpdateQuerySuggestionsBlockList',
                'UpdateQuerySuggestionsConfig',
                'UpdateThesaurus'
            ],
            Read: [
                'BatchGetDocumentStatus',
                'DescribeAccessControlConfiguration',
                'DescribeDataSource',
                'DescribeExperience',
                'DescribeFaq',
                'DescribeFeaturedResultsSet',
                'DescribeIndex',
                'DescribePrincipalMapping',
                'DescribeQuerySuggestionsBlockList',
                'DescribeQuerySuggestionsConfig',
                'DescribeThesaurus',
                'GetQuerySuggestions',
                'GetSnapshots',
                'ListTagsForResource',
                'Query',
                'Retrieve'
            ],
            List: [
                'ListAccessControlConfigurations',
                'ListDataSourceSyncJobs',
                'ListDataSources',
                'ListEntityPersonas',
                'ListExperienceEntities',
                'ListExperiences',
                'ListFaqs',
                'ListFeaturedResultsSets',
                'ListGroupsOlderThanOrderingId',
                'ListIndices',
                'ListQuerySuggestionsBlockLists',
                'ListThesauri'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Kendra = Kendra;
//# sourceMappingURL=data:application/json;base64,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