"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kms = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [kms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awskeymanagementservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kms extends shared_1.PolicyStatement {
    /**
     * Controls permission to cancel the scheduled deletion of an AWS KMS key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CancelKeyDeletion.html
     */
    toCancelKeyDeletion() {
        return this.to('CancelKeyDeletion');
    }
    /**
     * Controls permission to connect or reconnect a custom key store to its associated AWS CloudHSM cluster or external key manager outside of AWS
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ConnectCustomKeyStore.html
     */
    toConnectCustomKeyStore() {
        return this.to('ConnectCustomKeyStore');
    }
    /**
     * Controls permission to create an alias for an AWS KMS key. Aliases are optional friendly names that you can associate with KMS keys
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateAlias.html
     */
    toCreateAlias() {
        return this.to('CreateAlias');
    }
    /**
     * Controls permission to create a custom key store that is backed by an AWS CloudHSM cluster or an external key manager outside of AWS
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     *
     * Dependent actions:
     * - cloudhsm:DescribeClusters
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateCustomKeyStore.html
     */
    toCreateCustomKeyStore() {
        return this.to('CreateCustomKeyStore');
    }
    /**
     * Controls permission to add a grant to an AWS KMS key. You can use grants to add permissions without changing the key policy or IAM policy
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionContext()
     * - .ifEncryptionContextKeys()
     * - .ifGrantConstraintType()
     * - .ifGranteePrincipal()
     * - .ifGrantIsForAWSResource()
     * - .ifGrantOperations()
     * - .ifRetiringPrincipal()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateGrant.html
     */
    toCreateGrant() {
        return this.to('CreateGrant');
    }
    /**
     * Controls permission to create an AWS KMS key that can be used to protect data keys and other sensitive information
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifBypassPolicyLockoutSafetyCheck()
     * - .ifCallerAccount()
     * - .ifKeySpec()
     * - .ifKeyUsage()
     * - .ifKeyOrigin()
     * - .ifMultiRegion()
     * - .ifMultiRegionKeyType()
     * - .ifViaService()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - kms:PutKeyPolicy
     * - kms:TagResource
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_CreateKey.html
     */
    toCreateKey() {
        return this.to('CreateKey');
    }
    /**
     * Controls permission to decrypt ciphertext that was encrypted under an AWS KMS key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContext()
     * - .ifEncryptionContextKeys()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Decrypt.html
     */
    toDecrypt() {
        return this.to('Decrypt');
    }
    /**
     * Controls permission to delete an alias. Aliases are optional friendly names that you can associate with AWS KMS keys
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteAlias.html
     */
    toDeleteAlias() {
        return this.to('DeleteAlias');
    }
    /**
     * Controls permission to delete a custom key store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteCustomKeyStore.html
     */
    toDeleteCustomKeyStore() {
        return this.to('DeleteCustomKeyStore');
    }
    /**
     * Controls permission to delete cryptographic material that you imported into an AWS KMS key. This action makes the key unusable
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeleteImportedKeyMaterial.html
     */
    toDeleteImportedKeyMaterial() {
        return this.to('DeleteImportedKeyMaterial');
    }
    /**
     * Controls permission to use the specified AWS KMS key to derive shared secrets
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifKeyAgreementAlgorithm()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DeriveSharedSecret.html
     */
    toDeriveSharedSecret() {
        return this.to('DeriveSharedSecret');
    }
    /**
     * Controls permission to view detailed information about custom key stores in the account and region
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeCustomKeyStores.html
     */
    toDescribeCustomKeyStores() {
        return this.to('DescribeCustomKeyStores');
    }
    /**
     * Controls permission to view detailed information about an AWS KMS key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DescribeKey.html
     */
    toDescribeKey() {
        return this.to('DescribeKey');
    }
    /**
     * Controls permission to disable an AWS KMS key, which prevents it from being used in cryptographic operations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKey.html
     */
    toDisableKey() {
        return this.to('DisableKey');
    }
    /**
     * Controls permission to disable automatic rotation of a customer managed AWS KMS key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisableKeyRotation.html
     */
    toDisableKeyRotation() {
        return this.to('DisableKeyRotation');
    }
    /**
     * Controls permission to disconnect the custom key store from its associated AWS CloudHSM cluster or external key manager outside of AWS
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_DisconnectCustomKeyStore.html
     */
    toDisconnectCustomKeyStore() {
        return this.to('DisconnectCustomKeyStore');
    }
    /**
     * Controls permission to change the state of an AWS KMS key to enabled. This allows the KMS key to be used in cryptographic operations
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKey.html
     */
    toEnableKey() {
        return this.to('EnableKey');
    }
    /**
     * Controls permission to enable automatic rotation of the cryptographic material in an AWS KMS key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifRotationPeriodInDays()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_EnableKeyRotation.html
     */
    toEnableKeyRotation() {
        return this.to('EnableKeyRotation');
    }
    /**
     * Controls permission to use the specified AWS KMS key to encrypt data and data keys
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContext()
     * - .ifEncryptionContextKeys()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Encrypt.html
     */
    toEncrypt() {
        return this.to('Encrypt');
    }
    /**
     * Controls permission to use the AWS KMS key to generate data keys. You can use the data keys to encrypt data outside of AWS KMS
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContext()
     * - .ifEncryptionContextKeys()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKey.html
     */
    toGenerateDataKey() {
        return this.to('GenerateDataKey');
    }
    /**
     * Controls permission to use the AWS KMS key to generate data key pairs
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifDataKeyPairSpec()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContext()
     * - .ifEncryptionContextKeys()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPair.html
     */
    toGenerateDataKeyPair() {
        return this.to('GenerateDataKeyPair');
    }
    /**
     * Controls permission to use the AWS KMS key to generate data key pairs. Unlike the GenerateDataKeyPair operation, this operation returns an encrypted private key without a plaintext copy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifDataKeyPairSpec()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContext()
     * - .ifEncryptionContextKeys()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyPairWithoutPlaintext.html
     */
    toGenerateDataKeyPairWithoutPlaintext() {
        return this.to('GenerateDataKeyPairWithoutPlaintext');
    }
    /**
     * Controls permission to use the AWS KMS key to generate a data key. Unlike the GenerateDataKey operation, this operation returns an encrypted data key without a plaintext version of the data key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContext()
     * - .ifEncryptionContextKeys()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateDataKeyWithoutPlaintext.html
     */
    toGenerateDataKeyWithoutPlaintext() {
        return this.to('GenerateDataKeyWithoutPlaintext');
    }
    /**
     * Controls permission to use the AWS KMS key to generate message authentication codes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifMacAlgorithm()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateMac.html
     */
    toGenerateMac() {
        return this.to('GenerateMac');
    }
    /**
     * Controls permission to get a cryptographically secure random byte string from AWS KMS
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     * - .ifRecipientAttestation()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GenerateRandom.html
     */
    toGenerateRandom() {
        return this.to('GenerateRandom');
    }
    /**
     * Controls permission to view the key policy for the specified AWS KMS key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyPolicy.html
     */
    toGetKeyPolicy() {
        return this.to('GetKeyPolicy');
    }
    /**
     * Controls permission to view the key rotation status for an AWS KMS key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetKeyRotationStatus.html
     */
    toGetKeyRotationStatus() {
        return this.to('GetKeyRotationStatus');
    }
    /**
     * Controls permission to get data that is required to import cryptographic material into a customer managed key, including a public key and import token
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     * - .ifWrappingAlgorithm()
     * - .ifWrappingKeySpec()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetParametersForImport.html
     */
    toGetParametersForImport() {
        return this.to('GetParametersForImport');
    }
    /**
     * Controls permission to download the public key of an asymmetric AWS KMS key
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_GetPublicKey.html
     */
    toGetPublicKey() {
        return this.to('GetPublicKey');
    }
    /**
     * Controls permission to import cryptographic material into an AWS KMS key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifExpirationModel()
     * - .ifValidTo()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ImportKeyMaterial.html
     */
    toImportKeyMaterial() {
        return this.to('ImportKeyMaterial');
    }
    /**
     * Controls permission to view the aliases that are defined in the account. Aliases are optional friendly names that you can associate with AWS KMS keys
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListAliases.html
     */
    toListAliases() {
        return this.to('ListAliases');
    }
    /**
     * Controls permission to view all grants for an AWS KMS key
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListGrants.html
     */
    toListGrants() {
        return this.to('ListGrants');
    }
    /**
     * Controls permission to view the names of key policies for an AWS KMS key
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeyPolicies.html
     */
    toListKeyPolicies() {
        return this.to('ListKeyPolicies');
    }
    /**
     * Controls permission to view the list of key materials for an AWS KMS key
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeyRotations.html
     */
    toListKeyRotations() {
        return this.to('ListKeyRotations');
    }
    /**
     * Controls permission to view the key ID and Amazon Resource Name (ARN) of all AWS KMS keys in the account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListKeys.html
     */
    toListKeys() {
        return this.to('ListKeys');
    }
    /**
     * Controls permission to view all tags that are attached to an AWS KMS key
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListResourceTags.html
     */
    toListResourceTags() {
        return this.to('ListResourceTags');
    }
    /**
     * Controls permission to view grants in which the specified principal is the retiring principal. Other principals might be able to retire the grant and this principal might be able to retire other grants
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ListRetirableGrants.html
     */
    toListRetirableGrants() {
        return this.to('ListRetirableGrants');
    }
    /**
     * Controls permission to replace the key policy for the specified AWS KMS key
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifBypassPolicyLockoutSafetyCheck()
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_PutKeyPolicy.html
     */
    toPutKeyPolicy() {
        return this.to('PutKeyPolicy');
    }
    /**
     * Controls permission to decrypt data as part of the process that decrypts and reencrypts the data within AWS KMS
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContext()
     * - .ifEncryptionContextKeys()
     * - .ifReEncryptOnSameKey()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html
     */
    toReEncryptFrom() {
        return this.to('ReEncryptFrom');
    }
    /**
     * Controls permission to encrypt data as part of the process that decrypts and reencrypts the data within AWS KMS
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionAlgorithm()
     * - .ifEncryptionContext()
     * - .ifEncryptionContextKeys()
     * - .ifReEncryptOnSameKey()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReEncrypt.html
     */
    toReEncryptTo() {
        return this.to('ReEncryptTo');
    }
    /**
     * Controls permission to replicate a multi-Region primary key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifReplicaRegion()
     * - .ifViaService()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - kms:CreateKey
     * - kms:PutKeyPolicy
     * - kms:TagResource
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ReplicateKey.html
     */
    toReplicateKey() {
        return this.to('ReplicateKey');
    }
    /**
     * Controls permission to retire a grant. The RetireGrant operation is typically called by the grant user after they complete the tasks that the grant allowed them to perform
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifEncryptionContext()
     * - .ifEncryptionContextKeys()
     * - .ifGrantConstraintType()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RetireGrant.html
     */
    toRetireGrant() {
        return this.to('RetireGrant');
    }
    /**
     * Controls permission to revoke a grant, which denies permission for all operations that depend on the grant
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifGrantIsForAWSResource()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RevokeGrant.html
     */
    toRevokeGrant() {
        return this.to('RevokeGrant');
    }
    /**
     * Controls permission to invoke on-demand rotation of the cryptographic material in an AWS KMS key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_RotateKeyOnDemand.html
     */
    toRotateKeyOnDemand() {
        return this.to('RotateKeyOnDemand');
    }
    /**
     * Controls permission to schedule deletion of an AWS KMS key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifScheduleKeyDeletionPendingWindowInDays()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_ScheduleKeyDeletion.html
     */
    toScheduleKeyDeletion() {
        return this.to('ScheduleKeyDeletion');
    }
    /**
     * Controls permission to produce a digital signature for a message
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifMessageType()
     * - .ifRequestAlias()
     * - .ifSigningAlgorithm()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Sign.html
     */
    toSign() {
        return this.to('Sign');
    }
    /**
     * Controls access to internal APIs that synchronize multi-Region keys
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/multi-region-keys-auth.html#multi-region-auth-slr
     */
    toSynchronizeMultiRegionKey() {
        return this.to('SynchronizeMultiRegionKey');
    }
    /**
     * Controls permission to create or update tags that are attached to an AWS KMS key
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Controls permission to delete tags that are attached to an AWS KMS key
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Controls permission to associate an alias with a different AWS KMS key. An alias is an optional friendly name that you can associate with a KMS key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateAlias.html
     */
    toUpdateAlias() {
        return this.to('UpdateAlias');
    }
    /**
     * Controls permission to change the properties of a custom key store
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateCustomKeyStore.html
     */
    toUpdateCustomKeyStore() {
        return this.to('UpdateCustomKeyStore');
    }
    /**
     * Controls permission to delete or change the description of an AWS KMS key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdateKeyDescription.html
     */
    toUpdateKeyDescription() {
        return this.to('UpdateKeyDescription');
    }
    /**
     * Controls permission to update the primary Region of a multi-Region primary key
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifPrimaryRegion()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_UpdatePrimaryRegion.html
     */
    toUpdatePrimaryRegion() {
        return this.to('UpdatePrimaryRegion');
    }
    /**
     * Controls permission to use the specified AWS KMS key to verify digital signatures
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifMessageType()
     * - .ifRequestAlias()
     * - .ifSigningAlgorithm()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_Verify.html
     */
    toVerify() {
        return this.to('Verify');
    }
    /**
     * Controls permission to use the AWS KMS key to verify message authentication codes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifCallerAccount()
     * - .ifMacAlgorithm()
     * - .ifRequestAlias()
     * - .ifViaService()
     *
     * https://docs.aws.amazon.com/kms/latest/APIReference/API_VerifyMac.html
     */
    toVerifyMac() {
        return this.to('VerifyMac');
    }
    /**
     * Adds a resource of type alias to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#alias-concept
     *
     * @param alias - Identifier for the alias.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAlias(alias, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:alias/${alias}`);
    }
    /**
     * Adds a resource of type key to the statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#kms_keys
     *
     * @param keyId - Identifier for the keyId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifKeyOrigin()
     * - .ifKeySpec()
     * - .ifKeyUsage()
     * - .ifMultiRegion()
     * - .ifMultiRegionKeyType()
     * - .ifResourceAliases()
     */
    onKey(keyId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:kms:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:key/${keyId}`);
    }
    /**
     * Filters access to the specified AWS KMS operations based on both the key and value of the tag in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateKey()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the specified AWS KMS operations based on tags assigned to the AWS KMS key
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/tag-authorization.html
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * Applies to resource types:
     * - key
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the specified AWS KMS operations based on tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateKey()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the CreateKey and PutKeyPolicy operations based on the value of the BypassPolicyLockoutSafetyCheck parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-bypass-policy-lockout-safety-check
     *
     * Applies to actions:
     * - .toCreateKey()
     * - .toPutKeyPolicy()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifBypassPolicyLockoutSafetyCheck(value) {
        return this.if(`BypassPolicyLockoutSafetyCheck`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access to specified AWS KMS operations based on the AWS account ID of the caller. You can use this condition key to allow or deny access to all IAM users and roles in an AWS account in a single policy statement
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-caller-account
     *
     * Applies to actions:
     * - .toCancelKeyDeletion()
     * - .toConnectCustomKeyStore()
     * - .toCreateAlias()
     * - .toCreateCustomKeyStore()
     * - .toCreateGrant()
     * - .toCreateKey()
     * - .toDecrypt()
     * - .toDeleteAlias()
     * - .toDeleteCustomKeyStore()
     * - .toDeleteImportedKeyMaterial()
     * - .toDeriveSharedSecret()
     * - .toDescribeCustomKeyStores()
     * - .toDescribeKey()
     * - .toDisableKey()
     * - .toDisableKeyRotation()
     * - .toDisconnectCustomKeyStore()
     * - .toEnableKey()
     * - .toEnableKeyRotation()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toGenerateMac()
     * - .toGetKeyPolicy()
     * - .toGetKeyRotationStatus()
     * - .toGetParametersForImport()
     * - .toGetPublicKey()
     * - .toImportKeyMaterial()
     * - .toListGrants()
     * - .toListKeyPolicies()
     * - .toListKeyRotations()
     * - .toListResourceTags()
     * - .toPutKeyPolicy()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     * - .toReplicateKey()
     * - .toRetireGrant()
     * - .toRevokeGrant()
     * - .toRotateKeyOnDemand()
     * - .toScheduleKeyDeletion()
     * - .toSign()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAlias()
     * - .toUpdateCustomKeyStore()
     * - .toUpdateKeyDescription()
     * - .toUpdatePrimaryRegion()
     * - .toVerify()
     * - .toVerifyMac()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCallerAccount(value, operator) {
        return this.if(`CallerAccount`, value, operator ?? 'StringLike');
    }
    /**
     * The kms:CustomerMasterKeySpec condition key is deprecated. Instead, use the kms:KeySpec condition key
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-key-spec-replaced
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMasterKeySpec(value, operator) {
        return this.if(`CustomerMasterKeySpec`, value, operator ?? 'StringLike');
    }
    /**
     * The kms:CustomerMasterKeyUsage condition key is deprecated. Instead, use the kms:KeyUsage condition key
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-key-usage-replaced
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifCustomerMasterKeyUsage(value, operator) {
        return this.if(`CustomerMasterKeyUsage`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to GenerateDataKeyPair and GenerateDataKeyPairWithoutPlaintext operations based on the value of the KeyPairSpec parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-data-key-pair-spec
     *
     * Applies to actions:
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataKeyPairSpec(value, operator) {
        return this.if(`DataKeyPairSpec`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to encryption operations based on the value of the encryption algorithm in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-encryption-algorithm
     *
     * Applies to actions:
     * - .toDecrypt()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionAlgorithm(value, operator) {
        return this.if(`EncryptionAlgorithm`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to a symmetric AWS KMS key based on the encryption context in a cryptographic operation. This condition evaluates the key and value in each key-value encryption context pair
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-encryption-context
     *
     * Applies to actions:
     * - .toCreateGrant()
     * - .toDecrypt()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     * - .toRetireGrant()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionContext(value, operator) {
        return this.if(`EncryptionContext`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to a symmetric AWS KMS key based on the encryption context in a cryptographic operation. This condition key evaluates only the key in each key-value encryption context pair
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-encryption-context-keys
     *
     * Applies to actions:
     * - .toCreateGrant()
     * - .toDecrypt()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     * - .toRetireGrant()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEncryptionContextKeys(value, operator) {
        return this.if(`EncryptionContextKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the ImportKeyMaterial operation based on the value of the ExpirationModel parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-expiration-model
     *
     * Applies to actions:
     * - .toImportKeyMaterial()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExpirationModel(value, operator) {
        return this.if(`ExpirationModel`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the CreateGrant operation based on the grant constraint in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-grant-constraint-type
     *
     * Applies to actions:
     * - .toCreateGrant()
     * - .toRetireGrant()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGrantConstraintType(value, operator) {
        return this.if(`GrantConstraintType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the CreateGrant operation when the request comes from a specified AWS service
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-grant-is-for-aws-resource
     *
     * Applies to actions:
     * - .toCreateGrant()
     * - .toListGrants()
     * - .toRevokeGrant()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifGrantIsForAWSResource(value) {
        return this.if(`GrantIsForAWSResource`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access to the CreateGrant operation based on the operations in the grant
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-grant-operations
     *
     * Applies to actions:
     * - .toCreateGrant()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGrantOperations(value, operator) {
        return this.if(`GrantOperations`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the CreateGrant operation based on the grantee principal in the grant
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-grantee-principal
     *
     * Applies to actions:
     * - .toCreateGrant()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifGranteePrincipal(value, operator) {
        return this.if(`GranteePrincipal`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the DeriveSharedSecret operation based on the value of the KeyAgreementAlgorithm parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-key-agreement-algorithm
     *
     * Applies to actions:
     * - .toDeriveSharedSecret()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyAgreementAlgorithm(value, operator) {
        return this.if(`KeyAgreementAlgorithm`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to an API operation based on the Origin property of the AWS KMS key created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a KMS key
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-key-origin
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * Applies to resource types:
     * - key
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyOrigin(value, operator) {
        return this.if(`KeyOrigin`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to an API operation based on the KeySpec property of the AWS KMS key that is created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a KMS key resource
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-key-spec
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * Applies to resource types:
     * - key
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeySpec(value, operator) {
        return this.if(`KeySpec`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to an API operation based on the KeyUsage property of the AWS KMS key created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a KMS key resource
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-key-usage
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * Applies to resource types:
     * - key
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifKeyUsage(value, operator) {
        return this.if(`KeyUsage`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the GenerateMac and VerifyMac operations based on the MacAlgorithm parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-mac-algorithm
     *
     * Applies to actions:
     * - .toGenerateMac()
     * - .toVerifyMac()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMacAlgorithm(value, operator) {
        return this.if(`MacAlgorithm`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the Sign and Verify operations based on the value of the MessageType parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-message-type
     *
     * Applies to actions:
     * - .toSign()
     * - .toVerify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMessageType(value, operator) {
        return this.if(`MessageType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to an API operation based on the MultiRegion property of the AWS KMS key created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a KMS key resource
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-multi-region
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * Applies to resource types:
     * - key
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifMultiRegion(value) {
        return this.if(`MultiRegion`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access to an API operation based on the MultiRegionKeyType property of the AWS KMS key created by or used in the operation. Use it to qualify authorization of the CreateKey operation or any operation that is authorized for a KMS key resource
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-multi-region-key-type
     *
     * Applies to actions:
     * - .toCreateKey()
     *
     * Applies to resource types:
     * - key
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifMultiRegionKeyType(value, operator) {
        return this.if(`MultiRegionKeyType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the UpdatePrimaryRegion operation based on the value of the PrimaryRegion parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-primary-region
     *
     * Applies to actions:
     * - .toUpdatePrimaryRegion()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrimaryRegion(value, operator) {
        return this.if(`PrimaryRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the ReEncrypt operation when it uses the same AWS KMS key that was used for the Encrypt operation
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-reencrypt-on-same-key
     *
     * Applies to actions:
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifReEncryptOnSameKey(value) {
        return this.if(`ReEncryptOnSameKey`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the platform configuration register (PCR) 9 in the attestation document in the request. PCR9 is a custom PCR that can be defined by the user for specific use cases
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-nitro-enclaves.html#conditions-kms-recipient-pcrs
     *
     * Applies to actions:
     * - .toDecrypt()
     * - .toDeriveSharedSecret()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateRandom()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRecipientAttestation(value, operator) {
        return this.if(`RecipientAttestation`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the ReplicateKey operation based on the value of the ReplicaRegion parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-replica-region
     *
     * Applies to actions:
     * - .toReplicateKey()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifReplicaRegion(value, operator) {
        return this.if(`ReplicaRegion`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to cryptographic operations, DescribeKey, and GetPublicKey based on the alias in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-request-alias
     *
     * Applies to actions:
     * - .toDecrypt()
     * - .toDeriveSharedSecret()
     * - .toDescribeKey()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toGenerateMac()
     * - .toGetPublicKey()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     * - .toSign()
     * - .toVerify()
     * - .toVerifyMac()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestAlias(value, operator) {
        return this.if(`RequestAlias`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to specified AWS KMS operations based on aliases associated with the AWS KMS key
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-resource-aliases
     *
     * Applies to resource types:
     * - key
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceAliases(value, operator) {
        return this.if(`ResourceAliases`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the CreateGrant operation based on the retiring principal in the grant
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-retiring-principal
     *
     * Applies to actions:
     * - .toCreateGrant()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRetiringPrincipal(value, operator) {
        return this.if(`RetiringPrincipal`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the EnableKeyRotation operation based on the value of the RotationPeriodInDays parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-rotation-period-in-days
     *
     * Applies to actions:
     * - .toEnableKeyRotation()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRotationPeriodInDays(value, operator) {
        return this.if(`RotationPeriodInDays`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access to the ScheduleKeyDeletion operation based on the value of the PendingWindowInDays parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-schedule-key-deletion-pending-window-in-days
     *
     * Applies to actions:
     * - .toScheduleKeyDeletion()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifScheduleKeyDeletionPendingWindowInDays(value, operator) {
        return this.if(`ScheduleKeyDeletionPendingWindowInDays`, value, operator ?? 'NumericEquals');
    }
    /**
     * Filters access to the Sign and Verify operations based on the signing algorithm in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-signing-algorithm
     *
     * Applies to actions:
     * - .toSign()
     * - .toVerify()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSigningAlgorithm(value, operator) {
        return this.if(`SigningAlgorithm`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the ImportKeyMaterial operation based on the value of the ValidTo parameter in the request. You can use this condition key to allow users to import key material only when it expires by the specified date
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-valid-to
     *
     * Applies to actions:
     * - .toImportKeyMaterial()
     *
     * @param value The value(s) to check
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateEquals`
     */
    ifValidTo(value, operator) {
        if (typeof value.getMonth === "function") {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === "function") {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if(`ValidTo`, value, operator ?? 'DateEquals');
    }
    /**
     * Filters access when a request made on the principal's behalf comes from a specified AWS service
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-via-service
     *
     * Applies to actions:
     * - .toCancelKeyDeletion()
     * - .toCreateAlias()
     * - .toCreateGrant()
     * - .toCreateKey()
     * - .toDecrypt()
     * - .toDeleteAlias()
     * - .toDeleteImportedKeyMaterial()
     * - .toDeriveSharedSecret()
     * - .toDescribeKey()
     * - .toDisableKey()
     * - .toDisableKeyRotation()
     * - .toEnableKey()
     * - .toEnableKeyRotation()
     * - .toEncrypt()
     * - .toGenerateDataKey()
     * - .toGenerateDataKeyPair()
     * - .toGenerateDataKeyPairWithoutPlaintext()
     * - .toGenerateDataKeyWithoutPlaintext()
     * - .toGenerateMac()
     * - .toGetKeyPolicy()
     * - .toGetKeyRotationStatus()
     * - .toGetParametersForImport()
     * - .toGetPublicKey()
     * - .toImportKeyMaterial()
     * - .toListGrants()
     * - .toListKeyPolicies()
     * - .toListKeyRotations()
     * - .toListResourceTags()
     * - .toPutKeyPolicy()
     * - .toReEncryptFrom()
     * - .toReEncryptTo()
     * - .toReplicateKey()
     * - .toRetireGrant()
     * - .toRevokeGrant()
     * - .toRotateKeyOnDemand()
     * - .toScheduleKeyDeletion()
     * - .toSign()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAlias()
     * - .toUpdateKeyDescription()
     * - .toUpdatePrimaryRegion()
     * - .toVerify()
     * - .toVerifyMac()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifViaService(value, operator) {
        return this.if(`ViaService`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the GetParametersForImport operation based on the value of the WrappingAlgorithm parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-wrapping-algorithm
     *
     * Applies to actions:
     * - .toGetParametersForImport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingAlgorithm(value, operator) {
        return this.if(`WrappingAlgorithm`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access to the GetParametersForImport operation based on the value of the WrappingKeySpec parameter in the request
     *
     * https://docs.aws.amazon.com/kms/latest/developerguide/conditions-kms.html#conditions-kms-wrapping-key-spec
     *
     * Applies to actions:
     * - .toGetParametersForImport()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifWrappingKeySpec(value, operator) {
        return this.if(`WrappingKeySpec`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [kms](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awskeymanagementservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'kms';
        this.accessLevelList = {
            Write: [
                'CancelKeyDeletion',
                'ConnectCustomKeyStore',
                'CreateAlias',
                'CreateCustomKeyStore',
                'CreateKey',
                'Decrypt',
                'DeleteAlias',
                'DeleteCustomKeyStore',
                'DeleteImportedKeyMaterial',
                'DeriveSharedSecret',
                'DisableKey',
                'DisableKeyRotation',
                'DisconnectCustomKeyStore',
                'EnableKey',
                'EnableKeyRotation',
                'Encrypt',
                'GenerateDataKey',
                'GenerateDataKeyPair',
                'GenerateDataKeyPairWithoutPlaintext',
                'GenerateDataKeyWithoutPlaintext',
                'GenerateMac',
                'GenerateRandom',
                'ImportKeyMaterial',
                'ReEncryptFrom',
                'ReEncryptTo',
                'ReplicateKey',
                'RotateKeyOnDemand',
                'ScheduleKeyDeletion',
                'Sign',
                'SynchronizeMultiRegionKey',
                'UpdateAlias',
                'UpdateCustomKeyStore',
                'UpdateKeyDescription',
                'UpdatePrimaryRegion',
                'Verify',
                'VerifyMac'
            ],
            'Permissions management': [
                'CreateGrant',
                'PutKeyPolicy',
                'RetireGrant',
                'RevokeGrant'
            ],
            Read: [
                'DescribeCustomKeyStores',
                'DescribeKey',
                'GetKeyPolicy',
                'GetKeyRotationStatus',
                'GetParametersForImport',
                'GetPublicKey'
            ],
            List: [
                'ListAliases',
                'ListGrants',
                'ListKeyPolicies',
                'ListKeyRotations',
                'ListKeys',
                'ListResourceTags',
                'ListRetirableGrants'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Kms = Kms;
//# sourceMappingURL=data:application/json;base64,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