"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cassandra = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [cassandra](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkeyspacesforapachecassandra.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cassandra extends shared_1.PolicyStatement {
    /**
     * Grants permission to alter a keyspace or table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toAlter() {
        return this.to('Alter');
    }
    /**
     * Grants permission to alter a multiregion keyspace or table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toAlterMultiRegionResource() {
        return this.to('AlterMultiRegionResource');
    }
    /**
     * Grants permission to create a keyspace or table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toCreate() {
        return this.to('Create');
    }
    /**
     * Grants permission to create a multiregion keyspace or table
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toCreateMultiRegionResource() {
        return this.to('CreateMultiRegionResource');
    }
    /**
     * Grants permission to drop a keyspace or table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toDrop() {
        return this.to('Drop');
    }
    /**
     * Grants permission to drop a multiregion keyspace or table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toDropMultiRegionResource() {
        return this.to('DropMultiRegionResource');
    }
    /**
     * Grants permission to retrieve the CDC stream records from a given shard
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toGetRecords() {
        return this.to('GetRecords');
    }
    /**
     * Grants permission to return a shard iterator
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toGetShardIterator() {
        return this.to('GetShardIterator');
    }
    /**
     * Grants permission to return information about a CDC stream, including the composition of its shards
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toGetStream() {
        return this.to('GetStream');
    }
    /**
     * Grants permission to return an array of CDC stream ARNs associated with the current account and endpoint
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toListStreams() {
        return this.to('ListStreams');
    }
    /**
     * Grants permission to INSERT, UPDATE or DELETE data in a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toModify() {
        return this.to('Modify');
    }
    /**
     * Grants permission to INSERT, UPDATE or DELETE data in a multiregion table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toModifyMultiRegionResource() {
        return this.to('ModifyMultiRegionResource');
    }
    /**
     * Grants permission to restore table from a backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toRestore() {
        return this.to('Restore');
    }
    /**
     * Grants permission to restore multiregion table from a backup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toRestoreMultiRegionTable() {
        return this.to('RestoreMultiRegionTable');
    }
    /**
     * Grants permission to SELECT data from a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toSelect() {
        return this.to('Select');
    }
    /**
     * Grants permission to SELECT data from a multiregion table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toSelectMultiRegionResource() {
        return this.to('SelectMultiRegionResource');
    }
    /**
     * Grants permission to tag a multiregion keyspace or table
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toTagMultiRegionResource() {
        return this.to('TagMultiRegionResource');
    }
    /**
     * Grants permission to tag a keyspace, table, or stream
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a multiregion keyspace or table
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toUnTagMultiRegionResource() {
        return this.to('UnTagMultiRegionResource');
    }
    /**
     * Grants permission to untag a keyspace, table or stream
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to UPDATE the partitioner in a system table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/
     */
    toUpdatePartitioner() {
        return this.to('UpdatePartitioner');
    }
    /**
     * Adds a resource of type keyspace to the statement
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/what-is.html
     *
     * @param keyspaceName - Identifier for the keyspaceName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onKeyspace(keyspaceName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cassandra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/keyspace/${keyspaceName}`);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/what-is.html
     *
     * @param keyspaceName - Identifier for the keyspaceName.
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTable(keyspaceName, tableName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cassandra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/keyspace/${keyspaceName}/table/${tableName}`);
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/what-is.html
     *
     * @param keyspaceName - Identifier for the keyspaceName.
     * @param tableName - Identifier for the tableName.
     * @param streamLabel - Identifier for the streamLabel.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStream(keyspaceName, tableName, streamLabel, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cassandra:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/keyspace/${keyspaceName}/table/${tableName}/stream/${streamLabel}`);
    }
    /**
     * Filters actions based on the presence of tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toAlter()
     * - .toAlterMultiRegionResource()
     * - .toCreate()
     * - .toCreateMultiRegionResource()
     * - .toTagMultiRegionResource()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - keyspace
     * - table
     * - stream
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the presence of tag keys in the request
     *
     * https://docs.aws.amazon.com/keyspaces/latest/devguide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toAlter()
     * - .toAlterMultiRegionResource()
     * - .toCreate()
     * - .toCreateMultiRegionResource()
     * - .toTagMultiRegionResource()
     * - .toTagResource()
     * - .toUnTagMultiRegionResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [cassandra](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonkeyspacesforapachecassandra.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'cassandra';
        this.accessLevelList = {
            Write: [
                'Alter',
                'AlterMultiRegionResource',
                'Create',
                'CreateMultiRegionResource',
                'Drop',
                'DropMultiRegionResource',
                'Modify',
                'ModifyMultiRegionResource',
                'Restore',
                'RestoreMultiRegionTable',
                'UpdatePartitioner'
            ],
            Read: [
                'GetRecords',
                'GetShardIterator',
                'GetStream',
                'Select',
                'SelectMultiRegionResource'
            ],
            List: [
                'ListStreams'
            ],
            Tagging: [
                'TagMultiRegionResource',
                'TagResource',
                'UnTagMultiRegionResource',
                'UntagResource'
            ]
        };
    }
}
exports.Cassandra = Cassandra;
//# sourceMappingURL=data:application/json;base64,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