"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LexV2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [lex-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlexv2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class LexV2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to create new items in an existing custom vocabulary
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_BatchCreateCustomVocabularyItem.html
     */
    toBatchCreateCustomVocabularyItem() {
        return this.to('BatchCreateCustomVocabularyItem');
    }
    /**
     * Grants permission to delete existing items in an existing custom vocabulary
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_BatchDeleteCustomVocabularyItem.html
     */
    toBatchDeleteCustomVocabularyItem() {
        return this.to('BatchDeleteCustomVocabularyItem');
    }
    /**
     * Grants permission to update existing items in an existing custom vocabulary
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_BatchUpdateCustomVocabularyItem.html
     */
    toBatchUpdateCustomVocabularyItem() {
        return this.to('BatchUpdateCustomVocabularyItem');
    }
    /**
     * Grants permission to build an existing bot locale in a bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_BuildBotLocale.html
     */
    toBuildBotLocale() {
        return this.to('BuildBotLocale');
    }
    /**
     * Grants permission to create a new bot and a test bot alias pointing to the DRAFT bot version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateBot.html
     */
    toCreateBot() {
        return this.to('CreateBot');
    }
    /**
     * Grants permission to create a new bot alias in a bot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateBotAlias.html
     */
    toCreateBotAlias() {
        return this.to('CreateBotAlias');
    }
    /**
     * Grants permission to create a bot channel in an existing bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/deploying-messaging-platform.html
     */
    toCreateBotChannel() {
        return this.to('CreateBotChannel');
    }
    /**
     * Grants permission to create a new bot locale in an existing bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateBotLocale.html
     */
    toCreateBotLocale() {
        return this.to('CreateBotLocale');
    }
    /**
     * Grants permission to create bot replica for a bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateBotReplica.html
     */
    toCreateBotReplica() {
        return this.to('CreateBotReplica');
    }
    /**
     * Grants permission to create a new version of an existing bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateBotVersion.html
     */
    toCreateBotVersion() {
        return this.to('CreateBotVersion');
    }
    /**
     * Grants permission to create a new custom vocabulary in an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/vocab.html
     */
    toCreateCustomVocabulary() {
        return this.to('CreateCustomVocabulary');
    }
    /**
     * Grants permission to create an export for an existing resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateExport.html
     */
    toCreateExport() {
        return this.to('CreateExport');
    }
    /**
     * Grants permission to create a new intent in an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateIntent.html
     */
    toCreateIntent() {
        return this.to('CreateIntent');
    }
    /**
     * Grants permission to create a new resource policy for a Lex resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateResourcePolicy.html
     */
    toCreateResourcePolicy() {
        return this.to('CreateResourcePolicy');
    }
    /**
     * Grants permission to create a new resource policy statement for a Lex resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateResourcePolicyStatement.html
     */
    toCreateResourcePolicyStatement() {
        return this.to('CreateResourcePolicyStatement');
    }
    /**
     * Grants permission to create a new slot in an intent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateSlot.html
     */
    toCreateSlot() {
        return this.to('CreateSlot');
    }
    /**
     * Grants permission to create a new slot type in an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateSlotType.html
     */
    toCreateSlotType() {
        return this.to('CreateSlotType');
    }
    /**
     * Grants permission to import a new test-set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/create-test-set-from-CSV.html
     */
    toCreateTestSet() {
        return this.to('CreateTestSet');
    }
    /**
     * Grants permission to create a test set discrepancy report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateTestSetDiscrepancyReport.html
     */
    toCreateTestSetDiscrepancyReport() {
        return this.to('CreateTestSetDiscrepancyReport');
    }
    /**
     * Grants permission to create an upload url for import file
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_CreateUploadUrl.html
     */
    toCreateUploadUrl() {
        return this.to('CreateUploadUrl');
    }
    /**
     * Grants permission to delete an existing bot
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - lex:DeleteBotAlias
     * - lex:DeleteBotChannel
     * - lex:DeleteBotLocale
     * - lex:DeleteBotVersion
     * - lex:DeleteIntent
     * - lex:DeleteSlot
     * - lex:DeleteSlotType
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteBot.html
     */
    toDeleteBot() {
        return this.to('DeleteBot');
    }
    /**
     * Grants permission to delete an existing bot alias in a bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteBotAlias.html
     */
    toDeleteBotAlias() {
        return this.to('DeleteBotAlias');
    }
    /**
     * Grants permission to delete an existing bot channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/deploying-messaging-platform.html
     */
    toDeleteBotChannel() {
        return this.to('DeleteBotChannel');
    }
    /**
     * Grants permission to delete an existing bot locale in a bot
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - lex:DeleteIntent
     * - lex:DeleteSlot
     * - lex:DeleteSlotType
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteBotLocale.html
     */
    toDeleteBotLocale() {
        return this.to('DeleteBotLocale');
    }
    /**
     * Grants permission to delete an existing bot replica
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteBotReplica.html
     */
    toDeleteBotReplica() {
        return this.to('DeleteBotReplica');
    }
    /**
     * Grants permission to delete an existing bot version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteBotVersion.html
     */
    toDeleteBotVersion() {
        return this.to('DeleteBotVersion');
    }
    /**
     * Grants permission to delete an existing custom vocabulary in a bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteCustomVocabulary.html
     */
    toDeleteCustomVocabulary() {
        return this.to('DeleteCustomVocabulary');
    }
    /**
     * Grants permission to delete an existing export
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteExport.html
     */
    toDeleteExport() {
        return this.to('DeleteExport');
    }
    /**
     * Grants permission to delete an existing import
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteImport.html
     */
    toDeleteImport() {
        return this.to('DeleteImport');
    }
    /**
     * Grants permission to delete an existing intent in a bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteIntent.html
     */
    toDeleteIntent() {
        return this.to('DeleteIntent');
    }
    /**
     * Grants permission to delete an existing resource policy for a Lex resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete an existing resource policy statement for a Lex resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteResourcePolicyStatement.html
     */
    toDeleteResourcePolicyStatement() {
        return this.to('DeleteResourcePolicyStatement');
    }
    /**
     * Grants permission to delete session information for a bot alias and user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_DeleteSession.html
     */
    toDeleteSession() {
        return this.to('DeleteSession');
    }
    /**
     * Grants permission to delete an existing slot in an intent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteSlot.html
     */
    toDeleteSlot() {
        return this.to('DeleteSlot');
    }
    /**
     * Grants permission to delete an existing slot type in a bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteSlotType.html
     */
    toDeleteSlotType() {
        return this.to('DeleteSlotType');
    }
    /**
     * Grants permission to delete an existing test set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteTestSet.html
     */
    toDeleteTestSet() {
        return this.to('DeleteTestSet');
    }
    /**
     * Grants permission to delete utterance data for a bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DeleteUtterances.html
     */
    toDeleteUtterances() {
        return this.to('DeleteUtterances');
    }
    /**
     * Grants permission to retrieve an existing bot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBot.html
     */
    toDescribeBot() {
        return this.to('DescribeBot');
    }
    /**
     * Grants permission to retrieve an existing bot alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotAlias.html
     */
    toDescribeBotAlias() {
        return this.to('DescribeBotAlias');
    }
    /**
     * Grants permission to retrieve an existing bot channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/deploying-messaging-platform.html
     */
    toDescribeBotChannel() {
        return this.to('DescribeBotChannel');
    }
    /**
     * Grants permission to retrieve an existing bot locale
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotLocale.html
     */
    toDescribeBotLocale() {
        return this.to('DescribeBotLocale');
    }
    /**
     * Grants permission to retrieve metadata information about a bot recommendation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotRecommendation.html
     */
    toDescribeBotRecommendation() {
        return this.to('DescribeBotRecommendation');
    }
    /**
     * Grants permission to retrieve an existing bot replica
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotReplica.html
     */
    toDescribeBotReplica() {
        return this.to('DescribeBotReplica');
    }
    /**
     * Grants permission to retrieve metadata information for a bot resource generation
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotResourceGeneration.html
     */
    toDescribeBotResourceGeneration() {
        return this.to('DescribeBotResourceGeneration');
    }
    /**
     * Grants permission to retrieve an existing bot version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeBotVersion.html
     */
    toDescribeBotVersion() {
        return this.to('DescribeBotVersion');
    }
    /**
     * Grants permission to retrieve an existing custom vocabulary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/vocab.html
     */
    toDescribeCustomVocabulary() {
        return this.to('DescribeCustomVocabulary');
    }
    /**
     * Grants permission to retrieve metadata of an existing custom vocabulary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeCustomVocabularyMetadata.html
     */
    toDescribeCustomVocabularyMetadata() {
        return this.to('DescribeCustomVocabularyMetadata');
    }
    /**
     * Grants permission to retrieve an existing export
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - lex:DescribeBot
     * - lex:DescribeBotLocale
     * - lex:DescribeIntent
     * - lex:DescribeSlot
     * - lex:DescribeSlotType
     * - lex:ListBotLocales
     * - lex:ListIntents
     * - lex:ListSlotTypes
     * - lex:ListSlots
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeExport.html
     */
    toDescribeExport() {
        return this.to('DescribeExport');
    }
    /**
     * Grants permission to retrieve an existing import
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeImport.html
     */
    toDescribeImport() {
        return this.to('DescribeImport');
    }
    /**
     * Grants permission to retrieve an existing intent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeIntent.html
     */
    toDescribeIntent() {
        return this.to('DescribeIntent');
    }
    /**
     * Grants permission to retrieve an existing resource policy for a Lex resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeResourcePolicy.html
     */
    toDescribeResourcePolicy() {
        return this.to('DescribeResourcePolicy');
    }
    /**
     * Grants permission to retrieve an existing slot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeSlot.html
     */
    toDescribeSlot() {
        return this.to('DescribeSlot');
    }
    /**
     * Grants permission to retrieve an existing slot type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeSlotType.html
     */
    toDescribeSlotType() {
        return this.to('DescribeSlotType');
    }
    /**
     * Grants permission to retrieve test execution metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeTestExecution.html
     */
    toDescribeTestExecution() {
        return this.to('DescribeTestExecution');
    }
    /**
     * Grants permission to retrieve an existing test set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeTestSet.html
     */
    toDescribeTestSet() {
        return this.to('DescribeTestSet');
    }
    /**
     * Grants permission to retrieve test set discrepancy report metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeTestSetDiscrepancyReport.html
     */
    toDescribeTestSetDiscrepancyReport() {
        return this.to('DescribeTestSetDiscrepancyReport');
    }
    /**
     * Grants permission to retrieve test set generation metadata
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_DescribeTestSetGeneration.html
     */
    toDescribeTestSetGeneration() {
        return this.to('DescribeTestSetGeneration');
    }
    /**
     * Grants permission to generate supported fields or elements for a bot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_GenerateBotElement.html
     */
    toGenerateBotElement() {
        return this.to('GenerateBotElement');
    }
    /**
     * Grants permission to retrieve session information for a bot alias and user ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_GetSession.html
     */
    toGetSession() {
        return this.to('GetSession');
    }
    /**
     * Grants permission to retrieve artifacts URL for a test execution
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_GetTestExecutionArtifactsUrl.html
     */
    toGetTestExecutionArtifactsUrl() {
        return this.to('GetTestExecutionArtifactsUrl');
    }
    /**
     * Grants permission to list utterances and statistics for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListAggregatedUtterances.html
     */
    toListAggregatedUtterances() {
        return this.to('ListAggregatedUtterances');
    }
    /**
     * Grants permission to list alias replicas in a bot replica
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotAliasReplicas.html
     */
    toListBotAliasReplicas() {
        return this.to('ListBotAliasReplicas');
    }
    /**
     * Grants permission to list bot aliases in an bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotAliases.html
     */
    toListBotAliases() {
        return this.to('ListBotAliases');
    }
    /**
     * Grants permission to list bot channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/deploying-messaging-platform.html
     */
    toListBotChannels() {
        return this.to('ListBotChannels');
    }
    /**
     * Grants permission to list bot locales in a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotLocales.html
     */
    toListBotLocales() {
        return this.to('ListBotLocales');
    }
    /**
     * Grants permission to get a list of bot recommendations that meet the specified criteria
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotRecommendations.html
     */
    toListBotRecommendations() {
        return this.to('ListBotRecommendations');
    }
    /**
     * Grants permission to list replicas of a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotReplicas.html
     */
    toListBotReplicas() {
        return this.to('ListBotReplicas');
    }
    /**
     * Grants permission to list the resource generations for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotResourceGenerations.html
     */
    toListBotResourceGenerations() {
        return this.to('ListBotResourceGenerations');
    }
    /**
     * Grants permission to list version replicas in a bot replica
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotVersionReplicas.html
     */
    toListBotVersionReplicas() {
        return this.to('ListBotVersionReplicas');
    }
    /**
     * Grants permission to list existing bot versions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBotVersions.html
     */
    toListBotVersions() {
        return this.to('ListBotVersions');
    }
    /**
     * Grants permission to list existing bots
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBots.html
     */
    toListBots() {
        return this.to('ListBots');
    }
    /**
     * Grants permission to list built-in intents
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBuiltInIntents.html
     */
    toListBuiltInIntents() {
        return this.to('ListBuiltInIntents');
    }
    /**
     * Grants permission to list built-in slot types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListBuiltInSlotTypes.html
     */
    toListBuiltInSlotTypes() {
        return this.to('ListBuiltInSlotTypes');
    }
    /**
     * Grants permission to list items of an existing custom vocabulary
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListCustomVocabularyItems.html
     */
    toListCustomVocabularyItems() {
        return this.to('ListCustomVocabularyItems');
    }
    /**
     * Grants permission to list existing exports
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListExports.html
     */
    toListExports() {
        return this.to('ListExports');
    }
    /**
     * Grants permission to list existing imports
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListImports.html
     */
    toListImports() {
        return this.to('ListImports');
    }
    /**
     * Grants permission to list intent analytics metrics for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListIntentMetrics.html
     */
    toListIntentMetrics() {
        return this.to('ListIntentMetrics');
    }
    /**
     * Grants permission to list intent path analytics for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListIntentPaths.html
     */
    toListIntentPaths() {
        return this.to('ListIntentPaths');
    }
    /**
     * Grants permission to list intentStage analytics metrics for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListIntentStageMetrics.html
     */
    toListIntentStageMetrics() {
        return this.to('ListIntentStageMetrics');
    }
    /**
     * Grants permission to list intents in a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListIntents.html
     */
    toListIntents() {
        return this.to('ListIntents');
    }
    /**
     * Grants permission to get a list of recommended intents provided by the bot recommendation
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListRecommendedIntents.html
     */
    toListRecommendedIntents() {
        return this.to('ListRecommendedIntents');
    }
    /**
     * Grants permission to list session analytics data for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListSessionAnalyticsData.html
     */
    toListSessionAnalyticsData() {
        return this.to('ListSessionAnalyticsData');
    }
    /**
     * Grants permission to list session analytics metrics for a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListSessionMetrics.html
     */
    toListSessionMetrics() {
        return this.to('ListSessionMetrics');
    }
    /**
     * Grants permission to list slot types in a bot
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListSlotTypes.html
     */
    toListSlotTypes() {
        return this.to('ListSlotTypes');
    }
    /**
     * Grants permission to list slots in an intent
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListSlots.html
     */
    toListSlots() {
        return this.to('ListSlots');
    }
    /**
     * Grants permission to lists tags for a Lex resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve test results data for a test execution
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - lex:ListTestSetRecords
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListTestExecutionResultItems.html
     */
    toListTestExecutionResultItems() {
        return this.to('ListTestExecutionResultItems');
    }
    /**
     * Grants permission to list test executions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListTestExecutions.html
     */
    toListTestExecutions() {
        return this.to('ListTestExecutions');
    }
    /**
     * Grants permission to retrieve records inside an existing test set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListTestSetRecords.html
     */
    toListTestSetRecords() {
        return this.to('ListTestSetRecords');
    }
    /**
     * Grants permission to list test sets
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_ListTestSets.html
     */
    toListTestSets() {
        return this.to('ListTestSets');
    }
    /**
     * Grants permission to create a new session or modify an existing session for a bot alias and user ID
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_PutSession.html
     */
    toPutSession() {
        return this.to('PutSession');
    }
    /**
     * Grants permission to send user input (text-only) to an bot alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_RecognizeText.html
     */
    toRecognizeText() {
        return this.to('RecognizeText');
    }
    /**
     * Grants permission to send user input (text or speech) to an bot alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_RecognizeUtterance.html
     */
    toRecognizeUtterance() {
        return this.to('RecognizeUtterance');
    }
    /**
     * Grants permission to search for associated transcripts that meet the specified criteria
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_SearchAssociatedTranscripts.html
     */
    toSearchAssociatedTranscripts() {
        return this.to('SearchAssociatedTranscripts');
    }
    /**
     * Grants permission to start a bot recommendation for an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StartBotRecommendation.html
     */
    toStartBotRecommendation() {
        return this.to('StartBotRecommendation');
    }
    /**
     * Grants permission to start a resource generation for an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StartBotResourceGeneration.html
     */
    toStartBotResourceGeneration() {
        return this.to('StartBotResourceGeneration');
    }
    /**
     * Grants permission to stream user input (speech/text/DTMF) to a bot alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_runtime_StartConversation.html
     */
    toStartConversation() {
        return this.to('StartConversation');
    }
    /**
     * Grants permission to start a new import with the uploaded import file
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - lex:CreateBot
     * - lex:CreateBotLocale
     * - lex:CreateCustomVocabulary
     * - lex:CreateIntent
     * - lex:CreateSlot
     * - lex:CreateSlotType
     * - lex:CreateTestSet
     * - lex:DeleteBotLocale
     * - lex:DeleteCustomVocabulary
     * - lex:DeleteIntent
     * - lex:DeleteSlot
     * - lex:DeleteSlotType
     * - lex:UpdateBot
     * - lex:UpdateBotLocale
     * - lex:UpdateCustomVocabulary
     * - lex:UpdateIntent
     * - lex:UpdateSlot
     * - lex:UpdateSlotType
     * - lex:UpdateTestSet
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StartImport.html
     */
    toStartImport() {
        return this.to('StartImport');
    }
    /**
     * Grants permission to start a test execution using a test set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StartTestExecution.html
     */
    toStartTestExecution() {
        return this.to('StartTestExecution');
    }
    /**
     * Grants permission to generate a test set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StartTestSetGeneration.html
     */
    toStartTestSetGeneration() {
        return this.to('StartTestSetGeneration');
    }
    /**
     * Grants permission to stop a bot recommendation for an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_StopBotRecommendation.html
     */
    toStopBotRecommendation() {
        return this.to('StopBotRecommendation');
    }
    /**
     * Grants permission to add or overwrite tags of a Lex resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from a Lex resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an existing bot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateBot.html
     */
    toUpdateBot() {
        return this.to('UpdateBot');
    }
    /**
     * Grants permission to update an existing bot alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateBotAlias.html
     */
    toUpdateBotAlias() {
        return this.to('UpdateBotAlias');
    }
    /**
     * Grants permission to update an existing bot locale
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateBotLocale.html
     */
    toUpdateBotLocale() {
        return this.to('UpdateBotLocale');
    }
    /**
     * Grants permission to update an existing bot recommendation request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateBotRecommendation.html
     */
    toUpdateBotRecommendation() {
        return this.to('UpdateBotRecommendation');
    }
    /**
     * Grants permission to update an existing custom vocabulary
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/vocab.html
     */
    toUpdateCustomVocabulary() {
        return this.to('UpdateCustomVocabulary');
    }
    /**
     * Grants permission to update an existing export
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateExport.html
     */
    toUpdateExport() {
        return this.to('UpdateExport');
    }
    /**
     * Grants permission to update an existing intent
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateIntent.html
     */
    toUpdateIntent() {
        return this.to('UpdateIntent');
    }
    /**
     * Grants permission to update an existing resource policy for a Lex resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateResourcePolicy.html
     */
    toUpdateResourcePolicy() {
        return this.to('UpdateResourcePolicy');
    }
    /**
     * Grants permission to update an existing slot
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateSlot.html
     */
    toUpdateSlot() {
        return this.to('UpdateSlot');
    }
    /**
     * Grants permission to update an existing slot type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateSlotType.html
     */
    toUpdateSlotType() {
        return this.to('UpdateSlotType');
    }
    /**
     * Grants permission to update an existing test set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/lexv2/latest/APIReference/API_UpdateTestSet.html
     */
    toUpdateTestSet() {
        return this.to('UpdateTestSet');
    }
    /**
     * Adds a resource of type bot to the statement
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/how-it-works.html
     *
     * @param botId - Identifier for the botId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBot(botId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lex:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bot/${botId}`);
    }
    /**
     * Adds a resource of type bot alias to the statement
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/how-it-works.html
     *
     * @param botId - Identifier for the botId.
     * @param botAliasId - Identifier for the botAliasId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBotAlias(botId, botAliasId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lex:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:bot-alias/${botId}/${botAliasId}`);
    }
    /**
     * Adds a resource of type test set to the statement
     *
     * https://docs.aws.amazon.com/lexv2/latest/dg/test-workbench.html
     *
     * @param testSetId - Identifier for the testSetId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTestSet(testSetId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:lex:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:test-set/${testSetId}`);
    }
    /**
     * Filters access by the tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateBot()
     * - .toCreateBotAlias()
     * - .toStartImport()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags attached to a Lex resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - bot
     * - bot alias
     * - test set
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the set of tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateBot()
     * - .toCreateBotAlias()
     * - .toStartImport()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [lex-v2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonlexv2.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'lex';
        this.accessLevelList = {
            Write: [
                'BatchCreateCustomVocabularyItem',
                'BatchDeleteCustomVocabularyItem',
                'BatchUpdateCustomVocabularyItem',
                'BuildBotLocale',
                'CreateBot',
                'CreateBotAlias',
                'CreateBotChannel',
                'CreateBotLocale',
                'CreateBotReplica',
                'CreateBotVersion',
                'CreateCustomVocabulary',
                'CreateExport',
                'CreateIntent',
                'CreateResourcePolicy',
                'CreateResourcePolicyStatement',
                'CreateSlot',
                'CreateSlotType',
                'CreateTestSet',
                'CreateTestSetDiscrepancyReport',
                'CreateUploadUrl',
                'DeleteBot',
                'DeleteBotAlias',
                'DeleteBotChannel',
                'DeleteBotLocale',
                'DeleteBotReplica',
                'DeleteBotVersion',
                'DeleteCustomVocabulary',
                'DeleteExport',
                'DeleteImport',
                'DeleteIntent',
                'DeleteResourcePolicy',
                'DeleteResourcePolicyStatement',
                'DeleteSession',
                'DeleteSlot',
                'DeleteSlotType',
                'DeleteTestSet',
                'DeleteUtterances',
                'PutSession',
                'RecognizeText',
                'RecognizeUtterance',
                'StartBotRecommendation',
                'StartBotResourceGeneration',
                'StartConversation',
                'StartImport',
                'StartTestExecution',
                'StartTestSetGeneration',
                'StopBotRecommendation',
                'UpdateBot',
                'UpdateBotAlias',
                'UpdateBotLocale',
                'UpdateBotRecommendation',
                'UpdateCustomVocabulary',
                'UpdateExport',
                'UpdateIntent',
                'UpdateResourcePolicy',
                'UpdateSlot',
                'UpdateSlotType',
                'UpdateTestSet'
            ],
            Read: [
                'DescribeBot',
                'DescribeBotAlias',
                'DescribeBotChannel',
                'DescribeBotLocale',
                'DescribeBotRecommendation',
                'DescribeBotReplica',
                'DescribeBotResourceGeneration',
                'DescribeBotVersion',
                'DescribeCustomVocabulary',
                'DescribeCustomVocabularyMetadata',
                'DescribeExport',
                'DescribeImport',
                'DescribeIntent',
                'DescribeResourcePolicy',
                'DescribeSlot',
                'DescribeSlotType',
                'DescribeTestExecution',
                'DescribeTestSet',
                'DescribeTestSetDiscrepancyReport',
                'DescribeTestSetGeneration',
                'GenerateBotElement',
                'GetSession',
                'GetTestExecutionArtifactsUrl',
                'ListTagsForResource',
                'ListTestExecutionResultItems',
                'ListTestSetRecords'
            ],
            List: [
                'ListAggregatedUtterances',
                'ListBotAliasReplicas',
                'ListBotAliases',
                'ListBotChannels',
                'ListBotLocales',
                'ListBotRecommendations',
                'ListBotReplicas',
                'ListBotResourceGenerations',
                'ListBotVersionReplicas',
                'ListBotVersions',
                'ListBots',
                'ListBuiltInIntents',
                'ListBuiltInSlotTypes',
                'ListCustomVocabularyItems',
                'ListExports',
                'ListImports',
                'ListIntentMetrics',
                'ListIntentPaths',
                'ListIntentStageMetrics',
                'ListIntents',
                'ListRecommendedIntents',
                'ListSessionAnalyticsData',
                'ListSessionMetrics',
                'ListSlotTypes',
                'ListSlots',
                'ListTestExecutions',
                'ListTestSets',
                'SearchAssociatedTranscripts'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.LexV2 = LexV2;
//# sourceMappingURL=data:application/json;base64,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