"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Apptest = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [apptest](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmainframemodernizationapplicationtesting.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Apptest extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a test case
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_CreateTestCase.html
     */
    toCreateTestCase() {
        return this.to('CreateTestCase');
    }
    /**
     * Grants permission to create a test configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_CreateTestConfiguration.html
     */
    toCreateTestConfiguration() {
        return this.to('CreateTestConfiguration');
    }
    /**
     * Grants permission to create a test suite
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_CreateTestSuite.html
     */
    toCreateTestSuite() {
        return this.to('CreateTestSuite');
    }
    /**
     * Grants permission to delete a test case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_DeleteTestCase.html
     */
    toDeleteTestCase() {
        return this.to('DeleteTestCase');
    }
    /**
     * Grants permission to delete a test configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_DeleteTestConfiguration.html
     */
    toDeleteTestConfiguration() {
        return this.to('DeleteTestConfiguration');
    }
    /**
     * Grants permission to delete a test run
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:DeleteObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_DeleteTestRun.html
     */
    toDeleteTestRun() {
        return this.to('DeleteTestRun');
    }
    /**
     * Grants permission to delete a test suite
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_DeleteTestSuite.html
     */
    toDeleteTestSuite() {
        return this.to('DeleteTestSuite');
    }
    /**
     * Grants permission to get a test case
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_GetTestCase.html
     */
    toGetTestCase() {
        return this.to('GetTestCase');
    }
    /**
     * Grants permission to get a test configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_GetTestConfiguration.html
     */
    toGetTestConfiguration() {
        return this.to('GetTestConfiguration');
    }
    /**
     * Grants permission to get test run step
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_GetTestRunStep.html
     */
    toGetTestRunStep() {
        return this.to('GetTestRunStep');
    }
    /**
     * Grants permission to get a test suite
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_GetTestSuite.html
     */
    toGetTestSuite() {
        return this.to('GetTestSuite');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list test cases
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_ListTestCases.html
     */
    toListTestCases() {
        return this.to('ListTestCases');
    }
    /**
     * Grants permission to list test configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_ListTestConfigurations.html
     */
    toListTestConfigurations() {
        return this.to('ListTestConfigurations');
    }
    /**
     * Grants permission to list steps for a test run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_ListTestRunSteps.html
     */
    toListTestRunSteps() {
        return this.to('ListTestRunSteps');
    }
    /**
     * Grants permission to list test cases for a test run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_ListTestRunTestCases.html
     */
    toListTestRunTestCases() {
        return this.to('ListTestRunTestCases');
    }
    /**
     * Grants permission to list test runs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_ListTestRuns.html
     */
    toListTestRuns() {
        return this.to('ListTestRuns');
    }
    /**
     * Grants permission to list test suites
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_ListTestSuites.html
     */
    toListTestSuites() {
        return this.to('ListTestSuites');
    }
    /**
     * Grants permission to start a test run
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - cloudformation:CreateStack
     * - cloudformation:DeleteStack
     * - cloudformation:DescribeStacks
     * - dms:DescribeReplicationTasks
     * - dms:StartReplicationTask
     * - dms:StopReplicationTask
     * - ec2:DescribeAvailabilityZones
     * - ec2:DescribeVpcEndpointServiceConfigurations
     * - ec2:DescribeVpcEndpointServices
     * - m2:CreateDataSetImportTask
     * - m2:GetApplication
     * - m2:GetApplicationVersion
     * - m2:GetBatchJobExecution
     * - m2:GetDataSetDetails
     * - m2:GetDataSetImportTask
     * - m2:StartApplication
     * - m2:StartBatchJob
     * - m2:StopApplication
     * - s3:CreateBucket
     * - s3:DeleteObject
     * - s3:GetObject
     * - s3:ListBucket
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_StartTestRun.html
     */
    toStartTestRun() {
        return this.to('StartTestRun');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a test case
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_UpdateTestCase.html
     */
    toUpdateTestCase() {
        return this.to('UpdateTestCase');
    }
    /**
     * Grants permission to update a test configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_UpdateTestConfiguration.html
     */
    toUpdateTestConfiguration() {
        return this.to('UpdateTestConfiguration');
    }
    /**
     * Grants permission to update a test suite
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/apptest/latest/APIReference/API_UpdateTestSuite.html
     */
    toUpdateTestSuite() {
        return this.to('UpdateTestSuite');
    }
    /**
     * Adds a resource of type TestCase to the statement
     *
     * https://docs.aws.amazon.com/m2/latest/userguide/concepts-apptest.html#TestCase-concept
     *
     * @param testCaseId - Identifier for the testCaseId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTestCase(testCaseId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apptest:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:testcase/${testCaseId}`);
    }
    /**
     * Adds a resource of type TestConfiguration to the statement
     *
     * https://docs.aws.amazon.com/m2/latest/userguide/concepts-apptest.html#TestConfiguration-concept
     *
     * @param testConfigurationId - Identifier for the testConfigurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTestConfiguration(testConfigurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apptest:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:testconfiguration/${testConfigurationId}`);
    }
    /**
     * Adds a resource of type TestRun to the statement
     *
     * https://docs.aws.amazon.com/m2/latest/userguide/concepts-apptest.html#TestRun-concept
     *
     * @param testRunId - Identifier for the testRunId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTestRun(testRunId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apptest:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:testrun/${testRunId}`);
    }
    /**
     * Adds a resource of type TestSuite to the statement
     *
     * https://docs.aws.amazon.com/m2/latest/userguide/concepts-apptest.html#TestSuite-concept
     *
     * @param testSuiteId - Identifier for the testSuiteId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTestSuite(testSuiteId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apptest:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:testsuite/${testSuiteId}`);
    }
    /**
     * Filters access by a tag key and value pair that is allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateTestCase()
     * - .toCreateTestConfiguration()
     * - .toCreateTestSuite()
     * - .toStartTestRun()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a tag key and value pair of a resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - TestCase
     * - TestConfiguration
     * - TestRun
     * - TestSuite
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by a list of tag keys that are allowed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateTestCase()
     * - .toCreateTestConfiguration()
     * - .toCreateTestSuite()
     * - .toStartTestRun()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [apptest](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmainframemodernizationapplicationtesting.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'apptest';
        this.accessLevelList = {
            Write: [
                'CreateTestCase',
                'CreateTestConfiguration',
                'CreateTestSuite',
                'DeleteTestCase',
                'DeleteTestConfiguration',
                'DeleteTestRun',
                'DeleteTestSuite',
                'StartTestRun',
                'UpdateTestCase',
                'UpdateTestConfiguration',
                'UpdateTestSuite'
            ],
            Read: [
                'GetTestCase',
                'GetTestConfiguration',
                'GetTestRunStep',
                'GetTestSuite',
                'ListTagsForResource',
                'ListTestRunSteps',
                'ListTestRunTestCases'
            ],
            List: [
                'ListTestCases',
                'ListTestConfigurations',
                'ListTestRuns',
                'ListTestSuites'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Apptest = Apptest;
//# sourceMappingURL=data:application/json;base64,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