"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Aps = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [aps](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedserviceforprometheus.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Aps extends shared_1.PolicyStatement {
    /**
     * Grants permission to create alerts
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-CreateAlertManagerAlerts.html
     */
    toCreateAlertManagerAlerts() {
        return this.to('CreateAlertManagerAlerts');
    }
    /**
     * Grants permission to create an alert manager definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_CreateAlertManagerDefinition.html
     */
    toCreateAlertManagerDefinition() {
        return this.to('CreateAlertManagerDefinition');
    }
    /**
     * Grants permission to create a logging configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_CreateLoggingConfiguration.html
     */
    toCreateLoggingConfiguration() {
        return this.to('CreateLoggingConfiguration');
    }
    /**
     * Grants permission to create a query logging configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_CreateQueryLoggingConfiguration.html
     */
    toCreateQueryLoggingConfiguration() {
        return this.to('CreateQueryLoggingConfiguration');
    }
    /**
     * Grants permission to create a rule groups namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_CreateRuleGroupsNamespace.html
     */
    toCreateRuleGroupsNamespace() {
        return this.to('CreateRuleGroupsNamespace');
    }
    /**
     * Grants permission to create a scraper
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - aps:TagResource
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - eks:DescribeCluster
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_CreateScraper.html
     */
    toCreateScraper() {
        return this.to('CreateScraper');
    }
    /**
     * Grants permission to create a workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_CreateWorkspace.html
     */
    toCreateWorkspace() {
        return this.to('CreateWorkspace');
    }
    /**
     * Grants permission to delete an alert manager definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DeleteAlertManagerDefinition.html
     */
    toDeleteAlertManagerDefinition() {
        return this.to('DeleteAlertManagerDefinition');
    }
    /**
     * Grants permission to delete a silence
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-DeleteAlertManagerSilence.html
     */
    toDeleteAlertManagerSilence() {
        return this.to('DeleteAlertManagerSilence');
    }
    /**
     * Grants permission to delete a logging configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DeleteLoggingConfiguration.html
     */
    toDeleteLoggingConfiguration() {
        return this.to('DeleteLoggingConfiguration');
    }
    /**
     * Grants permission to delete a query logging configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DeleteQueryLoggingConfiguration.html
     */
    toDeleteQueryLoggingConfiguration() {
        return this.to('DeleteQueryLoggingConfiguration');
    }
    /**
     * Grants permission to delete workspace resource policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a rule groups namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DeleteRuleGroupsNamespace.html
     */
    toDeleteRuleGroupsNamespace() {
        return this.to('DeleteRuleGroupsNamespace');
    }
    /**
     * Grants permission to delete a scraper
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DeleteScraper.html
     */
    toDeleteScraper() {
        return this.to('DeleteScraper');
    }
    /**
     * Grants permission to delete a workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DeleteWorkspace.html
     */
    toDeleteWorkspace() {
        return this.to('DeleteWorkspace');
    }
    /**
     * Grants permission to describe an alert manager definition
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DescribeAlertManagerDefinition.html
     */
    toDescribeAlertManagerDefinition() {
        return this.to('DescribeAlertManagerDefinition');
    }
    /**
     * Grants permission to describe a logging configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DescribeLoggingConfiguration.html
     */
    toDescribeLoggingConfiguration() {
        return this.to('DescribeLoggingConfiguration');
    }
    /**
     * Grants permission to describe a query logging configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DescribeQueryLoggingConfiguration.html
     */
    toDescribeQueryLoggingConfiguration() {
        return this.to('DescribeQueryLoggingConfiguration');
    }
    /**
     * Grants permission to describe workspace resource policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DescribeResourcePolicy.html
     */
    toDescribeResourcePolicy() {
        return this.to('DescribeResourcePolicy');
    }
    /**
     * Grants permission to describe a rule groups namespace
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DescribeRuleGroupsNamespace.html
     */
    toDescribeRuleGroupsNamespace() {
        return this.to('DescribeRuleGroupsNamespace');
    }
    /**
     * Grants permission to describe a scraper
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DescribeScraper.html
     */
    toDescribeScraper() {
        return this.to('DescribeScraper');
    }
    /**
     * Grants permission to describe a workspace
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DescribeWorkspace.html
     */
    toDescribeWorkspace() {
        return this.to('DescribeWorkspace');
    }
    /**
     * Grants permission to describe workspace configuration
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_DescribeWorkspaceConfiguration.html
     */
    toDescribeWorkspaceConfiguration() {
        return this.to('DescribeWorkspaceConfiguration');
    }
    /**
     * Grants permission to get a silence
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-GetAlertManagerSilence.html
     */
    toGetAlertManagerSilence() {
        return this.to('GetAlertManagerSilence');
    }
    /**
     * Grants permission to get current status of an alertmanager
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-GetAlertManagerStatus.html
     */
    toGetAlertManagerStatus() {
        return this.to('GetAlertManagerStatus');
    }
    /**
     * Grants permission to get default scraper configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_GetDefaultScraperConfiguration.html
     */
    toGetDefaultScraperConfiguration() {
        return this.to('GetDefaultScraperConfiguration');
    }
    /**
     * Grants permission to retrieve AMP workspace labels
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-GetLabels.html
     */
    toGetLabels() {
        return this.to('GetLabels');
    }
    /**
     * Grants permission to retrieve the metadata for AMP workspace metrics
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-GetMetricMetadata.html
     */
    toGetMetricMetadata() {
        return this.to('GetMetricMetadata');
    }
    /**
     * Grants permission to retrieve AMP workspace time series data
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-GetSeries.html
     */
    toGetSeries() {
        return this.to('GetSeries');
    }
    /**
     * Grants permission to list groups
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-ListAlertManagerAlertGroups.html
     */
    toListAlertManagerAlertGroups() {
        return this.to('ListAlertManagerAlertGroups');
    }
    /**
     * Grants permission to list alerts
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-ListAlertManagerAlerts.html
     */
    toListAlertManagerAlerts() {
        return this.to('ListAlertManagerAlerts');
    }
    /**
     * Grants permission to list receivers
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-ListAlertManagerReceivers.html
     */
    toListAlertManagerReceivers() {
        return this.to('ListAlertManagerReceivers');
    }
    /**
     * Grants permission to list silences
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-ListAlertManagerSilences.html
     */
    toListAlertManagerSilences() {
        return this.to('ListAlertManagerSilences');
    }
    /**
     * Grants permission to list active alerts
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-ListAlerts.html
     */
    toListAlerts() {
        return this.to('ListAlerts');
    }
    /**
     * Grants permission to list rule groups namespaces
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_ListRuleGroupsNamespaces.html
     */
    toListRuleGroupsNamespaces() {
        return this.to('ListRuleGroupsNamespaces');
    }
    /**
     * Grants permission to list alerting and recording rules
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-ListRules.html
     */
    toListRules() {
        return this.to('ListRules');
    }
    /**
     * Grants permission to list scrapers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_ListScrapers.html
     */
    toListScrapers() {
        return this.to('ListScrapers');
    }
    /**
     * Grants permission to list tags on an AMP resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list workspaces
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_ListWorkspaces.html
     */
    toListWorkspaces() {
        return this.to('ListWorkspaces');
    }
    /**
     * Grants permission to update an alert manager definition
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_PutAlertManagerDefinition.html
     */
    toPutAlertManagerDefinition() {
        return this.to('PutAlertManagerDefinition');
    }
    /**
     * Grants permission to create or update a silence
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-PutAlertManagerSilences.html
     */
    toPutAlertManagerSilences() {
        return this.to('PutAlertManagerSilences');
    }
    /**
     * Grants permission to create and update workspace resource policy
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to update a rule groups namespace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_PutRuleGroupsNamespace.html
     */
    toPutRuleGroupsNamespace() {
        return this.to('PutRuleGroupsNamespace');
    }
    /**
     * Grants permission to run a query on AMP workspace metrics
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-QueryMetrics.html
     */
    toQueryMetrics() {
        return this.to('QueryMetrics');
    }
    /**
     * Grants permission to perform a remote write operation to initiate the streaming of metrics to AMP workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/AMP-APIReference-RemoteWrite.html
     */
    toRemoteWrite() {
        return this.to('RemoteWrite');
    }
    /**
     * Grants permission to tag an AMP resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag an AMP resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a logging configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_UpdateLoggingConfiguration.html
     */
    toUpdateLoggingConfiguration() {
        return this.to('UpdateLoggingConfiguration');
    }
    /**
     * Grants permission to update a query logging configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_UpdateQueryLoggingConfiguration.html
     */
    toUpdateQueryLoggingConfiguration() {
        return this.to('UpdateQueryLoggingConfiguration');
    }
    /**
     * Grants permission to update a scraper
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - aps:CreateScraper
     * - aps:TagResource
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_UpdateScraper.html
     */
    toUpdateScraper() {
        return this.to('UpdateScraper');
    }
    /**
     * Grants permission to modify the alias of existing AMP workspace
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_UpdateWorkspaceAlias.html
     */
    toUpdateWorkspaceAlias() {
        return this.to('UpdateWorkspaceAlias');
    }
    /**
     * Grants permission to update workspace configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/prometheus/latest/APIReference/API_UpdateWorkspaceConfiguration.html
     */
    toUpdateWorkspaceConfiguration() {
        return this.to('UpdateWorkspaceConfiguration');
    }
    /**
     * Adds a resource of type workspace to the statement
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/security-iam.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onWorkspace(workspaceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:aps:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workspace/${workspaceId}`);
    }
    /**
     * Adds a resource of type rulegroupsnamespace to the statement
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/security-iam.html
     *
     * @param workspaceId - Identifier for the workspaceId.
     * @param namespace - Identifier for the namespace.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onRulegroupsnamespace(workspaceId, namespace, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:aps:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:rulegroupsnamespace/${workspaceId}/${namespace}`);
    }
    /**
     * Adds a resource of type scraper to the statement
     *
     * https://docs.aws.amazon.com/prometheus/latest/userguide/security-iam.html
     *
     * @param scraperId - Identifier for the scraperId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     */
    onScraper(scraperId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:aps:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:scraper/${scraperId}`);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/eks/latest/userguide/clusters.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:eks:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster/${clusterName}`);
    }
    /**
     * Filters access based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateRuleGroupsNamespace()
     * - .toCreateScraper()
     * - .toCreateWorkspace()
     * - .toListTagsForResource()
     * - .toTagResource()
     *
     * Applies to resource types:
     * - workspace
     * - rulegroupsnamespace
     * - scraper
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toCreateAlertManagerAlerts()
     * - .toCreateAlertManagerDefinition()
     * - .toCreateLoggingConfiguration()
     * - .toCreateQueryLoggingConfiguration()
     * - .toCreateScraper()
     * - .toDeleteAlertManagerDefinition()
     * - .toDeleteAlertManagerSilence()
     * - .toDeleteLoggingConfiguration()
     * - .toDeleteQueryLoggingConfiguration()
     * - .toDeleteResourcePolicy()
     * - .toDeleteRuleGroupsNamespace()
     * - .toDeleteScraper()
     * - .toDeleteWorkspace()
     * - .toDescribeAlertManagerDefinition()
     * - .toDescribeLoggingConfiguration()
     * - .toDescribeQueryLoggingConfiguration()
     * - .toDescribeResourcePolicy()
     * - .toDescribeRuleGroupsNamespace()
     * - .toDescribeScraper()
     * - .toDescribeWorkspace()
     * - .toDescribeWorkspaceConfiguration()
     * - .toGetAlertManagerSilence()
     * - .toGetAlertManagerStatus()
     * - .toGetLabels()
     * - .toGetMetricMetadata()
     * - .toGetSeries()
     * - .toListAlertManagerAlertGroups()
     * - .toListAlertManagerAlerts()
     * - .toListAlertManagerReceivers()
     * - .toListAlertManagerSilences()
     * - .toListAlerts()
     * - .toListRuleGroupsNamespaces()
     * - .toListRules()
     * - .toPutAlertManagerDefinition()
     * - .toPutAlertManagerSilences()
     * - .toPutResourcePolicy()
     * - .toPutRuleGroupsNamespace()
     * - .toQueryMetrics()
     * - .toRemoteWrite()
     * - .toUpdateLoggingConfiguration()
     * - .toUpdateQueryLoggingConfiguration()
     * - .toUpdateScraper()
     * - .toUpdateWorkspaceAlias()
     * - .toUpdateWorkspaceConfiguration()
     *
     * Applies to resource types:
     * - workspace
     * - rulegroupsnamespace
     * - scraper
     * - cluster
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateRuleGroupsNamespace()
     * - .toCreateScraper()
     * - .toCreateWorkspace()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * Applies to resource types:
     * - workspace
     * - rulegroupsnamespace
     * - scraper
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [aps](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmanagedserviceforprometheus.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'aps';
        this.accessLevelList = {
            Write: [
                'CreateAlertManagerAlerts',
                'CreateAlertManagerDefinition',
                'CreateLoggingConfiguration',
                'CreateQueryLoggingConfiguration',
                'CreateRuleGroupsNamespace',
                'CreateScraper',
                'CreateWorkspace',
                'DeleteAlertManagerDefinition',
                'DeleteAlertManagerSilence',
                'DeleteLoggingConfiguration',
                'DeleteQueryLoggingConfiguration',
                'DeleteResourcePolicy',
                'DeleteRuleGroupsNamespace',
                'DeleteScraper',
                'DeleteWorkspace',
                'PutAlertManagerDefinition',
                'PutAlertManagerSilences',
                'PutResourcePolicy',
                'PutRuleGroupsNamespace',
                'RemoteWrite',
                'UpdateLoggingConfiguration',
                'UpdateQueryLoggingConfiguration',
                'UpdateScraper',
                'UpdateWorkspaceAlias',
                'UpdateWorkspaceConfiguration'
            ],
            Read: [
                'DescribeAlertManagerDefinition',
                'DescribeLoggingConfiguration',
                'DescribeQueryLoggingConfiguration',
                'DescribeResourcePolicy',
                'DescribeRuleGroupsNamespace',
                'DescribeScraper',
                'DescribeWorkspace',
                'DescribeWorkspaceConfiguration',
                'GetAlertManagerSilence',
                'GetAlertManagerStatus',
                'GetDefaultScraperConfiguration',
                'GetLabels',
                'GetMetricMetadata',
                'GetSeries',
                'ListAlertManagerAlertGroups',
                'ListAlertManagerAlerts',
                'ListAlertManagerReceivers',
                'ListAlertManagerSilences',
                'ListAlerts',
                'ListRules',
                'ListTagsForResource',
                'QueryMetrics'
            ],
            List: [
                'ListRuleGroupsNamespaces',
                'ListScrapers',
                'ListWorkspaces'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Aps = Aps;
//# sourceMappingURL=data:application/json;base64,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