"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsMarketplace = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [aws-marketplace](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplace.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AwsMarketplace extends shared_1.PolicyStatement {
    /**
     * Grants permission to users to approve an incoming subscription request (for providers who provide products that require subscription verification)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toAcceptAgreementApprovalRequest() {
        return this.to('AcceptAgreementApprovalRequest');
    }
    /**
     * Grants permission to users to accept their agreement requests. Note that this action is not applicable to Marketplace purchases
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toAcceptAgreementRequest() {
        return this.to('AcceptAgreementRequest');
    }
    /**
     * Grants permission to users to cancel their agreements. Note that this action is not applicable to Marketplace purchases
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toCancelAgreement() {
        return this.to('CancelAgreement');
    }
    /**
     * Grants permission to users to cancel pending subscription requests for products that require subscription verification
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toCancelAgreementRequest() {
        return this.to('CancelAgreementRequest');
    }
    /**
     * Grants permission to users to create an agreement request. Note that this action is not applicable to Marketplace purchases
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toCreateAgreementRequest() {
        return this.to('CreateAgreementRequest');
    }
    /**
     * Grants permission to users to describe the metadata about the agreement
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toDescribeAgreement() {
        return this.to('DescribeAgreement');
    }
    /**
     * Grants permission to users to view the details of their incoming subscription requests (for providers who provide products that require subscription verification)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toGetAgreementApprovalRequest() {
        return this.to('GetAgreementApprovalRequest');
    }
    /**
     * Grants permission to users to view the entitlements associated with an agreement
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toGetAgreementEntitlements() {
        return this.to('GetAgreementEntitlements');
    }
    /**
     * Grants permission to users to view the details of their subscription requests for data products that require subscription verification
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toGetAgreementRequest() {
        return this.to('GetAgreementRequest');
    }
    /**
     * Grants permission to users to get a list of terms for an agreement
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toGetAgreementTerms() {
        return this.to('GetAgreementTerms');
    }
    /**
     * Grants permission to users to list their incoming subscription requests (for providers who provide products that require subscription verification)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toListAgreementApprovalRequests() {
        return this.to('ListAgreementApprovalRequests');
    }
    /**
     * Grants permission to users to view charges associated with their agreements
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toListAgreementCharges() {
        return this.to('ListAgreementCharges');
    }
    /**
     * Grants permission to users to list their subscription requests for products that require subscription verification
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toListAgreementRequests() {
        return this.to('ListAgreementRequests');
    }
    /**
     * Grants permission to users to view details of the entitlements associated with an agreement. Note that this action is not applicable to Marketplace purchases
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toListEntitlementDetails() {
        return this.to('ListEntitlementDetails');
    }
    /**
     * Grants permission to users to decline an incoming subscription requests (for providers who provide products that require subscription verification)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toRejectAgreementApprovalRequest() {
        return this.to('RejectAgreementApprovalRequest');
    }
    /**
     * Grants permission to users to search their agreements
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toSearchAgreements() {
        return this.to('SearchAgreements');
    }
    /**
     * Grants permission to users to subscribe to AWS Marketplace products. Includes the ability to send a subscription request for products that require subscription verification. Includes the ability to enable auto-renewal for an existing subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toSubscribe() {
        return this.to('Subscribe');
    }
    /**
     * Grants permission to users to remove subscriptions to AWS Marketplace products. Includes the ability to disable auto-renewal for an existing subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toUnsubscribe() {
        return this.to('Unsubscribe');
    }
    /**
     * Grants permission to users to make changes to an incoming subscription request, including the ability to delete the prospective subscriber's information (for providers who provide products that require subscription verification)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toUpdateAgreementApprovalRequest() {
        return this.to('UpdateAgreementApprovalRequest');
    }
    /**
     * Grants permission to users to update purchase orders for charges associated with their agreements
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toUpdatePurchaseOrders() {
        return this.to('UpdatePurchaseOrders');
    }
    /**
     * Grants permission to users to see their account's subscriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     */
    toViewSubscriptions() {
        return this.to('ViewSubscriptions');
    }
    /**
     * Filters access by the type of the agreement
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAgreementType(value, operator) {
        return this.if(`AgreementType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the party type of the agreement
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPartyType(value, operator) {
        return this.if(`PartyType`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by product id for AWS Marketplace RedHat OpenShift and Bedrock Products. Note: Using this condition key will not restrict access to products in AWS Marketplace
     *
     * https://docs.aws.amazon.com/marketplace/latest/buyerguide/buyer-iam-users-groups-policies.html
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProductId(value, operator) {
        return this.if(`ProductId`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [aws-marketplace](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsmarketplace.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'aws-marketplace';
        this.accessLevelList = {
            Write: [
                'AcceptAgreementApprovalRequest',
                'AcceptAgreementRequest',
                'CancelAgreement',
                'CancelAgreementRequest',
                'CreateAgreementRequest',
                'RejectAgreementApprovalRequest',
                'Subscribe',
                'Unsubscribe',
                'UpdateAgreementApprovalRequest',
                'UpdatePurchaseOrders'
            ],
            Read: [
                'DescribeAgreement',
                'GetAgreementApprovalRequest',
                'GetAgreementEntitlements',
                'GetAgreementRequest',
                'ListEntitlementDetails'
            ],
            List: [
                'GetAgreementTerms',
                'ListAgreementApprovalRequests',
                'ListAgreementCharges',
                'ListAgreementRequests',
                'SearchAgreements',
                'ViewSubscriptions'
            ]
        };
    }
}
exports.AwsMarketplace = AwsMarketplace;
//# sourceMappingURL=data:application/json;base64,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