"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Memorydb = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [memorydb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmemorydb.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Memorydb extends shared_1.PolicyStatement {
    /**
     * Grants permissions to apply service updates
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_BatchUpdateCluster.html
     */
    toBatchUpdateCluster() {
        return this.to('BatchUpdateCluster');
    }
    /**
     * Allows an IAM user or role to connect as a specified MemoryDB user to a node in a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/auth-iam.html
     */
    toConnect() {
        return this.to('Connect');
    }
    /**
     * Grants permissions to make a copy of an existing snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        return this.to('CopySnapshot');
    }
    /**
     * Grants permissions to create a new access control list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateAcl.html
     */
    toCreateAcl() {
        return this.to('CreateAcl');
    }
    /**
     * Grants permissions to create a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifTLSEnabled()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - memorydb:TagResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateCluster.html
     */
    toCreateCluster() {
        return this.to('CreateCluster');
    }
    /**
     * Grants permissions to create a Multi-Region cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifTLSEnabled()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateMultiRegionCluster.html
     */
    toCreateMultiRegionCluster() {
        return this.to('CreateMultiRegionCluster');
    }
    /**
     * Grants permissions to create a new parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateParameterGroup.html
     */
    toCreateParameterGroup() {
        return this.to('CreateParameterGroup');
    }
    /**
     * Grants permissions to create a backup of a cluster at the current point in time
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permissions to create a new subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateSubnetGroup.html
     */
    toCreateSubnetGroup() {
        return this.to('CreateSubnetGroup');
    }
    /**
     * Grants permissions to create a new user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifUserAuthenticationMode()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permissions to delete an access control list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteAcl.html
     */
    toDeleteAcl() {
        return this.to('DeleteAcl');
    }
    /**
     * Grants permissions to delete a previously provisioned cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteCluster.html
     */
    toDeleteCluster() {
        return this.to('DeleteCluster');
    }
    /**
     * Grants permissions to delete a Multi-Region cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteMultiRegionCluster.html
     */
    toDeleteMultiRegionCluster() {
        return this.to('DeleteMultiRegionCluster');
    }
    /**
     * Grants permissions to delete a parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteParameterGroup.html
     */
    toDeleteParameterGroup() {
        return this.to('DeleteParameterGroup');
    }
    /**
     * Grants permissions to delete a snapshot
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        return this.to('DeleteSnapshot');
    }
    /**
     * Grants permissions to delete a subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteSubnetGroup.html
     */
    toDeleteSubnetGroup() {
        return this.to('DeleteSubnetGroup');
    }
    /**
     * Grants permissions to delete a user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permissions to retrieve information about access control lists
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeAcls.html
     */
    toDescribeAcls() {
        return this.to('DescribeAcls');
    }
    /**
     * Grants permissions to retrieve information about all provisioned clusters if no cluster identifier is specified, or about a specific cluster if a cluster identifier is supplied
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeClusters.html
     */
    toDescribeClusters() {
        return this.to('DescribeClusters');
    }
    /**
     * Grants permissions to list of the available engines and their versions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeEngineVersions.html
     */
    toDescribeEngineVersions() {
        return this.to('DescribeEngineVersions');
    }
    /**
     * Grants permissions to retrieve events related to clusters, subnet groups, and parameter groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        return this.to('DescribeEvents');
    }
    /**
     * Grants permissions to retrieve information about all Multi-Region clusters if no cluster identifier is specified, or about a specific Multi-Region cluster if a cluster identifier is supplied
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeMultiRegionClusters.html
     */
    toDescribeMultiRegionClusters() {
        return this.to('DescribeMultiRegionClusters');
    }
    /**
     * Grants permissions to retrieve information about Multi-Region parameter groups
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeMultiRegionParameterGroups.html
     */
    toDescribeMultiRegionParameterGroups() {
        return this.to('DescribeMultiRegionParameterGroups');
    }
    /**
     * Grants permissions to retrieve a detailed parameter list for a particular Multi-Region parameter group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeMultiRegionParameters.html
     */
    toDescribeMultiRegionParameters() {
        return this.to('DescribeMultiRegionParameters');
    }
    /**
     * Grants permissions to retrieve information about parameter groups
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeParameterGroups.html
     */
    toDescribeParameterGroups() {
        return this.to('DescribeParameterGroups');
    }
    /**
     * Grants permissions to retrieve a detailed parameter list for a particular parameter group
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeParameters.html
     */
    toDescribeParameters() {
        return this.to('DescribeParameters');
    }
    /**
     * Grants permissions to retrieve reserved nodes
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeReservedNodes.html
     */
    toDescribeReservedNodes() {
        return this.to('DescribeReservedNodes');
    }
    /**
     * Grants permissions to retrieve reserved nodes offerings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeReservedNodesOfferings.html
     */
    toDescribeReservedNodesOfferings() {
        return this.to('DescribeReservedNodesOfferings');
    }
    /**
     * Grants permissions to retrieve details of the service updates
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeServiceUpdates.html
     */
    toDescribeServiceUpdates() {
        return this.to('DescribeServiceUpdates');
    }
    /**
     * Grants permissions to retrieve information about cluster snapshots
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        return this.to('DescribeSnapshots');
    }
    /**
     * Grants permissions to retrieve a list of subnet group
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeSubnetGroups.html
     */
    toDescribeSubnetGroups() {
        return this.to('DescribeSubnetGroups');
    }
    /**
     * Grants permissions to retrieve information about users
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_DescribeUsers.html
     */
    toDescribeUsers() {
        return this.to('DescribeUsers');
    }
    /**
     * Grants permissions to test automatic failover on a specified shard in a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_FailoverShard.html
     */
    toFailoverShard() {
        return this.to('FailoverShard');
    }
    /**
     * Grants permissions to list available Multi-Region cluster updates
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_ListAllowedMultiRegionClusterUpdates.html
     */
    toListAllowedMultiRegionClusterUpdates() {
        return this.to('ListAllowedMultiRegionClusterUpdates');
    }
    /**
     * Grants permissions to list available node type updates
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_ListAllowedNodeTypeUpdates.html
     */
    toListAllowedNodeTypeUpdates() {
        return this.to('ListAllowedNodeTypeUpdates');
    }
    /**
     * Grants permissions to list cost allocation tags
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_ListTags.html
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to pause replication for a Multi-Region cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/fis/latest/userguide/fis-actions-reference.html#memorydb-actions-reference
     */
    toPauseMultiRegionClusterReplication() {
        return this.to('PauseMultiRegionClusterReplication');
    }
    /**
     * Grants permissions to purchase a new reserved node
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - memorydb:TagResource
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_PurchaseReservedNodesOffering.html
     */
    toPurchaseReservedNodesOffering() {
        return this.to('PurchaseReservedNodesOffering');
    }
    /**
     * Grants permissions to modify the parameters of a parameter group to the engine or system default value
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_ResetParameterGroup.html
     */
    toResetParameterGroup() {
        return this.to('ResetParameterGroup');
    }
    /**
     * Grants permissions to add up to 10 cost allocation tags to the named resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permissions to remove the tags identified by the TagKeys list from a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permissions to update an access control list
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateAcl.html
     */
    toUpdateAcl() {
        return this.to('UpdateAcl');
    }
    /**
     * Grants permissions to update the settings for a cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateCluster.html
     */
    toUpdateCluster() {
        return this.to('UpdateCluster');
    }
    /**
     * Grants permissions to update the settings for a Multi-Region cluster
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateMultiRegionCluster.html
     */
    toUpdateMultiRegionCluster() {
        return this.to('UpdateMultiRegionCluster');
    }
    /**
     * Grants permissions to update parameters in a parameter group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateParameterGroup.html
     */
    toUpdateParameterGroup() {
        return this.to('UpdateParameterGroup');
    }
    /**
     * Grants permissions to update a subnet group
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateSubnetGroup.html
     */
    toUpdateSubnetGroup() {
        return this.to('UpdateSubnetGroup');
    }
    /**
     * Grants permissions to update a user
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifUserAuthenticationMode()
     *
     * https://docs.aws.amazon.com/memorydb/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Adds a resource of type multiregionparametergroup to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param multiRegionParameterGroupName - Identifier for the multiRegionParameterGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onMultiregionparametergroup(multiRegionParameterGroupName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb::${account ?? this.defaultAccount}:multiregionparametergroup/${multiRegionParameterGroupName}`);
    }
    /**
     * Adds a resource of type parametergroup to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param parameterGroupName - Identifier for the parameterGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onParametergroup(parameterGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:parametergroup/${parameterGroupName}`);
    }
    /**
     * Adds a resource of type subnetgroup to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param subnetGroupName - Identifier for the subnetGroupName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSubnetgroup(subnetGroupName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:subnetgroup/${subnetGroupName}`);
    }
    /**
     * Adds a resource of type multiregioncluster to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifTLSEnabled()
     */
    onMultiregioncluster(clusterName, account, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb::${account ?? this.defaultAccount}:multiregioncluster/${clusterName}`);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param clusterName - Identifier for the clusterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCluster(clusterName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:cluster/${clusterName}`);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSnapshot(snapshotName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:snapshot/${snapshotName}`);
    }
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param userName - Identifier for the userName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUser(userName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:user/${userName}`);
    }
    /**
     * Adds a resource of type acl to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param aclName - Identifier for the aclName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAcl(aclName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:acl/${aclName}`);
    }
    /**
     * Adds a resource of type reservednode to the statement
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/WhatIs.Components.html
     *
     * @param reservationID - Identifier for the reservationID.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReservednode(reservationID, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:memorydb:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:reservednode/${reservationID}`);
    }
    /**
     * Filters actions based on the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCopySnapshot()
     * - .toCreateAcl()
     * - .toCreateCluster()
     * - .toCreateMultiRegionCluster()
     * - .toCreateParameterGroup()
     * - .toCreateSnapshot()
     * - .toCreateSubnetGroup()
     * - .toCreateUser()
     * - .toPurchaseReservedNodesOffering()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toBatchUpdateCluster()
     * - .toConnect()
     * - .toCopySnapshot()
     * - .toCreateAcl()
     * - .toCreateCluster()
     * - .toCreateMultiRegionCluster()
     * - .toCreateSnapshot()
     * - .toDeleteAcl()
     * - .toDeleteCluster()
     * - .toDeleteMultiRegionCluster()
     * - .toDeleteParameterGroup()
     * - .toDeleteSnapshot()
     * - .toDeleteSubnetGroup()
     * - .toDeleteUser()
     * - .toDescribeAcls()
     * - .toDescribeClusters()
     * - .toDescribeMultiRegionClusters()
     * - .toDescribeParameterGroups()
     * - .toDescribeParameters()
     * - .toDescribeReservedNodes()
     * - .toDescribeSnapshots()
     * - .toDescribeSubnetGroups()
     * - .toDescribeUsers()
     * - .toFailoverShard()
     * - .toListAllowedMultiRegionClusterUpdates()
     * - .toListAllowedNodeTypeUpdates()
     * - .toListTags()
     * - .toPauseMultiRegionClusterReplication()
     * - .toPurchaseReservedNodesOffering()
     * - .toResetParameterGroup()
     * - .toTagResource()
     * - .toUntagResource()
     * - .toUpdateAcl()
     * - .toUpdateCluster()
     * - .toUpdateMultiRegionCluster()
     * - .toUpdateParameterGroup()
     * - .toUpdateSubnetGroup()
     * - .toUpdateUser()
     *
     * Applies to resource types:
     * - parametergroup
     * - subnetgroup
     * - multiregioncluster
     * - cluster
     * - snapshot
     * - user
     * - acl
     * - reservednode
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters actions based on the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCopySnapshot()
     * - .toCreateAcl()
     * - .toCreateCluster()
     * - .toCreateMultiRegionCluster()
     * - .toCreateParameterGroup()
     * - .toCreateSnapshot()
     * - .toCreateSubnetGroup()
     * - .toCreateUser()
     * - .toPurchaseReservedNodesOffering()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the TLSEnabled parameter present in the request or defaults to true value if parameter is not present
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateCluster()
     * - .toCreateMultiRegionCluster()
     *
     * Applies to resource types:
     * - multiregioncluster
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifTLSEnabled(value) {
        return this.if(`TLSEnabled`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * Filters access by the UserAuthenticationMode.Type parameter in the request
     *
     * https://docs.aws.amazon.com/memorydb/latest/devguide/IAM.ConditionKeys.html#IAM.SpecifyingConditions
     *
     * Applies to actions:
     * - .toCreateUser()
     * - .toUpdateUser()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUserAuthenticationMode(value, operator) {
        return this.if(`UserAuthenticationMode`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [memorydb](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmemorydb.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'memorydb';
        this.accessLevelList = {
            Write: [
                'BatchUpdateCluster',
                'Connect',
                'CopySnapshot',
                'CreateAcl',
                'CreateCluster',
                'CreateMultiRegionCluster',
                'CreateParameterGroup',
                'CreateSnapshot',
                'CreateSubnetGroup',
                'CreateUser',
                'DeleteAcl',
                'DeleteCluster',
                'DeleteMultiRegionCluster',
                'DeleteParameterGroup',
                'DeleteSnapshot',
                'DeleteSubnetGroup',
                'DeleteUser',
                'FailoverShard',
                'PauseMultiRegionClusterReplication',
                'PurchaseReservedNodesOffering',
                'ResetParameterGroup',
                'UpdateAcl',
                'UpdateCluster',
                'UpdateMultiRegionCluster',
                'UpdateParameterGroup',
                'UpdateSubnetGroup',
                'UpdateUser'
            ],
            Read: [
                'DescribeAcls',
                'DescribeClusters',
                'DescribeEngineVersions',
                'DescribeEvents',
                'DescribeMultiRegionClusters',
                'DescribeMultiRegionParameterGroups',
                'DescribeMultiRegionParameters',
                'DescribeParameterGroups',
                'DescribeParameters',
                'DescribeReservedNodes',
                'DescribeReservedNodesOfferings',
                'DescribeServiceUpdates',
                'DescribeSnapshots',
                'DescribeSubnetGroups',
                'DescribeUsers',
                'ListAllowedMultiRegionClusterUpdates',
                'ListAllowedNodeTypeUpdates',
                'ListTags'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Memorydb = Memorydb;
//# sourceMappingURL=data:application/json;base64,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