"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ssmmessages = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [ssmmessages](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmessagegatewayservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ssmmessages extends shared_1.PolicyStatement {
    /**
     * Grants permission to register a control channel for an instance to send control messages to Systems Manager service
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifSsmSourceInstanceARN()
     * - .ifEc2SourceInstanceARN()
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-create-iam-instance-profile.html
     */
    toCreateControlChannel() {
        return this.to('CreateControlChannel');
    }
    /**
     * Grants permission to register a data channel for an instance to send data messages to Systems Manager service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-create-iam-instance-profile.html
     */
    toCreateDataChannel() {
        return this.to('CreateDataChannel');
    }
    /**
     * Grants permission to open a websocket connection for a registered control channel stream from an instance to Systems Manager service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-create-iam-instance-profile.html
     */
    toOpenControlChannel() {
        return this.to('OpenControlChannel');
    }
    /**
     * Grants permission to open a websocket connection for a registered data channel stream from an instance to Systems Manager service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/getting-started-create-iam-instance-profile.html
     */
    toOpenDataChannel() {
        return this.to('OpenDataChannel');
    }
    /**
     * Filters access by the ARN of the instance from which the request originated
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#amazon-ec2-keys
     *
     * Applies to actions:
     * - .toCreateControlChannel()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifEc2SourceInstanceARN(value, operator) {
        return this.if(`ec2:SourceInstanceARN`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by verifying the Amazon Resource Name (ARN) of the AWS Systems Manager's managed instance from which the request is made. This key is not present when the request comes from the managed instance authenticated with an IAM role associated with EC2 instance profile
     *
     * https://docs.aws.amazon.com/systems-manager/latest/userguide/security_iam_service-with-iam.html#policy-conditions
     *
     * Applies to actions:
     * - .toCreateControlChannel()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifSsmSourceInstanceARN(value, operator) {
        return this.if(`ssm:SourceInstanceARN`, value, operator ?? 'ArnLike');
    }
    /**
     * Statement provider for service [ssmmessages](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmessagegatewayservice.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'ssmmessages';
        this.accessLevelList = {
            Write: [
                'CreateControlChannel',
                'CreateDataChannel',
                'OpenControlChannel',
                'OpenDataChannel'
            ]
        };
    }
}
exports.Ssmmessages = Ssmmessages;
//# sourceMappingURL=data:application/json;base64,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