"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Monitron = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [monitron](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmonitron.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Monitron extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a user with the project as an administrator
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - sso-directory:DescribeUsers
     * - sso:AssociateProfile
     * - sso:GetManagedApplicationInstance
     * - sso:GetProfile
     * - sso:ListDirectoryAssociations
     * - sso:ListProfileAssociations
     * - sso:ListProfiles
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/user-management-chapter.html
     */
    toAssociateProjectAdminUser() {
        return this.to('AssociateProjectAdminUser');
    }
    /**
     * Grants permission to create a project
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     * - kms:CreateGrant
     * - sso:CreateManagedApplicationInstance
     * - sso:DeleteManagedApplicationInstance
     * - sso:DescribeRegisteredRegions
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/mp-creating-project.html
     */
    toCreateProject() {
        return this.to('CreateProject');
    }
    /**
     * Grants permission to associate a user with the project
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - sso-directory:DescribeUsers
     * - sso:AssociateProfile
     * - sso:GetManagedApplicationInstance
     * - sso:GetProfile
     * - sso:ListDirectoryAssociations
     * - sso:ListProfileAssociations
     * - sso:ListProfiles
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/adding-user.html
     */
    toCreateProjectUserAssociation() {
        return this.to('CreateProjectUserAssociation');
    }
    /**
     * Grants permission to associate an access role with the user
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - sso-directory:DescribeUsers
     * - sso:GetManagedApplicationInstance
     * - sso:GetProfile
     * - sso:ListDirectoryAssociations
     * - sso:ListProfileAssociations
     * - sso:ListProfiles
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/adding-user.html
     */
    toCreateUserAccessRoleAssociation() {
        return this.to('CreateUserAccessRoleAssociation');
    }
    /**
     * Grants permission to delete a project
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - sso:DeleteManagedApplicationInstance
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/mp-delete-project.html
     */
    toDeleteProject() {
        return this.to('DeleteProject');
    }
    /**
     * Grants permission to disassociate a user from the project
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - sso-directory:DescribeUsers
     * - sso:DisassociateProfile
     * - sso:GetManagedApplicationInstance
     * - sso:GetProfile
     * - sso:ListDirectoryAssociations
     * - sso:ListProfiles
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/deleting-user.html
     */
    toDeleteProjectUserAssociation() {
        return this.to('DeleteProjectUserAssociation');
    }
    /**
     * Grants permission to disassociate an access role from the user
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/deleting-user.html
     */
    toDeleteUserAccessRoleAssociation() {
        return this.to('DeleteUserAccessRoleAssociation');
    }
    /**
     * Grants permission to disassociate an administrator from the project
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - sso-directory:DescribeUsers
     * - sso:DisassociateProfile
     * - sso:GetManagedApplicationInstance
     * - sso:GetProfile
     * - sso:ListDirectoryAssociations
     * - sso:ListProfiles
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/mu-remove-project-admin.html
     */
    toDisassociateProjectAdminUser() {
        return this.to('DisassociateProjectAdminUser');
    }
    /**
     * Grants permission to get information about a project
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/mp-project-tasks.html
     */
    toGetProject() {
        return this.to('GetProject');
    }
    /**
     * Grants permission to describe an administrator who is associated with the project
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - sso-directory:DescribeUsers
     * - sso:GetManagedApplicationInstance
     * - sso:ListProfileAssociations
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/mp-project-tasks.html
     */
    toGetProjectAdminUser() {
        return this.to('GetProjectAdminUser');
    }
    /**
     * Grants permission to list all administrators associated with the project
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - sso-directory:DescribeUsers
     * - sso:GetManagedApplicationInstance
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/user-management-chapter.html
     */
    toListProjectAdminUsers() {
        return this.to('ListProjectAdminUsers');
    }
    /**
     * Grants permission to list all users associated with the project
     *
     * Access Level: List
     *
     * Dependent actions:
     * - sso:GetManagedApplicationInstance
     * - sso:GetProfile
     * - sso:ListDirectoryAssociations
     * - sso:ListProfileAssociations
     * - sso:ListProfiles
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/user-management-chapter.html
     */
    toListProjectUserAssociations() {
        return this.to('ListProjectUserAssociations');
    }
    /**
     * Grants permission to list all projects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/mp-project-tasks.html
     */
    toListProjects() {
        return this.to('ListProjects');
    }
    /**
     * Grants permission to list all tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/tagging.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all access roles associated with the user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/user-management-chapter.html
     */
    toListUserAccessRoleAssociations() {
        return this.to('ListUserAccessRoleAssociations');
    }
    /**
     * Grants permission to tag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/tagging.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/tagging.html#modify-tag-1
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a project
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/mp-updating-project.html
     */
    toUpdateProject() {
        return this.to('UpdateProject');
    }
    /**
     * Adds a resource of type project to the statement
     *
     * https://docs.aws.amazon.com/Monitron/latest/user-guide/projects-chapter.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProject(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:monitron:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:project/${resourceId}`);
    }
    /**
     * Filters access by the tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - project
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateProject()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [monitron](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmonitron.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'monitron';
        this.accessLevelList = {
            'Permissions management': [
                'AssociateProjectAdminUser',
                'CreateProjectUserAssociation',
                'CreateUserAccessRoleAssociation',
                'DeleteProjectUserAssociation',
                'DeleteUserAccessRoleAssociation',
                'DisassociateProjectAdminUser',
                'ListProjectAdminUsers'
            ],
            Write: [
                'CreateProject',
                'DeleteProject',
                'UpdateProject'
            ],
            Read: [
                'GetProject',
                'GetProjectAdminUser',
                'ListTagsForResource'
            ],
            List: [
                'ListProjectUserAssociations',
                'ListProjects',
                'ListUserAccessRoleAssociations'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Monitron = Monitron;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibW9uaXRyb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJtb25pdHJvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsUUFBUyxTQUFRLHdCQUFlO0lBRzNDOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7O09BaUJHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxpQ0FBaUM7UUFDdEMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7SUFDcEQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUNBQWlDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksdUJBQXVCO1FBQzVCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksNkJBQTZCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQ0FBZ0M7UUFDckMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFpQ0Q7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksU0FBUyxDQUFDLFVBQWtCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDeEYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsYUFBYyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsWUFBYSxVQUFXLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZLLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxlQUFlLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDM0YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFtQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLGdCQUFnQixDQUFDLE1BQWMsRUFBRSxLQUF3QixFQUFFLFFBQTRCO1FBQzVGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBb0IsTUFBTyxFQUFFLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksWUFBWSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDeEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQTlYUixrQkFBYSxHQUFHLFVBQVUsQ0FBQztRQXVSeEIsb0JBQWUsR0FBb0I7WUFDM0Msd0JBQXdCLEVBQUU7Z0JBQ3hCLDJCQUEyQjtnQkFDM0IsOEJBQThCO2dCQUM5QixpQ0FBaUM7Z0JBQ2pDLDhCQUE4QjtnQkFDOUIsaUNBQWlDO2dCQUNqQyw4QkFBOEI7Z0JBQzlCLHVCQUF1QjthQUN4QjtZQUNELEtBQUssRUFBRTtnQkFDTCxlQUFlO2dCQUNmLGVBQWU7Z0JBQ2YsZUFBZTthQUNoQjtZQUNELElBQUksRUFBRTtnQkFDSixZQUFZO2dCQUNaLHFCQUFxQjtnQkFDckIscUJBQXFCO2FBQ3RCO1lBQ0QsSUFBSSxFQUFFO2dCQUNKLDZCQUE2QjtnQkFDN0IsY0FBYztnQkFDZCxnQ0FBZ0M7YUFDakM7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsYUFBYTtnQkFDYixlQUFlO2FBQ2hCO1NBQ0YsQ0FBQztJQTJFRixDQUFDO0NBQ0Y7QUFqWUQsNEJBaVlDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQWNjZXNzTGV2ZWxMaXN0IH0gZnJvbSAnLi4vLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbCc7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSAnLi4vLi4vc2hhcmVkJztcbmltcG9ydCB7IGF3c19pYW0gYXMgaWFtIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFttb25pdHJvbl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NlcnZpY2UtYXV0aG9yaXphdGlvbi9sYXRlc3QvcmVmZXJlbmNlL2xpc3RfYW1hem9ubW9uaXRyb24uaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBNb25pdHJvbiBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ21vbml0cm9uJztcblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXNzb2NpYXRlIGEgdXNlciB3aXRoIHRoZSBwcm9qZWN0IGFzIGFuIGFkbWluaXN0cmF0b3JcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHNzby1kaXJlY3Rvcnk6RGVzY3JpYmVVc2Vyc1xuICAgKiAtIHNzbzpBc3NvY2lhdGVQcm9maWxlXG4gICAqIC0gc3NvOkdldE1hbmFnZWRBcHBsaWNhdGlvbkluc3RhbmNlXG4gICAqIC0gc3NvOkdldFByb2ZpbGVcbiAgICogLSBzc286TGlzdERpcmVjdG9yeUFzc29jaWF0aW9uc1xuICAgKiAtIHNzbzpMaXN0UHJvZmlsZUFzc29jaWF0aW9uc1xuICAgKiAtIHNzbzpMaXN0UHJvZmlsZXNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL01vbml0cm9uL2xhdGVzdC91c2VyLWd1aWRlL3VzZXItbWFuYWdlbWVudC1jaGFwdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Fzc29jaWF0ZVByb2plY3RBZG1pblVzZXIoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Fzc29jaWF0ZVByb2plY3RBZG1pblVzZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBwcm9qZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gaWFtOkNyZWF0ZVNlcnZpY2VMaW5rZWRSb2xlXG4gICAqIC0ga21zOkNyZWF0ZUdyYW50XG4gICAqIC0gc3NvOkNyZWF0ZU1hbmFnZWRBcHBsaWNhdGlvbkluc3RhbmNlXG4gICAqIC0gc3NvOkRlbGV0ZU1hbmFnZWRBcHBsaWNhdGlvbkluc3RhbmNlXG4gICAqIC0gc3NvOkRlc2NyaWJlUmVnaXN0ZXJlZFJlZ2lvbnNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL01vbml0cm9uL2xhdGVzdC91c2VyLWd1aWRlL21wLWNyZWF0aW5nLXByb2plY3QuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUHJvamVjdCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlUHJvamVjdCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSBhIHVzZXIgd2l0aCB0aGUgcHJvamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gc3NvLWRpcmVjdG9yeTpEZXNjcmliZVVzZXJzXG4gICAqIC0gc3NvOkFzc29jaWF0ZVByb2ZpbGVcbiAgICogLSBzc286R2V0TWFuYWdlZEFwcGxpY2F0aW9uSW5zdGFuY2VcbiAgICogLSBzc286R2V0UHJvZmlsZVxuICAgKiAtIHNzbzpMaXN0RGlyZWN0b3J5QXNzb2NpYXRpb25zXG4gICAqIC0gc3NvOkxpc3RQcm9maWxlQXNzb2NpYXRpb25zXG4gICAqIC0gc3NvOkxpc3RQcm9maWxlc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vTW9uaXRyb24vbGF0ZXN0L3VzZXItZ3VpZGUvYWRkaW5nLXVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUHJvamVjdFVzZXJBc3NvY2lhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlUHJvamVjdFVzZXJBc3NvY2lhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSBhbiBhY2Nlc3Mgcm9sZSB3aXRoIHRoZSB1c2VyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBzc28tZGlyZWN0b3J5OkRlc2NyaWJlVXNlcnNcbiAgICogLSBzc286R2V0TWFuYWdlZEFwcGxpY2F0aW9uSW5zdGFuY2VcbiAgICogLSBzc286R2V0UHJvZmlsZVxuICAgKiAtIHNzbzpMaXN0RGlyZWN0b3J5QXNzb2NpYXRpb25zXG4gICAqIC0gc3NvOkxpc3RQcm9maWxlQXNzb2NpYXRpb25zXG4gICAqIC0gc3NvOkxpc3RQcm9maWxlc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vTW9uaXRyb24vbGF0ZXN0L3VzZXItZ3VpZGUvYWRkaW5nLXVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlVXNlckFjY2Vzc1JvbGVBc3NvY2lhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlVXNlckFjY2Vzc1JvbGVBc3NvY2lhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIHByb2plY3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBzc286RGVsZXRlTWFuYWdlZEFwcGxpY2F0aW9uSW5zdGFuY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL01vbml0cm9uL2xhdGVzdC91c2VyLWd1aWRlL21wLWRlbGV0ZS1wcm9qZWN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVByb2plY3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVByb2plY3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgYSB1c2VyIGZyb20gdGhlIHByb2plY3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBQZXJtaXNzaW9ucyBtYW5hZ2VtZW50XG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHNzby1kaXJlY3Rvcnk6RGVzY3JpYmVVc2Vyc1xuICAgKiAtIHNzbzpEaXNhc3NvY2lhdGVQcm9maWxlXG4gICAqIC0gc3NvOkdldE1hbmFnZWRBcHBsaWNhdGlvbkluc3RhbmNlXG4gICAqIC0gc3NvOkdldFByb2ZpbGVcbiAgICogLSBzc286TGlzdERpcmVjdG9yeUFzc29jaWF0aW9uc1xuICAgKiAtIHNzbzpMaXN0UHJvZmlsZXNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL01vbml0cm9uL2xhdGVzdC91c2VyLWd1aWRlL2RlbGV0aW5nLXVzZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUHJvamVjdFVzZXJBc3NvY2lhdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlUHJvamVjdFVzZXJBc3NvY2lhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSBhbiBhY2Nlc3Mgcm9sZSBmcm9tIHRoZSB1c2VyXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vTW9uaXRyb24vbGF0ZXN0L3VzZXItZ3VpZGUvZGVsZXRpbmctdXNlci5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVVc2VyQWNjZXNzUm9sZUFzc29jaWF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVVc2VyQWNjZXNzUm9sZUFzc29jaWF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGlzYXNzb2NpYXRlIGFuIGFkbWluaXN0cmF0b3IgZnJvbSB0aGUgcHJvamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gc3NvLWRpcmVjdG9yeTpEZXNjcmliZVVzZXJzXG4gICAqIC0gc3NvOkRpc2Fzc29jaWF0ZVByb2ZpbGVcbiAgICogLSBzc286R2V0TWFuYWdlZEFwcGxpY2F0aW9uSW5zdGFuY2VcbiAgICogLSBzc286R2V0UHJvZmlsZVxuICAgKiAtIHNzbzpMaXN0RGlyZWN0b3J5QXNzb2NpYXRpb25zXG4gICAqIC0gc3NvOkxpc3RQcm9maWxlc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vTW9uaXRyb24vbGF0ZXN0L3VzZXItZ3VpZGUvbXUtcmVtb3ZlLXByb2plY3QtYWRtaW4uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlUHJvamVjdEFkbWluVXNlcigpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGlzYXNzb2NpYXRlUHJvamVjdEFkbWluVXNlcicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGdldCBpbmZvcm1hdGlvbiBhYm91dCBhIHByb2plY3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9Nb25pdHJvbi9sYXRlc3QvdXNlci1ndWlkZS9tcC1wcm9qZWN0LXRhc2tzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFByb2plY3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFByb2plY3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZXNjcmliZSBhbiBhZG1pbmlzdHJhdG9yIHdobyBpcyBhc3NvY2lhdGVkIHdpdGggdGhlIHByb2plY3RcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHNzby1kaXJlY3Rvcnk6RGVzY3JpYmVVc2Vyc1xuICAgKiAtIHNzbzpHZXRNYW5hZ2VkQXBwbGljYXRpb25JbnN0YW5jZVxuICAgKiAtIHNzbzpMaXN0UHJvZmlsZUFzc29jaWF0aW9uc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vTW9uaXRyb24vbGF0ZXN0L3VzZXItZ3VpZGUvbXAtcHJvamVjdC10YXNrcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRQcm9qZWN0QWRtaW5Vc2VyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRQcm9qZWN0QWRtaW5Vc2VyJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgYWRtaW5pc3RyYXRvcnMgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcm9qZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBzc28tZGlyZWN0b3J5OkRlc2NyaWJlVXNlcnNcbiAgICogLSBzc286R2V0TWFuYWdlZEFwcGxpY2F0aW9uSW5zdGFuY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL01vbml0cm9uL2xhdGVzdC91c2VyLWd1aWRlL3VzZXItbWFuYWdlbWVudC1jaGFwdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RQcm9qZWN0QWRtaW5Vc2VycygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFByb2plY3RBZG1pblVzZXJzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgdXNlcnMgYXNzb2NpYXRlZCB3aXRoIHRoZSBwcm9qZWN0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBzc286R2V0TWFuYWdlZEFwcGxpY2F0aW9uSW5zdGFuY2VcbiAgICogLSBzc286R2V0UHJvZmlsZVxuICAgKiAtIHNzbzpMaXN0RGlyZWN0b3J5QXNzb2NpYXRpb25zXG4gICAqIC0gc3NvOkxpc3RQcm9maWxlQXNzb2NpYXRpb25zXG4gICAqIC0gc3NvOkxpc3RQcm9maWxlc1xuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vTW9uaXRyb24vbGF0ZXN0L3VzZXItZ3VpZGUvdXNlci1tYW5hZ2VtZW50LWNoYXB0ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFByb2plY3RVc2VyQXNzb2NpYXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0UHJvamVjdFVzZXJBc3NvY2lhdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCBwcm9qZWN0c1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL01vbml0cm9uL2xhdGVzdC91c2VyLWd1aWRlL21wLXByb2plY3QtdGFza3MuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFByb2plY3RzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0UHJvamVjdHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBsaXN0IGFsbCB0YWdzIGZvciBhIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vTW9uaXRyb24vbGF0ZXN0L3VzZXItZ3VpZGUvdGFnZ2luZy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCBhbGwgYWNjZXNzIHJvbGVzIGFzc29jaWF0ZWQgd2l0aCB0aGUgdXNlclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL01vbml0cm9uL2xhdGVzdC91c2VyLWd1aWRlL3VzZXItbWFuYWdlbWVudC1jaGFwdGVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RVc2VyQWNjZXNzUm9sZUFzc29jaWF0aW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdFVzZXJBY2Nlc3NSb2xlQXNzb2NpYXRpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdGFnIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9Nb25pdHJvbi9sYXRlc3QvdXNlci1ndWlkZS90YWdnaW5nLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdUYWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVudGFnIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9Nb25pdHJvbi9sYXRlc3QvdXNlci1ndWlkZS90YWdnaW5nLmh0bWwjbW9kaWZ5LXRhZy0xXG4gICAqL1xuICBwdWJsaWMgdG9VbnRhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVbnRhZ1Jlc291cmNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gdXBkYXRlIGEgcHJvamVjdFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9Nb25pdHJvbi9sYXRlc3QvdXNlci1ndWlkZS9tcC11cGRhdGluZy1wcm9qZWN0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVByb2plY3QoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZVByb2plY3QnKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICAnUGVybWlzc2lvbnMgbWFuYWdlbWVudCc6IFtcbiAgICAgICdBc3NvY2lhdGVQcm9qZWN0QWRtaW5Vc2VyJyxcbiAgICAgICdDcmVhdGVQcm9qZWN0VXNlckFzc29jaWF0aW9uJyxcbiAgICAgICdDcmVhdGVVc2VyQWNjZXNzUm9sZUFzc29jaWF0aW9uJyxcbiAgICAgICdEZWxldGVQcm9qZWN0VXNlckFzc29jaWF0aW9uJyxcbiAgICAgICdEZWxldGVVc2VyQWNjZXNzUm9sZUFzc29jaWF0aW9uJyxcbiAgICAgICdEaXNhc3NvY2lhdGVQcm9qZWN0QWRtaW5Vc2VyJyxcbiAgICAgICdMaXN0UHJvamVjdEFkbWluVXNlcnMnXG4gICAgXSxcbiAgICBXcml0ZTogW1xuICAgICAgJ0NyZWF0ZVByb2plY3QnLFxuICAgICAgJ0RlbGV0ZVByb2plY3QnLFxuICAgICAgJ1VwZGF0ZVByb2plY3QnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnR2V0UHJvamVjdCcsXG4gICAgICAnR2V0UHJvamVjdEFkbWluVXNlcicsXG4gICAgICAnTGlzdFRhZ3NGb3JSZXNvdXJjZSdcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0UHJvamVjdFVzZXJBc3NvY2lhdGlvbnMnLFxuICAgICAgJ0xpc3RQcm9qZWN0cycsXG4gICAgICAnTGlzdFVzZXJBY2Nlc3NSb2xlQXNzb2NpYXRpb25zJ1xuICAgIF0sXG4gICAgVGFnZ2luZzogW1xuICAgICAgJ1RhZ1Jlc291cmNlJyxcbiAgICAgICdVbnRhZ1Jlc291cmNlJ1xuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgcHJvamVjdCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9Nb25pdHJvbi9sYXRlc3QvdXNlci1ndWlkZS9wcm9qZWN0cy1jaGFwdGVyLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHJlc291cmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgcmVzb3VyY2VJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKi9cbiAgcHVibGljIG9uUHJvamVjdChyZXNvdXJjZUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06bW9uaXRyb246JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnByb2plY3QvJHsgcmVzb3VyY2VJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHRhZyBrZXktdmFsdWUgcGFpcnMgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZVByb2plY3QoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1JlcXVlc3RUYWcodGFnS2V5OiBzdHJpbmcsIHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6UmVxdWVzdFRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWdzIGF0dGFjaGVkIHRvIHRoZSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVzb3VyY2V0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byByZXNvdXJjZSB0eXBlczpcbiAgICogLSBwcm9qZWN0XG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXNvdXJjZVRhZyh0YWdLZXk6IHN0cmluZywgdmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpSZXNvdXJjZVRhZy8keyB0YWdLZXkgfWAsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSB0YWcga2V5cyB0aGF0IGFyZSBwYXNzZWQgaW4gdGhlIHJlcXVlc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXRhZ2tleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZVByb2plY3QoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VbnRhZ1Jlc291cmNlKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NUYWdLZXlzKHZhbHVlOiBzdHJpbmcgfCBzdHJpbmdbXSwgb3BlcmF0b3I/OiBPcGVyYXRvciB8IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmlmKGBhd3M6VGFnS2V5c2AsIHZhbHVlLCBvcGVyYXRvciA/PyAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbbW9uaXRyb25dKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2FtYXpvbm1vbml0cm9uLmh0bWwpLlxuICAgKlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJvcHM/OiBpYW0uUG9saWN5U3RhdGVtZW50UHJvcHMpIHtcbiAgICBzdXBlcihwcm9wcyk7XG4gIH1cbn1cbiJdfQ==