"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mq = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [mq](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmq.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mq extends shared_1.PolicyStatement {
    /**
     * Grants permission to create a broker
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:CreateNetworkInterfacePermission
     * - ec2:CreateSecurityGroup
     * - ec2:CreateVpcEndpoint
     * - ec2:DescribeInternetGateways
     * - ec2:DescribeNetworkInterfacePermissions
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSecurityGroups
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcEndpoints
     * - ec2:DescribeVpcs
     * - ec2:ModifyNetworkInterfaceAttribute
     * - iam:CreateServiceLinkedRole
     * - route53:AssociateVPCWithHostedZone
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-brokers.html#rest-api-brokers-methods-post
     */
    toCreateBroker() {
        return this.to('CreateBroker');
    }
    /**
     * Grants permission to create a new configuration for the specified configuration name. Amazon MQ uses the default configuration (the engine type and engine version)
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configurations.html#rest-api-configurations-methods-post
     */
    toCreateConfiguration() {
        return this.to('CreateConfiguration');
    }
    /**
     * Grants permission to create a replica broker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-brokers.html#rest-api-brokers-methods-post
     */
    toCreateReplicaBroker() {
        return this.to('CreateReplicaBroker');
    }
    /**
     * Grants permission to create tags
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-post
     */
    toCreateTags() {
        return this.to('CreateTags');
    }
    /**
     * Grants permission to create an ActiveMQ user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-post
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to delete a broker
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:DeleteNetworkInterface
     * - ec2:DeleteNetworkInterfacePermission
     * - ec2:DeleteVpcEndpoints
     * - ec2:DetachNetworkInterface
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-delete
     */
    toDeleteBroker() {
        return this.to('DeleteBroker');
    }
    /**
     * Grants permission to delete a configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configurations.html#rest-api-configurations-methods-delete
     */
    toDeleteConfiguration() {
        return this.to('DeleteConfiguration');
    }
    /**
     * Grants permission to delete tags
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-delete
     */
    toDeleteTags() {
        return this.to('DeleteTags');
    }
    /**
     * Grants permission to delete an ActiveMQ user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-delete
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to return information about the specified broker
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-get
     */
    toDescribeBroker() {
        return this.to('DescribeBroker');
    }
    /**
     * Grants permission to return information about broker engines
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/broker-engine-types.html#broker-engine-types-http-methods
     */
    toDescribeBrokerEngineTypes() {
        return this.to('DescribeBrokerEngineTypes');
    }
    /**
     * Grants permission to return information about the broker instance options
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/broker-instance-options.html#broker-engine-types-http-methods
     */
    toDescribeBrokerInstanceOptions() {
        return this.to('DescribeBrokerInstanceOptions');
    }
    /**
     * Grants permission to return information about the specified configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration.html#rest-api-configuration-methods-get
     */
    toDescribeConfiguration() {
        return this.to('DescribeConfiguration');
    }
    /**
     * Grants permission to return the specified configuration revision for the specified configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration-revision.html#rest-api-configuration-revision-methods-get
     */
    toDescribeConfigurationRevision() {
        return this.to('DescribeConfigurationRevision');
    }
    /**
     * Grants permission to return information about an ActiveMQ user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-get
     */
    toDescribeUser() {
        return this.to('DescribeUser');
    }
    /**
     * Grants permission to return a list of all brokers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-brokers.html#rest-api-brokers-methods-get
     */
    toListBrokers() {
        return this.to('ListBrokers');
    }
    /**
     * Grants permission to return a list of all existing revisions for the specified configuration
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-revisions.html#rest-api-revisions-methods-get
     */
    toListConfigurationRevisions() {
        return this.to('ListConfigurationRevisions');
    }
    /**
     * Grants permission to return a list of all configurations
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configurations.html#rest-api-configurations-methods-get
     */
    toListConfigurations() {
        return this.to('ListConfigurations');
    }
    /**
     * Grants permission to return a list of tags
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-get
     */
    toListTags() {
        return this.to('ListTags');
    }
    /**
     * Grants permission to return a list of all ActiveMQ users
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-users.html#rest-api-users-methods-get
     */
    toListUsers() {
        return this.to('ListUsers');
    }
    /**
     * Grants permission to promote a broker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-promote.html#rest-api-promote-methods-post
     */
    toPromote() {
        return this.to('Promote');
    }
    /**
     * Grants permission to reboot a broker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-restart.html#rest-api-reboot-methods-post
     */
    toRebootBroker() {
        return this.to('RebootBroker');
    }
    /**
     * Grants permission to add a pending configuration change to a broker
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-get
     */
    toUpdateBroker() {
        return this.to('UpdateBroker');
    }
    /**
     * Grants permission to update RabbitMQ broker authentication and authorization configuration
     *
     * Access Level: Write
     */
    toUpdateBrokerAccessConfiguration() {
        return this.to('UpdateBrokerAccessConfiguration');
    }
    /**
     * Grants permission to update the specified configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration.html#rest-api-configuration-methods-put
     */
    toUpdateConfiguration() {
        return this.to('UpdateConfiguration');
    }
    /**
     * Grants permission to update the information for an ActiveMQ user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-put
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Adds a resource of type brokers to the statement
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/amazon-mq-how-it-works.html
     *
     * @param brokerName - Identifier for the brokerName.
     * @param brokerId - Identifier for the brokerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBrokers(brokerName, brokerId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mq:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:broker:${brokerName}:${brokerId}`);
    }
    /**
     * Adds a resource of type configurations to the statement
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/amazon-mq-how-it-works.html
     *
     * @param configurationId - Identifier for the configurationId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurations(configurationId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:mq:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:configuration:${configurationId}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to actions:
     * - .toCreateBroker()
     * - .toCreateConfiguration()
     * - .toCreateTags()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to resource types:
     * - brokers
     * - configurations
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/developer-guide/security_iam_service-with-iam.html#security_iam_service-with-iam-tags
     *
     * Applies to actions:
     * - .toCreateBroker()
     * - .toCreateConfiguration()
     * - .toCreateTags()
     * - .toDeleteTags()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [mq](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonmq.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'mq';
        this.accessLevelList = {
            Write: [
                'CreateBroker',
                'CreateConfiguration',
                'CreateReplicaBroker',
                'CreateUser',
                'DeleteBroker',
                'DeleteConfiguration',
                'DeleteUser',
                'Promote',
                'RebootBroker',
                'UpdateBroker',
                'UpdateBrokerAccessConfiguration',
                'UpdateConfiguration',
                'UpdateUser'
            ],
            Tagging: [
                'CreateTags',
                'DeleteTags'
            ],
            Read: [
                'DescribeBroker',
                'DescribeBrokerEngineTypes',
                'DescribeBrokerInstanceOptions',
                'DescribeConfiguration',
                'DescribeConfigurationRevision',
                'DescribeUser'
            ],
            List: [
                'ListBrokers',
                'ListConfigurationRevisions',
                'ListConfigurations',
                'ListTags',
                'ListUsers'
            ]
        };
    }
}
exports.Mq = Mq;
//# sourceMappingURL=data:application/json;base64,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